﻿<#
    $ powershell -Version 2.0 -ExecutionPolicy Unrestricted WaitCupAuto.ps1
#>

# main 引数
param (
    # Qualify the platforms
    [string] $Platform = "NX-NXFP2-a64",

    # Qualify the build types
    [string] $BuildType = "Develop",

    # Qualify the application identifer with the cup archives.
    [string] $ApplicationId = "01001a500005e020",

    # Qualify the system update meta identifier.
    [string] $UpdateId = "01001a500005e021",

    # Qualify the system program meta identifier.
    [string] $ProgramId = "01001a500005e022",

    # Qualify the test option.
    [string] $TestOption = "",

    # Qualify the input file path of cup configurations.
    [string] $InputCupConfigFile,

    [string]
    $TargetName,

    [string]
    $TargetAddress
)

# スクリプトパス関連 ( 全部文字列型らしい [string] )
$MyScriptPath          = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($MyScriptPath)

Import-Module "${MyScriptDirectoryPath}\Modules\PlatformUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Error"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Path"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\HostBridge"

Import-Module "${MyScriptDirectoryPath}\Modules\FileUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\SdkToolUtilities"

if ([string]::IsNullOrEmpty($TargetAddress)) {
    if ([string]::IsNullOrEmpty($TargetName)) {
        throw "TargetAddress and TargetName not specified"
    } else {
        $TargetAddress = Get-TargetAddressFromSerial $TargetName
    }
}

# プラットフォームチェック
$Platform = Resolve-PlatformConstants( ${Platform} )

# ControlTarget.exe の検索
[string] $ControlTargetExe = $(Search-AvailableControlTarget ${NintendoSdkRootPath})

# RunOnTarget.exe の検索
[string] $RunOnTargetExe = $(Search-AvailableRunOnTarget ${NintendoSdkRootPath})

# DevMenuCommandSystem.nsp の検索
[string] $DevMenuCommandNsp = $(Search-AvailableDevMenuCommandSystem ${NintendoSdkRootPath} ${Platform} ${BuildType})

# ConfigFile 確認
$(Test-ExistFile ${InputCupConfigFile})

# ログ接頭辞
$LogPrefox = "`r`n[$($MyInvocation.MyCommand.Name)]"

# 定数定義
$IsTestDebug = $FALSE
$CupTimeout = 120

# ================================
# ログサービスのアドレスとポートを見つける
# ================================
function Get-LogService( [string]$executor, [string]$target )
{
    $address = ""
    $port = ""

    # サービス一覧を取得
    Write-Host "list services...  ${target}"
    $result = Invoke-Expression "& `"${executor}`" list-service"
    if ( $LastExitCode -ne 0 ) {
        throw "ControlTarget failed. ExitCode = $LastExitCode"
    }

    # アドレスとポートを検索
    $pattern = "^${target}.*@Log[^\d]*(?<address>[\d\.]+?):(?<port>\d+?)$"
    foreach ($line in $result) {
        Write-Host "${line}"
        if ($line -imatch "${pattern}") {
            $address = $matches["address"]
            $port = $matches["port"]
            break
        }
    }

    if ([string]::IsNullOrEmpty($address) -or [string]::IsNullOrEmpty($address)) {
        throw "Not found target log service."
    }
    Write-Host "address : ${address}"
    Write-Host "port : ${port}"

    return [string[]]("${address}", "${port}")
}

# ================================
# ログに対象の文字列が表示さえるまで待機する
# ================================
function Wait-LogMessage( [string[]]$service, [string]$pattern, [int]$timeout )
{
    $mode = [System.Net.Sockets.SelectMode]::SelectRead
    $ip = [System.Net.IPAddress]::Parse($service[0])
    $port = [int]$service[1]

    Write-Host "connect ${ip}:${port}"
    $socket = New-Object System.Net.Sockets.TcpClient
    if ($socket.Connect($ip, $port) -eq $FALSE) {
        throw "Connection failed."
    }

    $stream = $socket.GetStream()
    $stream.ReadTimeout = $timeout * 1000
    $reader = New-Object System.IO.StreamReader($stream)

    $match = [string]::IsNullOrEmpty($pattern) -eq $FALSE
    $found = $FALSE
    $begin = (Get-Date)
    while (1) {

        # 読み込み
        $line = $reader.ReadLine()
        if ([string]::IsNullOrEmpty($line) -eq $FALSE) {
            Write-Host $line
        }

        # メッセージのマッチング
        if ($match -and $line -imatch "${pattern}") {
            $found = $TRUE
            break
        }

        # タイムアウト確認
        $elapsed = [int](New-TimeSpan $begin(Get-Date)).TotalSeconds
        if ($elapsed -gt $timeout) {
            Write-Host "Timeout..."
            break
        }

        # 接続確認
        if (($socket.Client.Poll(1000, $mode)) -and ($socket.Client.Available) -eq 0) {
            Write-Host "Disconnected..."
            break
        }
    }
    $reader.Close()
    $stream.Close()
    $socket.Close()

    return $found
}

# ================================
# ExitCode 宣言
# ================================
[int] $defExitCode = 0

# ================================
# main
# ================================
try {

    # 例外の英語化（※CI での例外発生解析のため）
    #[System.Threading.Thread]::CurrentThread.CurrentCulture = New-Object System.Globalization.CultureInfo("en-US")
    #[System.Threading.Thread]::CurrentThread.CurrentUICulture = New-Object System.Globalization.CultureInfo("en-US")

    # CIジョブでのTestRunnerは、nact経由でステップ実行し、nactはUTF8で出力を受信している。
    # そのため、強制的にコンソールをUTF-8としている。
    # 但し通常のコード932のDOS窓上では本スクリプトから、Write-Hostなどに日本語指定が使えないので注意。
    $defUseEncoding = New-Object System.Text.UTF8Encoding($False)
    $defBackupConsoleEncoding = [System.Console]::OutputEncoding
    $defBackupEncode = $OutputEncoding
    [System.Console]::OutputEncoding = ${defUseEncoding}
    $OutputEncoding = ${defUseEncoding}

    # エンコード情報表示
    [string] $defConsoleEncode = [System.Console]::OutputEncoding.EncodingName
    [string] $defStartupEncode = $defBackupEncode.EncodingName
    [string] $defCurrentEncode = $OutputEncoding.EncodingName
    Write-Host "[Console]::OutputEncoding : ${defConsoleEncode}"
    Write-Host "Startup `$OutputEncoding   : ${defStartupEncode}"
    Write-Host "Current `$OutputEncoding   : ${defCurrentEncode}"

    Write-Host "============== Script configuration. =============="
    Write-Host "Platform        : ${Platform}"
    Write-Host "BuildType       : ${BuildType}"
    Write-Host "SDK Root        : ${NintendoSdkRootPath}"
    Write-Host "ScriptPath      : ${MyScriptPath}"
    Write-Host "ScriptDirectory : ${MyScriptDirectoryPath}"

    # ゲームカード書き込み
    Write-Host "$LogPrefox write gamecard..."
    $configs = @{}
    Import-Csv -Path ${InputCupConfigFile} -Header "Key","Value" -Delimiter "=" | % {
        $configs.Add( $_.Key.Trim(), $_.Value.Trim() )
    }
    Invoke-Expression "& `"${RunOnTargetExe}`" `"--target`" `"${TargetAddress}`" `"${DevMenuCommandNsp}`" `"--`" `"gamecard`" `"write`" `"$($configs.AppNsp)`" `"--update-partition`" `"$($configs.CupNsp)`" `"--auto`""
    if ( $LastExitCode -ne 0 ) {
        throw "ControlTarget failed. ExitCode = $LastExitCode"
    }

    # CUP 適用前の再起動
    Write-Host "$LogPrefox reset and connect..."
    Invoke-Expression "& `"${ControlTargetExe}`" connect --reset -t ${TargetAddress}"
    if ( $LastExitCode -ne 0 ) {
        throw "ControlTarget failed. ExitCode = $LastExitCode"
    }

    if ($TestOption -eq "RestartDuringUpdate") {

        # ログ待機（推定：CUP適用中）
        Write-Host "$LogPrefox wait for system update progress..."
        $service = $(Get-LogService ${ControlTargetExe} ${TargetName})
        $result = $(Wait-LogMessage $service "\[GameCardManager\] SystemUpdateProgress" $CupTimeout)
        if ( $result -eq $FALSE ) {
            throw "Cup ditn't started."
        }

        # CUP 適用中の再起動
        Write-Host "$LogPrefox reset and connect..."
        Invoke-Expression "& `"${ControlTargetExe}`" connect --reset -t ${TargetAddress}"
        if ( $LastExitCode -ne 0 ) {
            throw "ControlTarget failed. ExitCode = $LastExitCode"
        }
    }

    # ログ待機（推定：CUP適用）
    Write-Host "$LogPrefox wait for system update closing..."
    $service = $(Get-LogService ${ControlTargetExe} ${TargetName})
    $result = $(Wait-LogMessage $service "\[SystemUpdateInterfaceServer\] Close system update control" $CupTimeout)
    if ( $result -eq $FALSE ) {
        throw "Cup ditn't started."
    }
    Start-Sleep -s 10

    # CUP 適用直後に再起動されるので再接続
    Write-Host "$LogPrefox connect after restart..."
    Invoke-Expression "& `"${ControlTargetExe}`" connect -t ${TargetAddress}"

    # ログ待機（推定：アプリ起動）
    Write-Host "$LogPrefox wait for application start..."
    $service = $(Get-LogService ${ControlTargetExe} ${TargetName})
    $result = $(Wait-LogMessage $service "ApplicationID  : 0x${ApplicationId}" $CupTimeout)
    if ( $result -eq $FALSE ) {
        throw "Application ditn't started."
    }

    # CUP 適用後の再起動
    Write-Host "$LogPrefox reset and connect..."
    $result = Invoke-Expression "& `"${ControlTargetExe}`" connect --reset -t ${TargetAddress}"
    if ( $LastExitCode -ne 0 ) {
        throw "ControlTarget failed. ExitCode = $LastExitCode"
    }

    # ログ待機（推定：アプリ起動 ※CUP適用済なので途中に再起動による切断はない）
    Write-Host "$LogPrefox wait for application start..."
    $service = $(Get-LogService ${ControlTargetExe} ${TargetName})
    $result = $(Wait-LogMessage $service "ApplicationID  : 0x${ApplicationId}" $CupTimeout)

} catch [Exception] {

    Write-ErrorRecord $_
    $defExitCode = 1

    # 例外の詳細（※CI での例外発生解析のため）
    Write-Host "$LogPrefox error detail"
    $_ | Format-List -force

    # 接続不良の状況を想定してリセット＆再接続
    Write-Host "$LogPrefox reset and wating..."
    Invoke-Expression "& `"${ControlTargetExe}`" connect --reset -t ${TargetAddress}"
    Start-Sleep -s $CupTimeout

    # 上記の再起動中に CUP されて切断されたことを考慮して再接続
    Invoke-Expression "& `"${ControlTargetExe}`" connect -t ${TargetAddress}"

} finally {

    # 自動起動のまま残ると他のテストに悪影響をもたらすので必ず削除
    Write-Host "$LogPrefox erase gamecard..."
    Invoke-Expression "& `"${RunOnTargetExe}`" `"--target`" `"${TargetAddress}`" `"${DevMenuCommandNsp}`" `"--`" `"gamecard`" `"erase`""

    if ( $IsTestDebug ) {

        # このスクリプトデバッグ用にインストールしたコンテンツを削除
        Write-Host "$LogPrefox erase system update..."
        Invoke-Expression "& `"${RunOnTargetExe}`" `"--target`" `"${TargetAddress}`" `"${DevMenuCommandNsp}`" `"--`" `"systemprogram`" `"uninstall`" `"0x${UpdateId}`""
        Invoke-Expression "& `"${RunOnTargetExe}`" `"--target`" `"${TargetAddress}`" `"${DevMenuCommandNsp}`" `"--`" `"systemprogram`" `"uninstall`" `"0x${ProgramId}`""
    }

    Write-Host "$LogPrefox reset and connect..."
    Invoke-Expression "& `"${ControlTargetExe}`" connect --reset -t ${TargetAddress}"

    $OutputEncoding = $defBackupEncode
    [System.Console]::OutputEncoding = $defBackupConsoleEncoding
}
exit ${defExitCode}
