﻿<#
    $ powershell -Version 2.0 -ExecutionPolicy Unrestricted SetupWetherDelivery.ps1
#>

# main 引数
param (
    # Qualify the platforms
    [string] $Platform = "NX-NXFP2-a64",

    # Qualify the build types
    [string] $BuildType = "Develop",

    # Qualify the system update identifier specified.
    [string] $TitleId = "01001a500005e080",

    # Qualify the system update version specified.
    [string] $TitleVersion = "0",

    # Qualify the configuration of delivery id on weather server api.
    [string] $DeliveryId = "18",

    # Qualify the proxy configuration
    [string] $Proxy = "http://proxy.nintendo.co.jp:8080",

    # Qualify the input file path
    [string] $InputDeviceIdFile
)

# スクリプトパス関連 ( 全部文字列型らしい [string] )
$MyScriptPath           = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath  = [System.IO.Path]::GetDirectoryName($MyScriptPath)

Import-Module "${MyScriptDirectoryPath}\Modules\PlatformUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\SdkToolUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\OptionUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Error"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Path"

# -----------------------------------------------------------------------
# HTTP response check.
# -----------------------------------------------------------------------
function Test-HttpResponse( [string]$defResponseStream )
{
    # tee-object で変数格納された response は改行なしで渡される事も想定.
    $defMatchResult = [regex]::Match( ${defResponseStream}, "HTTP\/[0-9]\.[0-9] 200" )
    if ( $defMatchResult.Success -eq $False )
    {
        throw "Failed http request."
    }
}

# ================================
# main
# ================================
try {
    # CIジョブでのTestRunnerは、nact経由でステップ実行し、nactはUTF8で出力を受信している。
    # そのため、強制的にコンソールをUTF-8としている。
    # 但し通常のコード932のDOS窓上では本スクリプトから、Write-Hostなどに日本語指定が使えないので注意。
    $defUseEncoding = New-Object System.Text.UTF8Encoding($False)
    $defBackupConsoleEncoding = [System.Console]::OutputEncoding
    $defBackupEncode = $OutputEncoding
    [System.Console]::OutputEncoding = ${defUseEncoding}
    $OutputEncoding = ${defUseEncoding}

    # エンコード情報表示
    [string] $defConsoleEncode = [System.Console]::OutputEncoding.EncodingName
    [string] $defStartupEncode = $defBackupEncode.EncodingName
    [string] $defCurrentEncode = $OutputEncoding.EncodingName
    Write-Host "[Console]::OutputEncoding : ${defConsoleEncode}"
    Write-Host "Startup `$OutputEncoding   : ${defStartupEncode}"
    Write-Host "Current `$OutputEncoding   : ${defCurrentEncode}"

    # プラットフォームチェック
    $Platform = Resolve-PlatformConstants( ${Platform} )

    # DeviceID 出力ファイル存在確認
    if ( -not $(Test-Path ${InputDeviceIdFile}) )
    {
        throw "Could not found the file with the device identifier string.`n >> ${InputDeviceIdFile}"
    }

    Write-Host "Platform        : ${Platform}"
    Write-Host "BuildType       : ${BuildType}"
    Write-Host "SDK Root        : ${NintendoSdkRootPath}"
    Write-Host "ScriptPath      : ${MyScriptPath}"
    Write-Host "ScriptDirectory : ${MyScriptDirectoryPath}"

    # cURL.exe のパス取得
    [string] $defCurlExec = $(Get-CurlExec)

    # Device ID 読み込み
    # 有効行が2以上ならSystem.Array、1行なら String が返される
    $fileStream = Get-Content ${InputDeviceIdFile} -Encoding UTF8

    # 1行目だけ取る
    $defExpectDeviceIdString = "none"
    foreach( $line in $fileStream )
    {
        $defExpectDeviceIdString = $line
        break
    }
    # 1行目に有効なデバイス識別子があるかフィルタリング
    $matchResult = [regex]::Match(${defExpectDeviceIdString}, "[A-Fa-f0-9]{16}")
    if ( $false -eq $matchResult.Success )
    {
        throw "Failed query the device identifier : ${defExpectDeviceIdString}."
    }

    # 有効なデバイスIDが見つかった
    $defDeviceIdentifier = $matchResult.Value
    Write-Host "Available device identifier : ${defDeviceIdentifier}"

    # プロキシ設定
    [string] $defProxy = $(New-ProxyOption ${Proxy})

    # 配信サーバーAPI ( td1/dev6 )
    [string] $defEnv = "dev6"
    [string] $defRequestUrl = "https://weather.wc.${defEnv}.d4c.nintendo.net/v1/delivery_configurations/${DeliveryId}"

    # CLIトークン取得
    [string] $defTokenJson = $(Get-CliBasicTokenAsJson ${defProxy} ${defEnv})
    [string] $defAccessToken = $(Export-CliTokenFromJson ${defTokenJson} "access_token")
    [string] $defAuthorization = "Authorization: Bearer ${defAccessToken}"

    # 配信設定リクエスト
    [string] $defRequestCommand = "& ${defCurlExec} ${defProxy} --dump-header - -k -H `"${defAuthorization}`" -H `"Content-type: application/json`" -X PUT -d `'{ \`"device_ids\`" : [ \`"${defDeviceIdentifier}\`" ], \`"title_id\`" : \`"${TitleId}\`", \`"title_version\`" : \`"${TitleVersion}\`" }`' `"${defRequestUrl}`""
    Write-Host "Request command : ${defRequestCommand}"
    ( [string]$defDeliverResponse = Invoke-Expression ${defRequestCommand} )
    $(Test-HttpResponse ${defDeliverResponse})

    # 承認リクエスト
    [string] $defApproveCommand = "& ${defCurlExec} ${defProxy} --dump-header - -k -H `"${defAuthorization}`" -X PUT `"${defRequestUrl}/approve`""
    Write-Host "Approve command : ${defApproveCommand}"
    ( [string]$defApproveResponse = Invoke-Expression ${defApproveCommand} )
    $(Test-HttpResponse ${defApproveResponse})

} catch [Exception] {

    Write-ErrorRecord $_
    $LastExitCode = 1

} finally {

    $OutputEncoding = $defBackupEncode
    [System.Console]::OutputEncoding = $defBackupConsoleEncoding

}
exit $LastExitCode
