﻿<#
    $ powershell -Version 2.0 -ExecutionPolicy Unrestricted MakeVerifyProperties.ps1
#>

# main 引数
param (
    # Qualify the intermediate working folder path
    [string] $IntermediateDirectory,

    # Qualify the platforms
    [string] $Platform = "NX-NXFP2-a64",

    # 出力ファイル名
    [string] $OutputCollectionFileName = "ExpectContents.xml",

    # 入力本体更新メタファイルパス
    [string] $SourceSystemUpdateMetaFile = "",

    # 入力コンテンツディレクトリ名
    [string] $SourceNspDirectoryName = "",

    # Qualify the content meta keys.
    # メタキー表現書式 = "${ID}:${VERSION}:${TYPE}"
    # 複数のメタキー区切りはスラッシュ"/"で行います.
    # "${ID}:${VERSION}:${TYPE}/${ID}:${VERSION}:${TYPE}"
    [string] $ContentMetaKeys = ""
)

# スクリプトパス関連 ( 全部文字列型らしい [string] )
$MyScriptPath           = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath  = [System.IO.Path]::GetDirectoryName($MyScriptPath)

Import-Module "${MyScriptDirectoryPath}\Modules\PlatformUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Error"
Import-Module "${MyScriptDirectoryPath}\Modules\FileUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\SdkToolUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\XmlFileUtilities"

# -----------------------------------------------------------------------
# Check source directory for nsp files.
# -----------------------------------------------------------------------
function Test-SourceNspDirectory( [string]$defIntermediateDirectory, [string]$defSourceNspDirectoryName )
{
    [string] $defSourceNspPath = ${defIntermediateDirectory}
    if ( $defSourceNspDirectoryName.Length -gt 0 )
    {
        [string] $defExpectSourcePath = "${defIntermediateDirectory}/${defSourceNspDirectoryName}"
        if ( $(Test-Path ${defExpectSourcePath}) )
        {
            [string] $defSourceNspPath = ${defExpectSourcePath}
        }
    }
    return ${defSourceNspPath}
}

# -----------------------------------------------------------------------
# Make expect contents xml.
# -----------------------------------------------------------------------
function Out-MetaKeyCollectionWithContent( [ref]$refMetaKeyDoc, [string]$defIntermediateDirectory, [string]$defSourceNspDirectoryName, [string] $defOutFileName, [string]$defArchSignature )
{
    # 出力XMLファイルの事前削除
    [string] $defOutCollectionFilePath = "${defIntermediateDirectory}\${defOutFileName}"
    $(Remove-ExistFile ${defOutCollectionFilePath})

    # MetaKeyCollection の各 <MetaKey> 要素に合致する <Content>を検出。
    [xml] $defXmlDoc = $refMetaKeyDoc.Value
    [array] $defMetaKeyNodes = $defXmlDoc.MetaKeyCollection.MetaKey
    ${defMetaKeyNodes} | % {
        [System.Xml.XmlElement] $defMetaKeyNode = $_
        [string] $defMetaId = $defMetaKeyNode.id
        [string] $defMetaType = $defMetaKeyNode.type
        [string] $defMetaVersion = $defMetaKeyNode.version
        [string] $defArch = ${defArchSignature}
        if ( ${defMetaType} -ne "SystemProgram" )
        {
            [string] $defArch = ""
        }
        Write-Host ""
        Write-Host "Try extract the [${defMetaId}:${defMetaVersion}] as ${defMetaType}."
        [string] $defSourceNspDirectory = $(Test-SourceNspDirectory ${defIntermediateDirectory} ${defSourceNspDirectoryName})
        [string] $defOutPath = $(Expand-ContentNsp ${defSourceNspDirectory} ${defMetaId} ${defMetaVersion} ${defArch})
        Write-Host "Extract to ${defOutPath}"

        # 展開先の*.cnmt.xml から、type, id が合致するものを検出。
        Get-ChildItem ${defOutPath} -include "*.cnmt.xml" -recurse -force | % {
            [System.IO.FileInfo] $defCnmtFilePath = $_
            $defXmlCnmt = [xml] $(Get-Content ${defCnmtFilePath})
            if (( $defXmlCnmt.ContentMeta.Id -eq "0x${defMetaId}" ) -and
                ( $defXmlCnmt.ContentMeta.Type -eq ${defMetaType} ) -and
                ( $defXmlCnmt.ContentMeta.Version -eq ${defMetaVersion} ))
            {
                Write-Host "Has content in [${defMetaId}:${defMetaVersion}:${defMetaType}] = {"

                # 検出した cnmt.xml から<Content>を抽出。
                [array] $defXmlCnmtContentMetaNodes = $defXmlCnmt.ContentMeta.Content
                ${defXmlCnmtContentMetaNodes} | % {
                    [System.Xml.XmlElement] $defXmlCnmtContentMetaNode = $_
                    [string] $defContentId = $defXmlCnmtContentMetaNode.id
                    [string] $defContentType = $defXmlCnmtContentMetaNode.type
                    $defContentElement = $defXmlDoc.CreateElement( "Content" )
                    $defContentElement.SetAttribute( "id", ${defContentId} ) | Out-Null
                    $defContentElement.SetAttribute( "type", ${defContentType} ) | Out-Null
                    $defMetaKeyNode.AppendChild( ${defContentElement} ) | Out-Null
                    Write-Host "    <Id=`"${defContentId}`", Type=`"${defContentType}`">"
                } | Out-Null

                Write-Host "}"
            }
        } | Out-Null
    }

    # 出力
    $(Out-WriteXml ${refMetaKeyDoc} ${defOutCollectionFilePath})
}



# ================================
# main
# ================================
try {
    # CIジョブでのTestRunnerは、nact経由でステップ実行し、nactはUTF8で出力を受信している。
    # そのため、強制的にコンソールをUTF-8としている。
    # 但し通常のコード932のDOS窓上では本スクリプトから、Write-Hostなどに日本語指定が使えないので注意。
    $defUseEncoding = New-Object System.Text.UTF8Encoding($False)
    $defBackupConsoleEncoding = [System.Console]::OutputEncoding
    $defBackupEncode = $OutputEncoding
    [System.Console]::OutputEncoding = ${defUseEncoding}
    $OutputEncoding = ${defUseEncoding}

    [string] $defConsoleEncode = [System.Console]::OutputEncoding.EncodingName
    [string] $defStartupEncode = $defBackupEncode.EncodingName
    [string] $defCurrentEncode = $OutputEncoding.EncodingName
    Write-Host "[Console]::OutputEncoding : ${defConsoleEncode}"
    Write-Host "Startup `$OutputEncoding   : ${defStartupEncode}"
    Write-Host "Current `$OutputEncoding   : ${defCurrentEncode}"

    # 中間ファイル作成作業フォルダ存在確認
    [string] $IntermediateDirectory = $(Get-AbsoluteFullPath ${IntermediateDirectory})
    $(Edit-MakeDirectory ${IntermediateDirectory})

    Write-Host "============== Script configuration. =============="
    Write-Host "Platform        : ${Platform}"
    Write-Host "SDK Root        : ${NintendoSdkRootPath}"
    Write-Host "ScriptPath      : ${MyScriptPath}"
    Write-Host "ScriptDirectory : ${MyScriptDirectoryPath}"
    Write-Host "OutputFile      : ${OutputCollectionFileName}"

    # 64/32 アーキテクトシグネチャ取得
    [string] $ArchSignature = $(Get-ArchitectSignature ${Platform})
    Write-Host "Architexture    : ${ArchSignature}"

    if ( $ContentMetaKeys.Length -gt 0 )
    {
        # Option of "-ContentMetaKeys"
        [string[]] $defArrayMetaKeys = ${ContentMetaKeys} -csplit "/"
        if ( $defArrayMetaKeys.Length -gt 0 )
        {
            Write-Host "Expect meta key = {"
            ${defArrayMetaKeys} | % {
                Write-Host "    [$_],"
            }
            Write-Host "}"

            # XMLオブジェクト作成
            [xml] $defXmlDoc = $(New-MakeMetaKeyCollectionXml ${defArrayMetaKeys})
            [ref] $refXmlDoc = ${defXmlDoc}
            $(Out-MetaKeyCollectionWithContent ${refXmlDoc} ${IntermediateDirectory} ${SourceNspDirectoryName} ${OutputCollectionFileName} ${ArchSignature})
        }
        else
        {
            throw "Did not has specified the meta key collection."
        }
    }
    elseif ( $SourceSystemUpdateMetaFile.Length -gt 0 )
    {
        # Option of "-SourceSystemUpdateMetaFile"
        [string] $SourceSystemUpdateMetaFile = $(Get-AbsoluteFullPath ${SourceSystemUpdateMetaFile})
        Write-Host "Expect meta key files = ${SourceSystemUpdateMetaFile}"
        [xml] $defXmlDoc = $(New-MakeMetaKeyCollectionXmlFromMetaXml ${SourceSystemUpdateMetaFile})
        [ref] $refXmlDoc = ${defXmlDoc}
        $(Out-MetaKeyCollectionWithContent ${refXmlDoc} ${IntermediateDirectory} ${SourceNspDirectoryName} ${OutputCollectionFileName} ${ArchSignature})
    }
    else
    {
        throw "There not has specified the source meta key collection."
    }

} catch [Exception] {

    Write-ErrorRecord $_
    $LastExitCode = 1

} finally {

    $OutputEncoding = $defBackupEncode
    [System.Console]::OutputEncoding = $defBackupConsoleEncoding

}
exit $LastExitCode
