﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CsTestAssistants;
using System.Reflection;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace testNs_ShopDeliveryBasic
{

    /// <summary>
    /// 仕様変更などにより検証項目として不要になった過去テストの集積所です。
    /// 過去経緯などの履歴として残します。
    /// </summary>
    class DisabledTestMethods
    {

#if false
        /// SIGLO-63868, SIGLO-65171 にて本テストの検証対象機能「NUP通知が検出された状態では自動更新が走らない」が却下されるため、
        /// 本テストは無効化します。
        /// 実装状態での呼び出し箇所は以下。
        /// TestForVerifyConsistencyOfStorageCapacity
        /// TestForVerifyToExpectAutoUpdateBeSuspendedByNupDetection

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 「必須システムバージョンが現バージョンよりも高いパッチは自動更新の対象外になる」検証。
        /// 厳密には 「NUP通知が検出された状態では自動更新が走らない」検証。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty( "JIRA", "SIGLO-61975" )]
        public void TestForVerifyToExpectAutoUpdateBeSuspendedByNupDetection()
        {
            using ( var scope = new TestMethodLog() )
            {
                var executor = new SigloHelper.CommodityExecutor.Context( ActiveConfiguration );
                var intermediate = GenerateIntermediateDirectoryAsMethod( executor );
                var method = MethodBase.GetCurrentMethod();

                // 本テストのメタ構成は以下の通り。
                // Application : 0x01003ab001e30048, version 0
                // Patch       : 0x01003ab001e30848, version 65536
                var IdForApplication = new ID64( 0x01003ab001e30048 );

                // make contents.
                var genApp = new TestApplication.ApplicationParameter( IdForApplication, 0, 1 * 1024 * 1024 );
                genApp.UseSmallCode = true;
                genApp.AddPatch( 1 );

                // アップロード
                D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment
                );

#if true    // コンテンツの生成及び、アップロードを行う場合はこちらのコードを有効にする。

                var uploadContents = TestApplication.MakeContents( intermediate, genApp );
                var app = uploadContents[ 0 ];
                var patch = uploadContents[ 1 ];

                uploader.RejectRoms( CliTokenAccount, IdForApplication );
                uploader.Publish( CliTokenAccount, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 5 * 60 );
                uploader.Publish( CliTokenAccount, patch.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, 5 * 60 );

#else   // 既にアップロード済でアップロード時に生成したnspファイルがOutput/に残されていた場合アップロードスキップで利用可能。

                var uploadContents = TestApplication.MakeFromExistsContents( new AuthoringExecutor(), intermediate,
                    new List<TestApplication.GenerateParameter<int>>( 1 ) { genApp }
                );
                var app = uploadContents[ 0 ];
                var patch = uploadContents[ 1 ];

#endif

                /// --------------------------------------------------------------------------
                /// Common prepared.
                /// --------------------------------------------------------------------------
                // クライアントの SystemUpdateMeta の version 取得
                var nupId = new ID64( 0x0100000000000816 );
                ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"systemprogram list-details 0x{nupId}" ) );
                var lMatchedClientSystemUpdate = Regex.Match( executor.OutputStream.Standard.ToString(), $@"\[target\] 0x{nupId}, ver\.[0-9]+" );
                var lClientVersion = uint.Parse( Regex.Match( lMatchedClientSystemUpdate.Value, "[0-9]+$" ).Value );
                Log.WriteLine( $"Detected client systemupdate meta version : {nupId}-{lClientVersion}" );
                var lLatestNupVersion = lClientVersion + 1;

                // npns 通知プロキシ生成
                var notifier = new NintendoServices.Npns.Notifier( "c17e59e1df28556e62b837d5a36e87bc", ProxyConfiguration );

                // "DevMenuCommandSystem.nsp npns create-token-test --id 0x010000000000001f" を使って、通知トークンを発行。
                // 0x010000000000001f は通知受信リスナが存在する。
                var notifyId = new ID64( 0x010000000000001f );
                ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"npns create-token-test --id 0x{notifyId}" ) );
                var lMatched = Regex.Match( executor.OutputStream.Standard.ToString(), @"Test token created:\s*[A-Za-z0-9]+" );
                var tokenForNotify = Regex.Match( lMatched.Value, "[A-Za-z0-9]+$" ).Value;
                ExpectEvaluator.IsFalse( string.IsNullOrEmpty( tokenForNotify ) );
                Log.WriteLine( "Detected notification token : {0}", tokenForNotify );

                // npns 受信モニタ用 DevMenuCommandSystem 起動待ちインターバル ( ミリ秒 )
                var waitIntervalForMonitorLaunch = 30 * 1000;
                // npns 受信待ちインターバル ( ミリ秒 )
                var waitIntervalForVersionListUpdate = ( 5 * 60 * 1000 ) + waitIntervalForMonitorLaunch;
                var waitIntervalForVersionListUpdateSecond = ( uint )waitIntervalForVersionListUpdate / 1000;

                // common command parameters.
                var comRecord = $"application list-record 0x{IdForApplication} --installed";
                var matchStorageAny = SigloHelper.ResultMatcher.ContentValue.Storage.Any;

                /// --------------------------------------------------------------------------
                /// finally
                /// --------------------------------------------------------------------------
                scope.AddDisposer( () =>
                {
                    uploader.DeleteVersion( patch.NspPath );
                    ExpectEvaluator.IsTrue( D4cHelper.WaitUntilVersionListDeleted( executor, patch, true ) );
                    executor.RunDevMenuCommandSystem( "application clear-task-status-list AND application uninstall --all" );
                } );

                /// --------------------------------------------------------------------------
                /// 自動更新がNUP更新要求で中断されたかを検出。
                /// 以下ログ検知で判断します。
                /// [ApplicationInstallRequestList] NeedsProcessVersionListRequest
                /// [ApplicationInstallRequestList]   Requested        : true
                /// [ApplicationInstallRequestList]   SystemUpdateEvent: true
                /// [ApplicationInstallRequestList]   => NeedsProcess  : false
                /// --------------------------------------------------------------------------
                System.Func<string, bool> HasSuspendAutoUpdateByNup = ( lStream ) =>
                {
                    var expectMatchPattern = @"\[ApplicationInstallRequestList\] NeedsProcessVersionListRequest[^\a]*" +
                        @"\[ApplicationInstallRequestList\]   Requested        : true[^\a]*" +
                        @"\[ApplicationInstallRequestList\]   SystemUpdateEvent: true[^\a]*" +
                        @"\[ApplicationInstallRequestList\]   => NeedsProcess  : false";
                    var match = Regex.Match( lStream, expectMatchPattern );
                    if ( false == match.Success )
                    {
                        Log.WriteLine( $"Not found an expected result => {expectMatchPattern}." );
                    }
                    return match.Success;
                };

                /// --------------------------------------------------------------------------
                /// 自動更新の中断検証メソッド
                /// --------------------------------------------------------------------------
                System.Func<int> SuspendAutoUpdateByNupVerifier = () =>
                {
                    // superfly → tagaya への反映待ちをダウンロード待ちで実装。( td1 は1分更新 )
                    // application version-list は更新対象にならなければリストアップされないため、NUP更新検知時は使えない。
                    if ( false == executor.RunDevMenuCommandSystemCustomOption( $"--time application wait-download 0x{app.Identifier} " +
                        $"--version {patch.Version} --timeout {waitIntervalForVersionListUpdate}",
                        new List<string>( 1 )
                        {
                            @"--pattern-success-exit ""\[NotificationTaskManager\] Process version list notification\."""
                        },
                        waitIntervalForVersionListUpdateSecond
                    ) )
                    {
                        // timeout した場合にはリトライしない.
                        Log.WriteLine( "SuspendAutoUpdateByNupVerifier: Notification receiving wait timeout." );
                        return -1;  // critical failure for verification.
                    }
                    // NUPにより自動更新が中断したか確認。
                    if ( false == HasSuspendAutoUpdateByNup( executor.OutputStream.Standard.ToString() ) )
                    {
                        return 1;   // retry trigger.
                    }
                    // patch ( version 1 ) が見つからない事を確認。
                    if ( false == executor.RunDevMenuCommandSystem( comRecord ) ||
                        1 != SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find( executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue( patch, matchStorageAny ) ) )
                    {
                        // 期待しない結果になってるのでリトライしない.
                        Log.WriteLine( $"SuspendAutoUpdateByNupVerifier: Unexpect result, Found the content => [ {patch} ]." );
                        return -1;  // critical failure for verification.
                    }
                    return 0;   // expect success.
                };

                /// --------------------------------------------------------------------------
                /// ケース 1
                /// - NUP通知検知状態シーンでの自動更新は実施されない ( npns 経由 )
                /// - NUPあり状態の再起動での自動更新は実施されない ( set-debug-id 経由 )
                /// - 再起動後に自動更新が行われる
                /// --------------------------------------------------------------------------
                scope.WriteLineWithTimeStamp( "Case1: Verification to expect auto update be suspended by NUP detection via npns." );
                {
                    // ターゲットデバイスのバージョンリストを削除する。
                    uploader.DeleteVersion( patch.NspPath );
                    ExpectEvaluator.IsTrue( D4cHelper.WaitUntilVersionListDeleted( executor, patch, true ) );

                    // appv0 をインストールして起動( 自動更新対象化 )する
                    ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"application clear-task-status-list AND application uninstall --all" ) );
                    ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"application install {app.NspPath} AND application launch 0x{app.Identifier}" ) );

                    // RunOnTarget での受信待ち中と並列通知させるためにタスク化
                    System.Threading.Tasks.Task.Run( () =>
                    {
                        // 受信待機アプリ( wait-download )起動待ち。
                        System.Threading.Thread.Sleep( waitIntervalForMonitorLaunch );
                        try
                        {
                            // NUP 通知を行う。
                            var webResult = notifier.RequestExecution( tokenForNotify, new NintendoServices.Npns.Notifier.Nup( nupId, lLatestNupVersion, nupId, 0 ) );
                            ExpectEvaluator.IsTrue( null != webResult && System.Net.HttpStatusCode.OK == webResult.StatusCode );
                            // ターゲットデバイスのバージョンリストに登録する。
                            uploader.RegisterVersion( patch.NspPath );
                        }
                        catch ( System.Exception e )
                        {
                            Log.WriteLine( $"Detected exception on sub thread => {e.ToString()}" );
                        }
                    } );

                    // VersionList / NUP 通知待ち
                    var retry = 3;
                    var result = -1;
                    while ( retry > 0 && ( result = SuspendAutoUpdateByNupVerifier() ) > 0 )
                    {
                        --retry;
                        // 正規経路でのVersionList通知で検出できなかった場合、手動でバージョンリスト更新通知を行う。
                        // 自身のバージョン更新要求以外でのVersionList通知などで受信待ちを途中退場した場合などを想定しています。
                        var webResult = notifier.RequestExecution( tokenForNotify, new NintendoServices.Npns.Notifier.VersionList( "0123456789abcdef0123456789abcdef" ) );
                        ExpectEvaluator.IsTrue( null != webResult && System.Net.HttpStatusCode.OK == webResult.StatusCode );
                    }
                    Log.WriteLine( $"SuspendAutoUpdateByNupVerifier [ result={result}, remain retry count={retry} ]" );
                    ExpectEvaluator.IsEqual( 0, result );

                    // 再起動時のNUP検出設定と、npns経由NUP受信ハンドラを無効にする
                    ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem(
                        $"debug set-boolean-fwdbg --name \"ns.notification\" --key \"enable_network_update\" false AND " +
                        $"systemupdate set-debug-id 0x{nupId}-{lLatestNupVersion}"
                    ) );

                    // 再起動で自動更新が走らない事を確認。
                    ExpectEvaluator.IsTrue( executor.ResetTarget() );
                    ExpectEvaluator.IsEqual( 0, SuspendAutoUpdateByNupVerifier() );

                    // NUP受信ハンドラ、NUP検出設定を元に戻す。
                    ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem(
                        $"debug set-boolean-fwdbg --name \"ns.notification\" --key \"enable_network_update\" true AND " +
                        $"systemupdate set-debug-id 0x0000000000000000-0"
                    ) );

                    // 再起動すると自動更新が走ることを確認。
                    ExpectEvaluator.IsTrue( executor.ResetTarget() );
                    ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"application wait-download 0x{app.Identifier} " +
                        $"--version {patch.Version} --timeout {waitIntervalForVersionListUpdate} AND " +
                        $"application verify 0x{app.Identifier}"
                    ) );
                }

                /// --------------------------------------------------------------------------
                /// ケース 2
                /// - NUP通知未検地状態シーンでの自動更新は実施される ( 通常の自動更新 )
                /// --------------------------------------------------------------------------
                scope.WriteLineWithTimeStamp( "Case2: Verification to expect auto update be normally." );
                {
                    // ターゲットデバイスのバージョンリストを削除する。
                    uploader.DeleteVersion( patch.NspPath );
                    ExpectEvaluator.IsTrue( D4cHelper.WaitUntilVersionListDeleted( executor, patch, true ) );

                    // appv0 をインストールして起動( 自動更新対象化 )する
                    ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"application clear-task-status-list AND application uninstall --all" ) );
                    ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"application install {app.NspPath} AND application launch 0x{app.Identifier}" ) );

                    // superfly → tagaya への反映待ちを含めてダウンロード待ちさせる。( td1 は1分更新 )
                    var waitIntervalForAutoUpdate = waitIntervalForVersionListUpdate + ( 5 * 60 * 1000 );
                    var waitIntervalForAutoUpdateSecond = ( uint )waitIntervalForAutoUpdate / 1000;
                    System.Threading.Tasks.Task.Run( () =>
                    {
                        System.Threading.Thread.Sleep( waitIntervalForMonitorLaunch );
                        try
                        {
                            uploader.RegisterVersion( patch.NspPath );
                        }
                        catch ( System.Exception e )
                        {
                            Log.WriteLine( $"Detected exception on sub thread => {e.ToString()}" );
                        }
                    } );

                    // VersionList 通知待ち
                    ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"application wait-download 0x{app.Identifier} " +
                        $"--version {patch.Version} --timeout {waitIntervalForAutoUpdate} AND " +
                        $"application verify 0x{app.Identifier}",
                        waitIntervalForAutoUpdateSecond
                    ) );

                    // patch ( version 1 ) が見つかる事を確認。
                    ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( comRecord ) );
                    ExpectEvaluator.IsEqual( 0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find( executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue( patch, matchStorageAny )
                    ) );
                }
            }
        }
#endif

    }
}
