﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CsTestAssistants;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Reflection;

namespace testNs_ShopCleanup
{
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;

    [TestClass]
    public class UnitCleanup : TestClassBase
    {
        public TestContext TestContext { get; set; }

        private const string LogPrefix = "[eShopCleanup]";

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        //!----------------------------------------------------------------------------
        [ClassInitialize]
        public static void TestClassInitialize( TestContext context )
        {
            ClassInitialize( context, LogPrefix );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        //!----------------------------------------------------------------------------
        [ClassCleanup]
        public static void TestClassCleanup()
        {
            ClassCleanup();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="methodName">
        /// 参照対象TestProperty属性を保持するメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// 明示的な null 指定の場合、もしくは "JIRA" キーの TestPropertyが見つからない場合には、TestPropertyに依存しないベース作業用ディレクトリが返却されます。
        /// </param>
        /// <returns>
        /// TestProperty( "JIRA", * ) に依存した作業用ディレクトリパス、もしくはベース作業用ディレクトリパス。
        /// </returns>
        //!----------------------------------------------------------------------------
        private string GenerateIntermediateDirectoryAsMethod( SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null )
        {
            var runAssembly = Assembly.GetExecutingAssembly();
            var asmTitle = ( AssemblyTitleAttribute )System.Attribute.GetCustomAttribute( runAssembly, typeof( AssemblyTitleAttribute ) );
            return GenerateIntermediateDirectory( context, asmTitle.Title, methodName );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 全テストの最後にアカウントの解除を行う。
        /// 本テストは直前までのテストでデバイスアカウントリンクが解除されないケースを回避する目的のための実施です。
        /// 新しいテスト追加は、本テストよりも前に宣言定義をお願いいたします。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty( "JIRA", "SIGLO-48563" )]
        public void TestForAccountDeviceUnregistration()
        {
            using ( var scope = new TestMethodLog() )
            {
                var account = NintendoAccount;
                var executor = new SigloHelper.CommodityExecutor.Context( ActiveConfiguration );

                // finally
                scope.AddDisposer( () => { executor.RunDevMenuCommandSystem( "shop unlink-device-all" ); } );
                scope.AddDisposer( () => { executor.RunDevMenuCommandSystem( "account clear_all" ); } );
                scope.AddDisposer( () => { executor.RunDevMenuCommandSystem( "shop unregister-device-account" ); } );

                // start test
                ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem(
                    "account add --register_nsa" ) );
                ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem(
                    $"account link --index 0 --id {account.ID} --password {account.PASSWORD}" ) );
                ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem(
                    "systemupdate get-device-id" ) );

                // デバイスリンク解除が目的であれば不要。
                // リンク状態不正な状態で呼び出すと、finally 時の unlink-device-all が成功しない事がある。
                //ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( "shop device-link-status 0" ) );
            }
        }
    }
}
