﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>

#include <nn/account.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_ApplicationDownloadApi.h>
#include <nn/ns/ns_DeviceLinkApi.h>
#include <nn/ns/ns_DownloadTaskApi.h>
#include <nn/ns/ns_DownloadTaskSystemApi.h>
#include <nn/ns/ns_GameCardRegistrationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/ns/ns_TicketApi.h>

#include <nn/es/es_Api.h>
#include <nn/es/es_InitializationApi.h>
#include <nn/nifm.h>
#include <nn/oe/oe_Common.h>
#include <nn/oe/oe_ApplicationControlApis.h>
#include <nn/util/util_Endian.h>

#include "testNs_ScopedMountAoc.h"
#include "testNs_PropertyCollection.h"
#include "libraries/testNs_MountHost.h"
#include "libraries/testNs_RawFileLoader.h"

namespace {

    //!--------------------------------------------------------------------------------------
    //! 定数
    //!--------------------------------------------------------------------------------------
    const char* MOUNT_VOLUME_HOST = "Host";     //! HOSTマウントボリューム
    const char* MOUNT_VOLUME_AOC = "Aoc";       //! AOCマウントボリューム

    const nn::aoc::AddOnContentIndex INVALID_AOC_INDEX = -1;

    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    const size_t MallocHeapSize = 12 * 1024 * 1024;

    void* Allocate( size_t size ) NN_NOEXCEPT
    {
        return std::malloc( size );
    }

    void Deallocate( void* addr, size_t ) NN_NOEXCEPT
    {
        std::free( addr );
    }

    //!--------------------------------------------------------------------------------------
    //! 指定プロパティファイルがホストに作成されるまで待機する
    //!--------------------------------------------------------------------------------------
    bool WaitUntilPropertyFileCreated( testns::KeyValueCollection& properties, const char* pPropertyFilePath, const int64_t timeoutSeconds = 120 ) NN_NOEXCEPT
    {
        nn::os::TimerEvent timer( nn::os::EventClearMode_ManualClear );
        if ( timeoutSeconds > 0 )
        {
            timer.StartOneShot( nn::TimeSpan::FromSeconds( timeoutSeconds ) );
        }
        while ( !timer.TryWait() )
        {
            if ( properties.LoadFromFile( pPropertyFilePath ) )
            {
                break;
            }
            LOG_OUT( "Waiting... create file => [ %s ].\n", pPropertyFilePath );
            nn::os::TimedWaitAny( nn::TimeSpan::FromSeconds( 3 ), timer.GetBase() );
        }
        if ( timer.TryWait() )
        {
            LOG_OUT( "Could not find the the file `%s` within the specified interval time => %lld sec.\n", pPropertyFilePath, timeoutSeconds );
            return false;
        }
        return true;
    }

    //!--------------------------------------------------------------------------------------
    //! 指定インデックスの AoC が存在するか確認。
    //!--------------------------------------------------------------------------------------
    bool HasRequiredAocIndex( const nn::aoc::AddOnContentIndex requiredIndex = INVALID_AOC_INDEX ) NN_NOEXCEPT
    {
        int total = 0;
        if ( 0 >= ( total = nn::aoc::CountAddOnContent() ) )
        {
            return false;   // Aoc がそもそも見つからん。
        }

        if ( requiredIndex <= 0 )
        {
            // インデクス指定がないから、何かAoc 見つかった時点でOK.
            return true;
        }

        // 指定のインデクス捜索.
        nn::aoc::AddOnContentIndex indexes[ 64 ];
        for ( int offset = 0; offset < total; offset += testns::CountOf( indexes ) )
        {
            const auto count = nn::aoc::ListAddOnContent( indexes, offset, testns::CountOf( indexes ) );
            for ( int i = 0; i < count; ++i )
            {
                if ( requiredIndex == indexes[ i ] )
                {
                    return true;
                }
            }
        }
        return false;
    }

    //!--------------------------------------------------------------------------------------
    //! ホストにマウント要求ファイル( プロパティファイル )が作成されたら、対象要求内容の Aoc がマウントできるかチェックする。
    //! ホストにマウント要求ファイル( プロパティファイル )が作成されるまで待機する。
    //!--------------------------------------------------------------------------------------
    bool CheckAocMountableByPropertySync( const char* pPropertyFilePath ) NN_NOEXCEPT
    {
        // ホストからの通知 ( ファイル監視 )を待つ。
        testns::KeyValueCollection properties;
        const uint32_t timeoutSeconds = testns::GetOptionAsInteger( "--timeout", 0 );
        if ( false == WaitUntilPropertyFileCreated( properties, pPropertyFilePath, timeoutSeconds ) )
        {
            return false;   // マウント検証プロパティファイルが時間内に見つからなかった。
        }

        NN_UTIL_SCOPE_EXIT
        {
            // ホストプロパティファイルが見つかってたら、成否に関わらず処理終了時に削除.
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::DeleteFile( pPropertyFilePath ) );
        };

        // ホストからの Aoc マウント要求( ファイル作成 )検知でマウント開始。
        const int32_t INVALID = static_cast< int32_t >( INVALID_AOC_INDEX );
        const auto requiredAocIndex = properties.GetValueAsInteger( "AocIndex", INVALID );
        const auto requiredAocVersion = properties.GetValueAsInteger( "AocVersion", INVALID );

        // とりあえず Aoc あるか検索.
        if ( false == HasRequiredAocIndex( requiredAocIndex ) )
        {
            return false;
        }

        // インデクスとバージョンの指定があったら、バージョンも一致しているか確認。
        if ( INVALID != requiredAocVersion && INVALID != requiredAocIndex )
        {
            // MakeTestApplication の "aoc:/_Version_.txt" が利用できるはず。
            testns::ScopedMountAoc mountAoc( MOUNT_VOLUME_AOC, requiredAocIndex );
            if ( false == mountAoc.IsAvailable() )
            {
                // Aoc マウントで失敗。
                char pMessageBuffer[ 128 ];
                nn::util::SNPrintf( pMessageBuffer, testns::CountOf( pMessageBuffer ),
                    "AddOnContent try mount failed => [ AocIndex=%d, AocVersion=0x%x ].", requiredAocIndex, requiredAocVersion );
                testns::LogoutResultValue( mountAoc.GetLastResult(), pMessageBuffer );
                return false;
            }

            // バージョンファイル取得 ( MakeTestApplication生成物 )
            const char* pStream;
            RawFileLoader loader;
            const auto versionFilePath = mountAoc.MakeMountPath( "/_Version_.txt" );
            if ( nullptr == ( pStream = loader.LoadFromFile< char >( versionFilePath.c_str() ) ) )
            {
                LOG_OUT( "Do not have a version file within AddOnContent at index [ %d ].\n", requiredAocIndex );
                return false;
            }

            // BOM付き想定変換
            std::string versionString( &pStream[ 3 ], loader.GetFileSize() - 3 );
            const auto version = static_cast< int32_t >( std::strtoull( versionString.c_str(), nullptr, 10 ) );
            if ( version != requiredAocVersion )
            {
                LOG_OUT( "AddOnContent of index [ %d ] has version as [ 0x%x ], It is difference to with the required version [ 0x%x ].\n",
                    requiredAocIndex, version, requiredAocVersion
                );
                return false;
            }
            LOG_OUT( "AddOnContent mount check for `index [ %d ]` was succeeded, requiredVersion=0x%x <=> installVersion=0x%x\n",
                requiredAocIndex, requiredAocVersion, version
            );
        }
        else
        {
            LOG_OUT( "AddOnContent mount check for any was succeeded\n" );
        }
        return true;
    }


    //!--------------------------------------------------------------------------------------
    //! Aocの数、Index数、バージョンファイルの内容確認
    //!--------------------------------------------------------------------------------------
    bool CheckAocContents( int checkIndexNo ) NN_NOEXCEPT
    {
        int total = 0; //Aocの数
        bool isIndex = false; //指定したインデックスが存在するか

        //追加コンテンツの数を取得
        total = nn::aoc::CountAddOnContent();
        LOG_OUT("CountAddOnContent: %d \n", total);

        // AOC がない場合はエラーとする。
        if (total <= 0)
        {
            LOG_OUT("ListAddOnContent:Not existence Aoc files. \n");
            return false;
        }

        //AOC の Index を表にまとめる
        nn::aoc::AddOnContentIndex indexes[64];
        for (int offset = 0; offset < total; offset += testns::CountOf(indexes))
        {
            int indexCount = nn::aoc::ListAddOnContent(indexes, offset, testns::CountOf(indexes));

            LOG_OUT("ListAddOnContent:totalIndex= %d offset= %d \n", indexCount, offset);
            LOG_OUT("ListAddOnContent:Index\n");
            for ( int i = 0; i < indexCount; ++i )
            {
                if (checkIndexNo == indexes[i])
                {
                    isIndex = true;
                }
                //Aoc の Indexの列挙
                LOG_OUT("                :%3d\n", indexes[i] );
            }
        }

        // 指定したインデックスが発見できない場合はエラーとする。
        if ( false == isIndex )
        {
            LOG_OUT("The specified index could not be found. CheckindexNo:%d\n", checkIndexNo );
            return false;
        }

        // MakeTestApplication の "aoc:/_Version_.txt" が利用できるはず。
        testns::ScopedMountAoc mountAoc(MOUNT_VOLUME_AOC, checkIndexNo);
        if (false == mountAoc.IsAvailable())
        {
            // Aoc マウントで失敗。
            char pMessageBuffer[128];
            nn::util::SNPrintf(pMessageBuffer, testns::CountOf(pMessageBuffer),
                "AddOnContent try mount failed => [ AocIndex=%d ].", checkIndexNo);
            testns::LogoutResultValue(mountAoc.GetLastResult(), pMessageBuffer);
            return false;
        }

        // バージョンファイル取得 ( MakeTestApplication生成物 )
        const char* pStream;
        RawFileLoader loader;
        const auto versionFilePath = mountAoc.MakeMountPath("/_Version_.txt");
        if (nullptr == (pStream = loader.LoadFromFile< char >(versionFilePath.c_str())))
        {
            LOG_OUT("Do not have a version file within AddOnContent at index [ %d ].\n", checkIndexNo);
            return false;
        }

        // BOM付き想定変換
        std::string versionString(&pStream[3], loader.GetFileSize() - 3);
        const auto version = static_cast< int32_t >(std::strtoull(versionString.c_str(), nullptr, 10));
        LOG_OUT("AddOnContent mount check for `index %d, installVersion= [ 0x%x ] \n",
            checkIndexNo, version
        );

        return true;
    }

    //!--------------------------------------------------------------------------------------
    //! Aocのハッシュ値の内容確認
    //!--------------------------------------------------------------------------------------
    bool CheckAocHash(int checkIndexNo) NN_NOEXCEPT
    {

        // MakeTestApplication の "aoc:/_Version_.txt" が利用できるはず。
        testns::ScopedMountAoc mountAoc(MOUNT_VOLUME_AOC, checkIndexNo);
        if (false == mountAoc.IsAvailable())
        {
            // Aoc マウントで失敗。
            char pMessageBuffer[128];
            nn::util::SNPrintf(pMessageBuffer, testns::CountOf(pMessageBuffer),
                "AddOnContent try mount failed => [ AocIndex=%d ].", checkIndexNo);
            testns::LogoutResultValue(mountAoc.GetLastResult(), pMessageBuffer);
            return false;
        }

        // ハッシュ値取得 ( MakeTestApplication生成物 )
        char* pStream;
        RawFileLoader loader;
        const auto hashFilePath = mountAoc.MakeMountPath("/HashVal.txt");
        if (nullptr == (pStream = loader.LoadFromFile< char >(hashFilePath.c_str())))
        {
            LOG_OUT("Do not have a hash file within AddOnContent at hash .\n");
            return false;
        }
        // BOM付き想定変換
        std::string hashString(&pStream[3], loader.GetFileSize() - 3);
        LOG_OUT("AddOnContent mount check for `hash [ %s ]` .\n", hashString.c_str() );

        return true;
    }

    //}
    //!--------------------------------------------------------------------------------------
    //! Aoc マウント可否確認コマンドメイン
    //!--------------------------------------------------------------------------------------
    bool CheckAocMountable() NN_NOEXCEPT
    {
        char pPropertyFilePath[ 256 ];
        if ( false == testns::MakeHostFileSystemPath( pPropertyFilePath, "/AocMountRequest.xml" ) )
        {
            // ホストがマウントされてない
            LOG_OUT( "Could not load the request properties file, because did not mount the host file system.\n" );
            return false;
        }

        // 最初消す
        if ( fsutil::IsExistPath( pPropertyFilePath ) )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::DeleteFile( pPropertyFilePath ) );
        }

        // 周期的に要求待ちを行う。
        while ( NN_STATIC_CONDITION( true ) )
        {
            if ( CheckAocMountableByPropertySync( pPropertyFilePath ) )
            {
                LOG_OUT( "[CheckAocMountable] Success.\n" );
            }
            else
            {
                LOG_OUT( "[CheckAocMountable] Failure.\n" );
            }
            nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
        }
        return true;
    }

    //!--------------------------------------------------------------------------------------
    //! Aoc でマウントする Index をコマンド引数から取得し、Indexが使用可能か確認
    //!--------------------------------------------------------------------------------------
    int GetAocIndexForCommand( char* pIndexValue ) NN_NOEXCEPT
    {
        const char* pCheckIndexNo = "CheckIndexNo";
        bool isIndex = false; //指定したインデックスが存在するか

        int indexNo = 0;

        if (0 == std::strncmp( pCheckIndexNo, pIndexValue, std::strlen(pCheckIndexNo) ))
        {
            // "CheckIndexNo" の値を取得
            indexNo = pIndexValue[ std::strlen(pCheckIndexNo) + 1 ] - '0';
            if ((indexNo < 1) || (indexNo > 9))
            {
                // Index 指定してない場合はエラー
                LOG_OUT("+++++ ShopRuntimeAssistants: Unknown CheckIndexNo +++++ pattern unmatch string \n");
                return 0;
            }
        }
        else
        {
            // Index 指定してない場合はエラー
            LOG_OUT("+++++ ShopRuntimeAssistants: Unknown CheckIndexNo +++++ pattern Non string \n");
            return 0;
        }

        //追加コンテンツの数を取得
        int total = nn::aoc::CountAddOnContent();

        // AOC がない場合はエラーとする。
        if (total <= 0)
        {
            LOG_OUT("ListAddOnContent:Not existence Aoc files. \n");
            return 0;
        }

        //指定した Index が AOC に存在するか調査
        nn::aoc::AddOnContentIndex indexes[64];
        for (int offset = 0; offset < total; offset += testns::CountOf(indexes))
        {
            int indexCount = nn::aoc::ListAddOnContent(indexes, offset, testns::CountOf(indexes));

            for (int i = 0; i < indexCount; ++i)
            {
                if (indexNo == indexes[i])
                {
                    isIndex = true;
                }
            }
        }

        // 指定したインデックスが発見できない場合はエラーとする。
        if (false == isIndex)
        {
            LOG_OUT("The specified index could not be found. CheckindexNo:%d\n", indexNo);
            return 0;
        }

        return indexNo;
    }

    //!--------------------------------------------------------------------------------------
    //! Aoc の Dynamic Commit チェックコマンドメイン
    //!--------------------------------------------------------------------------------------
    bool CheckDynamicCommitAoc() NN_NOEXCEPT
    {
        nn::os::SystemEvent chengedEvent;
        nn::aoc::GetAddOnContentListChangedEvent(&chengedEvent);

        // イベントが来るのを待つ
        // 最大 10 秒待つようにし、それを超えると false を返す
        if (false == chengedEvent.TimedWait(nn::TimeSpan::FromSeconds(10)))
        {
            // Aoc の ListChanged イベントが来ない
            LOG_OUT("[CheckDynamicCommitAoc] Failure. ChangeEvent Wait Timeout\n");
            return false;
        }

        // マウント要求のファイルパス (本アプリ実行前に作っているはず)
        char pPropertyFilePath[256] = { 0 };
        if (false == testns::MakeHostFileSystemPath(pPropertyFilePath, "/AocMountRequest.xml"))
        {
            // ホストがマウントされてない
            LOG_OUT("Could not load the request properties file, because did not mount the host file system.\n");
            return false;
        }

        // Aoc がマウントできるかチェックする
        if (false == CheckAocMountableByPropertySync(pPropertyFilePath))
        {
            LOG_OUT("[CheckDynamicCommitAoc] Failure. Mount Aoc\n");
            return false;
        }

        LOG_OUT("[CheckDynamicCommitAoc] Success.\n");
        return true;
    }

    //!--------------------------------------------------------------------------------------
    //! UserIndex から Uid を取得
    //!--------------------------------------------------------------------------------------
    bool GetUidFromIndex(nn::account::Uid* outValue, int userIndex) NN_NOEXCEPT
    {
        auto index = static_cast<size_t>(userIndex);

        std::vector<nn::account::Uid> uidList(nn::account::UserCountMax);
        int count;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, uidList.data(), static_cast<int>(uidList.size())));
        if (count == 0)
        {
            return false;
        }
        uidList.resize(static_cast<size_t>(count));

        if (index >= uidList.size())
        {
            return false;
        }

        *outValue = uidList[index];
        return true;
    }

    //!--------------------------------------------------------------------------------------
    //! Async の NN_RESULT_TRY 簡略化マクロ
    //!--------------------------------------------------------------------------------------
#define HANDLE_REQUEST_RESULT(result)                                   \
    NN_RESULT_TRY(result)                                               \
        NN_RESULT_CATCH_ALL                                             \
        {                                                               \
            LOG_OUT("[CheckErrorContext] Failure, request async failed %08x\n", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug()); \
            return false;                                               \
        }                                                               \
    NN_RESULT_END_TRY;

    //!--------------------------------------------------------------------------------------
    //! ErrorContext 表示
    //!--------------------------------------------------------------------------------------
    template<typename AsyncType>
    bool ShowErrorContext(nn::Result result, AsyncType& async) NN_NOEXCEPT
    {
        if (result.IsSuccess())
        {
            LOG_OUT("[CheckErrorContext] Failure, async task seems to succeed unexpectedly\n");
            return false;
        }
        nn::err::ErrorContext errorContext;
        async.GetErrorContext(&errorContext);
        switch (errorContext.type)
        {
        case nn::err::ErrorContextType::Http:
            LOG_OUT("[CheckErrorContext] ErrorContextJson: { \"result\": \"0x%08x\", \"fqdn\": \"%s\", \"ip\": \"%s\"}\n", result.GetInnerValueForDebug(), errorContext.http.fqdn, errorContext.http.ip);
            return true;
        default:
            // Http 以外は、このテストでは想定していない
            LOG_OUT("[CheckErrorContext] Failure, unexpected ErrorContextType: %d, result: %08x\n", static_cast<int>(errorContext.type), result.GetInnerValueForDebug());
            return false;
        }
    }

    //!--------------------------------------------------------------------------------------
    //! AsyncResult 用のテンプレート
    //!--------------------------------------------------------------------------------------
    template<typename AsyncType>
    bool WaitAndOutputErrorContext1(AsyncType& async, int timeoutSeconds) NN_NOEXCEPT
    {
        for (int i = 0; i < timeoutSeconds; ++i)
        {
            if (async.TryWait())
            {
                return ShowErrorContext(async.Get(), async);
            }
            nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
        }
        LOG_OUT( "[CheckErrorContext] Failure, timeout\n" );
        return false;
    }

    //!--------------------------------------------------------------------------------------
    //! AsyncValue 用のテンプレート
    //!--------------------------------------------------------------------------------------
    template<typename AsyncType>
    bool WaitAndOutputErrorContext2(AsyncType& async, int timeoutSeconds) NN_NOEXCEPT
    {
        for (int i = 0; i < timeoutSeconds; ++i)
        {
            if (async.TryWait())
            {
                // サイズが大きい場合があるので、ヒープから確保
                std::unique_ptr<typename AsyncType::AsyncValueType> value(new typename AsyncType::AsyncValueType());
                return ShowErrorContext(async.Get(value.get()), async);
            }
            nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
        }
        LOG_OUT( "[CheckErrorContext] Failure, timeout\n" );
        return false;
    }

    //!--------------------------------------------------------------------------------------
    //! UserIdの取得
    //!--------------------------------------------------------------------------------------
    bool GetUserId(nn::account::Uid* uid , const std::string* asyncName )
    {
        char userIndexBuffer[64] = "";
        if (!testns::QueryCommandLineOptions("--UserIndex", sizeof(userIndexBuffer), userIndexBuffer))
        {
            LOG_OUT("[CheckErrorContext] Failure, %s needs --UserIndex option\n", asyncName->c_str());
            return false;
        }

        auto userIndex = std::strtol(userIndexBuffer, nullptr, 10);
        if (!GetUidFromIndex(uid, userIndex))
        {
            LOG_OUT("[CheckErrorContext] Getting uid failed\n", asyncName->c_str());
            return false;
        }
        return true;
    }

    //!--------------------------------------------------------------------------------------
    //! ErrorContext の取得チェックメイン
    //!--------------------------------------------------------------------------------------
    bool CheckErrorContext() NN_NOEXCEPT
    {
        // ns の各種 async タスクを作成・実行し、ErrorContext が取れることを確認する
        // あらかじめ nim のエラーシミュレートでエラーが設定されている想定

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::nifm::SubmitNetworkRequestAndWait();
        if (!nn::nifm::IsNetworkAvailable())
        {
            LOG_OUT("[CheckErrorContext] Failure, network is unavailable\n");
            return false;
        }

        nn::ncm::ApplicationId id = {0x0100f6900049a000}; // PatchTest を借用

        char asyncNameBuffer[64] = "";
        if (testns::QueryCommandLineOptions("--AsyncName", sizeof(asyncNameBuffer), asyncNameBuffer))
        {
            std::string asyncName(asyncNameBuffer);
            if (asyncName == "ApplicationControlData")
            {
                nn::ns::AsyncResult async;
                HANDLE_REQUEST_RESULT(nn::ns::RequestDownloadApplicationControlData(&async, id));
                return WaitAndOutputErrorContext1(async, 10);
            }
            else if (asyncName == "EnsureDownloadTask")
            {
                nn::ns::AsyncResult async;
                nn::ns::RequestEnsureDownloadTask(&async);
                return WaitAndOutputErrorContext1(async, 10);
            }
            else if (asyncName == "DownloadTaskListData")
            {
                nn::ns::AsyncDownloadTaskListData async;
                HANDLE_REQUEST_RESULT(nn::ns::RequestDownloadTaskListData(&async));
                return WaitAndOutputErrorContext2(async, 10);
            }
            else if (asyncName == "CheckLatestUpdate")
            {
                nn::ns::DestroySystemUpdateTask();
                nn::ns::SystemUpdateControl control;
                NN_RESULT_TRY(control.Occupy())
                    NN_RESULT_CATCH_ALL
                    {
                        LOG_OUT("[CheckErrorContext] Failure, system update control occupy failed %08x\n", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                        return false;
                    }
                NN_RESULT_END_TRY;
                nn::ns::AsyncLatestSystemUpdate async;
                HANDLE_REQUEST_RESULT(control.RequestCheckLatestUpdate(&async));
                return WaitAndOutputErrorContext2(async, 10);
            }
            else if (asyncName == "DownloadLatestUpdate")
            {
                nn::ns::DestroySystemUpdateTask();
                nn::ns::SystemUpdateControl control;
                NN_RESULT_TRY(control.Occupy())
                    NN_RESULT_CATCH_ALL
                    {
                        LOG_OUT("[CheckErrorContext] Failure, system update control occupy failed %08x\n", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                        return false;
                    }
                NN_RESULT_END_TRY;
                nn::ns::AsyncResult async;
                HANDLE_REQUEST_RESULT(control.RequestDownloadLatestUpdate(&async));
                return WaitAndOutputErrorContext1(async, 10);
            }
            else if (asyncName == "ApplicationUpdateInfo")
            {
                nn::ns::AsyncApplicationUpdateInfo async;
                HANDLE_REQUEST_RESULT(nn::ns::RequestApplicationUpdateInfo(&async, id));
                return WaitAndOutputErrorContext2(async, 10);
            }
            else if (asyncName == "UpdateApplication")
            {
                nn::ns::AsyncResult async;
                HANDLE_REQUEST_RESULT(nn::ns::RequestUpdateApplication(&async, id));
                return WaitAndOutputErrorContext1(async, 10);
            }
            else if (asyncName == "CheckGameCardRegistration")
            {
                nn::ns::AsyncResult async;
                HANDLE_REQUEST_RESULT(nn::ns::RequestCheckGameCardRegistration(&async, id));
                return WaitAndOutputErrorContext1(async, 10);
            }
            else if (asyncName == "GameCardRegistrationGoldPoint")
            {
                // ユーザーIDの取得
                nn::account::Uid uid;
                bool isUserId = GetUserId(&uid, &asyncName);
                if (!isUserId)
                {
                    return false;
                }
                nn::ns::AsyncGameCardGoldPoint async;
                HANDLE_REQUEST_RESULT(nn::ns::RequestGameCardRegistrationGoldPoint(&async, uid, id));
                return WaitAndOutputErrorContext2(async, 10);
            }
            else if (asyncName == "RegisterGameCard")
            {
                // ユーザーIDの取得
                nn::account::Uid uid;
                bool isUserId = GetUserId(&uid, &asyncName);
                if (!isUserId)
                {
                    return false;
                }
                nn::ns::AsyncResult async;
                int goldPoint = 0;
                HANDLE_REQUEST_RESULT(nn::ns::RequestRegisterGameCard(&async, uid, id, goldPoint));
                return WaitAndOutputErrorContext1(async, 10);
            }
            else if (asyncName == "LinkDevice")
            {
                // ユーザーIDの取得
                nn::account::Uid uid;
                bool isUserId = GetUserId(&uid, &asyncName);
                if (!isUserId)
                {
                    return false;
                }
                nn::ns::AsyncResult async;
                HANDLE_REQUEST_RESULT(nn::ns::RequestLinkDevice(&async, uid));
                return WaitAndOutputErrorContext1(async, 10);
            }
            else if (asyncName == "DownloadApplicationPrepurchasedRights")
            {
                nn::ns::AsyncResult async;
                HANDLE_REQUEST_RESULT(nn::ns::RequestDownloadApplicationPrepurchasedRights(&async, id));
                return WaitAndOutputErrorContext1(async, 10);
            }
            else
            {
                LOG_OUT("[CheckErrorContext] Failure, unknown async name %s\n", asyncName.c_str());
                return false;
            }
        }
        else
        {
            LOG_OUT("[CheckErrorContext] Failure, please specify the async name option => \"--AsyncName=XXX\"\n");
            return false;
        }
    } // NOLINT(impl/function_size)

    //!--------------------------------------------------------------------------------------
    //! 予約情報の登録、デバッグ用で今は ID 決めうち
    //!--------------------------------------------------------------------------------------
    bool RegisterPrepurchaseInfo() NN_NOEXCEPT
    {
        nn::ncm::ApplicationId id = {0x0100f6900049a000}; // PatchTest を借用

        nn::es::PrepurchaseRecord record = {};

        // 上位 8byte に application Id を埋める
        nn::util::SwapEndian(&id);
        std::memcpy(&(record.rightsId), &id, sizeof(id));
        record.ticketId = 0x12345678;
        record.accountId = 0x00000000;

        nn::es::Initialize();
        NN_RESULT_TRY(nn::es::ImportPrepurchaseRecord(record))
            NN_RESULT_CATCH_ALL
            {
                LOG_OUT("[RegisterPrepurchaseInfo] Failure, %08x\n", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                return false;
            }
        NN_RESULT_END_TRY;
        return true;
    }

    //!--------------------------------------------------------------------------------------
    //! ListAddOnContent を実行し内容をログ出力
    //!--------------------------------------------------------------------------------------
    bool CheckListAddOnContent() NN_NOEXCEPT
    {
        //追加コンテンツの数を取得
        int total = nn::aoc::CountAddOnContent();

        const int listAcquisitionCount = 1;
        nn::aoc::AddOnContentIndex indexes[listAcquisitionCount];
        LOG_OUT("ListAddOnContent:start \n");
        for ( int offset = 0; offset < total; offset++ )
        {
            int indexCount = nn::aoc::ListAddOnContent(indexes, offset, listAcquisitionCount);

            for ( int i = 0; i < indexCount; ++i )
            {
                LOG_OUT("ListAddOnContent:indexCount= %d offset= %d index= %d \n",
                    indexCount, offset, indexes[i]);
            }
        }
        LOG_OUT("ListAddOnContent:end \n");
        return true;
    }


    //!--------------------------------------------------------------------------------------
    //! 要求処理振り分け
    //!--------------------------------------------------------------------------------------
    bool RunCommand() NN_NOEXCEPT
    {
        // 処理開始
        const char* pCheckAocContents = "CheckAocContents";
        const char* pCheckAocHash = "CheckAocHash";
        char pCaseValue[ 64 ] = "";


        if ( testns::QueryCommandLineOptions( "--Command", sizeof( pCaseValue ), pCaseValue ) && 0 != std::strlen( pCaseValue ) )
        {
            LOG_OUT( "+++++ ShopRuntimeAssistants run command as [ %s ] +++++\n", pCaseValue );
            if ( 0 == std::strcmp( pCaseValue, "CheckAocMountable" ) )
            {
                return CheckAocMountable();
            }
            //CheckAocContents の判定
            else if (0 == std::strncmp( pCheckAocContents, pCaseValue, std::strlen(pCheckAocContents) ) )
            {
                int indexNo = GetAocIndexForCommand( &pCaseValue[ std::strlen(pCheckAocContents) + 1 ] );
                if (indexNo < 1)
                {
                    return false;
                }
                return CheckAocContents(indexNo);
            }
            //CheckAocHash の判定
            else if (0 == std::strncmp(pCheckAocHash, pCaseValue, std::strlen(pCheckAocHash) ))
            {
                int indexNo = GetAocIndexForCommand( &pCaseValue[ std::strlen(pCheckAocHash) + 1 ] );
                if (indexNo < 1)
                {
                    return false;
                }
                return CheckAocHash(indexNo);
            }
            else if (0 == std::strcmp(pCaseValue, "CheckDynamicCommitAoc"))
            {
                return CheckDynamicCommitAoc();
            }
            else if (0 == std::strcmp(pCaseValue, "CheckErrorContext"))
            {
                return CheckErrorContext();
            }
            else if (0 == std::strcmp(pCaseValue, "RegisterPrepurchaseInfo"))
            {
                return RegisterPrepurchaseInfo();
            }
            else if (0 == std::strcmp(pCaseValue, "CheckListAddOnContent"))
            {
                return CheckListAddOnContent();
            }
            else
            {
                LOG_OUT( "+++++ ShopRuntimeAssistants: Unknown command detected as [ %s ] +++++\n", pCaseValue );
            }
        }
        else
        {
            LOG_OUT( "+++++ ShopRuntimeAssistants: Please specify the command option => \"--Command=XXX\" +++++\n" );
        }
        return false;
    }

} // ~unnamed

//!--------------------------------------------------------------------------------------
//! アプリケーションのメモリ管理機構を初期化
//!--------------------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::SetMemoryHeapSize( MemoryHeapSize ) );
    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::AllocateMemoryBlock( &address, MallocHeapSize ) );
    nn::init::InitializeAllocator( reinterpret_cast<void*>( address ), MallocHeapSize );
}

//!--------------------------------------------------------------------------------------
//! めいん
//!--------------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // nn::fs 初期化
    nn::fs::SetAllocator( Allocate, Deallocate );

    nn::oe::Initialize();

    nn::ns::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::ns::Finalize();
    };

    // Hostハンドリング
    NN_ABORT_UNLESS_RESULT_SUCCESS( testns::MountHostFileSystem( MOUNT_VOLUME_HOST ) );
    testns::InitializeCommandLineOptions();

    NN_UTIL_SCOPE_EXIT
    {
        testns::FinalizeCommandLineOptions();
        testns::UnmountHostFileSystem();
    };

    // aoc ライブラリ準備
    nn::aoc::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::aoc::Finalize();
    };

    nn::account::Initialize();

    // 要求コマンド振り分け＆実施
    if ( RunCommand() )
    {
        LOG_OUT( "[SUCCESS]\n" );
    }
    else
    {
        LOG_OUT( "[FAILURE]\n" );
    }
}
