﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace CsTestAssistants
{
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;

    public class TestClassBase
    {
        protected const string TestMethodLogBegin = "=== Begin the test of [ {0} ] ===";
        protected const string TestMethodLogFinish = "=== Finish the test of [ {0} ] ===";
        protected const string TestMethodLogFinally = "=== Finally the test of [ {0} ] ===";
        protected const int ReinitAfterInterval = 60000;

        protected static readonly BasicAccount CliTokenAccountForDEV1 = new BasicAccount( "pux_mori.tetsuya", "NjY2MDIyNDItMWY5" );
        protected static readonly BasicAccount CliTokenAccountForDEV6 = new BasicAccount( "pux_tsutsumi.daisuke", "NGEzYmM0YTUtODBk" );
        protected static readonly BasicAccount NintendoAccountForPUX = new BasicAccount( "eshop-test+puxdd1_001@exmx.nintendo.co.jp", "puxdd1_001" );
        protected static readonly string NetworkSettingJsonFilePath = System.IO.Path.Combine( SigloHelper.Path.GetTestsRoot(), "Ns", "Sources", "Tests", "ShopDeliveryBasic", "NetworkSettingsForCI.txt" );
        protected static readonly string NetworkSettingJsonOnlyEther = System.IO.Path.Combine( SigloHelper.Path.GetTestsRoot(), "Ns", "Sources", "Tests", "ShopDeliveryBasic", "NetworkSettingsForOnlyEther.txt" );

        protected static System.Net.WebProxy ProxyConfiguration;
        protected static EncodeContext ActiveEncode = new EncodeContext();
        protected static SigloHelper.Configuration.Context ActiveConfiguration = null;
        protected static bool IsSkipUpload = false;
        protected static bool IsUnitTestOnIDE = false;
        protected static BasicAccount CliTokenAccount = CliTokenAccountForDEV6;
        protected static D4cHelper.Configuration.ServerEnvironment ServerEnvironment = D4cHelper.Configuration.ServerEnvironment.DEV6;
        protected static readonly BasicAccount NintendoAccount = NintendoAccountForPUX;

        //!----------------------------------------------------------------------------
        /// <summary>
        /// Working stage scope definition class.
        /// </summary>
        //!----------------------------------------------------------------------------
        public class TestMethodLog : ScopeLog
        {
            public TestMethodLog( [System.Runtime.CompilerServices.CallerMemberName] string testName = "" )
                : base( TestMethodLogBegin, TestMethodLogFinish, TestMethodLogFinally, testName ) { }

            public override void WriteBeginLabel()
            {
                Log.WriteLine( $"=== Begin the test of [ {ScopeName} {System.DateTime.Now} ] ===  " );
            }

            public override void WriteEndLabel()
            {
                Log.WriteLine( $"=== Finish the test of [ {ScopeName} {System.DateTime.Now} ] ===  " );
            }

            public override void WriteDisposeLabel()
            {
                Log.WriteLine( $"=== Finally the test of [ {ScopeName} {System.DateTime.Now} ] ===  " );
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 実行環境に応じたプロキシ設定の取得.
        /// </summary>
        /// <param name="context">テストコンテキスト</param>
        /// <returns>プロキシ設定が無い場合は null, その他は対象の proxy instance.</returns>
        //!----------------------------------------------------------------------------
        private static System.Net.WebProxy PrepareProxyConfiguration( TestContext context )
        {
            try
            {
                if ( true == context.Properties.Contains( "EnableRunTestOnIDE_FromPUX" ) &&
                    true == bool.Parse( ( string )context.Properties[ "EnableRunTestOnIDE_FromPUX" ] ) )
                {
                    return D4cHelper.ProxyHelper.Create( null );
                }
            }
            catch ( System.Exception e )
            {
                Log.WriteLine( $"Critical failure of prepare proxy configuration => [ {e.Message} ]" );
            }
            return D4cHelper.ProxyHelper.Create();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        //!----------------------------------------------------------------------------
        public static void ClassInitialize( TestContext context, string logPrefix = null )
        {
            ActiveEncode.Apply( EncodeContext.UTF8 );
            if ( false == string.IsNullOrEmpty( logPrefix ) )
            {
                Log.Prefix = logPrefix;
            }
            Log.Verbose = true;

            ProxyConfiguration = PrepareProxyConfiguration( context );

            string targetName = ( string )context.Properties[ "TargetName" ];
            string platform = ( context.Properties.Contains( "Platform" ) ) ? ( string )context.Properties[ "Platform" ] : SigloHelper.Configuration.DefaultPlatform;
            string buildType = ( context.Properties.Contains( "BuildType" ) ) ? ( string )context.Properties[ "BuildType" ] : SigloHelper.Configuration.DefaultBuildType;
            ActiveConfiguration = new SigloHelper.Configuration.Context( targetName, platform, buildType );

            IsSkipUpload = context.Properties.Contains( "SkipUpload" ) && bool.Parse( ( string )context.Properties[ "SkipUpload" ] );
            IsUnitTestOnIDE = context.Properties.Contains( "UnitTestOnIDE" ) && bool.Parse( ( string )context.Properties[ "UnitTestOnIDE" ] );

            // CIテストの時、デバッグトレースを非表示にする。
            if ( false == IsUnitTestOnIDE )
            {
                // デバッグトレースログを非表示にする。
                Log.WithTrace = false;
            }

            Log.WriteLine( "=== Test Context ===" );
            Log.WriteLine( "TestDir                 : {0}", context.TestDir );
            Log.WriteLine( "TestLogsDir             : {0}", context.TestLogsDir );
            Log.WriteLine( "TestRunDirectory        : {0}", context.TestRunDirectory );
            Log.WriteLine( "TestResultsDirectory    : {0}", context.TestResultsDirectory );
            Log.WriteLine( "TestRunResultsDirectory : {0}", context.TestRunResultsDirectory );

            Log.WriteLine( "=== Configuration ===" );
            Log.WriteLine( "TargetName      : {0}", ActiveConfiguration.TargetName );
            Log.WriteLine( "TargetPlatform  : {0}", ActiveConfiguration.TargetPlatform );
            Log.WriteLine( "TargetBuildType : {0}", ActiveConfiguration.TargetBuildType );
            Log.WriteLine( "HostWebProxy    : {0}", ( null != ProxyConfiguration ) ? ProxyConfiguration.Address.ToString() : "None" );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        //!----------------------------------------------------------------------------
        public static void ClassCleanup()
        {
            ActiveEncode.Restore();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="baseWorkingFolderName">ベース作業用ディレクトリ名</param>
        /// <param name="methodName">
        /// 参照対象TestProperty属性を保持するメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// 明示的な null 指定の場合、もしくは "JIRA" キーの TestPropertyが見つからない場合には、TestPropertyに依存しないベース作業用ディレクトリが返却されます。
        /// </param>
        /// <returns>
        /// TestProperty( "JIRA", * ) に依存した作業用ディレクトリパス、もしくはベース作業用ディレクトリパス。
        /// </returns>
        //!----------------------------------------------------------------------------
        protected string GenerateIntermediateDirectory( SigloHelper.Configuration.Context context,
            string baseWorkingFolderName, [System.Runtime.CompilerServices.CallerMemberName] string methodName = null )
        {
            // ベース作業用ディレクトリ
            string outPath = context.GetOutputTestDirectory( baseWorkingFolderName );
            if ( false == string.IsNullOrEmpty( methodName ) )
            {
                string ticket = this.GetTestPropertyValue( methodName, "JIRA" );
                if ( !string.IsNullOrEmpty( ticket ) )
                {
                    outPath = System.IO.Path.Combine( outPath, ticket );
                }
            }
            Log.WriteLine( "intermediate directory : {0}", outPath );
            return outPath;
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// SDEV再初期化のためにネットワーク設定シーケンスをメソッド化
        /// </summary>
        /// <param name="executor"></param>
        /// <param name="networkJsonFile">インポート対象ネットワーク設定Jsonパラメータファイルパス( フルパス )</param>
        //!----------------------------------------------------------------------------
        protected void ImportSdevNetworkSettings( SigloHelper.CommodityExecutor.Context executor, string networkJsonFile, string passCode )
        {
            // Check file exists.
            FileHelper.TestExistsFile( networkJsonFile );

            // SettingsManager import `network`
            ExpectEvaluator.IsTrue( executor.RunSettingsManager( "Import \"" + networkJsonFile + "\"" ) );
            // reset sdev
            ExpectEvaluator.IsTrue( executor.ResetTarget() );
            // import service discovery setting
            ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"servicediscovery import-all {passCode}" ) );
            ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"debug set-integer-fwdbg --name ns.notification --key debug_waiting_limit 1" ) );
            ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem( $"debug set-integer-fwdbg --name ns.notification --key version_list_waiting_limit 1" ) );
            // reset sdev
            ExpectEvaluator.IsTrue( executor.ResetTarget() );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// SDEV再初期化のためにネットワーク設定シーケンスをメソッド化
        /// </summary>
        /// <param name="executor"></param>
        /// <param name="networkJsonFile">インポート対象ネットワーク設定Jsonパラメータファイルパス( フルパス ),
        /// null の場合は、無線設定が自動で選択されます。</param>
        //!----------------------------------------------------------------------------
        protected void SetupSdevEnvironments( SigloHelper.CommodityExecutor.Context executor, string networkJsonFile = null )
        {
            var path = ( string.IsNullOrEmpty( networkJsonFile ) ) ? NetworkSettingJsonFilePath : networkJsonFile;
            ImportSdevNetworkSettings( executor, path, ServerEnvironment.PassCode );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// SDEV再初期化
        /// </summary>
        /// <param name="executor"></param>
        //!----------------------------------------------------------------------------
        protected void InitializeSdev( SigloHelper.CommodityExecutor.Context executor )
        {
            // 再初期化
            ExpectEvaluator.IsTrue( executor.InvokeUpdateKernelOnSDEV() );
            // 起動待ち
            System.Threading.Thread.Sleep( ReinitAfterInterval );
            // 再設定
            SetupSdevEnvironments( executor );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// SdCard クリーンアップ.
        /// </summary>
        /// <param name="executor"></param>
        //!----------------------------------------------------------------------------
        protected void CleanupSdCard( SigloHelper.CommodityExecutor.Context executor )
        {
            // format / cleanup
            ExpectEvaluator.IsTrue( executor.RunDevMenuCommand( "sdcard format AND sdcard cleanup" ) );
            // reset sdev
            ExpectEvaluator.IsTrue( executor.ResetTarget() );
            // status check
            ExpectEvaluator.IsTrue( executor.RunDevMenuCommand( "sdcard status" ) );
        }

        /// <summary>
        /// コンテンツクリーンアップ
        /// </summary>
        protected void CleanupContents( SigloHelper.CommodityExecutor.Context executor, bool enableSdCard = false )
        {
            // gameCard
            var isInsertedCard = executor.RunDevMenuCommandSystem($"gamecard status", @"^Inserted$");
            if ( isInsertedCard )
            {
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase"));
            }

            // enable sdcard
            if ( enableSdCard )
            {
                ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem(
                    "debug enable-mount-sdcard " ) );
                ExpectEvaluator.IsTrue( executor.ResetTarget() );
            }

            // cleanup contents
            ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem(
                "application clear-task-status-list AND " +
                "application uninstall --all AND " +
                "application reset-required-version --all AND " +
                "sdcard format AND " +
                "sdcard cleanup " ) );
            ExpectEvaluator.IsTrue( executor.ResetTarget() );
        }

        /// <summary>
        /// コンテンツクリーンアップと待機
        /// </summary>
        protected void CleanupContentsAndWait( SigloHelper.CommodityExecutor.Context executor, System.TimeSpan waitTime )
        {
            CleanupContents( executor, false );
            System.Threading.Thread.Sleep( waitTime );
        }

        /// <summary>
        /// アプリケーションリストコマンド実行
        /// </summary>
        protected void CheckAppplicationList( SigloHelper.CommodityExecutor.Context executor )
        {
            ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem(
                "application list --detail AND " +
                "application list-record-detail AND " +
                "application list-view " ) );
        }

        /// <summary>
        /// 指定アプリケーションのベリファイとレコード表示
        /// </summary>
        protected void VerifyAppplication( SigloHelper.CommodityExecutor.Context executor, ID64 id )
        {
            ExpectEvaluator.IsTrue( executor.RunDevMenuCommandSystem(
                $"application verify 0x{id} AND " +
                $"application list-record 0x{id} --installed " ) );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 電源ボタン押下経由でのシャットダウン / 起動による再起動。
        /// </summary>
        /// <param name="executor">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="holdTime">電源遮断時の電源ボタン押下継続時間</param>
        /// <param name="intervalSeconds">電源遮断後のスレッドスリープ待機時間( 秒数 )</param>
        /// <param name="waitBootupSeconds">電源投入後のスレッドスリープ待機時間( 秒数 )</param>
        /// <param name="methodName">
        /// ログに出力する際のメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// </param>
        /// <returns>成否。 true は成功。</returns>
        //!----------------------------------------------------------------------------
        protected bool RebootTargetByPowerButton( SigloHelper.CommodityExecutor.Context executor,
            int holdTime = 10, int intervalSeconds = 30, int waitBootupSeconds = 15,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = "" )
        {
            var boot = false;
            var shutdown = false;
            var options = new System.Collections.Generic.List<string> { $"--hold-time {holdTime}" };
            if ( shutdown = executor.ControlTarget( "press-power-button", options ) )
            {
                Log.WriteLine( $"{methodName}: waiting {intervalSeconds} seconds, after press-power-button hold-time {holdTime}." );
                System.Threading.Thread.Sleep( intervalSeconds * 1000 );    // 電源再投入インターバル
                if ( boot = executor.ControlTarget( "press-power-button" ) )
                {
                    Log.WriteLine( $"{methodName}: waiting {waitBootupSeconds} seconds, after press-power-button." );
                    System.Threading.Thread.Sleep( waitBootupSeconds * 1000 );  // 起動シーケンス待ち
                }
            }
            return shutdown && boot;
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// ControlTarget power コマンド経由での電源遮断 / 電源投入による再起動。
        /// </summary>
        /// <param name="executor">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="intervalSeconds">電源遮断後のスレッドスリープ待機時間( 秒数 )</param>
        /// <param name="waitBootupSeconds">電源投入後のスレッドスリープ待機時間( 秒数 )</param>
        /// <param name="methodName">
        /// ログに出力する際のメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// </param>
        /// <returns>成否。 true は成功。</returns>
        //!----------------------------------------------------------------------------
        protected bool RebootTargetByPowerCommand( SigloHelper.CommodityExecutor.Context executor,
            int intervalSeconds = 30, int waitBootupSeconds = 15,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = "" )
        {
            var boot = false;
            var shutdown = false;
            if ( shutdown = executor.ControlTarget( "power-off" ) )
            {
                Log.WriteLine( $"{methodName}: waiting 30 seconds, after power-off." );
                System.Threading.Thread.Sleep( intervalSeconds * 1000 );    // 電源再投入インターバル
                if ( boot = executor.ControlTarget( "power-on" ) )
                {
                    Log.WriteLine( $"{methodName}: waiting 15 seconds, after power-on." );
                    System.Threading.Thread.Sleep( waitBootupSeconds * 1000 );  // 起動シーケンス待ち
                }
            }
            return shutdown && boot;
        }
    }
}
