﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Text.RegularExpressions;

namespace CsTestAssistants
{
    /// <summary>
    /// SIGLO プラットフォーム構造を対象としたヘルパです。
    /// NX 特化を目的とはしていません。
    /// </summary>
    public static class SigloHelper
    {
        //!----------------------------------------------------------------------------
        /// <summary>
        /// SDKコンフィグ関連
        /// </summary>
        //!----------------------------------------------------------------------------
        public static class Configuration
        {
            /// <summary>
            /// 定数
            /// </summary>
            public static class Constants
            {
                public static class BuildType
                {
                    public const string Debug   = "Debug";
                    public const string Develop = "Develop";
                    public const string Release = "Release";
                }

                public static class Platform
                {
                    public const string NX64 = "NX-NXFP2-a64";
                    public const string NX32 = "NX-NXFP2-a32";
                }

                public static class Architecture
                {
                    public const string A64 = "arch64.lp64";
                    public const string A32 = "arch32.ilp32";
                }
            }
            /// <summary>
            /// デフォルトプラットフォーム
            /// </summary>
            public const string DefaultPlatform = Constants.Platform.NX64;

            /// <summary>
            /// デフォルトビルドタイプ
            /// </summary>
            public const string DefaultBuildType = Constants.BuildType.Develop;

            // 辞書
            private static readonly List<string> PlatformKeywordNX64 = new List<string>
            {
                Constants.Platform.NX64.ToLower(), "nxfp2-a64", "nx64"
            };
            private static readonly List<string> PlatformKeywordNX32 = new List<string>
            {
                Constants.Platform.NX32.ToLower(), "nxfp2-a32", "nx32"
            };
            private static readonly Dictionary<string, string> BuildTypeKeyword = new Dictionary<string, string>
            {
                { Constants.BuildType.Debug.ToLower(),      Constants.BuildType.Debug },
                { Constants.BuildType.Develop.ToLower(),    Constants.BuildType.Develop },
                { Constants.BuildType.Release.ToLower(),    Constants.BuildType.Release },
            };

            /// <summary>
            /// 指定文字列を類似該当するプラットフォーム定数に補正します.
            /// 類似該当する定数がなければ System.IO.InvalidDataException 例外を発生させます.
            /// </summary>
            /// <param name="platform">定数に類似する任意のプラットフォーム文字列</param>
            /// <returns>プラットフォーム定数</returns>
            /// <see cref="Constants.Platform"/>
            /// <exception cref="System.IO.InvalidDataException">類似該当する定数が見つからない</exception>
            public static string CorrectPlatform( string platform )
            {
                string key = platform.ToLower();
                if ( PlatformKeywordNX64.Contains( key ) )
                {
                    return Constants.Platform.NX64;
                }
                else if ( PlatformKeywordNX32.Contains( key ) )
                {
                    return Constants.Platform.NX32;
                }
                throw new System.IO.InvalidDataException( string.Format( "Unavailable platform were specified : {0}", platform ) );
            }

            /// <summary>
            /// 指定文字列を類似該当するアーキテクト定数に補正します.
            /// 類似該当する定数がなければ System.IO.InvalidDataException 例外を発生させます.
            /// </summary>
            /// <param name="platform">定数に類似する任意のプラットフォーム文字列</param>
            /// <returns>アーキテクト定数</returns>
            /// <see cref="Constants.Architecture"/>
            /// <exception cref="System.IO.InvalidDataException">類似該当する定数が見つからない</exception>
            public static string GetArchitectSignature( string platform )
            {
                string key = platform.ToLower();
                if ( PlatformKeywordNX64.Contains( key ) )
                {
                    return Constants.Architecture.A64;
                }
                else if ( PlatformKeywordNX32.Contains( key ) )
                {
                    return Constants.Architecture.A32;
                }
                throw new System.IO.InvalidDataException( string.Format( "Unavailable platform were specified : {0}", platform ) );
            }

            /// <summary>
            /// 指定文字列を類似該当するビルドタイプ定数に補正します.
            /// 類似該当する定数がなければ System.IO.InvalidDataException 例外を発生させます.
            /// </summary>
            /// <param name="type">定数に類似する任意のビルドタイプ文字列</param>
            /// <returns>ビルドタイプ定数</returns>
            /// <see cref="Constants.BuildType"/>
            /// <exception cref="System.IO.InvalidDataException">類似該当する定数が見つからない</exception>
            public static string CorrectBuildType( string type )
            {
                string value;
                if ( BuildTypeKeyword.TryGetValue( type.ToLower(), out value ) )
                {
                    return value;
                }
                throw new System.IO.InvalidDataException( string.Format( "Unavailable build type were specified : {0}", type ) );
            }

            //!----------------------------------------------------------------------------
            /// <summary>
            /// Commodity configuration context.
            /// platform, buildtype, targetname などのコンテキストプロパティとして管理するインスタンス化可能なクラス.
            /// </summary>
            //!----------------------------------------------------------------------------
            public class Context
            {
                public string TargetBuildType { get; }
                public string TargetPlatform { get; }
                public string TargetName { get; }

                public Context( string targetName, string platform = DefaultPlatform, string buildType = DefaultBuildType )
                {
                    if ( string.IsNullOrEmpty( targetName ) )
                    {
                        throw new System.IO.InvalidDataException( "Could not construct the new instance by the specified argument because the targetName object was null or empty." );
                    }
                    TargetName = targetName;
                    TargetPlatform = CorrectPlatform( platform );
                    TargetBuildType = CorrectBuildType( buildType );
                }

                public Context( Context other )
                {
                    if ( null == other )
                    {
                        throw new System.IO.InvalidDataException( "Could not construct the new instance by the specified argument because the null object." );
                    }
                    TargetName = other.TargetName;
                    TargetPlatform = other.TargetPlatform;
                    TargetBuildType = other.TargetBuildType;
                }

                public string GetOutputTestDirectory( string module )
                {
                    // ${SIGLO_ROOT}\Tests\Outputs\${PLATFORM}\Tests\${MODULE_NAME}\${BUILD_TYPE} 固定。
                    return System.IO.Path.Combine( Path.GetTestsRoot(), "Outputs", TargetPlatform, "Tests", module, TargetBuildType );
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// Sigloパス関連
        /// </summary>
        //!----------------------------------------------------------------------------
        public static class Path
        {
            private const string SigloRootMarkFileName = "SigloRootMark";
            private const string RootPathPrograms = "Programs";
            private const string RootPathTests = "Tests";
            private const string RootPathTools = "Tools";

            private static Dictionary<string, string> CachedFoundSigloRoots = new Dictionary<string, string>();

            /// <summary>
            /// 指定基準パスから SigloRoot パスを検索します。
            /// </summary>
            /// <remarks>
            /// 指定基準パスから上位層に向かって SigloRootMark を検索します。
            /// キャッシュは影響しません。
            /// </remarks>
            /// <param name="sourcePath">検索基準パス</param>
            /// <returns></returns>
            /// <exception cref="System.IO.FileNotFoundException">SigloRootMarkが見つからなかった</exception>
            public static string FindSigloRoot( string sourcePath )
            {
                string currentDirectory = sourcePath;
                string rootDirectory = System.IO.Path.GetPathRoot( currentDirectory );

                while ( rootDirectory != currentDirectory )
                {
                    string sigloRootMarkFilePath = System.IO.Path.Combine( currentDirectory, SigloRootMarkFileName );
                    if ( System.IO.File.Exists( sigloRootMarkFilePath ) )
                    {
                        return currentDirectory;
                    }
                    currentDirectory = System.IO.Path.GetFullPath( System.IO.Path.Combine( currentDirectory, ".." ) );
                }
                throw new System.IO.FileNotFoundException( string.Format( "Could not found the `SigloRootMark` marker file, From the source path : {0}", sourcePath ) );
            }

            public static void ClearCachedSigloRoots( string key = null )
            {
                if ( string.IsNullOrEmpty( key ) )
                {
                    CachedFoundSigloRoots.Clear();
                }
                else if ( CachedFoundSigloRoots.ContainsKey( key ) )
                {
                    CachedFoundSigloRoots.Remove( key );
                }
            }

            public static string GetSigloRoot( string key = null )
            {
                string value = string.Empty;
                key = ( string.IsNullOrEmpty( key ) ) ? System.IO.Path.GetDirectoryName( System.Reflection.Assembly.GetExecutingAssembly().Location ) : key;
                if ( false == CachedFoundSigloRoots.TryGetValue( key, out value ) )
                {
                    value = FindSigloRoot( key );
                    CachedFoundSigloRoots.Add( key, value );
                }
                return value;
            }

            public static string GetToolsRoot()
            {
                return System.IO.Path.Combine( GetSigloRoot(), RootPathTools );
            }

            public static string GetTestsRoot()
            {
                return System.IO.Path.Combine( GetSigloRoot(), RootPathTests );
            }

            public static string GetProgramsRoot()
            {
                return System.IO.Path.Combine( GetSigloRoot(), RootPathPrograms );
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// Siglo Tools 系パス関連
        /// </summary>
        //!----------------------------------------------------------------------------
        public static class ToolPath
        {
            public static string FindTools( string path )
            {
                string expect = path;
                if ( false == string.IsNullOrEmpty( path ) && System.IO.File.Exists( expect = System.IO.Path.Combine( Path.GetToolsRoot(), path ) ) )
                {
                    return expect;
                }
                throw new System.IO.FileNotFoundException( string.Format( "Could not found the `{0}`, Expect path : {1}", path, expect ) );
            }

            public static string FindControlTarget()
            {
                return FindTools( System.IO.Path.Combine( "CommandLineTools", "ControlTarget.exe" ) );
            }

            public static string FindRunOnTarget()
            {
                return FindTools( System.IO.Path.Combine( "CommandLineTools", "RunOnTarget.exe" ) );
            }

            public static string FindRunOnTargetPrivate()
            {
                return FindTools( System.IO.Path.Combine( "CommandLineTools", "RunOnTargetPrivate.exe" ) );
            }

            public static string FindAuthoringTool()
            {
                return FindTools( System.IO.Path.Combine( "CommandLineTools", "AuthoringTool", "AuthoringTool.exe" ) );
            }

            public static string FindAuthoringOceanKeyConfigXml()
            {
                return FindTools( System.IO.Path.Combine( "CommandLineTools", "AuthoringTool", "AuthoringTool.ocean.keyconfig.xml" ) );
            }

            public static string FindProgramEris( string module, string platform, string buildtype, string category = "TargetTools" )
            {
                if ( false == string.IsNullOrEmpty( module ) && false == string.IsNullOrEmpty( category ) )
                {
                    string form = Configuration.CorrectPlatform( platform );
                    string type = Configuration.CorrectBuildType( buildtype );
                    string expect = System.IO.Path.Combine( Path.GetProgramsRoot(), "Eris", "Outputs", form, category, module, type, module ) + ".nsp";
                    if ( System.IO.File.Exists( expect ) )
                    {
                        return expect;
                    }
                    throw new System.IO.FileNotFoundException( string.Format( "Could not found the module of `{0}`, Expect path : {1}", module, expect ) );
                }
                throw new System.IO.InvalidDataException( string.Format( "Unavailable the module name were specified : {0}", module ) );
            }

            public static string FindDevMenuCommand( string platform, string buildtype )
            {
                return FindProgramEris( "DevMenuCommand", platform, buildtype );
            }

            public static string FindDevMenuCommandSystem( string platform, string buildtype )
            {
                return FindProgramEris( "DevMenuCommandSystem", platform, buildtype );
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// Siglo Tools Commodity Executors.
        /// 主に使うのは以下を想定。
        ///     ResetTarget
        ///     RunDevMenuCommand
        ///     RunDevMenuCommandSystem
        ///
        ///     以下は、汎用メソッド。
        ///     RunOnTarget
        ///     ControlTarget
        /// </summary>
        //!----------------------------------------------------------------------------
        public static class CommodityExecutor
        {
            public enum BootType : byte
            {
                List,       // from HOST
                Nand,       // from NAND( embedded )
            }

            public enum SignedType : byte
            {
                Signed,
                Unsigned,
            }

            public enum ConnectionType : byte
            {
                Hb,         // HostBridge
                Usb,        // USB
            }

            /// <summary>
            /// RunOnTarget default time out, that is 10 minute.
            /// Specified by unit of seconds.
            /// </summary>
            public const uint TimeoutOfStandard = 600;

            /// <summary>
            /// デフォルトの失敗パターン
            /// </summary>
            private const string RunOnTargetDefaultFailExitOption =
                " --pattern-failure-exit \"\\bAssertion Failure:\""      +
                " --pattern-failure-exit \"\\bPrecondition not met:\""   +
                " --pattern-failure-exit \"\\bUnexpected Default:\""     +
                " --pattern-failure-exit \"\\bAbort:\""
            ;

            public static bool RunDevMenuCommand( string targetName, string args,
                string platform = Configuration.DefaultPlatform, string buildtype = Configuration.DefaultBuildType,
                uint timeout = TimeoutOfStandard, OutputStreams output = null, string successPattern = null )
            {
                string nsp = ToolPath.FindProgramEris( "DevMenuCommand", platform, buildtype );
                string options = "--pattern-failure-exit \\[FAILURE\\]";
                if(successPattern != null)
                {
                    options += " --pattern-not-found-failure " + successPattern;
                }
                return RunOnTarget( targetName, nsp, options, args, timeout, output );
            }

            public static bool RunDevMenuCommandSystem( string targetName, string args,
                string platform = Configuration.DefaultPlatform, string buildtype = Configuration.DefaultBuildType,
                uint timeout = TimeoutOfStandard, OutputStreams output = null, string successPattern = null, string failurePattern = null )
            {
                string nsp = ToolPath.FindProgramEris( "DevMenuCommandSystem", platform, buildtype );
                string options = "--pattern-failure-exit \\[FAILURE\\]";
                if (successPattern != null)
                {
                    options += " --pattern-not-found-failure " + successPattern;
                }
                if (failurePattern != null)
                {
                    options += " --pattern-failure-exit " + failurePattern;
                }
                return RunOnTarget( targetName, nsp, options, args, timeout, output );
            }

            public static bool RunDevMenuCommandSystemCustomOption( string targetName, string args, List<string> appendixOptions,
                string platform = Configuration.DefaultPlatform, string buildtype = Configuration.DefaultBuildType,
                uint timeout = TimeoutOfStandard, OutputStreams output = null )
            {
                string nsp = ToolPath.FindProgramEris( "DevMenuCommandSystem", platform, buildtype );
                string options = "--pattern-failure-exit \\[FAILURE\\]";
                if ( null != appendixOptions && appendixOptions.Count > 0 )
                {
                    options += " " + string.Join( " ", appendixOptions );
                }
                return RunOnTarget( targetName, nsp, options, args, timeout, output );
            }

            public static bool RunDevMenuCommandSystemUseOnlyFilter( string targetName, string args,
                string platform = Configuration.DefaultPlatform, string buildtype = Configuration.DefaultBuildType,
                uint timeout = TimeoutOfStandard, OutputStreams output = null, string successPattern = null, string failurePattern = null )
            {
                string nsp = ToolPath.FindProgramEris( "DevMenuCommandSystem", platform, buildtype );
                string options = "--pattern-failure-exit \\[FAILURE\\]";
                if ( successPattern != null )
                {
                    options += " --pattern-not-found-failure " + successPattern;
                }
                if ( failurePattern != null )
                {
                    options += " --pattern-failure-exit " + failurePattern;
                }
                options += " --filter \"FileName =~ /DevMenuCommand/\"";
                return RunOnTarget( targetName, nsp, options, args, timeout, output );
            }

            public static bool RunDevMenuCommandSystemSuppressAutoKill(string targetName, string args,
                string platform = Configuration.DefaultPlatform, string buildtype = Configuration.DefaultBuildType,
                uint timeout = TimeoutOfStandard, OutputStreams output = null, string successPattern = null)
            {
                string nsp = ToolPath.FindProgramEris("DevMenuCommandSystem", platform, buildtype);
                string options = "--pattern-failure-exit \\[FAILURE\\]";
                options += " --suppress-auto-kill ";
                if (successPattern != null)
                {
                    options += " --pattern-not-found-failure " + successPattern;
                }
                return RunOnTarget(targetName, nsp, options, args, timeout, output);
            }

            public static bool RunSettingsManager( string targetName, string args,
                string platform = Configuration.DefaultPlatform, string buildtype = Configuration.DefaultBuildType,
                uint timeout = TimeoutOfStandard, OutputStreams output = null )
            {
                string nsp = ToolPath.FindProgramEris( "SettingsManager", platform, buildtype );
                string options = "--pattern-failure-exit \\[FAILURE\\]";
                return RunOnTarget( targetName, nsp, options, args, timeout, output );
            }

            public static bool ResetTarget( string targetName, int waitBootSequenceMilliseconds = 20 * 1000, OutputStreams output = null )
            {
                bool result;
                if ( ( result = ControlTarget( "reset", "-v -t " + targetName, output ) ) && waitBootSequenceMilliseconds > 0 )
                {
                    // Interval for wait a boot sequence complete.
                    System.Threading.Thread.Sleep( waitBootSequenceMilliseconds );
                }
                return result;
            }

            public static bool RunOnTarget( string targetName, string nsp, string options = null, string argsOfNsp = null, uint timeout = TimeoutOfStandard, OutputStreams output = null )
            {
                Log.WriteLineAsIs( System.Environment.NewLine );

                if ( string.IsNullOrEmpty( targetName ) || string.IsNullOrEmpty( nsp ) )
                {
                    Log.WriteLine( "===== RunOnTarget : Could not execution, because invalid arguments [ target:{0}, nsp:{1} ] =====", targetName, nsp );
                    return false;
                }

                Log.WriteLine( "===== Start of RunOnTarget : {0} =====", nsp );

                string args = " -t " + targetName + " " + RunOnTargetDefaultFailExitOption;
                if ( false == string.IsNullOrEmpty( options ) )
                {
                    args += " " + options;
                }
                args += " --failure-timeout " + timeout + " -v " + nsp;
                if ( false == string.IsNullOrEmpty( argsOfNsp ) )
                {
                    args += " -- " + argsOfNsp;
                }
                int result = CommandLineExecutor.ExecuteOnProcess( ToolPath.FindRunOnTarget(), args, output );

                Log.WriteLine( "===== Finish of RunOnTarget : {0} =====", nsp );
                return ( 0 == result );
            }

            public static bool RunOnTargetPrivate( string targetName, string nsp, string options = null, string argsOfNsp = null, uint timeout = TimeoutOfStandard, OutputStreams output = null )
            {
                Log.WriteLineAsIs( System.Environment.NewLine );

                if ( string.IsNullOrEmpty( targetName ) || string.IsNullOrEmpty( nsp ) )
                {
                    Log.WriteLine( "===== RunOnTargetPrivate : Could not execution, because invalid arguments [ target:{0}, nsp:{1} ] =====", targetName, nsp );
                    return false;
                }

                Log.WriteLine( "===== Start of RunOnTargetPrivate : {0} =====", nsp );

                string args = $"run {nsp} -t {targetName} {RunOnTargetDefaultFailExitOption}";
                if ( false == string.IsNullOrEmpty( options ) )
                {
                    args += " " + options;
                }
                args += $" --failure-timeout {timeout} -v";
                if ( false == string.IsNullOrEmpty( argsOfNsp ) )
                {
                    args += " -- " + argsOfNsp;
                }
                int result = CommandLineExecutor.ExecuteOnProcess( ToolPath.FindRunOnTargetPrivate(), args, output );

                Log.WriteLine( "===== Finish of RunOnTarget : {0} =====", nsp );
                return ( 0 == result );
            }

            public static bool ControlTarget( string command, string options, OutputStreams output = null )
            {
                Log.WriteLineAsIs( System.Environment.NewLine );

                if ( string.IsNullOrEmpty( command ) )
                {
                    Log.WriteLine( "===== ControlTarget : Could not execution, because invalid command. =====" );
                    return false;
                }

                Log.WriteLine( "===== Start of ControlTarget : {0} =====", command );

                string args = " " + command;
                if ( false == string.IsNullOrEmpty( options ) )
                {
                    args += " " + options;
                }
                int result = CommandLineExecutor.ExecuteOnProcess( ToolPath.FindControlTarget(), args, output );

                Log.WriteLine( "===== Finish of ControlTarget : {0} =====", command );
                return ( 0 == result );
            }

            //!----------------------------------------------------------------------------
            /// <summary>
            /// Siglo Tools Commodity Executors with context.
            /// platform, buildtype, targetname などをコンテキストプロパティとして管理するインスタンス化可能なクラス.
            /// </summary>
            //!----------------------------------------------------------------------------
            public class Context : Configuration.Context
            {
                public const int DefaultOutputStreamInitialCapacity = 64 * 1024;

                public OutputStreams OutputStream { get; }

                public PowerShellExecutor PowerShellInvoker { get; }

                public Context( string targetName, string platform = Configuration.DefaultPlatform, string buildType = Configuration.DefaultBuildType,
                    int outputStreamInitialCapacity = DefaultOutputStreamInitialCapacity )
                    : base( targetName, platform, buildType )
                {
                    OutputStream = new OutputStreams( outputStreamInitialCapacity, outputStreamInitialCapacity );
                    PowerShellInvoker = new PowerShellExecutor( 2.0f, PowerShellExecutor.ExecutionPolicy.RemoteSigned );
                }

                public Context( Configuration.Context other, int outputStreamInitialCapacity = DefaultOutputStreamInitialCapacity )
                    : base( other )
                {
                    OutputStream = new OutputStreams( outputStreamInitialCapacity, outputStreamInitialCapacity );
                    PowerShellInvoker = new PowerShellExecutor( 2.0f, PowerShellExecutor.ExecutionPolicy.RemoteSigned );
                }

                public bool RunDevMenuCommand( string args, uint timeout = TimeoutOfStandard )
                {
                    return CommodityExecutor.RunDevMenuCommand( TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream );
                }

                public bool RunDevMenuCommand( string args, string successPattern, uint timeout = TimeoutOfStandard )
                {
                    return CommodityExecutor.RunDevMenuCommand( TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream, successPattern );
                }

                public bool RunDevMenuCommandSystem( string args, uint timeout = TimeoutOfStandard )
                {
                    return CommodityExecutor.RunDevMenuCommandSystem( TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream );
                }

                public bool RunDevMenuCommandSystem( string args, string successPattern, uint timeout = TimeoutOfStandard )
                {
                    return CommodityExecutor.RunDevMenuCommandSystem( TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream, successPattern );
                }

                public bool RunDevMenuCommandSystem(string args, string successPattern, string failurePattern, uint timeout = TimeoutOfStandard)
                {
                    return CommodityExecutor.RunDevMenuCommandSystem(TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream, successPattern, failurePattern);
                }

                public bool RunDevMenuCommandSystemUseOnlyFilter( string args, uint timeout = TimeoutOfStandard )
                {
                    return CommodityExecutor.RunDevMenuCommandSystemUseOnlyFilter( TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream );
                }

                public bool RunDevMenuCommandSystemUseOnlyFilter( string args, string successPattern, string failurePattern, uint timeout = TimeoutOfStandard )
                {
                    return CommodityExecutor.RunDevMenuCommandSystemUseOnlyFilter( TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream, successPattern, failurePattern );
                }

                public bool RunDevMenuCommandSystemSuppressAutoKill(string args, uint timeout = TimeoutOfStandard)
                {
                    return CommodityExecutor.RunDevMenuCommandSystemSuppressAutoKill(TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream);
                }

                public bool RunDevMenuCommandSystemSuppressAutoKill(string args, string successPattern, uint timeout = TimeoutOfStandard)
                {
                    return CommodityExecutor.RunDevMenuCommandSystemSuppressAutoKill(TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream, successPattern);
                }

                public bool RunDevMenuCommandSystemCustomOption( string args, List<string> appendOptions, uint timeout = TimeoutOfStandard )
                {
                    return CommodityExecutor.RunDevMenuCommandSystemCustomOption( TargetName, args, appendOptions, TargetPlatform, TargetBuildType, timeout, OutputStream );
                }

                public bool RunSettingsManager( string args, uint timeout = TimeoutOfStandard )
                {
                    return CommodityExecutor.RunSettingsManager( TargetName, args, TargetPlatform, TargetBuildType, timeout, OutputStream );
                }

                public bool ResetTarget( int waitBootSequenceMilliseconds = 20 * 1000 )
                {
                    return CommodityExecutor.ResetTarget( TargetName, waitBootSequenceMilliseconds, OutputStream );
                }

                public bool RunOnTarget( string nsp, string options = null, string argsOfNsp = null, uint timeout = TimeoutOfStandard )
                {
                    return CommodityExecutor.RunOnTarget( TargetName, nsp, options, argsOfNsp, timeout, OutputStream );
                }

                public bool ControlTarget( string command, List<string> options = null )
                {
                    System.Text.StringBuilder b = new System.Text.StringBuilder( 128 );
                    b.Append( "-v -t " ).Append( TargetName );
                    if ( null != options && options.Count > 0 )
                    {
                        b.Append( ' ' ).Append( string.Join( " ", options ) );
                    }
                    return CommodityExecutor.ControlTarget( command, b.ToString(), OutputStream );
                }

                public bool ControlTargetNoTarget(string command, List<string> options = null)
                {
                    System.Text.StringBuilder b = new System.Text.StringBuilder(128);
                    if (null != options && options.Count > 0)
                    {
                        b.Append(' ').Append(string.Join(" ", options));
                    }
                    return CommodityExecutor.ControlTarget(command, b.ToString(), OutputStream);
                }

                public bool InvokeUpdateKernelOnSDEV( BootType boot = BootType.Nand, ConnectionType connection = ConnectionType.Hb, SignedType signed = SignedType.Unsigned )
                {
                    string command = System.IO.Path.Combine( Path.GetSigloRoot(), "Integrate", "Scripts", "Invoke-UpdateKernel.ps1" );
                    FileHelper.TestExistsFile( command );
                    List<string> arguments = new List<string>( 5 )
                    {
                        string.Format( "-Platform {0}", TargetPlatform ),
                        string.Format( "-BuildType {0}", TargetBuildType ),
                        string.Format( "-BootType {0}", boot ),
                        string.Format( "-ConnectionType {0}", connection ),
                        string.Format( "-SignedType {0}", signed )
                    };
                    return ( 0 == PowerShellInvoker.Execute( command, arguments, OutputStream ) );
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// DevMenuCommand などのコマンド実行ログ文字列とのパターンマッチング支援.
        /// </summary>
        //!----------------------------------------------------------------------------
        public static class ResultMatcher
        {
            /// <summary>
            /// 指定の実行結果出力文字列から接頭辞 "[target] " を削除します。
            /// </summary>
            /// <param name="source"></param>
            /// <returns></returns>
            public static string TrimPrefixTarget( string source )
            {
                return Regex.Replace( source, @"^\[target\] ", "", RegexOptions.Multiline );
            }

            /// <summary>
            /// 指定の実行結果出力文字列から接頭辞 "[trace] " を削除します。
            /// </summary>
            /// <param name="source"></param>
            /// <returns></returns>
            public static string TrimPrefixTrace( string source )
            {
                return Regex.Replace( source, @"^\[trace\] ", "", RegexOptions.Multiline );
            }


            /// <summary>
            /// コンテンツ情報マッチング, 汎用パラメータコンテナ.
            /// </summary>
            public class ContentValue : ContentMeta.Contexture.Basic
            {
                public static class Storage
                {
                    public const string Any = @"\w+";
                    public const string SdCard = @"SdCard";
                    public const string GameCard = @"GameCard";
                    public const string BuiltInAny = @"BuiltIn\w+";
                    public const string BuiltInUser = @"BuiltInUser";
                    public const string BuiltInSystem = @"BuiltInSystem";
                    public const string None = @"None";
                }

                /// <summary>
                /// ns_ApplicationContentMetaApi.h, enum ContentMetaRightsCheck.
                /// </summary>
                public static class RightsCheck
                {
                    public const string Any = @"\w+";                               //!< なんでも。
                    public const string NotChecked = @"NotChecked";                 //!< 権利がチェックされていません。
                    public const string NotNeeded = @"NotNeeded";                   //!< 権利が不要なコンテンツです。
                    public const string CommonRights = @"CommonRights";             //!< 共通権利を所有しています。
                    public const string PersonalizedRights = @"PersonalizedRights"; //!< 固有権利を所有しています。
                    public const string NoRights = @"NoRights";                     //!< 権利を所有していません。
                }

                /// <summary>
                /// ncm::ContentInstallType
                /// </summary>
                public static class InstallType
                {
                    public const string Full = "Full";
                    public const string Unknown = "Unknown";
                    public const string FragmentOnly = "FragmentOnly";
                }

                /// <summary>
                /// 期待するインストール済対象のストレージタイプ正規表現パターン
                /// </summary>
                public string StoragePattern { get; }
                public string RightsCheckPattern { get; }
                public string InstallTypePattern { get; }

                /// <summary>
                /// コンストラクタ
                /// </summary>
                /// <param name="id">期待する対象のコンテンツメタID</param>
                /// <param name="version">期待する対象のコンテンツメタバージョン</param>
                /// <param name="type">期待する対象のコンテンツメタタイプ</param>
                /// <param name="storagePattern">期待するインストール済対象のストレージタイプの正規表現パターン</param>
                /// <param name="rightsCheckPattern">期待する対象のコンテンツの権利状態の正規表現パターン</param>
                public ContentValue( ID64 id, int version, ContentMeta.Type type, string storagePattern = Storage.Any, string rightsCheckPattern = null, string installTypePattern = null ) : base( id, version, type )
                {
                    StoragePattern = storagePattern;
                    RightsCheckPattern = rightsCheckPattern;
                    InstallTypePattern = installTypePattern;
                }

                /// <summary>
                /// コンストラクタ
                /// </summary>
                /// <param name="meta">期待する対象のコンテンツメタ情報</param>
                /// <param name="storagePattern">期待するインストール済対象のストレージタイプの正規表現パターン</param>
                /// <param name="rightsCheckPattern">期待する対象のコンテンツの権利状態の正規表現パターン</param>
                public ContentValue( ContentMeta.Contexture.Basic meta, string storagePattern = Storage.Any, string rightsCheckPattern = null, string installTypePattern = null ) : base( meta )
                {
                    StoragePattern = storagePattern;
                    RightsCheckPattern = rightsCheckPattern;
                    InstallTypePattern = installTypePattern;
                }

                /// <summary>
                /// override ToString
                /// </summary>
                /// <returns></returns>
                public override string ToString()
                {
                    return ( string.IsNullOrEmpty( RightsCheckPattern ) )
                        ? string.Format( "{0}, [ Storage={1} ]", base.ToString(), StoragePattern )
                        : string.Format( "{0}, [ Storage={1}, RightsCheck={2} ]", base.ToString(), StoragePattern, RightsCheckPattern );
                }
            }

            /// <summary>
            /// DevMenuCommand, DevMenuCommandSystem "application" コマンド
            /// </summary>
            public static class Application
            {
                /// <summary>
                /// application list-record [id] --installed
                /// </summary>
                public static class ListRecordInstalled
                {
                    private const string MatchPattern = @"""id"": ""0x{0}"",\s*""version"": {1},\s*""type"": ""{2}"",\s*""storage"": ""{3}""";
                    private const string MatchPatternWithInstallType = @"""id"": ""0x{0}"",\s*""version"": {1},\s*""type"": ""{2}"",\s*""storage"": ""{3}"",\s*""installType"": ""{4}""";

                    /// <summary>
                    /// インストールコンテンツ検証用マッチング情報
                    /// </summary>
                    public class MatchingValue : ContentValue
                    {
                        /// <summary>
                        /// コンストラクタ
                        /// </summary>
                        /// <param name="id">期待するインストール済対象のコンテンツメタID</param>
                        /// <param name="version">期待するインストール済対象のコンテンツメタバージョン</param>
                        /// <param name="type">期待するインストール済対象のコンテンツメタタイプ</param>
                        /// <param name="storagePattern">期待するインストール済対象のストレージタイプの正規表現パターン</param>
                        public MatchingValue( ID64 id, int version, ContentMeta.Type type, string storagePattern = Storage.Any, string installTypePattern = null ) : base( id, version, type, storagePattern, null, installTypePattern )
                        {
                        }

                        /// <summary>
                        /// コンストラクタ
                        /// </summary>
                        /// <param name="meta">期待するインストール済対象のコンテンツメタ情報</param>
                        /// <param name="storagePattern">期待するインストール済対象のストレージタイプの正規表現パターン</param>
                        public MatchingValue( ContentMeta.Contexture.Basic meta, string storagePattern = Storage.Any, string installTypePattern = null ) : base( meta, storagePattern, null, installTypePattern )
                        {
                        }
                    }

                    /// <summary>
                    /// 期待するインストールメタ情報値とのマッチング
                    /// </summary>
                    /// <param name="output">マッチング対象ストリーム</param>
                    /// <param name="id">期待するインストール済対象のコンテンツメタID</param>
                    /// <param name="version">期待するインストール済対象のコンテンツメタバージョン</param>
                    /// <param name="type">期待するインストール済対象のコンテンツメタタイプ</param>
                    /// <param name="storagePattern">期待するインストール済対象のストレージタイプの正規表現パターン</param>
                    /// <returns>期待するインストールメタ情報が見つかった場合は、true.</returns>
                    public static Match Find( string output, ID64 id, int version, ContentMeta.Type type, string storagePattern )
                    {
                        output = TrimPrefixTarget( output );
                        return Regex.Match( output, string.Format( MatchPattern, id, version, type, storagePattern ) );
                    }

                    /// <summary>
                    /// 対象値の全マッチング
                    /// </summary>
                    /// <param name="output">マッチング対象ストリーム</param>
                    /// <param name="values">マッチング値配列</param>
                    /// <returns>見つからなかった対象値の数を返します。 0 の場合全て見つかった事を示します。</returns>
                    public static int Find( string output, params ContentValue[] values )
                    {
                        output = TrimPrefixTarget( output );
                        int unmatchedCount = 0;
                        foreach ( var value in values )
                        {
                            var pattern = ( string.IsNullOrEmpty( value.InstallTypePattern ) )
                                ? string.Format( MatchPattern, value.Identifier, value.Version, value.Type, value.StoragePattern )
                                : string.Format( MatchPatternWithInstallType, value.Identifier, value.Version, value.Type, value.StoragePattern, value.InstallTypePattern );
                            var matched = Regex.Match( output, pattern );
                            if ( false == matched.Success )
                            {
                                Log.WriteLine( $"Not found an installed contents => {value.ToString()}." );
                                ++unmatchedCount;
                            }
                            else
                            {
                                Log.WriteLine( $"Found an installed contents => {value.ToString()}.{System.Environment.NewLine}It be matched value => [ {matched.Value} ]" );
                            }
                        }
                        return unmatchedCount;
                    }
                }

                /// <summary>
                /// application content-meta-status --all --check-rights
                /// application content-meta-status [application_id] --check-rights
                /// </summary>
                public static class ContentMetaStatusCheckRights
                {
                    private const string MatchPattern = @"""id"": ""0x{0}"",\s*""type"": ""{1}"",\s*""version"": {2},\s*""installedStorage"": ""{3}"",\s*""rightsCheck"": ""{4}""";

                    /// <summary>
                    /// 対象値の全マッチング
                    /// </summary>
                    /// <param name="output">マッチング対象ストリーム</param>
                    /// <param name="values">マッチング値配列</param>
                    /// <returns>見つからなかった対象値の数を返します。 0 の場合全て見つかった事を示します。</returns>
                    public static int Find( string output, params ContentValue[] values )
                    {
                        output = TrimPrefixTarget( output );
                        int unmatchedCount = 0;
                        foreach ( var value in values )
                        {
                            var matched = Regex.Match( output, string.Format( MatchPattern, value.Identifier, value.Type, value.Version, value.StoragePattern, value.RightsCheckPattern ) );
                            if ( false == matched.Success )
                            {
                                Log.WriteLine( $"Not found an installed contents => {value.ToString()}." );
                                ++unmatchedCount;
                            }
                            else
                            {
                                Log.WriteLine( $"Found an installed contents => {value.ToString()}.{System.Environment.NewLine}It be matched value => [ {matched.Value} ]" );
                            }
                        }
                        return unmatchedCount;
                    }
                }

                /// <summary>
                /// application list-view
                /// </summary>
                public static class ListView
                {
                    /// <summary>
                    /// list-view 出力JSONキー
                    /// </summary>
                    public enum Property : byte
                    {
                        hasRecord,
                        hasMainRecord,
                        hasPatchRecord,
                        hasAddOnContentRecord,
                        hasMainInstallRecord,
                        isDownloading,
                        isGameCard,
                        hasGameCardEntity,
                        isLaunchable,
                        hasAllEntity,
                        hasMainEntity,
                        hasAllAddOnContentEntity,
                        hasAnyAddOnContentEntity,
                        hasPatchEntity,
                        maybeCorrupted,
                        isWaitingCommit,
                        isWaitingApplicationCommit,
                        isWaitingAocCommit,
                        isWaitingPatchInstall,
                        isAutoDeleteDisabled,
                        isApplyingDelta,
                        isCleanupAddOnContentWithNoRightsRecommended,
                        isSystemUpdateRequiredToCommit
                    }

                    /// <summary>
                    /// プロパティコレクション
                    /// </summary>
                    public class ExpectValues : Dictionary<Property, bool>
                    {
                    }

                    /// <summary>
                    /// ID/Version から該当ブロックを抜き出す用
                    /// </summary>
                    private const string MatchPatternBlock = @"\{{\s*""id"": ""0x{0}"",\s*""version"": {1},[^\}}]*\}}";

                    /// <summary>
                    /// json形式の出力内容から、指定のID/Versionを持つアプリケーションブロックを特定し、
                    /// 指定プロパティコレクションのステータスに合致するかどうか判定します。
                    /// </summary>
                    /// <param name="output">マッチング対象ストリーム</param>
                    /// <param name="applicationId">検出対象アプリケーションID</param>
                    /// <param name="version">検出対象アプリケーションバージョン</param>
                    /// <param name="values">マッチング対象プロパティステータスコレクション</param>
                    /// <returns>見つからなかった対象値の数を返します。 0 の場合全て見つかった事を示します。</returns>
                    /// <example>
                    /// ExpectEvaluator.IsEqual( 0, SigloHelper.ResultMatcher.Application.ListView.Find( executor.OutputStream.Standard.ToString(),
                    ///     new ID64( 0x01003ab001e30002 ), 0x20000,
                    ///     new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                    ///     {
                    ///         {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, true },
                    ///         {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                    ///         {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                    ///         {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                    ///     }
                    /// ) );
                    /// </example>
                    public static int Find( string output, ID64 applicationId, int version, ExpectValues values )
                    {
                        if ( null == values || 0 >= values.Count )
                        {
                            return -1;
                        }
                        Log.WriteLine( $"Find an expected view property in the content of => [ ID={applicationId}, VER={version} ]." );
                        output = TrimPrefixTarget( output );
                        var block = Regex.Match( output, string.Format( MatchPatternBlock, applicationId, version ) );
                        if ( false == block.Success )
                        {
                            Log.WriteLine( $"Not found an expected content meta => [ ID={applicationId}, VER={version} ]." );
                            return values.Count;
                        }
                        int unmatchedCount = 0;
                        var properties = block.Value;
                        foreach ( var value in values )
                        {
                            var pattern = string.Format( @"""{0}"": {1}", value.Key, value.Value );
                            var matched = Regex.Match( properties, pattern, RegexOptions.IgnoreCase );
                            if ( false == matched.Success )
                            {
                                Log.WriteLine( $"Not found an expected view property => {pattern}." );
                                ++unmatchedCount;
                            }
                            else
                            {
                                Log.WriteLine( $"Found an expected view property => {pattern}.{System.Environment.NewLine}It be matched value => [ {matched.Value} ]" );
                            }
                        }
                        return unmatchedCount;
                    }
                }

                /// <summary>
                /// application list-record-detail
                /// </summary>
                public static class ListRecordDetail
                {
                    public enum Property : byte
                    {
                        lastEvent,
                        isAutoDeleteDisabled,
                        isAutoUpdateEnabled,
                        lastUpdated,
                        terminateResult,
                        patchId,
                        currentVersion,
                    }

                    public class ExpectValues : Dictionary<Property, object>
                    {
                        public static string ToPattern( KeyValuePair<Property, object> expect )
                        {
                            if ( expect.Value == null )
                            {
                                return "";
                            }
                            else if ( expect.Value is bool )
                            {
                                var b = ( bool )expect.Value;
                                return string.Format( @"""{0}"": {1}", expect.Key, b ? "true" : "false" );
                            }
                            else if ( expect.Value is int )
                            {
                                var i = ( int )expect.Value;
                                return string.Format( @"""{0}"": {1}", expect.Key, i );
                            }
                            else if ( expect.Value is string )
                            {
                                var s = ( string )expect.Value;
                                return string.Format( @"""{0}"": ""{1}""", expect.Key, s );
                            }
                            else if ( expect.Value is ID64 )
                            {
                                var id = ( ID64 )expect.Value;
                                return string.Format( @"""{0}"": ""0x{1}""", expect.Key, id );
                            }
                            else
                            {
                                var s = expect.Value.ToString();
                                return string.Format( @"""{0}"": ""{1}""", expect.Key, s );
                            }
                        }
                    }

                    /// <summary>
                    /// 指定のIDを持つアプリケーションブロックのパターン（※外側の括弧{}を含んでいません）
                    /// </summary>
                    private const string MatchPatternBlock = "\\s+\"id\": \"0x{0}\",\\s+\"lastEvent\"" +
                                                             "[^\\{{\\}}]*" +
                                                             "(" +
                                                             "((?'open'\\{{)[^\\{{\\}}]*)+" +
                                                             "((?'close-open'\\}})[^\\{{\\}}]*)+" +
                                                             ")*" +
                                                             "(?(open)(?!))";

                    /// <summary>
                    /// json形式の出力内容から、指定のIDを持つアプリケーションブロックを特定し、
                    /// 指定プロパティコレクションのステータスに合致するかどうか判定します。
                    /// </summary>
                    /// <param name="output">マッチング対象ストリーム</param>
                    /// <param name="id">検出対象アプリケーションID</param>
                    /// <param name="values">マッチング対象プロパティステータスコレクション</param>
                    /// <returns>見つからなかった対象値の数を返します。 0 の場合全て見つかった事を示します。</returns>
                    /// <example>
                    /// ExpectEvaluator.IsEqual( 0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                    ///     executor.OutputStream.Standard.ToString(),
                    ///     new ID64( 0x01003ab001e30002 ), 0x20000,
                    ///     new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                    ///     {
                    ///         {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.lastEvent, "Launched" },
                    ///         {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, true },
                    ///     }
                    /// ) );
                    /// </example>
                    public static int Find( string output, ID64 id, ExpectValues values )
                    {
                        if ( null == values || 0 >= values.Count )
                        {
                            return -1;
                        }
                        Log.WriteLine( $"Find an expected property in the content of => [ ID={id} ]." );
                        var block = Regex.Match( TrimPrefixTarget( output ), string.Format( MatchPatternBlock, id ) );
                        if ( false == block.Success )
                        {
                            Log.WriteLine( $"Not found an expected content meta => [ ID={id} ]." );
                            return values.Count;
                        }
                        int unmatched = 0;
                        foreach ( var value in values )
                        {
                            var pattern = ExpectValues.ToPattern( value );
                            if ( string.IsNullOrEmpty( pattern ) )
                            {
                                return -1;
                            }
                            var matched = Regex.Match( block.Value, pattern, RegexOptions.IgnoreCase );
                            if ( matched.Success )
                            {
                                Log.WriteLine( $"Found an expected property => {pattern}.{System.Environment.NewLine}It be matched value => [ {matched.Value} ]" );
                            }
                            else
                            {
                                Log.WriteLine( $"Not found an expected property => {pattern}." );
                                ++unmatched;
                            }
                        }
                        return unmatched;
                    }
                }

                /// <summary>
                /// application occupied-size, storage-size
                /// </summary>
                public static class OccupiedSize
                {
                    public class ResultSizeValue
                    {
                        public double Application { get; }
                        public double Patch { get; }
                        public double AddOnContent { get; }

                        public ResultSizeValue( double app, double patch, double aoc )
                        {
                            Application = app;
                            Patch = patch;
                            AddOnContent = aoc;
                        }
                    }

                    public static class Storage
                    {
                        public const string GameCard = "Card";
                        public const string SdCard = "SD Card";
                        public const string BuiltInUser = "Built-in";
                    }

                    private const string LabelApplication = "Application";
                    private const string LabelPatch = "Patch";
                    private const string LabelAoc = "Add-On Content";
                    private const string MatchPatternForSizeValueField = @"[0-9]+[\.]?[0-9]*";

                    /// <summary>
                    /// Storage キーワード指定の該当ブロック抽出用
                    /// {0}: Storage
                    /// {1}: Unit of size
                    /// </summary>
                    private const string MatchPatternBlock = @"\[{0}\]\s*" +
                        LabelApplication + @"\s*" + MatchPatternForSizeValueField + @"\s*{1}\s*" +
                        LabelPatch + @"\s*" + MatchPatternForSizeValueField + @"\s*{1}\s*" +
                        LabelAoc + @"\s*" + MatchPatternForSizeValueField + @"\s*{1}";

                    /// <summary>
                    /// "label" "size" "unit" の書式から size を double で取得します。
                    /// </summary>
                    /// <param name="lStream"></param>
                    /// <param name="lLabel"></param>
                    /// <param name="lUnit"></param>
                    /// <returns></returns>
                    public static double ExtractSizeValue( string lStream, string lLabel, string lUnit)
                    {
                        var lMatched = Regex.Match( lStream, $@"{lLabel}\s*{MatchPatternForSizeValueField}\s*{lUnit}" );
                        return double.Parse( Regex.Match( lMatched.Value, MatchPatternForSizeValueField ).Value );
                    }

                    /// <summary>
                    /// 出力内容から指定のStorage キーワードを持つアプリケーションブロックを特定し、サイズ情報を取得します。
                    /// </summary>
                    /// <param name="output">マッチング対象ストリーム</param>
                    /// <param name="storage">抽出対象ストレージブロック</param>
                    /// <param name="unit">抽出対象サイズ単位 ( B, KB, MB, GB )</param>
                    /// <returns></returns>
                    public static ResultSizeValue ExtractBlock( string output, string storage, string unit )
                    {
                        output = TrimPrefixTarget( output );
                        var block = Regex.Match( output, string.Format( MatchPatternBlock, storage, unit ) );
                        if ( false == block.Success )
                        {
                            Log.WriteLine( $"Not found an expected storage block => [ Storage={storage}, Unit={unit} ]." );
                            return null;
                        }
                        var stream = block.Value;
                        var app = ExtractSizeValue( stream, LabelApplication, unit );
                        var patch = ExtractSizeValue( stream, LabelPatch, unit );
                        var aoc = ExtractSizeValue( stream, LabelAoc, unit );
                        return new ResultSizeValue( app, patch, aoc );
                    }
                }

                /// <summary>
                /// application list-download-task-status
                /// </summary>
                public static class ListDownloadTaskStatus
                {
                    public enum Status : byte
                    {
                        Created,
                        Added,
                        AlreadyExists,
                        Failed,
                    }

                    private const string MatchPattern = @"""uuid"": ""{0}"",\s*""applicationId"": ""0x{1}"",\s*""detail"": ""{2}""";

                    public static Match Find( string output, DownloadTaskList.Task task, Status status )
                    {
                        output = TrimPrefixTarget( output );
                        return Regex.Match( output, string.Format( MatchPattern, task.Uuid, task.OwnerApplication, status ) );
                    }

                    public static int Find( string output, DownloadTaskList dtl, Status status )
                    {
                        output = TrimPrefixTarget( output );
                        int count = 0; // unmatched
                        foreach ( var task in dtl.Tasks.Values )
                        {
                            var matched = Regex.Match( output, string.Format( MatchPattern, task.Uuid, task.OwnerApplication, status ) );
                            if ( !matched.Success )
                            {
                                Log.WriteLine( $"Not found download task status => {task.Uuid}." );
                                ++count;
                            }
                            else
                            {
                                Log.WriteLine( $"Found download task status => {task.ToString()}.{System.Environment.NewLine}It be matched value => [ {matched.Value} ]" );
                            }
                        }
                        return count;
                    }
                }

                /// <summary>
                /// application check-launch-rights
                /// </summary>
                public static class CheckLaunchRights
                {
                    public enum Status : byte
                    {
                        Available,
                        RightsNotFound,
                        InactiveNintendoAccount,
                        RecordNotFound,
                        ContentNotFound,
                        ContentPrepurchased,
                    }

                    public static Match Find(string output, Status status)
                    {
                        output = TrimPrefixTarget(output);
                        return Regex.Match(output, status.ToString());
                    }
                }

                /// <summary>
                /// application download-application-prepurchased-rights
                /// </summary>
                public static class DownloadApplicationPrepurchasedRights
                {
                    public enum Status : byte
                    {
                        Available,
                        ContentStillUnavailable,
                        ContentPartiallyStillUnavailable,
                        ContentRightsLost,
                        TryLater,
                    }

                    public static Match Find(string output, Status status)
                    {
                        output = TrimPrefixTarget(output);
                        return Regex.Match(output, status.ToString());
                    }
                }

                /// <summary>
                /// application wait-download-all
                /// </summary>
                public static class WaitDownloadAll
                {
                    /// <summary>
                    /// Failed application download requests から applicationId を取得します。
                    /// </summary>
                    /// <param name="executionLog">マッチング対象ストリーム</param>
                    /// <returns>発見された場合は Match 結果が返されます。</returns>
                    public static System.Text.RegularExpressions.Match FindApplicationIds(string executionLog)
                    {
                        // 接頭辞 [target] を削除
                        var output = TrimPrefixTarget(executionLog);

                        // Failed application download requests の後の[]内の文字のみ取り出す。
                        var mc = Regex.Match(output.ToString(), @"Failed application download requests \[[^\]]*\]", RegexOptions.Multiline);

                        // applicationId の値を取得する
                        return Regex.Match(mc.ToString(), @"""applicationId"": ""0x([0-9a-f]{16})""");
                    }
                }
            }

            /// <summary>
            /// DevMenuCommand, DevMenuCommandSystem "ticket" コマンド
            /// </summary>
            public static class Ticket
            {
                /// <summary>
                /// ticket list-all
                /// </summary>
                public static class ListAll
                {
                    /// <summary>
                    /// "RightsId:" の上位16桁が指定したID64( コンテントメタID )かどうかマッチングします。
                    /// </summary>
                    /// <param name="output">マッチング対象ストリーム</param>
                    /// <param name="expectRightsId">RightsId の上位16桁の配置を期待するコンテンツメタID</param>
                    /// <returns>マッチング結果</returns>
                    public static Match Find( string output, ID64 expectRightsId )
                    {
                        return Regex.Match( output, $@"RightsId:\s*0x{expectRightsId}[A-Za-z0-9]{{16}}" );
                    }

                    /// <summary>
                    /// "RightsId:" の上位16桁が指定したID64( コンテントメタID )かどうかマッチングします。
                    /// </summary>
                    /// <param name="output">マッチング対象ストリーム</param>
                    /// <param name="expectIds">マッチング対象値配列</param>
                    /// <returns>見つからなかった対象値の数を返します。 0 の場合全て見つかった事を示します。</returns>
                    public static int Find( string output, params ID64[] expectIds )
                    {
                        int unmatchedCount = 0;
                        foreach ( var id in expectIds )
                        {
                            var matched = Find( output, id );
                            if ( false == matched.Success )
                            {
                                Log.WriteLine( $"Not found a ticket in contents => {id}." );
                                ++unmatchedCount;
                            }
                            else
                            {
                                Log.WriteLine( $"Found a ticket in contents => {id}.{System.Environment.NewLine}It be matched value => [ {matched.Value} ]" );
                            }
                        }
                        return unmatchedCount;
                    }
                }
            }

            /// <summary>
            /// DevMenuCommand, DevMenuCommandSystem "addoncontent" コマンド
            /// </summary>
            public class AddOnContent
            {
                /// <summary>
                /// addoncontent list
                /// </summary>
                /// <remarks>
                /// 2 add-on-content(s) found.
                /// id                  index  release-version  required-app-release-version
                /// 0x01003ab001e30001      1                0                             0
                /// 0x01003ab001e30001      2                0                             0
                /// </remarks>
                public static class List
                {
                    /// <summary>
                    /// 指定の期待Aocがリスト中に存在するか確認します。
                    /// </summary>
                    /// <param name="stream">マッチング対象ストリーム</param>
                    /// <param name="expectAoc">期待する Aoc 情報構造体, <see cref="GeneratedContentResult"/></param>
                    /// <returns>発見された場合は true が返されます。</returns>
                    public static bool Find( string stream, GeneratedContentResult expectAoc )
                    {
                        // Multilineの対象改行は(LF)なので CRLFの場合、最後に( \r )で検出するようにしています。
                        var expectMatchPatternAocList = @"[0-9]+ add-on-content\(s\) found\.\r$";
                        var matchFound = Regex.Match( stream, expectMatchPatternAocList, RegexOptions.Multiline );
                        var foundAocCount = ( matchFound.Success ) ? uint.Parse( Regex.Match( matchFound.Value, @"[0-9]+" ).Value ) : 0U;

                        var prop = expectAoc.Properties;
                        var aocAppId = prop.GetValueAsID64( ContentMeta.Contexture.Advance.PropertyKey.ApplicationId.ToString() );
                        var aocIndex = prop.GetValueAsUI32( ContentMeta.Contexture.Advance.PropertyKey.Index.ToString() );
                        var aocRequiredAppVersion = prop.GetValueAsUI32( ContentMeta.Contexture.Advance.PropertyKey.RequiredApplicationVersion.ToString() );
                        var aocRequiredAppReleaseVersion = aocRequiredAppVersion >> 16;
                        var requiredAocSpec = $"[ID={aocAppId}, Index={aocIndex}, ReleaseVersion={expectAoc.ReleaseVersion}, RequiredAppReleaseVersion={aocRequiredAppReleaseVersion}]";
                        var hasFound = false;

                        if ( foundAocCount > 0 )
                        {
                            var matchPatternForItem = $@"^\[target\] 0x{aocAppId}\s+{aocIndex}\s+{expectAoc.ReleaseVersion}\s+{aocRequiredAppReleaseVersion}";
                            var matched = Regex.Match( stream, matchPatternForItem, RegexOptions.Multiline );
                            hasFound = matched.Success;
                        }
                        var resultMessage = ( hasFound ) ? "Found" : "Not found";
                        Log.WriteLine( $"Required aoc {requiredAocSpec} => {resultMessage}" );
                        return hasFound;
                    }
                }
            }

        }
    }
}
