﻿using System.Collections.Generic;
using System.Text;

namespace CsTestAssistants
{
    public class PowerShellExecutor
    {
        /// <summary>
        /// ツール実行パス取得。
        /// </summary>
        /// <returns>"powershell.exe"が返されます</returns>
        public string GetExecutionToolPath()
        {
            if (IsSetSysNativePath &&
                System.Environment.Is64BitOperatingSystem &&  // 64ビットOSかどうか
                (System.Environment.Is64BitProcess == false)  // 64ビットプロセスで動作しているか
               )
            {
                // ファイルパスのリダイレクション処理を回避する特殊パスを返す
                // (64ビットOSの32ビットプロセス動作時のみ設定可能なパス)
                return System.Environment.ExpandEnvironmentVariables(
                    @"%windir%\sysnative\WindowsPowerShell\v1.0\powershell.exe");
            }
            // 上記の特殊条件以外は従来通りのパスを返す
            return "powershell.exe";
        }

        /// <summary>
        /// PowerShell の実行ポリシー列挙値
        /// </summary>
        /// <remarks>
        /// | policy       | 署名付き | 署名なし/ローカル | 署名なし/リモート( ネット ) |
        /// | Restricted   |    ×    |        ×         |             ×              |
        /// | AllSigned    |    ○    |        ×         |             ×              |
        /// | RemoteSigned |    ○    |        ○         |             ×              |
        /// | Unrestricted |    ○    |        ○         |             △              |
        /// | Bypass       |    ○    |        ○         |             ○              |
        /// △：ユーザ確認が必要.
        /// </remarks>
        public enum ExecutionPolicy : byte
        {
            Restricted,
            AllSigned,
            RemoteSigned,
            Unrestricted,
            Bypass,
        }

        /// <summary>
        /// 文字列から ExecutionPolicy 列挙型へ変換.
        /// </summary>
        /// <param name="name">変換元文字列</param>
        /// <param name="defaultValue">変換できない文字列の場合、返される列挙値。</param>
        /// <returns>変換後の列挙値</returns>
        public static ExecutionPolicy ToExecutionPolicy( string name, ExecutionPolicy defaultValue = ExecutionPolicy.RemoteSigned )
        {
            return CsExtension.ToEnumValue( name, defaultValue );
        }

        public float Version { get; protected set; }

        public ExecutionPolicy Policy { get; protected set; }

        // SysNativeのパスを設定するかどうかの特殊設定フラグ (デフォルト値は false とする)
        public bool IsSetSysNativePath { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="policy">利用する実行ポリシーを指定します.</param>
        /// <param name="version">利用するPowerShellバージョンを指定します.</param>
        public PowerShellExecutor( float version = 2.0f, ExecutionPolicy policy = ExecutionPolicy.RemoteSigned )
        {
            Version = version;
            Policy = policy;
            IsSetSysNativePath = false;
        }

        /// <summary>
        /// ps1 スクリプトを実行します。
        /// </summary>
        /// <remarks>
        /// 以下のコマンドが生成されます。
        /// powershell.exe -Version ${Version} -NoProfile -ExecutionPolicy ${Policy} -Command "& '${command}' ${arguments}; exit $LastExitCode;"
        /// </remarks>
        /// <param name="command">PowerShellスクリプトコマンドファイルパス</param>
        /// <param name="arguments">スクリプトコマンド引数リスト</param>
        /// <returns>PowerShellの $LastExitCode が返されます。</returns>
        public int Execute( string commandPath, List<string> arguments = null, OutputStreams outputs = null )
        {
            // correct to full path string on the commandPath.
            commandPath = System.IO.Path.GetFullPath( commandPath );

            // make operands.
            StringBuilder b = new StringBuilder( 512 );
            b.AppendFormat( "-Version {0:F1} -NoProfile -ExecutionPolicy {1} -Command \"", Version, Policy.ToString() );
            b.Append( "& '" ).Append( commandPath ).Append( '\'' );
            if ( null != arguments && arguments.Count > 0 )
            {
                b.Append( ' ' ).Append( string.Join( " ", arguments ) );
            }
            b.Append( "; exit ${LastExitCode};\"" );

            // run process.
            return CommandLineExecutor.ExecuteOnProcess( GetExecutionToolPath(), b.ToString(), outputs );
        }
    }
}
