﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Globalization;
using System.Text;
using System.Runtime.Serialization.Json;

namespace CsTestAssistants
{
    /// <summary>
    /// DataContractJsonSerializer用JSONシリアライズエクステンション
    /// </summary>
    /// <example>
    /// ContractableJsonClass obj = new ContractableJsonClass();
    /// var stream = obj.SerializeToJson();
    /// Log.WriteLine( "=== Json ===\n{0}\n=== Json end ===\n", stream );
    /// var readed = stream.DeserializeFromJson< ContractableJsonClass >();
    /// Log.WriteLine( "=== Reload json ===\n{0}\n=== Reload json end ===\n", readed.SerializeToJson() );
    /// </example>
    public static class JsonExtension
    {
        private static string SerializeToJsonCore< T >( DataContractJsonSerializer serializer, T obj )
        {
            using ( var stream = new System.IO.MemoryStream( 16 * 1024 ) )
            {
                serializer.WriteObject( stream, obj );
                return Encoding.UTF8.GetString( stream.ToArray() );
            }
        }

        /// <summary>
        /// JSON シリアライザ
        /// </summary>
        /// <typeparam name="T">書込元型</typeparam>
        /// <param name="obj"></param>
        /// <returns></returns>
        public static string SerializeToJson< T >( this T obj ) where T : class
        {
            return SerializeToJsonCore( new DataContractJsonSerializer( typeof( T ) ), obj );
        }

        /// <summary>
        /// JSON シリアライザ ( Settings プロパティ対応 )
        /// </summary>
        /// <typeparam name="T">書込元型</typeparam>
        /// <param name="obj"></param>
        /// <returns></returns>
        public static string SerializeToJson< T >( this T obj, DataContractJsonSerializerSettings settings ) where T : class
        {
            return SerializeToJsonCore( new DataContractJsonSerializer( typeof( T ), settings ), obj );
        }

        /// <summary>
        /// JSON デシリアライザ
        /// </summary>
        /// <typeparam name="T">読込先型</typeparam>
        /// <param name="obj"></param>
        /// <returns></returns>
        public static T DeserializeFromJson< T >( this string obj )
        {
            using ( var stream = new System.IO.MemoryStream( Encoding.UTF8.GetBytes( obj ) ) )
            {
                var serializer = new DataContractJsonSerializer( typeof( T ) );
                return ( T )serializer.ReadObject( stream );
            }
        }

        /// <summary>
        /// JSON デシリアライザ ( Settings プロパティ対応 )
        /// </summary>
        /// <typeparam name="T">読込先型</typeparam>
        /// <param name="obj"></param>
        /// <returns></returns>
        public static T DeserializeFromJson< T >( this string obj, DataContractJsonSerializerSettings settings )
        {
            using ( var stream = new System.IO.MemoryStream( Encoding.UTF8.GetBytes( obj ) ) )
            {
                var serializer = new DataContractJsonSerializer( typeof( T ), settings );
                return ( T )serializer.ReadObject( stream );
            }
        }

        /// <summary>
        /// シリアライズされたJSONを段落インデント整形します。
        /// </summary>
        /// <param name="jsonStream"></param>
        /// <returns></returns>
        public static string FormatParagraphIndent( string jsonStream, int indentSpace = 4 )
        {
            if ( string.IsNullOrEmpty( jsonStream ) )
            {
                return jsonStream;
            }
            StringBuilder b = new StringBuilder( jsonStream.Length * 2 );
            List<char> returnKeyword = new List<char>() { ',', '{', '[' };
            List<char> indentBegin = new List<char>() { '{', '[' };
            List<char> indentEnd = new List<char>() { '}', ']' };

            TextElementEnumerator enumrator = StringInfo.GetTextElementEnumerator( jsonStream );
            enumrator.Reset();
            int indent = 0;
            bool lineStart = true;
            while ( enumrator.MoveNext() )
            {
                string text = enumrator.GetTextElement();
                bool isSurrogate = ( text.Length > 1 );
                char c = text[ 0 ];
                if ( false == isSurrogate && indentEnd.Contains( c ) )
                {
                    int newIndex;
                    indent = ( 0 > ( newIndex = indent - indentSpace ) ) ? 0 : newIndex;
                    lineStart = true;
                    b.Append( '\n' );
                }
                if ( lineStart )
                {
                    lineStart = false;
                    b.Append( "".PadLeft( indent ) );
                }
                // copy with surrogate.
                b.Append( text );

                // handle indent
                if ( false == isSurrogate && returnKeyword.Contains( c ) )
                {
                    indent += ( indentBegin.Contains( c ) ) ? indentSpace : 0;
                    lineStart = true;
                    b.Append( '\n' );
                }
            }
            return b.ToString();
        }
    }
}
