﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/npnsUtil.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include <nn/ns/ns_Result.h>

#include <nn/npns.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/bgtc/bgtc_Api.h>

#include <nn/ns/srv/ns_PushNotificationDisptcher.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_TypesRequirement.h>

using namespace nn;

namespace {
    class ScopedNetworkConnection
    {
        NN_DISALLOW_COPY(ScopedNetworkConnection);
    public:
        NN_IMPLICIT ScopedNetworkConnection() NN_NOEXCEPT :
            m_Connection(os::EventClearMode_ManualClear)
        {}

        Result WaitConnection() NN_NOEXCEPT
        {
            NN_RESULT_DO(nn::nifm::SetRequestRequirementPreset(m_Connection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));
            m_Connection.SubmitRequest();

            auto& connectionEvent = m_Connection.GetSystemEvent();

            while (!m_Connection.IsAvailable())
            {
                connectionEvent.Wait();
                connectionEvent.Clear();
            }

            NN_RESULT_SUCCESS;
        }

        ~ScopedNetworkConnection()
        {
            m_Connection.CancelRequest();
        }
    private:
        nifm::NetworkConnection m_Connection;
    };

    class NotificationReceiveTest : public nnt::npns::util::TestBase
    {
    public:
    protected:

        static void SetUpTestCase()
        {
            // INFO: nifm::Initialize が呼ばれるよりもまえに System で初期化しておく
            //       (PushNotificationDispatcher が System 権限で動くため)
            NNT_EXPECT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

            // INFO:
            // アプリ権限で SubmitAndWait されると、Dispatcher 内でシステム用の Persistent が通らなくなるので
            // SubmitAndWait しない状態で初期化
            TestBase::SetUpTestCaseWithoutConnection();

        }

        static void TearDownTestCase()
        {
            TestBase::TearDownTestCase();
        }

        virtual void SetUp()
        {
            WaitNpnsConnected();
            ncm::Initialize();
        }

        virtual void TearDown()
        {
            ncm::Finalize();
        }

    private:
        void WaitNpnsConnected()
        {
            nn::npns::State state;
            nn::os::SystemEvent stateChangedEvent;
            nn::npns::GetStateChangeEvent(stateChangedEvent);
            while ((state = nn::npns::GetState()) != nn::npns::State_Connected)
            {
                auto result = stateChangedEvent.TimedWait(nn::TimeSpan::FromSeconds(20));
                NN_ABORT_UNLESS(result);
            }
        }

        void WaitDispatcherReady(ns::srv::PushNotificationDispatcher* dispatcher)
        {
            auto tryCount = 0;
            bool completed = false;
            nn::sf::Out<bool> c = nn::sf::Out<bool>(&completed);
            while (!completed && tryCount < 100)
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                tryCount++;
                NNT_EXPECT_RESULT_SUCCESS(dispatcher->IsNotificationSetupCompleted(c));
            }
        }


    protected:
        typedef std::function<Result(const ns::NotificationInfo& info)> NotificationHandler;

        class AutoClearEvent : public nn::os::Event
        {
        public:
            AutoClearEvent() NN_NOEXCEPT
                : nn::os::Event(nn::os::EventClearMode_AutoClear)
            {}
        };

        void TestNotificationReceive(NotificationHandler handlers[], ns::AsyncTask::AsyncTaskType types[], int handlerCount, ApplicationId appId, const char* notifications[], int notificationCount, ns::srv::VulnerabilityManager* vulnerabilityManager = nullptr)
        {
            using NotificationHandlerHolder = nn::ns::srv::AsyncTaskManager::NotificationHandlerHolder;

            nn::nim::InitializeForNetworkInstallManager();
            NN_UTIL_SCOPE_EXIT{ nn::nim::FinalizeForNetworkInstallManager(); };

            nn::bgtc::Initialize();
            NN_UTIL_SCOPE_EXIT{ nn::bgtc::Finalize(); };

            std::unique_ptr<AutoClearEvent[]> handlerDoneEvents(new AutoClearEvent[handlerCount]);
            std::unique_ptr<NotificationHandlerHolder[]> holders(new NotificationHandlerHolder[handlerCount]);
            for (int i = 0; i < handlerCount; ++i)
            {
                holders[i] =
                {
                    [&, i](const ns::NotificationInfo& info) -> Result
                    {
                        NN_RESULT_DO(handlers[i](info));
                        handlerDoneEvents[i].Signal();
                        NN_RESULT_SUCCESS;
                    },
                    types[i]
                };
            }

            ScopedNetworkConnection connection;
            NNT_EXPECT_RESULT_SUCCESS(connection.WaitConnection());

            nn::npns::NotificationToken token;
            NNT_EXPECT_RESULT_SUCCESS(nn::npns::CreateToken(&token, {}, appId));


            nn::ns::srv::AsyncTaskManager::Config nmConfig =
            {
                false, false, false, false, false, 0, 0, true,
            };

            nn::ns::srv::PushNotificationDispatcher::Config config =
            {
                false, false, 60,
            };
            static NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 s_NotificationDispatcherStack[8 * 1024];
            static NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 s_AsyncTaskManagerStack[16 * 1024];

            if (!vulnerabilityManager)
            {
                vulnerabilityManager = new ns::srv::VulnerabilityManager();
                vulnerabilityManager->InitializeForTest(ns::srv::VulnerabilityManager::StorePolicy::OnMemory);
            }
            nn::ns::srv::ApplicationInstallRequestList requestList(nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr);
            nn::ns::srv::ApplicationVersionManager applicationVersionManager;
            nn::ns::srv::PushNotificationDispatcher dispatcher(s_NotificationDispatcherStack, sizeof(s_NotificationDispatcherStack), s_AsyncTaskManagerStack, sizeof(s_AsyncTaskManagerStack));
            dispatcher.Initialize(appId.value, config, &requestList, nullptr, vulnerabilityManager, &applicationVersionManager, holders.get(), handlerCount, nmConfig);
            NN_UTIL_SCOPE_EXIT{ dispatcher.Finalize(); };

            WaitDispatcherReady(&dispatcher);

            NN_LOG("Send notification\n");

            for (int i = 0; i < notificationCount; ++i)
            {
                PublishNotification(token, notifications[i]);
            }

            for (int i = 0; i < handlerCount; ++i)
            {
                EXPECT_TRUE(handlerDoneEvents[i].TimedWait(nn::TimeSpan::FromSeconds(200)));
            }
        }

        void TestNotificationReceive(NotificationHandler& handler, ns::AsyncTask::AsyncTaskType type, ApplicationId appId, const char* notification, ns::srv::VulnerabilityManager* vulnerabilityManager = nullptr)
        {
            TestNotificationReceive(&handler, &type, 1, appId, &notification, 1, vulnerabilityManager);
        }

    };

} // namespace

TEST_F(NotificationReceiveTest, NupTopic)
{
    NotificationHandler handler = [](const ns::NotificationInfo& info) -> Result
    {
        EXPECT_EQ(nn::ns::NotificationInfo::NotificationType::SystemUpdate, info.type);
        EXPECT_EQ(0x0100000000000816, info.nupInfo.titleId);
        EXPECT_EQ(1, info.nupInfo.titleVersion);
        EXPECT_EQ(0x0100000000000816, info.nupInfo.requiredTitleId);
        EXPECT_EQ(1, info.nupInfo.requiredVersion);
        NN_RESULT_SUCCESS;
    };
    nn::ApplicationId appId = { 0x0100000000000000 };
    auto notification = "{ \\\"type\\\" : \\\"Nup\\\", \\\"title_id\\\":\\\"0100000000000816\\\",\\\"title_version\\\": 1,\\\"required_title_id\\\":\\\"0100000000000816\\\",\\\"required_title_version\\\": 1 }";

    TestNotificationReceive(handler, ns::AsyncTask::AsyncTaskType::SystemUpdate, appId, notification);
}

TEST_F(NotificationReceiveTest, SafeSystemVersionUpdateImmediately)
{
    ns::srv::VulnerabilityManager vulnerabilityManager;
    vulnerabilityManager.InitializeForTest(ns::srv::VulnerabilityManager::StorePolicy::OnMemory);

    const Bit64 ValidSystemUpdateId = 0x0100000000000816;
    NNT_EXPECT_RESULT_SUCCESS(vulnerabilityManager.UpdateSafeSystemVersionInfo({ValidSystemUpdateId}, 0));
    EXPECT_FALSE(vulnerabilityManager.NeedsUpdateVulnerability());

    // 即時の DTL、5 秒待ちの NUP どちらからも呼び出される。
    // SafeSystemVersion が更新されてれば NeedsUpdateVulnerability が true になってるはず
    NotificationHandler handler = [&](const ns::NotificationInfo& info) -> Result
    {
        EXPECT_TRUE(vulnerabilityManager.NeedsUpdateVulnerability());
        NN_RESULT_SUCCESS;
    };
    nn::ApplicationId appId = { 0x0100000000000004 };

    // 5 秒待ちの NUP を受信した時点で 5 秒待たずして SafeSystemVersion が更新されることを確認するための通知セット
    const char* notifications[] = {
        // 5 秒待ちの NUP
        "{ \\\"type\\\" : \\\"Nup\\\", \\\"title_id\\\":\\\"0100000000000816\\\",\\\"title_version\\\": 1,\\\"required_title_id\\\":\\\"0000000000000000\\\",\\\"required_title_version\\\": 0,"
            "  \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 5  }}",

        // 即時の DTL
        "{ \\\"type\\\" : \\\"Dtl\\\",\\\"e_tag\\\":\\\"0123456789abcdef0123456789abcdef\\\"}",
    };
    const int NotificationCount = sizeof(notifications) / sizeof(notifications[0]);

    NotificationHandler handlers[NotificationCount] =
    {
        handler, handler,
    };

    nn::ns::AsyncTask::AsyncTaskType types[NotificationCount] =
    {
        nn::ns::AsyncTask::AsyncTaskType::SystemUpdate,
        nn::ns::AsyncTask::AsyncTaskType::DownloadTaskList,
    };

    TestNotificationReceive(handlers, types, NotificationCount, appId, notifications, NotificationCount, &vulnerabilityManager);
}

TEST_F(NotificationReceiveTest, DtlNotify)
{
    NotificationHandler handler = [](const ns::NotificationInfo& info) -> Result
    {
        EXPECT_EQ(nn::ns::NotificationInfo::NotificationType::DownloadTaskList, info.type);
        NN_RESULT_SUCCESS;
    };
    nn::ApplicationId appId = { 0x0100000000000001 };
    auto notification = "{ \\\"type\\\" : \\\"Dtl\\\",\\\"e_tag\\\":\\\"0123456789abcdef0123456789abcdef\\\"}";

    TestNotificationReceive(handler, ns::AsyncTask::AsyncTaskType::DownloadTaskList, appId, notification);
}

TEST_F(NotificationReceiveTest, VersionListTopic)
{
    NotificationHandler handler = [](const ns::NotificationInfo& info) -> Result
    {
        EXPECT_EQ(nn::ns::NotificationInfo::NotificationType::VersionList, info.type);
        NN_RESULT_SUCCESS;
    };
    nn::ApplicationId appId = { 0x0100000000000002 };
    auto notification = "{ \\\"type\\\" : \\\"VersionList\\\", \\\"e_tag\\\" : \\\"0123456789abcdef0123456789abcdef\\\"}";

    TestNotificationReceive(handler, ns::AsyncTask::AsyncTaskType::VersionList, appId, notification);
}

TEST_F(NotificationReceiveTest, ETicketAvailable)
{
    NotificationHandler handler = [](const ns::NotificationInfo& info) -> Result
    {
        EXPECT_EQ(nn::ns::NotificationInfo::NotificationType::ETicketAvailable, info.type);
        NN_RESULT_SUCCESS;
    };

    nn::ApplicationId appId = { 0x0100000000000008 };
    auto notification = "{ \\\"type\\\" : \\\"ETicketAvailable\\\", \\\"eci_account_id\\\" : \\\"12345678\\\", \\\"title_ids\\\" : [\\\"0x01004b9000490000\\\"]}";

    TestNotificationReceive(handler, ns::AsyncTask::AsyncTaskType::ETicketAvailable, appId, notification);
}

// Nup ... 30 秒待ち
// Dtl ... 20 秒待ち
// VerList ... 5 秒待ち
// Nup ... 10 秒待ち(30 秒待ちと異なる etag 値とし、上書きされることを確認)
// Dtl ... 1 秒待ち(20 秒待ちと同じ etag 値とし、上書きされないことを確認)
// 上記の通知を順に送り、VerList -> Nup -> Dtl の順で解決されることを確認
TEST_F(NotificationReceiveTest, MixBasic)
{
    std::string resultSequence = "";
    NotificationHandler nupHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        EXPECT_EQ(nn::ns::NotificationInfo::NotificationType::SystemUpdate, info.type);
        EXPECT_EQ(0x0100000000000816, info.nupInfo.titleId);
        EXPECT_EQ(1, info.nupInfo.titleVersion);
        EXPECT_EQ(0x0100000000000816, info.nupInfo.requiredTitleId);
        EXPECT_EQ(1, info.nupInfo.requiredVersion);
        resultSequence.push_back('N');
        NN_RESULT_SUCCESS;
    };

    NotificationHandler dtlHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        EXPECT_EQ(nn::ns::NotificationInfo::NotificationType::DownloadTaskList, info.type);
        resultSequence.push_back('D');
        NN_RESULT_SUCCESS;
    };

    NotificationHandler verListHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        EXPECT_EQ(nn::ns::NotificationInfo::NotificationType::VersionList, info.type);
        resultSequence.push_back('V');
        NN_RESULT_SUCCESS;
    };

    NotificationHandler handlers[] =
    {
        nupHandler, dtlHandler, verListHandler,
    };

    nn::ns::AsyncTask::AsyncTaskType types[] =
    {
        nn::ns::AsyncTask::AsyncTaskType::SystemUpdate,
        nn::ns::AsyncTask::AsyncTaskType::DownloadTaskList,
        nn::ns::AsyncTask::AsyncTaskType::VersionList,
    };

    nn::ApplicationId appId = { 0x0100000000000003 };

    const char* notifications[] = {
        "{ \\\"type\\\" : \\\"Nup\\\", \\\"e_tag\\\" : \\\"00000000000000000000000000000000\\\", \\\"title_id\\\":\\\"0100000000000816\\\",\\\"title_version\\\": 1,\\\"required_title_id\\\":\\\"0100000000000816\\\",\\\"required_title_version\\\": 1,"
        " \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 30  }}",

        "{ \\\"type\\\" : \\\"Dtl\\\",\\\"e_tag\\\":\\\"0123456789abcdef0123456789abcdef\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 20  }}",

        "{ \\\"type\\\" : \\\"VersionList\\\", \\\"e_tag\\\" : \\\"0123456789abcdef0123456789abcdef\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 5  }}",

        "{ \\\"type\\\" : \\\"Nup\\\", \\\"e_tag\\\" : \\\"00000000000000000000000000000001\\\", \\\"title_id\\\":\\\"0100000000000816\\\",\\\"title_version\\\": 1,\\\"required_title_id\\\":\\\"0100000000000816\\\",\\\"required_title_version\\\": 1,"
        " \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 10  }}",
        "{ \\\"type\\\" : \\\"Dtl\\\",\\\"e_tag\\\":\\\"0123456789abcdef0123456789abcdef\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 1  }}",
    };
    const int NotificationCount = sizeof(notifications) / sizeof(notifications[0]);

    TestNotificationReceive(handlers, types, 3, appId, notifications, NotificationCount);

    EXPECT_EQ("VND", resultSequence);
}

// (SIGLO-63388) 確認用
//
// Nup ... 5 秒待ち
// VerList ... 15 秒待ち
// Nup ハンドラの中で VerList 5 秒待ちの通知発行
// として、VersionList が重複しないことの確認
TEST_F(NotificationReceiveTest, NotDuplicate)
{
    nn::ApplicationId appId = { 0x0100000000000005 };

    NotificationHandler nupHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        const char* Notification = "{ \\\"type\\\" : \\\"VersionList\\\", \\\"e_tag\\\" : \\\"00000000000000000000000000000000\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 5  }}";
        nn::npns::NotificationToken token;
        NNT_EXPECT_RESULT_SUCCESS(nn::npns::CreateToken(&token, {}, appId));
        PublishNotification(token, Notification);

        NN_RESULT_SUCCESS;
    };

    NotificationHandler dtlHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        EXPECT_EQ(nn::ns::NotificationInfo::NotificationType::DownloadTaskList, info.type);
        NN_RESULT_SUCCESS;
    };

    os::Event verListEvent(nn::os::EventClearMode_ManualClear);
    NotificationHandler verListHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        // VerList の通知が 2 回以上呼ばれるならここで失敗する。
        EXPECT_FALSE(verListEvent.TryWait());
        verListEvent.Signal();
        NN_RESULT_SUCCESS;
    };

    NotificationHandler handlers[] =
    {
        nupHandler, dtlHandler, verListHandler,
    };

    nn::ns::AsyncTask::AsyncTaskType types[] =
    {
        nn::ns::AsyncTask::AsyncTaskType::SystemUpdate,
        nn::ns::AsyncTask::AsyncTaskType::DownloadTaskList,
        nn::ns::AsyncTask::AsyncTaskType::VersionList,
    };

    const char* notifications[] = {
        "{ \\\"type\\\" : \\\"Nup\\\", \\\"e_tag\\\" : \\\"00000000000000000000000000000000\\\", \\\"title_id\\\":\\\"0100000000000816\\\",\\\"title_version\\\": 1,\\\"required_title_id\\\":\\\"0100000000000816\\\",\\\"required_title_version\\\": 1,"
        " \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 5  }}",

        "{ \\\"type\\\" : \\\"Dtl\\\",\\\"e_tag\\\":\\\"0123456789abcdef0123456789abcdef\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 20  }}",

        "{ \\\"type\\\" : \\\"VersionList\\\", \\\"e_tag\\\" : \\\"0123456789abcdef0123456789abcdef\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 15  }}",
    };
    const int NotificationCount = sizeof(notifications) / sizeof(notifications[0]);

    TestNotificationReceive(handlers, types, 3, appId, notifications, NotificationCount);
}

// WaitingPolicy が等しい場合は、通知のリセットが発生しない
//
// Nup ... 5 秒待ち発行
// Dtl ... 15 秒待ち発行
// VerList ... 10 秒待ち発行
// Nup ハンドラの中で VerList 10秒待ち、etag 変更を発行（10秒待ちが一致なのでこのタイミングで VerList の待ちがリセットされないことの確認）
// VerList の待ちがリセットされると Nup -> Dtl -> VerList の順で処理され失敗。VerList の待ちがリセットされなければ Nup -> VerList -> Dtl の順で処理され成功。
TEST_F(NotificationReceiveTest, WaitingNotReset)
{
    nn::ApplicationId appId = { 0x0100000000000006 };
    std::string resultSequence = "";
    NotificationHandler nupHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        const char* Notification = "{ \\\"type\\\" : \\\"VersionList\\\", \\\"e_tag\\\" : \\\"00000000000000000000000000000000\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 10  }}";
        nn::npns::NotificationToken token;
        NNT_EXPECT_RESULT_SUCCESS(nn::npns::CreateToken(&token, {}, appId));
        PublishNotification(token, Notification);

        resultSequence.push_back('N');
        NN_RESULT_SUCCESS;
    };

    NotificationHandler dtlHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        resultSequence.push_back('D');
        NN_RESULT_SUCCESS;
    };

    NotificationHandler verListHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        const char expectedEtag[] = "00000000000000000000000000000000";
        EXPECT_TRUE(std::memcmp(expectedEtag, info.eTag.data, sizeof(expectedEtag)) == 0);
        resultSequence.push_back('V');
        NN_RESULT_SUCCESS;
    };

    NotificationHandler handlers[] =
    {
        nupHandler, dtlHandler, verListHandler,
    };

    nn::ns::AsyncTask::AsyncTaskType types[] =
    {
        nn::ns::AsyncTask::AsyncTaskType::SystemUpdate,
        nn::ns::AsyncTask::AsyncTaskType::DownloadTaskList,
        nn::ns::AsyncTask::AsyncTaskType::VersionList,
    };

    const char* notifications[] = {
        "{ \\\"type\\\" : \\\"Nup\\\", \\\"e_tag\\\" : \\\"00000000000000000000000000000000\\\", \\\"title_id\\\":\\\"0100000000000816\\\",\\\"title_version\\\": 1,\\\"required_title_id\\\":\\\"0100000000000816\\\",\\\"required_title_version\\\": 1,"
        " \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 5  }}",

        "{ \\\"type\\\" : \\\"Dtl\\\",\\\"e_tag\\\":\\\"0123456789abcdef0123456789abcdef\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 15  }}",

        "{ \\\"type\\\" : \\\"VersionList\\\", \\\"e_tag\\\" : \\\"0123456789abcdef0123456789abcdef\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 10  }}",
    };
    const int NotificationCount = sizeof(notifications) / sizeof(notifications[0]);

    TestNotificationReceive(handlers, types, 3, appId, notifications, NotificationCount);
    EXPECT_EQ("NVD", resultSequence);
}

// WaitingPolicy が異なる場合は、通知のリセットが発生する
//
// Nup ... 5 秒待ち発行
// Dtl ... 15 秒待ち発行
// VerList ... 10 秒待ち発行
// Nup ハンドラの中で VerList 11秒待ち、etag 変更を発行（10秒待ちと不一致なのでこのタイミングで VerList の待ちがリセットされる）
// VerList の待ちがリセットされると Nup -> Dtl -> VerList の順で処理され成功。VerList の待ちがリセットされなければ Nup -> VerList -> Dtl の順で処理され失敗。
TEST_F(NotificationReceiveTest, WaitingReset)
{
    nn::ApplicationId appId = { 0x0100000000000007 };
    std::string resultSequence = "";
    NotificationHandler nupHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        const char* Notification = "{ \\\"type\\\" : \\\"VersionList\\\", \\\"e_tag\\\" : \\\"00000000000000000000000000000000\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 11  }}";
        nn::npns::NotificationToken token;
        NNT_EXPECT_RESULT_SUCCESS(nn::npns::CreateToken(&token, {}, appId));
        PublishNotification(token, Notification);

        resultSequence.push_back('N');
        NN_RESULT_SUCCESS;
    };

    NotificationHandler dtlHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        resultSequence.push_back('D');
        NN_RESULT_SUCCESS;
    };

    NotificationHandler verListHandler = [&](const ns::NotificationInfo& info) -> Result
    {
        const char expectedEtag[] = "00000000000000000000000000000000";
        EXPECT_TRUE(std::memcmp(expectedEtag, info.eTag.data, sizeof(expectedEtag)) == 0);
        resultSequence.push_back('V');
        NN_RESULT_SUCCESS;
    };

    NotificationHandler handlers[] =
    {
        nupHandler, dtlHandler, verListHandler,
    };

    nn::ns::AsyncTask::AsyncTaskType types[] =
    {
        nn::ns::AsyncTask::AsyncTaskType::SystemUpdate,
        nn::ns::AsyncTask::AsyncTaskType::DownloadTaskList,
        nn::ns::AsyncTask::AsyncTaskType::VersionList,
    };

    const char* notifications[] = {
        "{ \\\"type\\\" : \\\"Nup\\\", \\\"e_tag\\\" : \\\"00000000000000000000000000000000\\\", \\\"title_id\\\":\\\"0100000000000816\\\",\\\"title_version\\\": 1,\\\"required_title_id\\\":\\\"0100000000000816\\\",\\\"required_title_version\\\": 1,"
        " \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 5  }}",

        "{ \\\"type\\\" : \\\"Dtl\\\",\\\"e_tag\\\":\\\"0123456789abcdef0123456789abcdef\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 15  }}",

        "{ \\\"type\\\" : \\\"VersionList\\\", \\\"e_tag\\\" : \\\"0123456789abcdef0123456789abcdef\\\", \\\"wait_policy\\\" : { \\\"type\\\": \\\"random\\\", \\\"limit\\\": 10  }}",
    };
    const int NotificationCount = sizeof(notifications) / sizeof(notifications[0]);

    TestNotificationReceive(handlers, types, 3, appId, notifications, NotificationCount);
    EXPECT_EQ("NDV", resultSequence);
}

// INFO: 正しいシステムアップデートトピックに通知が送られる。
//       ログを見て [ApplicationInstallRequestList] Version list updated  が出力されることを確認する。
#if 0
TEST_F(NotificationReceiveTest, SystemUpdate)
{
    PublishNotificationByTopic("nx_nup",
        "{\\\"type\\\":\\\"Nup\\\",\\\"title_id\\\":\\\"0100000000000816\\\",\\\"title_version\\\":0,\\\"required_title_id\\\":\\\"0100000000000816\\\",\\\"required_title_version\\\":0}");

}
#endif

// INFO: 正しいバージョンリストトピックに通知が送られる。
//       ログを見て [ApplicationInstallRequestList] Version list updated  が出力されることを確認する。
#if 0
TEST_F(NotificationReceiveTest, VersionListUpdate)
{
    PublishNotificationByTopic("nx_vlist",
        "{ \\\"type\\\" : \\\"VersionList\\\", \\\"e_tag\\\" : \\\"0123456789abcdef0123456789abcdef\\\"}");

}
#endif

// INFO: このテストは ns プロセス向けに DTL のダウンロード通知の送信のみ行う。
//       ログを見て ns(PushNotificationDispatcher) が RequestServer をキックしていることを確認する。
#if 0
TEST_F(NotificationReceiveTest, DtlNotifyForNs)
{
    nn::ApplicationId appId = { 0x010000000000001f };

    nn::npns::NotificationToken token;
    NNT_EXPECT_RESULT_SUCCESS(nn::npns::CreateToken(&token, {}, appId));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    PublishNotification(token,
        "{ \\\"type\\\" : \\\"Dtl\\\",\\\"e_tag\\\":\\\"0123456789abcdef0123456789abcdef\\\"}");
}
#endif

// INFO: このテストは ns プロセス向けにチケット利用可能通知の送信のみ行う。
//       ログを見て ns(PushNotificationDispatcher) が RequestServer をキックしていることを確認する。
#if 0
TEST_F(NotificationReceiveTest, TicketNotifyForNs)
{
    nn::ApplicationId appId = { 0x010000000000001f };

    nn::npns::NotificationToken token;
    NNT_EXPECT_RESULT_SUCCESS(nn::npns::CreateToken(&token, {}, appId));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    PublishNotification(token,
        "{ \\\"type\\\" : \\\"ETicketAvailable\\\", \\\"eci_account_id\\\" : \\\"12345678\\\", \\\"title_ids\\\" : [\\\"0x01004b9000490000\\\"]}");
}
#endif
