﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        スリープ中にバージョンリスト更新 ---> 途中起床してダウンロードが行われるか
        のテストを行います。

    .DESCRIPTION
        Setup-TargetEnvironment, Setup-TestApplication の事前実行が必要です。

    .PARAMETER Target
        ControlTarget, RunOnTarget の -t オプションに渡す文字列
#>

param
(
    [Parameter(Mandatory = $true)]
    [string]
    $Target,

    [switch]
    $AgingTest
)

$ErrorActionPreference = 'Stop'

Set-StrictMode -Version Latest

$ScriptPath      = $MyInvocation.MyCommand.Path
$ScriptDirectory = Split-Path $ScriptPath

# --------------------------------------------------------------------------
# 共通モジュール読み込み & 定数定義など
# --------------------------------------------------------------------------
Import-Module "$ScriptDirectory\Common"
. "$ScriptDirectory\Constants.ps1"
. "$ScriptDirectory\Functions.ps1"

$VersionListPushWait = 20

$DownloadTaskListJsonPath = "$IntermediatesDirectory\DownloadTaskList.json"
$SystemUpdateMetaNspPath = "$IntermediatesDirectory\SystemUpdateMeta.nsp"


# --------------------------------------------------------------------------
# テスト
# --------------------------------------------------------------------------
$TestSuccess = $false

try
{
    # 不具合調査用にデバイス ID を表示させる
    Write-InfoLog "Device ID = $(Get-TargetDeviceId $Target)"

    # バージョンリスト上のパッチバージョンを v1 にしておく (後の工程で v2 にした時に e_tag 値を確実に変えるため)
    Invoke-ContentsUploader register-version,--content-meta-id,$TestApplicationPatchId,--application-id,$TestApplicationId,--type,Patch,--content-meta-version,$TestApplicationV1Version,-e,td1

    # NUP ありの状態ではバージョンリストがダウンロードできないので
    # 最新の SystemUpdate をインストールして NUP 無し状態にする
    Invoke-DevMenuCommandSystem $Target systemprogram,install,$SystemUpdateMetaNspPath
    Invoke-TargetReset $Target

    # テストアプリ購入
    # DTL 更新の通知とダウンロードの開始を待つ
    Start-DemoDownload $Target $TestApplicationId
    Invoke-DevMenuCommandSystem $Target application,wait-download,$TestApplicationId -TrialCount 1 -Timeout 300

    # 現状 shop download-demo ではパッチが降りてこないので、とりあえず DTL を使う
    # 本来不要な処理
    $DownloadTaskList = @{tasks = @(@{
        id = '7c3714a1-4e0a-449c-8482-09c353b5b539'; owner_application = $TestApplicationId; titles = @(
            @{id = $TestApplicationPatchId; version = $TestApplicationV1Version; type = "Patch"}
        )})
    }
    if (-not (Test-Path (Split-Path $DownloadTaskListJsonPath)))
    {
        New-Item (Split-Path $DownloadTaskListJsonPath) -ItemType Directory | Out-Null
    }
    ConvertTo-Json20 $DownloadTaskList | Set-Content $DownloadTaskListJsonPath

    Invoke-DevMenuCommandSystem $Target application,push-download-task-list,$DownloadTaskListJsonPath
    Invoke-DevMenuCommandSystem $Target application,wait-download,$TestApplicationId -TrialCount 1 -Timeout 300

    # アプリとパッチが落ちてきていることを確認
    if (Test-ApplicationDownloadComplete $Target $TestApplicationId $TestApplicationV1Version)
    {
    }
    else
    {
        throw "App downloading didn't complete before sleep."
    }

    # バージョンリストを更新して、最新版が落ちてきていることを確認
    Invoke-DevMenuCommandSystem $Target application,request-version-list
    $IsUpdateRequestedLogs = Invoke-DevMenuCommandSystem $Target application,is-update-requested,$TestApplicationId
    if ($IsUpdateRequestedLogs -notcontains "Update is not requested.")
    {
        throw "App update is requested before sleep."
    }

    # アプリの自動更新がかかるように、一度起動する
    Invoke-RunOnTarget $Target "--failure-timeout",10,$TestApplicationId

    # 寝る
    Invoke-TargetSleep $Target

    # 適当な時間待つ
    Write-InfoLog "idle sleeping for $VersionListPushWait sec."
    Start-Sleep $VersionListPushWait

    # v2 パッチを最新版にして、バージョンリストの更新を通知
    Invoke-ContentsUploader register-version,--content-meta-id,$TestApplicationPatchId,--application-id,$TestApplicationId,--type,Patch,--content-meta-version,$TestApplicationV2Version,-e,td1

    for ($i = 0; ; ++$i)
    {
        # 何度かスリープさせてから起こす
        if ($AgingTest)
        {
            Wait-SleepCycle $Target -SleepCount 2
            Invoke-TargetWakeForAging $Target
        }
        else
        {
            Wait-SleepCycle $Target -SleepCount 3
            Invoke-TargetWake $Target
        }

        # アプリはダウンロードされているか、バージョンは想定通りか
        if (Test-ApplicationDownloadComplete $Target $TestApplicationId $TestApplicationV2Version -AllowVersionDifference)
        {
            break
        }
        else
        {
            Write-InfoLog "Download hasn't completed yet, but maybe because of some temporary network failures. Will enter sleep again."
        }

        # 寝る
        Invoke-TargetSleep $Target
    }

    # アプリがちゃんと起動するか確認
    Invoke-RunOnTarget $Target "--failure-timeout",10,$TestApplicationId

    Write-SuccessLog "TEST SUCCESS"
    $TestSuccess = $true
}
catch
{
    Write-FailureLog "TEST FAILURE"
    throw
}
finally
{
    # スリープから復帰しないことがあるので、テストに成功していないならリセットをかける
    if (-not $TestSuccess)
    {
        Invoke-TargetReset $Target
    }

    Teardown-DemoDownload $Target $TestApplicationId
    Teardown-SystemUpdate $Target
    Invoke-ContentsUploader register-version,--content-meta-id,$TestApplicationPatchId,--application-id,$TestApplicationId,--type,Patch,--content-meta-version,$TestApplicationV1Version,-e,td1
}
