﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <memory>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Abort.h>
#include <nn/es/es_InitializationApi.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ContentDeliveryApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_Result.h>

namespace {

    class ContentDeliveryManagerTest : public testing::Test
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT
        {
            nn::ncm::Initialize();
            nn::es::Initialize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ns::InitializeDependenciesForDfc();
#endif
            m_NspDirectoryPath = nnt::GetHostArgv()[1];
            NN_LOG("nspDirectoryPath %s\n", m_NspDirectoryPath);

        }

        virtual void TearDown() NN_NOEXCEPT
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ns::FinalizeDependenciesForDfc();
#endif
            nn::ncm::Finalize();
            nn::es::Finalize();
        }

        static void SetUpTestCase() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(nnt::GetHostArgc() > 1);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

// noclean でシステムセーブデータが消えない間のワークアラウンド
#if defined( NN_BUILD_CONFIG_OS_WIN)
            nn::fs::DeleteSaveData(0x8000000000000041ull);
#endif
        }

        static void TearDownTestCase() NN_NOEXCEPT
        {
            nn::fs::UnmountHostRoot();
        }

        const char* GetNspDirectoryPath() NN_NOEXCEPT
        {
            return m_NspDirectoryPath;
        }

        nn::Result DeleteContent(nn::Bit64 id, nn::ncm::StorageId storageId) NN_NOEXCEPT
        {
            nn::ncm::ContentMetaDatabase db;
            nn::ncm::ContentStorage storage;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, storageId));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, storageId));
            nn::ncm::ContentManagerAccessor accessor(&db, &storage);

            return accessor.DeleteAll(id);
        }

    private:
        const char* m_NspDirectoryPath;
    };

    class TestApplication
    {
    public:
        explicit TestApplication(const char* nspPath) NN_NOEXCEPT
        {
            // 現存する環境をクリアする
            {
                const auto flags = nn::ns::ApplicationEntityFlag_All | nn::ns::ApplicationEntityFlag_SkipGameCardCheck::Mask | nn::ns::ApplicationEntityFlag_SkipRunningCheck::Mask;
                nn::ns::ApplicationRecord recordList[128];
                auto recordCount = nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
                for (int i = 0; i < recordCount; i++)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::DeleteApplicationCompletelyForDebug(recordList[i].id, flags));
                }
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::DeleteRedundantApplicationEntity());

            nn::ncm::StorageId storage = nn::ncm::StorageId::BuiltInUser;

            // TORIAEZU: 指定されたディレクトリ内にある nsp を全てインストール
            std::vector<nn::ncm::ApplicationContentMetaKey> installed;
            nn::ncm::ApplicationId idList[128];
            int idCount = 0;
            {
                NNT_EXPECT_RESULT_SUCCESS(InstallAllTestApplications(&installed, nspPath, storage));

                nn::ns::ApplicationRecord recordList[128];
                auto recordCount = nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
                for (int i = 0; i < recordCount; i++)
                {
                    const auto& record = recordList[i];
                    idList[i] = record.id;
                }

                idCount = recordCount;
            }

            // TORIAEZU: インストールした中からパッチを見つける
            nn::util::optional<nn::ncm::ContentMetaKey> patchKey;
            nn::ncm::ApplicationId appId {};
            for (int i = 0; i < idCount; i++)
            {
                auto key = FindPatch(installed, idList[i]);
                if (key)
                {
                    patchKey = key;
                    appId = idList[i];
                    break;
                }
            }
            NN_ABORT_UNLESS(patchKey);
            m_AppId = appId;
            m_PatchKey = *patchKey;

            // m_AppId 以外のコンテンツを削除
            {
                for (int i = 0; i < idCount; i++)
                {
                    if (m_AppId != idList[i])
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::DeleteApplicationCompletely(idList[i]));
                    }
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::DeleteRedundantApplicationEntity());
            }
        }

        ~TestApplication() NN_NOEXCEPT
        {
            // Result は無視
            nn::ns::DeleteApplicationCompletely(m_AppId);
        }

        nn::ncm::ApplicationId GetApplicationId() NN_NOEXCEPT
        {
            return m_AppId;
        }

        nn::ncm::ContentMetaKey GetPatchContentMetaKey() NN_NOEXCEPT
        {
            return m_PatchKey;
        }


    private:
        nn::Result InstallAllTestApplications(std::vector<nn::ncm::ApplicationContentMetaKey>* outValue, const char* nspPath, nn::ncm::StorageId storage) NN_NOEXCEPT
        {
            return m_ApplicationInstaller.InstallAll(outValue, nspPath, storage);
        }

        nn::util::optional<nn::ncm::ContentMetaKey> FindPatch(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id) NN_NOEXCEPT
        {
            for (auto key : list)
            {
                if (key.applicationId.value == id.value && key.key.type == nn::ncm::ContentMetaType::Patch)
                {
                    return key.key;
                }
            }

            return nn::util::nullopt;
        }

        nnt::nsutil::ApplicationInstaller m_ApplicationInstaller;
        nn::ncm::ApplicationId m_AppId;
        nn::ncm::ContentMetaKey m_PatchKey;
    };

    // ns::InitializeDependenciesForDfc() がDEATH TEST でアボートしてしまうので初期化から全てをテスト関数で行う
    void TestCreateDownloadTask(const nn::ncm::ContentMetaKey& testKey) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(nnt::GetHostArgc() > 1);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

// noclean でシステムセーブデータが消えない間のワークアラウンド
#if defined( NN_BUILD_CONFIG_OS_WIN)
        nn::fs::DeleteSaveData(0x8000000000000041ull);
#endif

        nn::ncm::Initialize();
        nn::es::Initialize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
        nn::ns::InitializeDependenciesForDfc();
#endif
        const char* nspPath = nnt::GetHostArgv()[1];

        nn::ns::Initialize();

        {
            TestApplication app(nspPath);

            // ここでアボートするはず
            nn::ns::CreateDownloadTask(&testKey, 1, app.GetApplicationId());
        }

        nn::ns::Finalize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
        nn::ns::FinalizeDependenciesForDfc();
#endif
        nn::es::Finalize();
        nn::ncm::Finalize();
        nn::fs::UnmountHostRoot();
    }
}

#if !defined( NN_BUILD_CONFIG_OS_WIN)
TEST_F(ContentDeliveryManagerTest, BaseTest)
{
    nn::ns::Initialize();

    {
        TestApplication app(GetNspDirectoryPath());

        // SystemDeliveryInfo と ApplicationDeliveryInfo を集める
        nn::ns::SystemDeliveryInfo sysInfo;

        NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetSystemDeliveryInfo(&sysInfo));
        EXPECT_EQ(sysInfo.systemDeliveryProtocolVersion, 1);
        EXPECT_EQ(sysInfo.applicationDeliveryProtocolVersion, 1);

        auto invalidSysInfo = sysInfo;
        invalidSysInfo.systemUpdateVersion--;

        // SystemDeliveryInfo を比較
        int cmpResult;
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidSystemDeliveryInfo, nn::ns::CompareSystemDeliveryInfo(&cmpResult, sysInfo, invalidSysInfo));
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidSystemDeliveryInfo, nn::ns::CompareSystemDeliveryInfo(&cmpResult, invalidSysInfo, sysInfo));
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::CompareSystemDeliveryInfo(&cmpResult, sysInfo, sysInfo));
        EXPECT_EQ(cmpResult, 0);

        int appCount;
        nn::ns::ApplicationDeliveryInfo appInfo;
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationDeliveryInfo(&appCount, &appInfo, 1, app.GetApplicationId(), nn::ns::ApplicationDeliveryAttribute_RequestPatch::Mask));

        // プロトコルバージョンチェック
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::VerifyDeliveryProtocolVersion(sysInfo));
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidSystemDeliveryInfo, nn::ns::VerifyDeliveryProtocolVersion(invalidSysInfo));

        // アプリ配信可能の ApplicationDeliveryInfo かを調べる
        bool hasAll;
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::HasAllContentsToDeliver(&hasAll, &appInfo, 1));
        EXPECT_TRUE(hasAll);

        auto invalidAppInfo = appInfo;
        invalidAppInfo.version++;
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidApplicationDeliveryInfo, nn::ns::HasAllContentsToDeliver(&hasAll, &invalidAppInfo, 1));

        int cmp;
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::CompareApplicationDeliveryInfo(&cmp, &appInfo, 1, &appInfo, 1));
        EXPECT_EQ(0, cmp);
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidApplicationDeliveryInfo, nn::ns::CompareApplicationDeliveryInfo(&cmp, &appInfo, 1, &invalidAppInfo, 1));
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidApplicationDeliveryInfo, nn::ns::CompareApplicationDeliveryInfo(&cmp, &invalidAppInfo, 1, &appInfo, 1));

        bool canDeliver;
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::CanDeliverApplication(&canDeliver, &appInfo, 1, &appInfo, 1));
        EXPECT_TRUE(!canDeliver);
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidApplicationDeliveryInfo, nn::ns::CanDeliverApplication(&canDeliver, &invalidAppInfo, 1, &appInfo, 1));
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidApplicationDeliveryInfo, nn::ns::CanDeliverApplication(&canDeliver, &appInfo, 1, &invalidAppInfo, 1));

        bool needsUpdate;
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::NeedsSystemUpdateToDeliverApplication(&needsUpdate, &appInfo, 1, sysInfo));
        EXPECT_TRUE(!needsUpdate);
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidApplicationDeliveryInfo, nn::ns::NeedsSystemUpdateToDeliverApplication(&needsUpdate, &invalidAppInfo, 1, sysInfo));
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidSystemDeliveryInfo, nn::ns::NeedsSystemUpdateToDeliverApplication(&needsUpdate, &appInfo, 1, invalidSysInfo));

        // 最新の本体を探す
        int index;
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::SelectLatestSystemDeliveryInfo(&index, &sysInfo, 1, sysInfo, &appInfo, 1));
        EXPECT_EQ(0, index);
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidSystemDeliveryInfo, nn::ns::SelectLatestSystemDeliveryInfo(&index, &invalidSysInfo, 1, sysInfo, &appInfo, 1));
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidSystemDeliveryInfo, nn::ns::SelectLatestSystemDeliveryInfo(&index, &sysInfo, 1, invalidSysInfo, &appInfo, 1));
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidApplicationDeliveryInfo, nn::ns::SelectLatestSystemDeliveryInfo(&index, &sysInfo, 1, sysInfo, &invalidAppInfo, 1));

        const int ListCount = 16;
        nn::ncm::ContentMetaKey keyList[ListCount];
        int outKeyCount;
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::ListContentMetaKeyToDeliverApplication(&outKeyCount, keyList, ListCount, 0, &appInfo, 1));
        EXPECT_EQ(1, outKeyCount);
        int64_t contentSize;
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::EstimateRequiredSize(&contentSize, keyList, outKeyCount));
        EXPECT_GT(contentSize, 0);

        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultInvalidApplicationDeliveryInfo, nn::ns::ListContentMetaKeyToDeliverApplication(&outKeyCount, keyList, ListCount, 0, &invalidAppInfo, 1));
    }

    nn::ns::Finalize();
}
#endif

TEST(ContentDeliveryManagerDeathTest, CreateDownloadTask)
{
    const nn::ncm::ContentMetaKey invalidKeyList[] =
    {
        { 0x0100ffffffff0000, 0, nn::ncm::ContentMetaType::Unknown },
        { 0x0100ffffffff0000, 0, nn::ncm::ContentMetaType::SystemProgram },
        { 0x0100ffffffff0000, 0, nn::ncm::ContentMetaType::SystemData },
        { 0x0100ffffffff0000, 0, nn::ncm::ContentMetaType::SystemUpdate },
        { 0x0100ffffffff0000, 0, nn::ncm::ContentMetaType::BootImagePackage },
        { 0x0100ffffffff0000, 0, nn::ncm::ContentMetaType::BootImagePackageSafe },
        { 0x0100ffffffff0000, 0, nn::ncm::ContentMetaType::Delta },
        { 0x0100ffffffff0000, 0, static_cast<nn::ncm::ContentMetaType>(6) },
        { 0x0100ffffffff0000, 0, static_cast<nn::ncm::ContentMetaType>(127) },
        { 0x0100ffffffff0000, 0, static_cast<nn::ncm::ContentMetaType>(255) },
    };


    for (auto key : invalidKeyList)
    {
        EXPECT_DEATH_IF_SUPPORTED(TestCreateDownloadTask(key), "");
    }
}
