﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace ApplicationManagerOnTool
{
    [TestClass]
    public class ApplicationManagerToolTest
    {
        public TestContext TestContext { get; set; }

        private Tuple<StringBuilder, StringBuilder> SetupProcess(Process process, string filename, string arguments, FileStream fsOutput = null, FileStream fsErrorOutput = null)
        {
            var standardOutput = new StringBuilder();
            var standardError = new StringBuilder();

            process.StartInfo.FileName = filename;
            process.StartInfo.Arguments = arguments;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.StandardOutputEncoding = System.Text.Encoding.UTF8;
            process.OutputDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                if (fsOutput != null && e.Data != null)
                {
                    fsOutput.Write(Encoding.UTF8.GetBytes(e.Data), 0, e.Data.Length);
                    fsOutput.Write(Encoding.UTF8.GetBytes("\n"), 0, 1);
                }
                standardOutput.AppendLine(e.Data);
            };
            process.ErrorDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                if (fsErrorOutput != null && e.Data != null)
                {
                    fsErrorOutput.Write(Encoding.UTF8.GetBytes(e.Data), 0, e.Data.Length);
                    fsErrorOutput.Write(Encoding.UTF8.GetBytes("\n"), 0, 1);
                }
                standardError.AppendLine(e.Data);
            };

            return new Tuple<StringBuilder, StringBuilder>(standardOutput, standardError);
        }

        private void RunAndWaitProcess(Process process)
        {
            //Console.WriteLine("[ApplicationManagerOnTool] process start.");
            process.Start();
            process.BeginOutputReadLine();
            process.BeginErrorReadLine();
            //Console.WriteLine("[ApplicationManagerOnTool] process wait for exit.");
            process.WaitForExit();
        }

        private bool ExecuteTargetRebootProcess()
        {
            Console.WriteLine("[ApplicationManagerOnTool] ExecuteTargetRebootProcess Start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var exeFilePath = Path.Combine(testPath.GetSigloRoot(), @"Tools\CommandLineTools\ControlTarget.exe");

                var target = string.Empty;
                if (TestContext.Properties.Contains("TargetName"))
                {
                    target = string.Format(" -t {0}", (string)TestContext.Properties["TargetName"]);
                }

                var arguments = string.Format("connect --reset{0} -v", target);
                Console.WriteLine("[ApplicationManagerOnTool] ControlTarget argument : {0}", arguments);
                var outputs = SetupProcess(process, exeFilePath, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("[ApplicationManagerOnTool] Standard Output:\n{0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("[ApplicationManagerOnTool] Standard Error:\n{0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        private bool ExecuteMakeTestApplication(string xmlPath)
        {
            Console.WriteLine("[ApplicationManagerOnTool] ExecuteMakeTestApplication Start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var exeFilePath = Path.Combine(testPath.GetSigloRoot(), @"Tools\CommandLineTools\MakeTestApplication\MakeTestApplication.exe");
                var arguments = string.Format("--config-xml {0} -j 8 --small-code", xmlPath);
                Console.WriteLine("[ApplicationManagerOnTool] MakeTestApplication argument : {0}", arguments);
                var outputs = SetupProcess(process, exeFilePath, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("[ApplicationManagerOnTool] Standard Output:\n{0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("[ApplicationManagerOnTool] Standard Error:\n{0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        private bool ExecuteApplicationManagerTestTool(string outputPath, string csvPath, bool isSdFormat = false)
        {
            Console.WriteLine("[ApplicationManagerOnTool] ExecuteApplicationManagerTestTool Start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = Path.Combine(testPath.GetSigloRoot(), @"Tools\CommandLineTools\RunOnTarget.exe");
                var toolPath = Path.Combine(testPath.GetTestRoot(), @"Outputs\NX-NXFP2-a64\TargetTools\ApplicationManagerTestTool\Develop\ApplicationManagerTestTool.nsp");

                var target = string.Empty;
                if (TestContext.Properties.Contains("TargetName"))
                {
                    target = string.Format(" -t {0}", (string)TestContext.Properties["TargetName"]);
                }

                var formatOption = string.Empty;
                if (isSdFormat)
                {
                    formatOption = " --formatsd";
                }

                // -p オプションを付けて詳細なログを出力するようにしておく
                var arguments = string.Format("{0}{1} -- {2}{3} -p", toolPath, target, csvPath, formatOption);
                Console.WriteLine("[ApplicationManagerOnTool] ApplicationManagerTestTool argument : {0}", arguments);

                var csvSplitPath = csvPath.Split(new char[] { '\\' });
                var txtOutput = (outputPath + "\\TestResult_" + csvSplitPath[csvSplitPath.Length - 1]).Replace(".csv", ".txt");
                var txtErrorOutput = (outputPath + "\\ErrorTestResult_" + csvSplitPath[csvSplitPath.Length - 1]).Replace(".csv", ".txt");
                var fsOutput = new FileStream(txtOutput, FileMode.Create, FileAccess.ReadWrite);
                var fsErrorOutput = new FileStream(txtErrorOutput, FileMode.Create, FileAccess.ReadWrite);
                var outputs = SetupProcess(process, filename, arguments, fsOutput, fsErrorOutput);
                RunAndWaitProcess(process);

                fsOutput.Close();
                fsErrorOutput.Close();
                Console.WriteLine(string.Format("[ApplicationManagerOnTool] Standard Output:\n{0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("[ApplicationManagerOnTool] Standard Error:\n{0}", outputs.Item2.ToString()));

                var success = "[  PASSED  ] 1 test.";
                return outputs.Item1.ToString().Contains(success);
            }
        }

        private void ReplaceEnvirmentVariable(string inSrcFilePath, string inNewFilePath)
        {
            var srcStr = string.Empty;
            using (var sr = new StreamReader(inSrcFilePath))
            {
                // 巨大なファイルを読み込む訳ではないので、ファイル内容を文字列として(メモリに)格納しておく
                srcStr = sr.ReadToEnd();
            }

            // 念のためUTF8のBOM付き形式で保存する
            using (var sw = new StreamWriter(inNewFilePath, false, Encoding.UTF8))
            {
                // 環境変数文字列があれば置換した内容を新規ファイルに保存する
                sw.Write(System.Environment.ExpandEnvironmentVariables(srcStr));
            }
        }

        [TestMethod]
        [Timeout(170 * 1000)]
        public void TestProcess()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testWorkPath = Path.Combine(testPath.GetTestRoot(), @"Ns\Sources\Tests\ApplicationManagerOnTool");

            var testOutDirPath = Path.Combine(testWorkPath, @"testOutputDirectory");
            if (Directory.Exists(testOutDirPath))
            {
                Directory.Delete(testOutDirPath, true);
            }
            Directory.CreateDirectory(testOutDirPath);
            // SDカードのフォーマット処理
            Assert.IsTrue(this.ExecuteApplicationManagerTestTool(testOutDirPath, "dummyPath", true));
            // ターゲットの開発機を再起動させる
            Assert.IsTrue(this.ExecuteTargetRebootProcess());

            var configXmlPath = Path.Combine(testWorkPath, @"InputTestData\MakeTestApplication_TestConfig.xml");
            var nspOutDirPath = Path.Combine(testWorkPath, @"nspOutputDirectory");
            if (Directory.Exists(nspOutDirPath))
            {
                Directory.Delete(nspOutDirPath, true);
            }
            Directory.CreateDirectory(nspOutDirPath);

            Assert.IsTrue(this.ExecuteMakeTestApplication(configXmlPath));

            if (Directory.Exists(testOutDirPath))
            {
                Directory.Delete(testOutDirPath, true);
            }
            Directory.CreateDirectory(testOutDirPath);

            var srcTestCsvPath = Path.Combine(testWorkPath, @"InputTestData\ApplicationManagerTestTool_InputData.csv");
            var replacedTestCsvPath = Path.Combine(testWorkPath, @"InputTestData\ApplicationManagerTestTool_InputData_ReplacedEnvVar.csv");
            if (File.Exists(replacedTestCsvPath))
            {
                File.Delete(replacedTestCsvPath);
            }

            // (SIGLO-49026) 環境変数の置換処理をここでやってしまう
            ReplaceEnvirmentVariable(srcTestCsvPath, replacedTestCsvPath);

            // ApplicationManagerTestTool を実行
            Assert.IsTrue(ExecuteApplicationManagerTestTool(testOutDirPath, replacedTestCsvPath));

            // 念のため新たに作成した入力用ファイルは消しておく
            File.Delete(replacedTestCsvPath);

            // 念のため作成したテスト用の nsp ファイルは消しておく
            Directory.Delete(nspOutDirPath, true);
        }
    }
}
