﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/es/es_InitializationApi.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/nifm.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_DocumentApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_TerminateResultApi.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/result/testResult_Assert.h>

#if NN_BUILD_CONFIG_TOOLCHAIN_VC

#pragma execution_character_set("utf-8")
#define USTRING(s) s

#else
#define USTRING(s) u8 ## s
#endif

namespace {

    void TestApplicationControlProperty(const nn::ns::ApplicationControlProperty& property, nn::ncm::ApplicationId id)
    {
        switch (id.value)
        {
        // パッチを当てた時に値が反映されているかをテスト
        case 0x0005000c10000000:
            {
                auto& japaneseTitle = property.GetTitle(nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("パッチタイトル")), japaneseTitle.name);
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("パッチパブリッシャー")), japaneseTitle.publisher);

                auto& americanEnglishTitle = property.GetTitle(nn::settings::LanguageCode::Make(nn::settings::Language_AmericanEnglish));
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("Patch Title")), americanEnglishTitle.name);
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("Patch Publisher")), americanEnglishTitle.publisher);

                auto& defaultTitle = property.GetDefaultTitle();
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("Patch Title")), defaultTitle.name);
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("Patch Publisher")), defaultTitle.publisher);

                EXPECT_STREQ("012-3-45-678901-2", property.isbn);

                EXPECT_EQ(nn::ns::StartupUserAccount::Required, property.startupUserAccount);
                EXPECT_TRUE((property.attributeFlag & static_cast<nn::Bit32>(nn::ns::AttributeFlag::Demo)) == static_cast<nn::Bit32>(nn::ns::AttributeFlag::Demo));
                EXPECT_TRUE((property.attributeFlag & static_cast<nn::Bit32>(nn::ns::AttributeFlag::RetailInteractiveDisplay)) == static_cast<nn::Bit32>(nn::ns::AttributeFlag::RetailInteractiveDisplay));
                EXPECT_TRUE((property.parentalControlFlag & static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication)) != 0);
                EXPECT_TRUE(property.SupportsLanguage(nn::settings::LanguageCode::Make(nn::settings::Language_Japanese)));
                EXPECT_TRUE(property.SupportsLanguage(nn::settings::LanguageCode::Make(nn::settings::Language_AmericanEnglish)));
                EXPECT_EQ(0x0005000c10000003LL, property.presenceGroupId.value);
                EXPECT_EQ(nn::ns::Screenshot::Deny, property.screenShot);
                EXPECT_EQ(nn::ns::VideoCapture::Enable, property.videoCapture);
                EXPECT_EQ(18, property.GetAge(nn::ns::RatingOrganization::CERO));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::GRACGCRB));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::GSRMR));
                EXPECT_EQ(17, property.GetAge(nn::ns::RatingOrganization::ESRB));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::ClassInd));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::USK));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::PEGI));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::PEGIPortugal));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::PEGIBBFC));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::Russian));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::ACB));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::OFLC));

                EXPECT_EQ(0x0005000c10000000LL, property.saveDataOwnerId);
                EXPECT_EQ(0x0000000000400000LL, property.userAccountSaveDataSize);
                EXPECT_EQ(0x0000000000100000LL, property.userAccountSaveDataJournalSize);
                EXPECT_EQ(0x0000000001000000LL, property.deviceSaveDataSize);
                EXPECT_EQ(0x0000000000300000LL, property.deviceSaveDataJournalSize);
                EXPECT_EQ(0x0000000000800000LL, property.bcatDeliveryCacheStorageSize);
                EXPECT_EQ(nn::ns::LogoType::LicensedByNintendo, property.logoType);
                for (int i = 1; i < 8; i++)
                {
                    EXPECT_EQ(0x0005000c10000000LL, property.localCommunicationId[i]);
                }
                EXPECT_EQ(nn::ns::LogoHandling::Auto, property.logoHandling);
                EXPECT_EQ(0x0005000c1000000FLL, property.seedForPseudoDeviceId);
                EXPECT_STREQ("ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789", property.bcatPassphrase);
                EXPECT_EQ(0x0000000006000000LL, property.temporaryStorageSize);
                EXPECT_EQ(0x0000000005000000LL, property.cacheStorageSize);
                EXPECT_EQ(0x0000000004000000LL, property.cacheStorageJournalSize);
                EXPECT_EQ(nn::ns::Hdcp::Required, property.hdcp);
                EXPECT_EQ(nn::ns::CrashReport::Allow, property.crashReport);
                EXPECT_EQ(nn::ns::RuntimeAddOnContentInstall::AllowAppend, property.runtimeAddOnContentInstall);
                EXPECT_EQ(nn::ns::PlayLogQueryCapability::All, property.playLogQueryCapability);
                EXPECT_TRUE((property.repairFlag & static_cast<nn::Bit8>(nn::ns::RepairFlag::SuppressGameCardAccess)) == static_cast<nn::Bit8>(nn::ns::RepairFlag::SuppressGameCardAccess));
                EXPECT_TRUE((property.requiredNetworkServiceLicenseOnLaunchFlag & static_cast<nn::Bit8>(nn::ns::RequiredNetworkServiceLicenseOnLaunchFlag::Common)) == static_cast<nn::Bit8>(nn::ns::RequiredNetworkServiceLicenseOnLaunchFlag::Common));
            }
            break;
        // 設定した値が反映されているかをテスト
        case 0x0005000c10000001:
            {
                auto& japaneseTitle = property.GetTitle(nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７")), japaneseTitle.name);
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３")), japaneseTitle.publisher);

                auto& americanEnglishTitle = property.GetTitle(nn::settings::LanguageCode::Make(nn::settings::Language_AmericanEnglish));
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567")), americanEnglishTitle.name);
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("123456789012345678901234567890123456789012345678901234567890123")), americanEnglishTitle.publisher);

                auto& defaultTitle = property.GetDefaultTitle();
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567")), defaultTitle.name);
                EXPECT_STREQ(reinterpret_cast<const char*>(USTRING("123456789012345678901234567890123456789012345678901234567890123")), defaultTitle.publisher);

                EXPECT_STREQ("012-3-45-678901-2", property.isbn);

                EXPECT_EQ(nn::ns::StartupUserAccount::RequiredWithNetworkServiceAccountAvailable, property.startupUserAccount);
                EXPECT_TRUE((property.attributeFlag & static_cast<nn::Bit32>(nn::ns::AttributeFlag::Demo)) == static_cast<nn::Bit32>(nn::ns::AttributeFlag::Demo));
                EXPECT_TRUE((property.attributeFlag & static_cast<nn::Bit32>(nn::ns::AttributeFlag::RetailInteractiveDisplay)) == 0);
                EXPECT_TRUE((property.parentalControlFlag & static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication)) != 0);
                EXPECT_TRUE(property.SupportsLanguage(nn::settings::LanguageCode::Make(nn::settings::Language_Japanese)));
                EXPECT_TRUE(property.SupportsLanguage(nn::settings::LanguageCode::Make(nn::settings::Language_AmericanEnglish)));
                EXPECT_EQ(0x0005000c10000002LL, property.presenceGroupId.value);
                EXPECT_EQ(nn::ns::Screenshot::Deny, property.screenShot);
                EXPECT_EQ(nn::ns::VideoCapture::Manual, property.videoCapture);
                EXPECT_EQ(nn::ns::DataLossConfirmation::Required, property.dataLossConfirmation);
                EXPECT_EQ(nn::ns::PlayLogPolicy::None, property.playLogPolicy);
                EXPECT_EQ(18, property.GetAge(nn::ns::RatingOrganization::CERO));
                EXPECT_EQ(15, property.GetAge(nn::ns::RatingOrganization::GRACGCRB));
                EXPECT_EQ(16, property.GetAge(nn::ns::RatingOrganization::GSRMR));
                EXPECT_EQ(17, property.GetAge(nn::ns::RatingOrganization::ESRB));
                EXPECT_EQ(0, property.GetAge(nn::ns::RatingOrganization::ClassInd));
                EXPECT_EQ(3, property.GetAge(nn::ns::RatingOrganization::USK));
                EXPECT_EQ(7, property.GetAge(nn::ns::RatingOrganization::PEGI));
                EXPECT_EQ(6, property.GetAge(nn::ns::RatingOrganization::PEGIPortugal));
                EXPECT_EQ(4, property.GetAge(nn::ns::RatingOrganization::PEGIBBFC));
                EXPECT_EQ(12, property.GetAge(nn::ns::RatingOrganization::Russian));
                EXPECT_EQ(13, property.GetAge(nn::ns::RatingOrganization::ACB));
                EXPECT_EQ(8, property.GetAge(nn::ns::RatingOrganization::OFLC));
                EXPECT_STREQ("1.0.0", property.displayVersion);
                EXPECT_EQ(0x0005000c10000001LL, property.saveDataOwnerId);
                EXPECT_EQ(0x0000000000400000LL, property.userAccountSaveDataSize);
                EXPECT_EQ(0x0000000000100000LL, property.userAccountSaveDataJournalSize);
                EXPECT_EQ(0x0000000001000000LL, property.deviceSaveDataSize);
                EXPECT_EQ(0x0000000000300000LL, property.deviceSaveDataJournalSize);
                EXPECT_EQ(0x0000000000800000LL, property.bcatDeliveryCacheStorageSize);
                EXPECT_STREQ("ABC123", property.applicationErrorCodeCategory.value);
                EXPECT_EQ(nn::ns::LogoType::Nintendo, property.logoType);
                EXPECT_EQ(0x0005000c10000001LL, property.localCommunicationId[0]);
                EXPECT_EQ(0x0005000c10000002LL, property.localCommunicationId[1]);
                for (int i = 2; i < 8; i++)
                {
                    EXPECT_EQ(0x0005000c10000001LL, property.localCommunicationId[i]);
                }
                EXPECT_EQ(nn::ns::LogoHandling::Manual, property.logoHandling);
                EXPECT_EQ(0x0005000c1000000FLL, property.seedForPseudoDeviceId);
                EXPECT_STREQ("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF", property.bcatPassphrase);
                EXPECT_EQ(0x0000000003000000LL, property.temporaryStorageSize);
                EXPECT_EQ(0x0000000002000000LL, property.cacheStorageSize);
                EXPECT_EQ(0x0000000001000000LL, property.cacheStorageJournalSize);
                EXPECT_EQ(nn::ns::Hdcp::Required, property.hdcp);
                EXPECT_EQ(nn::ns::CrashReport::Allow, property.crashReport);
                EXPECT_EQ(nn::ns::RuntimeAddOnContentInstall::AllowAppend, property.runtimeAddOnContentInstall);
                EXPECT_EQ(nn::ns::PlayLogQueryCapability::WhiteList, property.playLogQueryCapability);
                EXPECT_EQ(0x0005000c10000001LL, property.playLogQueryableApplicationId[0].value);
                EXPECT_EQ(0x0005000c10000002LL, property.playLogQueryableApplicationId[1].value);
                for (int i = 2; i < 8; i++)
                {
                    EXPECT_EQ(nn::ncm::ApplicationId::GetInvalidId(), property.playLogQueryableApplicationId[i]);
                }
                EXPECT_TRUE((property.repairFlag & static_cast<nn::Bit8>(nn::ns::RepairFlag::SuppressGameCardAccess)) == static_cast<nn::Bit8>(nn::ns::RepairFlag::SuppressGameCardAccess));
                EXPECT_TRUE((property.requiredNetworkServiceLicenseOnLaunchFlag & static_cast<nn::Bit8>(nn::ns::RequiredNetworkServiceLicenseOnLaunchFlag::Common)) == static_cast<nn::Bit8>(nn::ns::RequiredNetworkServiceLicenseOnLaunchFlag::Common));
            }
            break;
        // デフォルト値をテスト
        case 0x0005000c10000002:
            {
                EXPECT_STREQ("", property.isbn);
                EXPECT_EQ(nn::ns::StartupUserAccount::None, property.startupUserAccount);
                EXPECT_TRUE(property.attributeFlag == 0);
                EXPECT_FALSE((property.parentalControlFlag & static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication)) != 0);
                EXPECT_TRUE(property.SupportsLanguage(nn::settings::LanguageCode::Make(nn::settings::Language_Japanese)));
                EXPECT_TRUE(property.SupportsLanguage(nn::settings::LanguageCode::Make(nn::settings::Language_AmericanEnglish)));
                EXPECT_TRUE(property.SupportsLanguage(nn::settings::LanguageCode::Make(nn::settings::Language_French)));
                EXPECT_TRUE(property.SupportsLanguage(nn::settings::LanguageCode::Make(nn::settings::Language_German)));
                EXPECT_EQ(0x0005000c10000002, property.presenceGroupId.value);
                EXPECT_EQ(nn::ns::Screenshot::Allow, property.screenShot);
                EXPECT_EQ(nn::ns::VideoCapture::Enable, property.videoCapture);
                EXPECT_EQ(nn::ns::DataLossConfirmation::None, property.dataLossConfirmation);
                EXPECT_EQ(nn::ns::PlayLogPolicy::All, property.playLogPolicy);
                EXPECT_EQ(0x0005000c10000002LL, property.saveDataOwnerId);
                EXPECT_EQ(0x0LL, property.userAccountSaveDataSize);
                EXPECT_EQ(0x0LL, property.userAccountSaveDataJournalSize);
                EXPECT_EQ(0x0LL, property.deviceSaveDataSize);
                EXPECT_EQ(0x0LL, property.deviceSaveDataJournalSize);
                EXPECT_EQ(0x0LL, property.bcatDeliveryCacheStorageSize);
                EXPECT_STREQ("", property.applicationErrorCodeCategory.value);
                EXPECT_EQ(nn::ns::LogoType::LicensedByNintendo, property.logoType);
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::CERO));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::GRACGCRB));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::GSRMR));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::ESRB));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::ClassInd));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::USK));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::PEGI));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::PEGIPortugal));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::PEGIBBFC));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::Russian));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::ACB));
                EXPECT_EQ(-1, property.GetAge(nn::ns::RatingOrganization::OFLC));
                for (int i = 1; i < 8; i++)
                {
                    EXPECT_EQ(0x0005000c10000002LL, property.localCommunicationId[i]);
                }
                EXPECT_EQ(nn::ns::LogoHandling::Auto, property.logoHandling);
                EXPECT_EQ(0x0005000c10000002LL, property.seedForPseudoDeviceId);
                EXPECT_STREQ("", property.bcatPassphrase);
                EXPECT_EQ(0x0ULL, property.temporaryStorageSize);
                EXPECT_EQ(0x0ULL, property.cacheStorageSize);
                EXPECT_EQ(0x0ULL, property.cacheStorageJournalSize);
                EXPECT_EQ(nn::ns::Hdcp::None, property.hdcp);
                EXPECT_EQ(nn::ns::CrashReport::Deny, property.crashReport);
                EXPECT_EQ(nn::ns::RuntimeAddOnContentInstall::Deny, property.runtimeAddOnContentInstall);
                EXPECT_EQ(nn::ns::PlayLogQueryCapability::None, property.playLogQueryCapability);
                EXPECT_TRUE(property.repairFlag == 0);
                EXPECT_TRUE(property.requiredNetworkServiceLicenseOnLaunchFlag == static_cast<nn::Bit8>(nn::ns::RequiredNetworkServiceLicenseOnLaunchFlag::None));
            }
            break;
        default:
            return;
        }
    } // NOLINT(impl/function_size)

    class ApplicationManagerTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();
            nn::es::Initialize();
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
            nn::ns::InitializeDependenciesForDfc();
#endif
            m_OutputDirectoryPath = std::string(nnt::GetHostArgv()[1]);
            NN_LOG("outputDirectoryPath %s\n", m_OutputDirectoryPath.c_str());
            m_ContentsDirectoryPath = std::string(nnt::GetHostArgv()[2]);
            NN_LOG("contentsDirectoryPath %s\n", m_ContentsDirectoryPath.c_str());
        }

        virtual void TearDown()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
            nn::ns::FinalizeDependenciesForDfc();
#endif
            nn::ncm::Finalize();
            nn::es::Finalize();
        }

        static void SetUpTestCase()
        {
            NN_ASSERT(nnt::GetHostArgc() > 2);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

// noclean でシステムセーブデータが消えない間のワークアラウンド
#if defined( NN_BUILD_CONFIG_OS_WIN)
            nn::fs::DeleteSaveData(0x8000000000000041ull);
#endif
        }

        static void TearDownTestCase()
        {
            nn::fs::UnmountHostRoot();
        }

        const char* GetSystemTitleOutputDirectoryPath()
        {
            return m_SystemTitleOutputDirectoryPath.c_str();
        }

        const char* GetOutputDirectoryPath()
        {
            return m_OutputDirectoryPath.c_str();
        }

        const char* GetContentsDirectoryPath()
        {
            return m_ContentsDirectoryPath.c_str();
        }

        std::string GetApplicationControlPropertyPath(nn::Bit64 id)
        {
            char idString[32];
            nn::util::SNPrintf(idString, sizeof(idString), "0x%016llx", id);

            return m_ContentsDirectoryPath + "/" + idString + ".nacp";
        }

        std::string GetIconDataPath(nn::Bit64 id)
        {
            char idString[32];
            nn::util::SNPrintf(idString, sizeof(idString), "0x%016llx", id);

            return m_ContentsDirectoryPath + "/" + idString + ".jpg";
        }

        std::string GetLogoDataPath()
        {
            return m_ContentsDirectoryPath + "/" + "logo.bin";
        }

        nn::Result InstallAllTestApplications(std::vector<nn::ncm::ApplicationContentMetaKey>* outValue, nn::ncm::StorageId storage)
        {
            return m_ApplicationInstaller.InstallAll(outValue, GetOutputDirectoryPath(), storage);
        }

        nn::Result InstallAllTestSystemTitles(std::vector<std::string>* outValue, nn::ncm::StorageId storage)
        {
            return m_SubmissionPackageInstaller.InstallAll(outValue, GetSystemTitleOutputDirectoryPath(), storage);
        }

        nn::Result InstallAllRedundantApplications()
        {
            std::vector<std::string> installed;
            return m_SubmissionPackageInstaller.InstallAll(&installed, GetOutputDirectoryPath());
        }

        nn::Result DeleteContent(nn::Bit64 id, nn::ncm::StorageId storageId)
        {
            nn::ncm::ContentMetaDatabase db;
            nn::ncm::ContentStorage storage;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, storageId));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, storageId));
            nn::ncm::ContentManagerAccessor accessor(&db, &storage);

            return accessor.DeleteAll(id);
        }

        bool Find(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            for (auto key : list)
            {
                if (key.applicationId.value == id.value)
                {
                    return true;
                }
            }

            return false;
        }

        bool FindMain(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            for (auto key : list)
            {
                if (key.applicationId.value == id.value && key.key.type == nn::ncm::ContentMetaType::Application)
                {
                    return true;
                }
            }

            return false;
        }

        nn::util::optional<nn::ncm::ContentMetaKey> FindPatch(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            for (auto key : list)
            {
                if (key.applicationId.value == id.value && key.key.type == nn::ncm::ContentMetaType::Patch)
                {
                    return key.key;
                }
            }

            return nn::util::nullopt;
        }

        nn::util::optional<nn::ncm::ContentMetaKey> FindAddOnContent(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            for (auto key : list)
            {
                if (key.applicationId.value == id.value && key.key.type == nn::ncm::ContentMetaType::AddOnContent)
                {
                    return key.key;
                }
            }

            return nn::util::nullopt;
        }

        uint32_t FindVersion(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            uint32_t version = 0;
            for (auto key : list)
            {
                if (key.applicationId.value == id.value && (key.key.type == nn::ncm::ContentMetaType::Patch || key.key.type == nn::ncm::ContentMetaType::Application))
                {
                    if (version < key.key.version)
                    {
                        version = key.key.version;
                    }
                }
            }

            return version;
        }

    private:
        nnt::nsutil::ApplicationInstaller m_ApplicationInstaller;
        nnt::ncmutil::SubmissionPackageFileInstaller m_SubmissionPackageInstaller;
        std::string m_OutputDirectoryPath;
        std::string m_ContentsDirectoryPath;
        std::string m_SystemTitleOutputDirectoryPath;
    };
}

TEST_F(ApplicationManagerTest, Basic)
{
    nn::ns::Initialize();
    nn::ncm::StorageId storageList[] = { nn::ncm::StorageId::BuildInUser, nn::ncm::StorageId::Any };

    for (auto storage : storageList)
    {
        nn::ns::InvalidateAllApplicationControlCache();

        // 現存する環境をクリアする
        {
            const auto flags = nn::ns::ApplicationEntityFlag_All | nn::ns::ApplicationEntityFlag_SkipGameCardCheck::Mask | nn::ns::ApplicationEntityFlag_SkipRunningCheck::Mask;
            nn::ns::ApplicationRecord recordList[128];
            auto recordCount = nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
            for (int i = 0; i < recordCount; i++)
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteApplicationCompletelyForDebug(recordList[i].id, flags));
            }
        }
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteRedundantApplicationEntity());

        std::vector<nn::ncm::ApplicationContentMetaKey> installed;
        NNT_EXPECT_RESULT_SUCCESS(InstallAllTestApplications(&installed, storage));

        EXPECT_FALSE(nn::ns::IsAnyApplicationEntityRedundant());

        nn::ncm::ApplicationId idList[128];
        int idCount;
        {
            nn::ns::ApplicationRecord recordList[128];
            auto recordCount = nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
            for (int i = 0; i < recordCount; i++)
            {
                const auto& record = recordList[i];
                EXPECT_EQ(nn::ns::ApplicationEvent::LocalInstalled, record.lastEvent);
                if (i > 0)
                {
                    EXPECT_LT(recordList[i].lastUpdated, recordList[i - 1].lastUpdated);
                }
                EXPECT_TRUE(Find(installed, record.id));
                idList[i] = record.id;
            }

            idCount = recordCount;
        }

        {
            nn::ns::ApplicationView viewList[128];
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationView(viewList, idList, idCount));
            for (int i = 0; i < idCount; i++)
            {
                const auto& view = viewList[i];
                EXPECT_EQ(FindVersion(installed, idList[i]), view.version);
                EXPECT_EQ((view.version & 0xffff), 0);
                EXPECT_EQ(idList[i].value, view.id.value);

                auto hasMain = FindMain(installed, idList[i]);
                auto hasPatch = FindPatch(installed, idList[i]);
                auto hasAddOnContent = FindAddOnContent(installed, idList[i]);
                EXPECT_TRUE(view.HasRecord());
                EXPECT_FALSE(view.IsDownloading());
                EXPECT_FALSE(view.IsGameCard());
                EXPECT_FALSE(view.HasGameCardEntity());
                EXPECT_EQ(hasMain, view.IsLaunchable());
                EXPECT_EQ(hasMain, view.HasMainRecord());
                EXPECT_EQ(hasMain, view.HasMainEntity());
                EXPECT_EQ(hasPatch, view.HasPatchRecord());
                EXPECT_EQ(hasPatch, view.HasPatchEntity());
                EXPECT_EQ(hasAddOnContent, view.HasAddOnContentRecord());
                EXPECT_EQ(hasAddOnContent, view.HasAddOnContentRecord() && view.HasAllAddOnContentEntity());
                EXPECT_TRUE(view.HasAllEntity());
                EXPECT_FALSE(view.MaybeCorrupted());
                EXPECT_FALSE(view.IsAutoDeleteDisabled());
                EXPECT_FALSE(view.IsWaitingAocCommit());
                EXPECT_FALSE(view.IsWaitingPatchInstall());

                NNT_EXPECT_RESULT_SUCCESS(nn::ns::SetApplicationTerminateResult(idList[i], nn::fs::ResultDataCorrupted()));
                NNT_EXPECT_RESULT_SUCCESS(nn::ns::DisableApplicationAutoDelete(idList[i]));

                nn::ns::ApplicationView newView;
                NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationView(&newView, &idList[i], 1));
                EXPECT_TRUE(newView.MaybeCorrupted());
                EXPECT_TRUE(newView.IsAutoDeleteDisabled());

            }
        }

        for (int i = 0; i < idCount; i++)
        {
            nn::ns::ApplicationOccupiedSize occupiedSize;
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::CalculateApplicationOccupiedSize(&occupiedSize, idList[i]));
            for (auto& storageEntry : occupiedSize.storage)
            {
                NN_LOG("occupiedSize.storageId %lld\n", storageEntry.storageId);
                NN_LOG("occupiedSize.appSize %lld\n", storageEntry.appSize);
                NN_LOG("occupiedSize.patchSize %lld\n", storageEntry.patchSize);
                NN_LOG("occupiedSize.aocSize %lld\n", storageEntry.aocSize);
            }
        }

        for (int i = 0; i < idCount; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::CheckApplicationLaunchRights(idList[i]));
        }

        // DevMenu がGetApplicationControlData を呼び出してしまうので、一旦キャッシュをクリアしておく
        // Invalidate してから DevMenu が起動して、呼び出した場合は対応できない
        nn::ns::InvalidateAllApplicationControlCache();

        for (int i = 0; i < idCount; i++)
        {
            if (!FindMain(installed, idList[i]))
            {
                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultApplicationContentNotFound, nn::ns::MountApplicationHtmlDocument("doc", idList[i]));
                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultApplicationContentNotFound, nn::ns::MountApplicationLegalInformation("doc", idList[i]));
                // TODO: 権利チェックの処理が実装されたら細かいテストを作る
                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultApplicationRecordNotFound, nn::ns::CheckApplicationLaunchRights(idList[i]));
                continue;
            }

            EXPECT_TRUE(nn::ns::IsAnyApplicationEntityInstalled(idList[i]));

            auto patchKey = FindPatch(installed, idList[i]);
            auto pathId = patchKey ? patchKey->id : idList[i].value;

            size_t bufferSize = 1024 * 1024;
            std::unique_ptr<nn::ns::ApplicationControlProperty> property(new nn::ns::ApplicationControlProperty());
            {
                nn::fs::FileHandle file;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&file, GetApplicationControlPropertyPath(pathId).c_str(), nn::fs::OpenMode_Read));
                NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, property.get(), sizeof(nn::ns::ApplicationControlProperty)));
            }

            std::unique_ptr<char[]> jpgBuffer(new char[bufferSize]);
            size_t jpgSize;
            {
                nn::fs::FileHandle file;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&file, GetIconDataPath(pathId).c_str(), nn::fs::OpenMode_Read));
                NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(&jpgSize, file, 0, jpgBuffer.get(), bufferSize));
            }

            // TODO: 権利チェックの処理が実装されたら細かいテストを作る
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::CheckApplicationLaunchRights(idList[i]));

            std::unique_ptr<char[]> buffer(new char[bufferSize]);
            size_t outSize;
            NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultApplicationControlDataNotFound, nn::ns::GetApplicationControlData(&outSize, buffer.get(), bufferSize, nn::ns::ApplicationControlSource::CacheOnly, idList[i]));
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationControlData(&outSize, buffer.get(), bufferSize, nn::ns::ApplicationControlSource::Storage, idList[i]));
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationControlData(&outSize, buffer.get(), outSize, nn::ns::ApplicationControlSource::Storage, idList[i]));
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationControlData(&outSize, buffer.get(), bufferSize, nn::ns::ApplicationControlSource::CacheOnly, idList[i]));
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationControlData(&outSize, buffer.get(), outSize, nn::ns::ApplicationControlSource::CacheOnly, idList[i]));
            nn::ns::ApplicationControlDataAccessor accessor(buffer.get(), outSize);
            EXPECT_TRUE(std::memcmp(&accessor.GetProperty(), property.get(), sizeof(property)) == 0);
            EXPECT_EQ(jpgSize, accessor.GetIconSize());
            EXPECT_TRUE(std::memcmp(accessor.GetIconData(), jpgBuffer.get(), jpgSize) == 0);
            TestApplicationControlProperty(accessor.GetProperty(), idList[i]);

            const size_t smallBufferSize = 1;
            std::unique_ptr<char[]> smallBuffer(new char[smallBufferSize]);
            NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultBufferNotEnough, nn::ns::GetApplicationControlData(&outSize, smallBuffer.get(), smallBufferSize, nn::ns::ApplicationControlSource::CacheOnly, idList[i]));
            NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultBufferNotEnough, nn::ns::GetApplicationControlData(&outSize, smallBuffer.get(), smallBufferSize, nn::ns::ApplicationControlSource::Storage, idList[i]));
            NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultBufferNotEnough, nn::ns::GetApplicationControlData(&outSize, smallBuffer.get(), smallBufferSize, nn::ns::ApplicationControlSource::StorageOnly, idList[i]));

            // nn::ns::MountApplicationHtmlDocument() がアプリ起動中であること前提となったため無効化
            // NNT_EXPECT_RESULT_SUCCESS(nn::ns::MountApplicationHtmlDocument("doc", idList[i]));
            // NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount("doc"); };

            NNT_EXPECT_RESULT_SUCCESS(nn::ns::MountApplicationLegalInformation("legaldoc", idList[i]));
            NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount("legaldoc"); };

            // ロゴ
            {
                size_t readSize;
                const size_t LogoBufferSize = 512 * 1024;
                std::unique_ptr<char[]> logoBuffer(new char[LogoBufferSize]);
                NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationLogoData(&readSize, logoBuffer.get(), LogoBufferSize, idList[i], "logo.bin"));

                size_t masterLogoSize;
                std::unique_ptr<char[]> masterLogoBuffer(new char[LogoBufferSize]);
                {
                    nn::fs::FileHandle file;
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&file, GetLogoDataPath().c_str(), nn::fs::OpenMode_Read));
                    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(&masterLogoSize, file, 0, masterLogoBuffer.get(), LogoBufferSize));
                }
                EXPECT_EQ(masterLogoSize, readSize);
                EXPECT_TRUE(std::memcmp(masterLogoBuffer.get(), logoBuffer.get(), masterLogoSize) == 0);

                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultLogoPathNotFound, nn::ns::GetApplicationLogoData(&readSize, logoBuffer.get(), LogoBufferSize, idList[i], "not_found.bin"));
            }
        }

        for (int i = 0; i < idCount; i++)
        {
            if (!FindMain(installed, idList[i]))
            {
                nn::util::optional<nn::Result> result;
                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultApplicationRecordNotFound, nn::ns::IsApplicationUpdateRequested(&result, idList[i]));
                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultApplicationRecordNotFound, nn::ns::RequestApplicationUpdate(idList[i], nn::ns::ResultApplicationControlDataNotFound()));
                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultApplicationRecordNotFound, nn::ns::WithdrawApplicationUpdateRequest(idList[i]));
                continue;
            }

            nn::util::optional<nn::Result> result;
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::IsApplicationUpdateRequested(&result, idList[i]));
            EXPECT_FALSE(result);

            NNT_EXPECT_RESULT_SUCCESS(nn::ns::RequestApplicationUpdate(idList[i], nn::ns::ResultApplicationControlDataNotFound()));
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::IsApplicationUpdateRequested(&result, idList[i]));
            EXPECT_TRUE(result);
            NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultApplicationControlDataNotFound, *result);

            NNT_EXPECT_RESULT_SUCCESS(nn::ns::WithdrawApplicationUpdateRequest(idList[i]));
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::IsApplicationUpdateRequested(&result, idList[i]));
            EXPECT_FALSE(result);
        }

        // ここから削除系
        for (int i = 0; i < idCount; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteApplicationEntity(idList[i]));

            EXPECT_FALSE(nn::ns::IsAnyApplicationEntityInstalled(idList[i]));
        }

        {
            nn::ns::ApplicationView viewList[128];
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationView(viewList, idList, idCount));
            for (int i = 0; i < idCount; i++)
            {
                const auto& view = viewList[i];
                EXPECT_EQ(FindVersion(installed, idList[i]), view.version);
                EXPECT_EQ(idList[i].value, view.id.value);

                auto hasMain = FindMain(installed, idList[i]);
                auto hasPatch = FindPatch(installed, idList[i]);
                auto hasAddOnContent = FindAddOnContent(installed, idList[i]);

                EXPECT_TRUE(view.HasRecord());
                EXPECT_FALSE(view.IsDownloading());
                EXPECT_FALSE(view.IsGameCard());
                EXPECT_FALSE(view.IsLaunchable());
                EXPECT_FALSE(view.HasAllEntity());
                EXPECT_EQ(hasMain, view.HasMainRecord());
                EXPECT_FALSE(view.HasMainEntity());
                EXPECT_EQ(hasPatch, view.HasPatchRecord());
                EXPECT_FALSE(view.HasPatchEntity());
                EXPECT_EQ(hasAddOnContent, view.HasAddOnContentRecord());
                EXPECT_EQ(hasAddOnContent, view.HasAddOnContentRecord() && !view.HasAllAddOnContentEntity());
                EXPECT_TRUE(view.MaybeCorrupted());
                EXPECT_TRUE(view.IsAutoDeleteDisabled());
                EXPECT_FALSE(view.IsWaitingAocCommit());
                EXPECT_FALSE(view.IsWaitingPatchInstall());

                NNT_EXPECT_RESULT_SUCCESS(nn::ns::ClearApplicationTerminateResult(idList[i]));
                NNT_EXPECT_RESULT_SUCCESS(nn::ns::EnableApplicationAutoDelete(idList[i]));

                nn::ns::ApplicationView newView;
                NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationView(&newView, &idList[i], 1));

                EXPECT_FALSE(newView.MaybeCorrupted());
                EXPECT_FALSE(newView.IsAutoDeleteDisabled());
            }
        }

        for (int i = 0; i < idCount; i++)
        {
            nn::ns::ApplicationOccupiedSize occupiedSize;
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::CalculateApplicationOccupiedSize(&occupiedSize, idList[i]));
            for (auto& storageEntry : occupiedSize.storage)
            {
                EXPECT_EQ(0, storageEntry.appSize);
                EXPECT_EQ(0, storageEntry.patchSize);
                EXPECT_EQ(0, storageEntry.aocSize);
            }
        }

        for (int i = 0; i < idCount; i++)
        {
            size_t bufferSize = 1024 * 1024;
            std::unique_ptr<char[]> buffer(new char[bufferSize]);
            size_t outSize;
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetApplicationControlData(&outSize, buffer.get(), bufferSize, nn::ns::ApplicationControlSource::Storage, idList[i]));
        }

        auto maxCacheCount = nn::ns::GetMaxApplicationControlCacheCount();
        EXPECT_LT(0, maxCacheCount);
        nn::ns::InvalidateAllApplicationControlCache();
        for (int i = 0; i < idCount; i++)
        {
            size_t bufferSize = 1024 * 1024;
            std::unique_ptr<char[]> buffer(new char[bufferSize]);
            size_t outSize;
            NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultApplicationControlDataNotFound, nn::ns::GetApplicationControlData(&outSize, buffer.get(), bufferSize, nn::ns::ApplicationControlSource::Storage, idList[i]));
        }

        for (int i = 0; i < idCount; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteApplicationCompletely(idList[i]));
        }

        {
            nn::ns::ApplicationRecord recordList[128];
            auto recordCount = nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
            EXPECT_EQ(0, recordCount);
        }

        NNT_EXPECT_RESULT_SUCCESS(InstallAllRedundantApplications());
        EXPECT_TRUE(nn::ns::IsAnyApplicationEntityRedundant());
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteRedundantApplicationEntity());
    }

    nn::ns::Finalize();
} // NOLINT(impl/function_size)

