﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>
#include <iostream>
#include <memory>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnt/result/testResult_Assert.h>

#include "ApplicationManagerTestTool_Common.h"
#include "ApplicationManagerTestTool_TestDataContainer.h"
#include "ApplicationManagerTestTool_FsUtilities.h"
#include "ApplicationManagerTestTool_NsTypeConverter.h"
#include "ApplicationManagerTestTool_HtcUtil.h"

// 読み込む CSV ファイルが UTF-16 (BOM付)である場合は以下の定義を有効化する
//#define UTF_16_CONVERT_PROCESS

#ifdef UTF_16_CONVERT_PROCESS
#include <nn/util/util_CharacterEncoding.h>
#else // UTF_16_CONVERT_PROCESS
#include <array>
#endif // UTF_16_CONVERT_PROCESS

const char* APPLICATION_TYPE         = "Type";
const char* INSTALL_FILE_PATH        = "InstallFilePath";
const char* INSTALL_STORAGE          = "InstallStorage";
const char* MOVE_DESTINATION_STORAGE = "MoveDestinationStorage";
const char* DELETE_FLAG              = "DeleteFlag";
const char* APPLICATION_ID           = "ApplicationId";
const char* APPLICATION_SIZE         = "Size";
//const char* LAUNCH_RIGHT = "LaunchRight";
//const char* ICON_FILE_PATH = "IconFilePath";

const char* APP_TITLE_LANGUAGE                       = "App_Title_Language";
const char* APP_TITLE_NAME                           = "App_Title_Name";
const char* APP_TITLE_PUBLISHER                      = "App_Title_Publisher";
const char* APP_VERSION                              = "App_Version";
const char* APP_DISPLAY_VERSION                      = "App_DisplayVersion";
const char* APP_ISBN                                 = "App_ISBN";
const char* APP_STARTUP_USER_ACCOUNT                 = "App_StartupUserAccount";
const char* APP_ATTRIBUTE                            = "App_Attribute";
const char* APP_SUPPORTED_LANGUAGE                   = "App_SupportedLanguage";
const char* APP_PARENTAL_CONTROL                     = "App_ParentalControl";
const char* APP_PRESENCE_GROUP_ID                    = "App_PresenceGroupId";
const char* APP_SCREEN_SHOT                          = "App_ScreenShot";
const char* APP_VIDEO_CAPTURE                        = "App_VideoCapture";
const char* APP_DATA_LOSS_CONFIRMATION               = "App_DataLossConfirmation";
const char* APP_PLAY_LOG_POLICY                      = "App_PlayLogPolicy";
const char* APP_ADD_ON_CONTENT_BASE_ID               = "App_AddOnContentBaseId";
const char* APP_SAVE_DATA_OWNER_ID                   = "App_SaveDataOwnerId";
const char* APP_USER_ACCOUNT_SAVE_DATA_SIZE          = "App_UserAccountSaveDataSize";
const char* APP_USER_ACCOUNT_SAVE_DATA_JOURNAL_SIZE  = "App_UserAccountSaveDataJournalSize";
const char* APP_DEVICE_SAVE_DATA_SIZE                = "App_DeviceSaveDataSize";
const char* APP_DEVICE_SAVE_DATA_JOURNAL_SIZE        = "App_DeviceSaveDataJournalSize";
const char* APP_BCAT_DELIVERY_CACHE_STORAGE_SIZE     = "App_BcatDeliveryCacheStorageSize";
const char* APP_ERROR_CODE_CATEGORY                  = "App_ApplicationErrorCodeCategory";
const char* APP_LOGO_TYPE                            = "App_LogoType";
const char* APP_LOGO_HANDLING                        = "App_LogoHandling";
const char* APP_RATING_ORGANIZATION                  = "App_Rating_Organization";
const char* APP_RATING_AGE                           = "App_Rating_Age";
const char* APP_LOCAL_COMMUNICATION_ID               = "App_LocalCommunicationId";
const char* APP_RELEASE_VERSION                      = "App_ReleaseVersion";
const char* APP_PRIVATE_VERSION                      = "App_PrivateVersion";
const char* APP_SEED_FOR_PSEUDO_DEVICE_ID            = "App_SeedForPseudoDeviceId";
const char* APP_BCAT_PASS_PHRASE                     = "App_BcatPassphrase";
const char* APP_HDCP                                 = "App_Hdcp";
const char* APP_USER_ACCOUNT_SAVE_DATA_SIZE_MAX      = "App_UserAccountSaveDataSizeMax";
const char* APP_USER_ACCOUNT_SAVE_DATA_JOURNAL_SIZE_MAX = "App_UserAccountSaveDataJournalSizeMax";
const char* APP_DEVICE_SAVE_DATA_SIZE_MAX            = "App_DeviceSaveDataSizeMax";
const char* APP_DEVICE_SAVE_DATA_JOURNAL_SIZE_MAX    = "App_DeviceSaveDataJournalSizeMax";
const char* APP_TEMPORARY_STORAGE_SIZE               = "App_TemporaryStorageSize";
const char* APP_CACHE_STORAGE_SIZE                   = "App_CacheStorageSize";
const char* APP_CACHE_STORAGE_JOURNAL_SIZE           = "App_CacheStorageJournalSize";
const char* APP_CRASH_REPORT                         = "App_CrashReport";
const char* APP_RUNTIME_ADD_ON_CONTENT_INSTALL       = "App_RuntimeAddOnContentInstall";
const char* APP_PLAY_LOG_QUERY_CAPABILITY            = "App_PlayLogQueryCapability";
const char* APP_PLAY_LOG_QUERYABLE_APPLICATION_ID    = "App_PlayLogQueryableApplicationId";
const char* APP_REPAIR_FLAG                          = "App_RepairFlag";
const char* APP_PROGRAM_INDEX                        = "App_ProgramIndex";
const char* APP_REQUIRED_NETWORK_SERVICE_LICENCE_ON_LAUNCH = "App_RequiredNetworkServiceLicenseOnLaunch";

const char* AOC_INDEX                        = "Aoc_Index";
const char* AOC_RELEASE_VERSION              = "Aoc_ReleaseVersion";
const char* AOC_APPLICAITON_ID               = "Aoc_ApplicatonId";
const char* AOC_REQUIRED_APPLICATION_VERSION = "Aoc_RequiredApplicationVersion";
const char* AOC_TAG                          = "Aoc_Tag";
const char* AOC_DATA_PATH                    = "Aoc_DataPath";

TestDataContainer::TestDataContainer() NN_NOEXCEPT
{
    this->InitializeTitleIndexMap();
    this->Clear();
}

void TestDataContainer::InitializeTitleIndexMap() NN_NOEXCEPT
{
    this->m_TitleIndexMap.clear();
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APPLICATION_TYPE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(INSTALL_FILE_PATH), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(INSTALL_STORAGE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(MOVE_DESTINATION_STORAGE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(DELETE_FLAG), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APPLICATION_SIZE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APPLICATION_ID), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_VERSION), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_TITLE_LANGUAGE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_TITLE_NAME), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_TITLE_PUBLISHER), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_DISPLAY_VERSION), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_ISBN), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_STARTUP_USER_ACCOUNT), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_ATTRIBUTE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_SUPPORTED_LANGUAGE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_PARENTAL_CONTROL), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_PRESENCE_GROUP_ID), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_SCREEN_SHOT), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_VIDEO_CAPTURE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_DATA_LOSS_CONFIRMATION), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_PLAY_LOG_POLICY), 0));
    //this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_ADD_ON_CONTENT_BASE_ID), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_SAVE_DATA_OWNER_ID), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_USER_ACCOUNT_SAVE_DATA_SIZE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_USER_ACCOUNT_SAVE_DATA_JOURNAL_SIZE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_DEVICE_SAVE_DATA_SIZE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_DEVICE_SAVE_DATA_JOURNAL_SIZE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_BCAT_DELIVERY_CACHE_STORAGE_SIZE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_ERROR_CODE_CATEGORY), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_LOGO_TYPE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_LOGO_HANDLING), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_RATING_ORGANIZATION), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_RATING_AGE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_LOCAL_COMMUNICATION_ID), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_RELEASE_VERSION), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_PRIVATE_VERSION), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_SEED_FOR_PSEUDO_DEVICE_ID), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_BCAT_PASS_PHRASE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_HDCP), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_USER_ACCOUNT_SAVE_DATA_SIZE_MAX), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_USER_ACCOUNT_SAVE_DATA_JOURNAL_SIZE_MAX), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_DEVICE_SAVE_DATA_SIZE_MAX), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_DEVICE_SAVE_DATA_JOURNAL_SIZE_MAX), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_TEMPORARY_STORAGE_SIZE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_CACHE_STORAGE_SIZE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_CACHE_STORAGE_JOURNAL_SIZE), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_CRASH_REPORT), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_RUNTIME_ADD_ON_CONTENT_INSTALL), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_PLAY_LOG_QUERY_CAPABILITY), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_PLAY_LOG_QUERYABLE_APPLICATION_ID), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_REPAIR_FLAG), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_PROGRAM_INDEX), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(APP_REQUIRED_NETWORK_SERVICE_LICENCE_ON_LAUNCH), 0));

    this->m_TitleIndexMap.insert(std::make_pair(std::string(AOC_INDEX), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(AOC_RELEASE_VERSION), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(AOC_APPLICAITON_ID), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(AOC_REQUIRED_APPLICATION_VERSION), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(AOC_TAG), 0));
    this->m_TitleIndexMap.insert(std::make_pair(std::string(AOC_DATA_PATH), 0));
}

TestDataContainer::~TestDataContainer() NN_NOEXCEPT
{
}

// テストデータの元になる一覧データの読み込み
// m_TestDataList に展開する
nn::Result TestDataContainer::ReadFromCSV(const std::string& inFilePath) NN_NOEXCEPT
{
    nn::Result result;

    //NN_LOG("ReadFromCSV() : path=%s\n", inFilePath.c_str());

    fsutil::File file;
    result = file.Open(inFilePath.c_str(), nn::fs::OpenMode_Read);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        NN_LOG("CSVFile Path Open Failed : path=%s", inFilePath.c_str());
        return result;
    }

    const size_t fileSize = static_cast<size_t>(file.GetSize());
    //NN_LOG("ReadFromCSV() : fileSize = %lld\n", fileSize);
    std::unique_ptr<char[]> buf(new char[fileSize + 1]);

    result = file.Read(0, buf.get(), fileSize);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        NN_LOG("CSVFile Path Read Failed : path=%s", inFilePath.c_str());
        return result;
    }
    // 念のため明示的にクローズしておく
    file.Close();

#ifdef UTF_16_CONVERT_PROCESS
    // UTF-16 BOM付きのファイルデータとして読み込まれているはず？なので、UTF-8 変換を行う
    const int bomSize = 2;
    auto utf16Size = (fileSize - bomSize) / sizeof(uint16_t);
    std::unique_ptr<uint16_t[]> buf16(new uint16_t[utf16Size + 1]);
    memcpy(buf16.get(), (buf.get() + bomSize), (fileSize - bomSize));
    // 文字列として扱うため、末尾にヌル文字を入れておく
    buf16[utf16Size] = L'\0';

    buf.reset();

    int utf8Size = 0;
    auto encResult = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&utf8Size, buf16.get(), static_cast<int>(utf16Size));
    if (encResult != nn::util::CharacterEncodingResult_Success)
    {
        NN_LOG(" [Error] GetLengthOfConvertedStringUtf16NativeToUtf8() Failed : %d\n", encResult);
    }

    std::unique_ptr<char[]> utf8buf(new char[utf8Size + 1]);
    if (utf8Size > 0 && encResult == nn::util::CharacterEncodingResult_Success)
    {
        encResult = nn::util::ConvertStringUtf16NativeToUtf8(utf8buf.get(), (utf8Size + 1), buf16.get());
        if (encResult != nn::util::CharacterEncodingResult_Success)
        {
            NN_LOG(" [Error] ConvertStringUtf16NativeToUtf8() Failed : %d\n", encResult);
        }
    }
    buf16.reset();
    // CSVデータの文字列としてパース処理を行う
    auto table = this->ParseTable(utf8buf.get(), ",");
    utf8buf.reset();
#else // UTF_16_CONVERT_PROCESS
    // UTF-8 のファイルデータがそのまま読み込まれた場合の処理
    // 文字列として扱うため、末尾にヌル文字を入れておく
    buf[fileSize] = '\0';

    // BOM 付きデータにも対応するための判定処理
    char* strPtr = buf.get();
    const std::array<uint8_t, 3> utf8BomData = { { 0xEF, 0xBB, 0xBF } };
    if (memcmp(buf.get(), utf8BomData.data(), utf8BomData.size()) == 0)
    {
        // BOM付きデータであるためBOMのサイズ分ずらす
        strPtr += utf8BomData.size();
    }

    // CSVデータの文字列としてパース処理を行う
    auto table = this->ParseTable(strPtr, ",");
    buf.reset();
#endif // UTF_16_CONVERT_PROCESS

    // パースしたCSVデータを本クラスのテストデータのコンテナ型へ変換する
    this->ToTestDataContainer(table);

    NN_RESULT_SUCCESS;
}

// テストデータリストのクリア処理
void TestDataContainer::Clear() NN_NOEXCEPT
{
    this->m_TestDataList.clear();
}


TestDataContainer::CsvDataTable
TestDataContainer::ParseTable(const std::string& inData, const std::string& delim) NN_NOEXCEPT
{
    std::stringstream ss(inData);
    CsvDataTable table;

    const auto delimSize = delim.size();
    CsvRecordTable record;
    std::string lineStr;
    while (getline(ss, lineStr))
    {
        // \r が末尾に存在する場合を考慮して、\r が存在すれば消しておく
        if (*lineStr.rbegin() == '\r')
        {
            lineStr.erase(lineStr.size() - 1);
        }

        if (lineStr.size() <= 1)
        {
            continue;
        }

        std::vector<std::string> tokens;

        size_t currentPos = 0;
        auto foundPos = lineStr.find(delim, currentPos);
        while (foundPos != std::string::npos)
        {
            tokens.emplace_back(lineStr, currentPos, foundPos - currentPos);
            currentPos = foundPos + delimSize;
            foundPos = lineStr.find(delim, currentPos);
        }
        tokens.emplace_back(lineStr, currentPos, lineStr.size() - currentPos);

        if (record.empty() == false && tokens[0] != "")
        {
            table.push_back(record);
            record.clear();
        }
        record.emplace_back(tokens);
    }

    if (record.empty() == false)
    {
        table.push_back(record);
    }

#if 0
    // デバッグ用のログ出力処理
    for (auto& record : table)
    {
        for (auto& line : record)
        {
            NN_LOG(":");
            for (auto& cell : line)
            {
                NN_LOG("\"%s\":", cell.c_str());
            }
            NN_LOG("\n");
        }
        NN_LOG("----------------------------\n");
    }
#endif

    return table;
}

void TestDataContainer::ToTestDataContainer(CsvDataTable& inCSVData) NN_NOEXCEPT
{
    if (inCSVData.size() < 2)
    {
        // CSVデータとしては不完全なデータ
        NN_LOG(" [Error] Not CSV Data\n");
        return;
    }

    auto iterRecord = inCSVData.begin();
    for (auto& titlePair : m_TitleIndexMap)
    {
        titlePair.second = this->FindIndexVector((*iterRecord)[0], titlePair.first);
    }

    // 2行目以降のデータが各テストデータのレコードとなる
    ++iterRecord;
    for ( ; iterRecord != inCSVData.end(); ++iterRecord)
    {
        TestDataRecord record;

        this->SetCsvRecordData(record, *iterRecord);

        m_TestDataList.push_back(record);
    }
}

int TestDataContainer::FindIndexVector(CsvRecord& inFindList, const std::string& inFindValue) NN_NOEXCEPT
{
    // 検索対象の値のイテレーターを取得
    auto iter = std::find(inFindList.begin(), inFindList.end(), inFindValue);
    // 取得したイテレーターのインデックス値を計算
    size_t index = std::distance(inFindList.begin(), iter);
    if (index == inFindList.size())
    {
        NN_LOG(" [Warning] Not Found CSV_Title : %s\n", inFindValue.c_str());
        // 該当なしであれば -1 を返す
        return -1;
    }
    return static_cast<int>(index);
}

std::string TestDataContainer::SafeAt(CsvRecord& vec, const std::string& titleStr) NN_NOEXCEPT
{
    const int index = m_TitleIndexMap[titleStr];
    if (index < 0 || static_cast<int>(vec.size()) <= index)
    {
        // リストの範囲外をアクセスしようとすると空文字を返しておく
        //NN_LOG(" [Warning] Invalid Index : %d (%s)\n", index, titleStr.c_str());
        return std::string("");
    }
    return vec.at(index);
}

nn::Bit32 TestDataContainer::GetFlagValue(CsvRecordTable& inRecordTable, const std::string& inTitleName) NN_NOEXCEPT
{
    nn::Bit32 flagValue = 0x00000000u;

    for (auto& record : inRecordTable)
    {
        auto val = SafeAt(record, inTitleName);
        if (val == "")
        {
            break;
        }

        if (inTitleName == APP_ATTRIBUTE)
        {
            auto attrVal = NsTypeConv::ToAttributeFlag(val);
            flagValue |= static_cast<nn::Bit32>(attrVal);
        }
        else if (inTitleName == APP_PARENTAL_CONTROL)
        {
            auto prVal = NsTypeConv::ToParentalControlFlag(val);
            flagValue |= static_cast<nn::Bit32>(prVal);
        }
        else
        {
            // ここに来ることはありえないと思うが強制的にループを抜ける
            break;
        }
    }

    return flagValue;
}

nn::Bit64 TestDataContainer::GetDefaultAppId(CsvRecord& inFindList, const std::string& inFindValue, nn::Bit64 inAppId) NN_NOEXCEPT
{
    auto strIdValue = SafeAt(inFindList, inFindValue);
    if (strIdValue == "")
    {
        return inAppId;
    }
    return StringUtil::ToNum<int64_t>(strIdValue);
}


void TestDataContainer::SetCsvRecordData(TestDataRecord& outRecord, CsvRecordTable& inCSVRecordData) NN_NOEXCEPT
{
    auto& rec = outRecord;
    auto& rd = inCSVRecordData[0];

    rec.type = SafeAt(rd, APPLICATION_TYPE);
    if (rec.type == "")
    {
        // もし空文字の場合はデフォルト値である Application を設定しておく
        rec.type = CONTENT_TYPE_APPLICATION;
    }

    rec.installFilePath = SafeAt(rd, INSTALL_FILE_PATH);
    // 環境変数が含まれている場合を考慮する
    rec.installFilePath = HtcUtil::ReplaceEnvironmentVariable(rec.installFilePath);

    rec.installStorage = SafeAt(rd, INSTALL_STORAGE);
    if (rec.installStorage == "")
    {
        // 空文字(設定なし)の場合は auto をデフォルト設定としておく
        rec.installStorage = "auto";
    }

    rec.moveDestinationStorage = SafeAt(rd, MOVE_DESTINATION_STORAGE);
    {
        rec.isDeleteFlag = true;
        auto isDeleteStr = SafeAt(rd, DELETE_FLAG);
        if (isDeleteStr.empty() == true || isDeleteStr == "0")
        {
            rec.isDeleteFlag = false;
        }
    }

    this->SetCsvRecordData_Size(rec, rd);

    rec.id.value = StringUtil::ToNum<int64_t>(SafeAt(rd, APPLICATION_ID));
    //auto str = At(rd, APPLICATION_ID);
    //NN_LOG("SetCsvRecordData [APPLICATION_ID] : %lld [0x%016llx] (%s)\n", rec.id.value, rec.id.value, str.c_str());
    rec.view.id.value = rec.id.value;
    rec.view.version = StringUtil::ToBit32(SafeAt(rd, APP_VERSION));

    {
        auto& ctl = rec.controlData;

        this->SetCsvRecordData_Title(ctl, inCSVRecordData);

        ctl.isbn = SafeAt(rd, APP_ISBN);
        ctl.displayVersion = SafeAt(rd, APP_DISPLAY_VERSION);

        ctl.startupUserAccount = NsTypeConv::ToStartupUserAccount(
            SafeAt(rd, APP_STARTUP_USER_ACCOUNT));

        ctl.attributeFlag = this->GetFlagValue(inCSVRecordData, APP_ATTRIBUTE);

        this->SetCsvRecordData_SupportedLanguage(ctl, inCSVRecordData);

        ctl.parentalControlFlag = this->GetFlagValue(inCSVRecordData, APP_PARENTAL_CONTROL);

        ctl.presenceGroupId.value = this->GetDefaultAppId(rd, APP_PRESENCE_GROUP_ID, rec.id.value);

        ctl.screenShot = NsTypeConv::ToScreenshot(
            SafeAt(rd, APP_SCREEN_SHOT));
        ctl.videoCapture = NsTypeConv::ToVideoCapture(
            SafeAt(rd, APP_VIDEO_CAPTURE));
        ctl.dataLossConfirmation = NsTypeConv::ToDataLossConfirmation(
            SafeAt(rd, APP_DATA_LOSS_CONFIRMATION));
        ctl.playLogPolicy = NsTypeConv::ToPlayLogPolicy(
            SafeAt(rd, APP_PLAY_LOG_POLICY));

        // ・Todo
        // addOnContentBaseId

        this->SetCsvRecordData_LocalCommunicationId(ctl, inCSVRecordData, rec.id.value);

        ctl.saveDataOwnerId = this->GetDefaultAppId(rd, APP_SAVE_DATA_OWNER_ID, rec.id.value);

        ctl.userAccountSaveDataSize = StringUtil::ToNum<int64_t>(
            SafeAt(rd, APP_USER_ACCOUNT_SAVE_DATA_SIZE));
        ctl.userAccountSaveDataJournalSize = StringUtil::ToNum<int64_t>(
            SafeAt(rd, APP_USER_ACCOUNT_SAVE_DATA_JOURNAL_SIZE));
        ctl.deviceSaveDataSize = StringUtil::ToNum<int64_t>(
            SafeAt(rd, APP_DEVICE_SAVE_DATA_SIZE));
        ctl.deviceSaveDataJournalSize = StringUtil::ToNum<int64_t>(
            SafeAt(rd, APP_DEVICE_SAVE_DATA_JOURNAL_SIZE));
        ctl.bcatDeliveryCacheStorageSize = StringUtil::ToNum<int64_t>(
            SafeAt(rd, APP_BCAT_DELIVERY_CACHE_STORAGE_SIZE));

        ctl.applicationErrorCodeCategory = SafeAt(rd, APP_ERROR_CODE_CATEGORY);

        ctl.logoType = NsTypeConv::ToLogoType(SafeAt(rd, APP_LOGO_TYPE));
        ctl.logoHandling = NsTypeConv::ToLogoHandling(SafeAt(rd, APP_LOGO_HANDLING));

        this->SetCsvRecordData_Rating(ctl, inCSVRecordData);

        ctl.releaseVersion = SafeAt(rd, APP_RELEASE_VERSION);
        ctl.privateVersion = SafeAt(rd, APP_PRIVATE_VERSION);

        // SIGLO-46420, SIGLO-46752 対応
        ctl.seedForPseudoDeviceId = this->GetDefaultAppId(rd, APP_SEED_FOR_PSEUDO_DEVICE_ID, rec.id.value);
        ctl.bcatPassphrase = SafeAt(rd, APP_BCAT_PASS_PHRASE);

        // (SIGLO-51085) AddOnContentRegistrationType は実質的に固定値(NXAddon 1.1.0 以降は OnDemand)
        ctl.aocRegistrationType = nn::ns::AddOnContentRegistrationType::OnDemand;

        // (SIGLO-61528) HDCP の追加
        ctl.hdcp = NsTypeConv::ToHdcp(SafeAt(rd, APP_HDCP));

        // (SIGLO-62037) 一時データ保存領域の追加
        ctl.userAccountSaveDataSizeMax = StringUtil::ToNum<int64_t>(SafeAt(rd, APP_USER_ACCOUNT_SAVE_DATA_SIZE_MAX));
        ctl.userAccountSaveDataJournalSizeMax = StringUtil::ToNum<int64_t>(SafeAt(rd, APP_USER_ACCOUNT_SAVE_DATA_JOURNAL_SIZE_MAX));
        ctl.deviceSaveDataSizeMax = StringUtil::ToNum<int64_t>(SafeAt(rd, APP_DEVICE_SAVE_DATA_SIZE_MAX));
        ctl.deviceSaveDataJournalSizeMax = StringUtil::ToNum<int64_t>(SafeAt(rd, APP_DEVICE_SAVE_DATA_JOURNAL_SIZE_MAX));
        ctl.temporaryStorageSize = StringUtil::ToNum<int64_t>(SafeAt(rd, APP_TEMPORARY_STORAGE_SIZE));
        ctl.cacheStorageSize = StringUtil::ToNum<int64_t>(SafeAt(rd, APP_CACHE_STORAGE_SIZE));
        ctl.cacheStorageJournalSize = StringUtil::ToNum<int64_t>(SafeAt(rd, APP_CACHE_STORAGE_JOURNAL_SIZE));

        // (SIGLO-64934) CrashReport の追加
        ctl.crashReport = NsTypeConv::ToCrashReport(SafeAt(rd, APP_CRASH_REPORT));

        // (SIGLO-67759) RuntimeAddOnContentInstall の追加
        ctl.runtimeAddOnContentInstall = NsTypeConv::ToRuntimeAddOnContentInstall(SafeAt(rd, APP_RUNTIME_ADD_ON_CONTENT_INSTALL));

        // (SIGLO-73968) プレイ情報の取得権限設定の追加
        this->SetCsvRecordData_PlayLogQueryCapability(ctl, inCSVRecordData);

        // (SIGLO-73969) 修理ツールの動作指定設定の追加
        ctl.repairFlag = NsTypeConv::ToRepairFlag(SafeAt(rd, APP_REPAIR_FLAG));

        ctl.programIndex = StringUtil::ToNum<uint32_t>(SafeAt(rd, APP_PROGRAM_INDEX));

        // (SIGLO-83936) 起動に必要なネットワークライセンス設定の追加(将来複数の値が出てきたときに、複数選択可能)
        this->SetCsvRecordData_RequiredNetworkServiceLicenseOnLaunch(ctl, inCSVRecordData);
    }

    this->SetCsvRecordData_Aoc(rec, inCSVRecordData);
}

void TestDataContainer::SetCsvRecordData_Title(ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT
{
    for (auto& record : inRecordTable)
    {
        auto lang = SafeAt(record, APP_TITLE_LANGUAGE);
        if (lang == "")
        {
            break;
        }

        auto settingLang = NsTypeConv::ToSettingLanguage(lang);

        ApplicationControlData::AppTitle title;
        title.name = SafeAt(record, APP_TITLE_NAME);
        title.publisher = SafeAt(record, APP_TITLE_PUBLISHER);

        outData.titleList.insert(std::make_pair(settingLang, title));
    }
}

void TestDataContainer::SetCsvRecordData_Rating(ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT
{
    for (auto& record : inRecordTable)
    {
        auto org = SafeAt(record, APP_RATING_ORGANIZATION);
        if (org == "")
        {
            break;
        }

        auto ratingOrganization = NsTypeConv::ToRatingOrganization(org);
        auto age = static_cast<int8_t>(StringUtil::ToBit32(
            SafeAt(record, APP_RATING_AGE)));

        outData.ratingList.insert(std::make_pair(ratingOrganization, age));
    }
}

void TestDataContainer::SetCsvRecordData_SupportedLanguage(ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT
{
    for (auto& record : inRecordTable)
    {
        auto lang = SafeAt(record, APP_SUPPORTED_LANGUAGE);
        if (lang == "")
        {
            break;
        }

        auto settingLang = NsTypeConv::ToSettingLanguage(lang);
        outData.supportedLanguageList.push_back(settingLang);
    }
}

void TestDataContainer::SetCsvRecordData_LocalCommunicationId(ApplicationControlData& outData, CsvRecordTable& inRecordTable, nn::Bit64 inAppId) NN_NOEXCEPT
{
    for (auto& record : inRecordTable)
    {
        auto strId = SafeAt(record, APP_LOCAL_COMMUNICATION_ID);
        if (strId == "")
        {
            break;
        }

        auto id = StringUtil::ToNum<int64_t>(strId);
        outData.localCommunicationId.push_back(id);
    }

    if (outData.localCommunicationId.empty() == true)
    {
        // もし設定値が存在しない場合はアプリケーションIDを設定する
        outData.localCommunicationId.push_back(inAppId);
    }
}

void TestDataContainer::SetCsvRecordData_PlayLogQueryCapability(ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT
{
    auto& rd = inRecordTable[0];
    outData.playLogQueryCapability = NsTypeConv::ToPlayLogQueryCapability(
        SafeAt(rd, APP_PLAY_LOG_QUERY_CAPABILITY));

    if (outData.playLogQueryCapability == nn::ns::PlayLogQueryCapability::WhiteList)
    {
        // WhiteList が設定されている場合のみ PlayLogQueryableApplicationId の中身を確認する
        for (auto& record : inRecordTable)
        {
            auto appId = SafeAt(record, APP_PLAY_LOG_QUERYABLE_APPLICATION_ID);
            if (appId == "")
            {
                break;
            }

            auto id = StringUtil::ToNum<int64_t>(appId);
            outData.playLogQueryableApplicationId.push_back(id);
        }
    }
}

void TestDataContainer::SetCsvRecordData_Size(TestDataRecord& outRecord, CsvRecord& inRecord) NN_NOEXCEPT
{
    auto& rec = outRecord;
    auto& rd = inRecord;

    std::memset(&rec.size, 0, sizeof(rec.size));

    auto sizeNum = StringUtil::ToNum<int64_t>(SafeAt(rd, APPLICATION_SIZE));
    if (sizeNum == 0)
    {
        // MakeTestApplication のデフォルト値を入れる
        sizeNum = 4;
    }

    // ひとまず Application のみ扱う
    if (rec.type == CONTENT_TYPE_APPLICATION)
    {
        if (rec.installStorage == INSTALL_STORAGE_SDCARD)
        {
            rec.size.storage[0].storageId = nn::ncm::StorageId::SdCard;
            rec.size.storage[0].appSize = sizeNum;
        }
        else
        {
            rec.size.storage[0].storageId = nn::ncm::StorageId::BuildInUser;
            rec.size.storage[0].appSize = sizeNum;
        }
    }
}

void TestDataContainer::SetCsvRecordData_Aoc(TestDataRecord& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT
{
    for (auto& record : inRecordTable)
    {
        auto index = SafeAt(record, AOC_INDEX);
        if (index == "")
        {
            break;
        }

        AddOnContentMeta aocMeta;
        aocMeta.index = StringUtil::ToBit32(index);
        aocMeta.releaseVersion = StringUtil::ToBit32(SafeAt(record, AOC_RELEASE_VERSION));
        aocMeta.id.value = StringUtil::ToNum<int64_t>(SafeAt(record, AOC_APPLICAITON_ID));
        aocMeta.requiredAppVersion = StringUtil::ToBit32(SafeAt(record, AOC_REQUIRED_APPLICATION_VERSION));
        aocMeta.tag = SafeAt(record, AOC_TAG);
        aocMeta.dataPath = SafeAt(record, AOC_DATA_PATH);
        // AddOnContent の属性ではないが、後々のチェック用にこのタイミングで保持させておく
        aocMeta.installStorage = outData.installStorage;

        outData.aocProperty.aocList.push_back(aocMeta);
    }
}

void TestDataContainer::SetCsvRecordData_RequiredNetworkServiceLicenseOnLaunch(
    ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT
{
    outData.requiredNetworkServiceLicenseOnLaunchFlag = 0x0;
    for (auto& record : inRecordTable)
    {
        auto rd = SafeAt(record, APP_REQUIRED_NETWORK_SERVICE_LICENCE_ON_LAUNCH);
        if (rd == "")
        {
            break;
        }

        outData.requiredNetworkServiceLicenseOnLaunchFlag |= NsTypeConv::ToRequiredNetworkServiceLicenseOnLaunchFlag(rd);
    }
}
