﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ApplicationManagerTestTool_HtcUtil.h"

// htc を Release でビルドする場合に必要な定義値 (Develop のみの使用だと想定しているが念のため定義しておく)
#define NN_ENABLE_HTC

#include <memory>

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/htc.h>

//#include <nnt/result/testResult_Assert.h>

void HtcUtil::Initialize() NN_NOEXCEPT
{
    nn::htc::Initialize();

    // htc の仕様上?ホストPC(TargetManager.exe)と接続するまで一定時間待つ必要があるようなので待機処理を入れる
    // (特に Generic 版ではこの処理を実施しないと、環境変数取得時に ConnectionFailure エラーとなる可能性が非常に高い)
    // Tests/Htc/Sources/Tests/GetEnvironmentVariable/testHtc_GetEnvironmentVariable/testHtc_GetEnvironmentVariable.cpp を参考に・・
    size_t count = 0;
    size_t tmpSize = 0;
    while (nn::htc::ResultConnectionFailure::Includes(nn::htc::GetEnvironmentVariableLength(&tmpSize, "")))
    {
        if (count > 12)
        {
            // TargetManager.exe が立ち上がっていない可能性が高いためループを抜けておく
            NN_LOG(" [Warning] Failed to connect TargetManager.exe\n");
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(250));
        ++count;
    }
}

void HtcUtil::Finalize() NN_NOEXCEPT
{
    nn::htc::Finalize();
}

std::string HtcUtil::ReplaceEnvironmentVariable(const std::string& inPath) NN_NOEXCEPT
{
    const auto ENV_DELM = "%";
    auto pos = inPath.find(ENV_DELM);
    if (pos == std::string::npos)
    {
        // そもそも環境変数の書式に該当する文字列が存在しない場合はそのまま返す
        return inPath;
    }

    std::string path = inPath;
    while (pos != std::string::npos)
    {
        auto nextPos = path.find(ENV_DELM, pos + 1);
        if (nextPos == std::string::npos)
        {
            // 環境変数の書式が存在しないため抜ける
            break;
        }

        // 環境変数( % に囲まれている箇所)の抽出
        auto envStr = path.substr(pos + 1, nextPos - pos - 1);

        // 環境変数の値のサイズを取得
        size_t size = 0;
        auto result = nn::htc::GetEnvironmentVariableLength(&size, envStr.c_str());
        //NNT_EXPECT_RESULT_SUCCESS(result);
        if (result.IsFailure())
        {
            // 指定した環境変数が存在しないなどの理由で取得できないため次の環境変数を探す
            //NN_LOG(" [Debug] GetEnvironmentVariableLength() Failed : envStr=%s\n", envStr.c_str());
            pos = nextPos;
            continue;
        }

        // 環境変数の値を取得
        std::unique_ptr<char[]> bufStr(new char[size]);
        size_t readSize = 0;
        result = nn::htc::GetEnvironmentVariable(&readSize, bufStr.get(), size, envStr.c_str());
        if (result.IsFailure())
        {
            NN_LOG(" [Error] GetEnvironmentVariable() Failed : envStr=%s\n", envStr.c_str());
            pos = nextPos;
            continue;
        }

        // 環境変数の値をパス文字列に置換(反映)する
        path.replace(pos, nextPos - pos + 1, bufStr.get());
        // 念のため先頭から % を検索しておく
        pos = path.find(ENV_DELM);
    }

    return path;
}
