﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNim_AccountServiceUtil.h"

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_Result.h>

#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/friends/friends_ApiAdmin.h>

namespace nnt { namespace account {

namespace
{
nn::Result TryUnlinkNa(const nn::account::Uid& user, nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    NN_UNUSED(user);

    bool linked;
    NN_RESULT_DO(admin.IsLinkedWithNintendoAccount(&linked));
    if (linked)
    {
        nn::account::AsyncContext ctx;
        NN_RESULT_DO(admin.DebugUnlinkNintendoAccountAsync(&ctx));

        nn::os::SystemEvent e;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        e.Wait();

        NN_RESULT_DO(ctx.GetResult());
    }
    NN_RESULT_SUCCESS;
}
} // ~namespace nnt::account::<anonymous>

/*
    追加したアカウント削除用ユーリティティ
*/
void Cleanup(const UserContext* pUsers) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    int count;
    nn::account::Uid users[nn::account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));

    for (auto& u : users)
    {
        if (!u)
        {
            break;
        }

        bool matchUser = false;
        int no = -1;
        for (int i=0; i < nn::account::UserCountMax; i++)
        {
            if (pUsers[i].id == u)
            {
                matchUser = true;
                no = i;
                break;
            }
        }
        if (matchUser == true)
        {
            if (pUsers[no].isAdded == false)
            {
                // 既存ユーザーは削除しない
                continue;
            }
        }
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, u));

        bool has;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&has));
        if (!has)
        {
            continue;
        }

        nn::nifm::InitializeSystem();
        if (nn::nifm::IsNetworkAvailable())
        {
            auto r = TryUnlinkNa(u, admin);
            if (!r.IsSuccess())
            {
                NN_LOG("[nnt::account] Unregister failed %03d-%04d\n", r.GetModule(), r.GetDescription());;
            }

            // BaaS アカウントの削除
            nn::account::AsyncContext ctx;
            NN_ABORT_UNLESS_RESULT_SUCCESS(admin.UnregisterAsync(&ctx));

            nn::os::SystemEvent e;
            NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
            e.Wait();

            NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetResult());
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(admin.DeleteRegistrationInfoLocally());
        }
    }
    for (auto& u : users)
    {
        if (!u)
        {
            break;
        }
        bool matchUser = false;
        int no = -1;
        for (int i=0; i < nn::account::UserCountMax; i++)
        {
            if (pUsers[i].id == u)
            {
                matchUser = true;
                no = i;
                break;
            }
        }
        if (matchUser == true)
        {
            if (pUsers[no].isAdded == false)
            {
                // 既存ユーザーは削除しない
                continue;
            }
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(u));
    }
}

/*
    ネットワークサービスアカウント周りのユーティリティ
*/
nn::Result RegisterNetworkServiceAccount(nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    nn::account::AsyncContext task;
    NN_RESULT_DO(admin.RegisterAsync(&task));

    nn::os::SystemEvent e;
    NN_RESULT_DO(task.GetSystemEvent(&e));
    e.Wait();
    NN_RESULT_DO(task.GetResult());

    EXPECT_TRUE(IsNetworkServiceAccountRegistered(admin));
    NN_RESULT_SUCCESS;
}
nn::Result UnregisterNetworkServiceAccount(nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    nn::account::AsyncContext task;
    NN_RESULT_DO(admin.UnregisterAsync(&task));

    nn::os::SystemEvent e;
    NN_RESULT_DO(task.GetSystemEvent(&e));
    e.Wait();
    NN_RESULT_DO(task.GetResult());

    EXPECT_FALSE(IsNetworkServiceAccountRegistered(admin));
    NN_RESULT_SUCCESS;
}
bool IsNetworkServiceAccountRegistered(nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    bool isRegistered;
    NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&isRegistered));
    return isRegistered;
}

}} // ~namespace nnt::account
