﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNim_AccountUtil.h"

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/fs.h>
#include <nn/nifm.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/system/settings_Account.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

#include "testNim_Nas.h"

namespace accounts {

bool IsLinkedNa(const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));

    bool registered;
    NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&registered));
    if (registered)
    {
        bool linkage;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsLinkedWithNintendoAccount(&linkage));
        if (linkage)
        {
            return true;
        }
    }
    return false;
}

bool LinkNa(const nn::account::Uid& uid, const char* id, const char* password)  NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));

    bool registered;
    NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&registered));
    if (registered)
    {
        bool linkage;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsLinkedWithNintendoAccount(&linkage));
        if (linkage)
        {
            NN_LOG("[accounts] Already linked\n");
            return true;
        }
    }
    else
    {
        // NSA 未作成の場合は作成する
        NN_LOG("[accounts] Cound not find the network service account.\n");
        return false;
    }

    nn::account::NintendoAccountLinkageProcedure proc;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(admin.CreateProcedureToLinkWithNintendoAccount(&proc));

    nn::account::RequestUrl requestUrl;
    nn::account::CallbackUri callbackUri;
    NN_ABORT_UNLESS_RESULT_SUCCESS(proc.GetRequest(&requestUrl, &callbackUri));

    Buffer response(4096);
    // [!注意!] 非サポートの手段による連携の認可処理
    bool verbose = false;
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
    verbose = true;
#endif
    ACCOUNTS_FAILURE_UNLESS(GetAuthorizationViaNasHelper(response, id, password, requestUrl.url, verbose));

    nn::account::AsyncContext ctx;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(proc.ApplyResponseAsync(&ctx, response.Get<char>(), response.GetSize()));
    nn::os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
    e.Wait();
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(ctx.GetResult());
    return true;
}

} // ~namespace accounts
