﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_LocalCommunicationDeliveryProtocol.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

namespace {

    const uint16_t TestPort = 12345;

    class LocalCommunicationProtocolTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
            nn::nifm::CancelNetworkRequest();
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::nifm::SubmitNetworkRequestAndWait();

            static nn::socket::ConfigDefaultWithMemory s_SocketConfig;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(s_SocketConfig));
        }

        static void TearDownTestCase()
        {
            nn::socket::Finalize();
            nn::nifm::CancelNetworkRequest();
        }
    };

    class BasicSession
    {
        NN_DISALLOW_COPY(BasicSession);

    public:
        BasicSession()
        {
            // サーバー・クライアントを初期化
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Server.Initialize(nn::socket::InAddr_Loopback, TestPort));
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Client.Initialize(nn::socket::InAddr_Loopback, TestPort));

            // セッション確立
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Server.WaitClient());
        }

        ~BasicSession()
        {
            m_Client.Finalize();
            m_Server.Finalize();
        }

        nn::nim::srv::LocalCommunicationDeliveryProtocolServer& GetServer()
        {
            return m_Server;
        }

        nn::nim::srv::LocalCommunicationDeliveryProtocolClient& GetClient()
        {
            return m_Client;
        }

    private:
        nn::nim::srv::LocalCommunicationDeliveryProtocolServer m_Server;
        nn::nim::srv::LocalCommunicationDeliveryProtocolClient m_Client;
    };
}

TEST_F(LocalCommunicationProtocolTest, Basic)
{
    // サーバー・クライアントを初期化できる
    nn::nim::srv::LocalCommunicationDeliveryProtocolServer server;
    NNT_EXPECT_RESULT_SUCCESS(server.Initialize(nn::socket::InAddr_Loopback, TestPort));

    nn::nim::srv::LocalCommunicationDeliveryProtocolClient client;
    NNT_EXPECT_RESULT_SUCCESS(client.Initialize(nn::socket::InAddr_Loopback, TestPort));

    // セッションを確立できる
    NNT_EXPECT_RESULT_SUCCESS(server.WaitClient());

    // リクエスト・レスポンスを繰り返せる
    for (int i = 0; i < 4; ++i)
    {
        nn::ncm::ContentId contentId = { { 0x0f, 0x1e, 0x2d, 0x3c, 0x4b, 0x5a, 0x69, 0x78, 0x00, 0x22, 0x44, 0x66, 0x88, 0xaa, 0xcc, 0xee } };

        // リクエストヘッダが送信できる
        auto requestHeader = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolRequestHeader(nn::nim::srv::GetContentTag, sizeof(contentId));
        NNT_EXPECT_RESULT_SUCCESS(client.SendHeader(requestHeader));

        // リクエストデータが送信できる
        NNT_EXPECT_RESULT_SUCCESS(client.SendData(&contentId, sizeof(contentId), sizeof(contentId)));

        // リクエストヘッダが受信できる
        nn::nim::srv::LocalCommunicationDeliveryProtocolHeader recvRequestHeader;
        NNT_EXPECT_RESULT_SUCCESS(server.ReceiveHeader(&recvRequestHeader));

        // リクエストヘッダ内容に相違がない
        ASSERT_TRUE(memcmp(requestHeader.signature, recvRequestHeader.signature, sizeof(requestHeader.signature)) == 0);
        ASSERT_EQ(requestHeader.tag, recvRequestHeader.tag);
        ASSERT_EQ(requestHeader.additionalHeaderSize, recvRequestHeader.additionalHeaderSize);
        ASSERT_EQ(requestHeader.size, recvRequestHeader.size);

        // リクエストデータが受信できる
        nn::ncm::ContentId recvContentId;
        NNT_EXPECT_RESULT_SUCCESS(server.ReceiveData(&recvContentId, sizeof(recvContentId), requestHeader.size));

        // リクエストデータに相違がない
        ASSERT_EQ(contentId, recvContentId);

        static const size_t ContentLength = 16 * 1024;

        // レスポンスの分割送信ができる
        uint8_t buffer[1 * 1024];
        auto responseHeader = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolResponseHeader(requestHeader.tag, ContentLength);
        auto sendContentFunc = [&](size_t writeSize, int64_t offset) -> nn::Result
        {
            for (size_t i = 0; i < writeSize; ++i)
            {
                buffer[i] = static_cast<uint8_t>(offset + i);
            }

            NN_RESULT_SUCCESS;
        };
        NNT_EXPECT_RESULT_SUCCESS(server.Send(responseHeader, &buffer, sizeof(buffer), sendContentFunc));

        // レスポンスヘッダの受信ができる
        nn::nim::srv::LocalCommunicationDeliveryProtocolHeader recvResponseHeader;
        NNT_EXPECT_RESULT_SUCCESS(client.ReceiveHeader(&recvResponseHeader));

        // レスポンスヘッダ内容に相違がない
        ASSERT_TRUE(memcmp(responseHeader.signature, recvResponseHeader.signature, sizeof(responseHeader.signature)) == 0);
        ASSERT_EQ(responseHeader.tag, recvResponseHeader.tag);
        ASSERT_EQ(responseHeader.additionalHeaderSize, recvResponseHeader.additionalHeaderSize);
        ASSERT_EQ(responseHeader.size, recvResponseHeader.size);

        // レスポンスデータの分割受信ができる
        bool isEqual = true;
        auto receiveContent = [&](size_t recvSize, int64_t offset) -> nn::Result
        {
            NN_UNUSED(offset);

            for (size_t i = 0; i < recvSize; ++i)
            {
                isEqual &= buffer[i] == static_cast<uint8_t>(offset + i);
            }

            NN_RESULT_SUCCESS;
        };
        NNT_EXPECT_RESULT_SUCCESS(client.ReceiveData(buffer, sizeof(buffer), recvResponseHeader.size, receiveContent));

        // レスポンスデータに相違がない
        ASSERT_TRUE(isEqual);
    }

    // サーバー・クライアントを破棄
    client.Finalize();
    server.Finalize();
}

TEST_F(LocalCommunicationProtocolTest, Cancel)
{
    uint8_t buffer[1024];

    // ヘッダ送信時のキャンセルができる
    {
        BasicSession session;

        session.GetClient().Cancel();

        auto requestHeader = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolRequestHeader(nn::nim::srv::GetContentTag, sizeof(buffer));
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationConnectionCanceled, session.GetClient().SendHeader(requestHeader));
    }

    // データ送信時のキャンセルができる
    {
        BasicSession session;

        session.GetClient().Cancel();

        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationConnectionCanceled, session.GetClient().SendData(buffer, sizeof(buffer), sizeof(buffer)));
    }

    // データ分割送信中のキャンセルができる
    {
        BasicSession session;

        static const size_t ContentLength = sizeof(buffer) * 4;

        auto header = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolResponseHeader(nn::nim::srv::GetContentTag, ContentLength);
        auto sendContentFunc = [&](size_t writeSize, int64_t offset) -> nn::Result
        {
            if (offset >= ContentLength / 2)
            {
                session.GetServer().Cancel();
            }

            NN_RESULT_SUCCESS;
        };
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationConnectionCanceled, session.GetServer().Send(header, &buffer, sizeof(buffer), sendContentFunc));
    }

    // ヘッダ受信時のキャンセルができる
    {
        BasicSession session;

        auto requestHeader = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolRequestHeader(nn::nim::srv::GetContentTag, sizeof(buffer));
        NNT_EXPECT_RESULT_SUCCESS(session.GetClient().SendHeader(requestHeader));

        session.GetServer().Cancel();

        nn::nim::srv::LocalCommunicationDeliveryProtocolHeader header;
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationConnectionCanceled, session.GetServer().ReceiveHeader(&header));
    }

    // データ受信時のキャンセルができる
    {
        BasicSession session;

        NNT_EXPECT_RESULT_SUCCESS(session.GetClient().SendData(buffer, sizeof(buffer), sizeof(buffer)));

        session.GetServer().Cancel();

        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationConnectionCanceled, session.GetServer().ReceiveData(buffer, sizeof(buffer), sizeof(buffer)));
    }

    // データ分割受信中のキャンセルができる
    {
        BasicSession session;

        static const size_t ContentLength = sizeof(buffer) * 4;

        auto header = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolResponseHeader(nn::nim::srv::GetContentTag, ContentLength);
        auto sendContentFunc = [&](size_t writeSize, int64_t offset) -> nn::Result
        {
            NN_RESULT_SUCCESS;
        };
        NNT_EXPECT_RESULT_SUCCESS(session.GetServer().Send(header, &buffer, sizeof(buffer), sendContentFunc));

        auto receiveContent = [&](size_t recvSize, int64_t offset) -> nn::Result
        {
            NN_UNUSED(offset);

            if (offset >= ContentLength / 2)
            for (size_t i = 0; i < recvSize; ++i)
            {
                session.GetClient().Cancel();
            }

            NN_RESULT_SUCCESS;
        };
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationConnectionCanceled, session.GetClient().ReceiveData(&buffer, sizeof(buffer), ContentLength, receiveContent));
    }
}

TEST_F(LocalCommunicationProtocolTest, SessionClose)
{
    uint8_t buffer[1024];

    // ヘッダ送信時の切断がハンドリングできる
    {
        BasicSession session;

        session.GetClient().Finalize();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        auto requestHeader = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolRequestHeader(nn::nim::srv::GetContentTag, sizeof(buffer));
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationSessionClosed, session.GetClient().SendHeader(requestHeader));
    }

    // データ送信時の切断がハンドリングできる
    {
        BasicSession session;

        session.GetClient().Finalize();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationSessionClosed, session.GetClient().SendData(buffer, sizeof(buffer), sizeof(buffer)));
    }

    // データ分割送信中の切断がハンドリングできる
    {
        BasicSession session;

        static const size_t ContentLength = sizeof(buffer) * 4;

        auto header = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolResponseHeader(nn::nim::srv::GetContentTag, ContentLength);
        auto sendContentFunc = [&](size_t writeSize, int64_t offset) -> nn::Result
        {
            if (offset >= ContentLength / 2)
            {
                session.GetServer().Finalize();
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }

            NN_RESULT_SUCCESS;
        };
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationSessionClosed, session.GetServer().Send(header, &buffer, sizeof(buffer), sendContentFunc));
    }

    // ヘッダ受信時の切断がハンドリングできる
    {
        BasicSession session;

        auto requestHeader = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolRequestHeader(nn::nim::srv::GetContentTag, sizeof(buffer));
        NNT_EXPECT_RESULT_SUCCESS(session.GetClient().SendHeader(requestHeader));

        session.GetServer().Finalize();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        nn::nim::srv::LocalCommunicationDeliveryProtocolHeader header;
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationSessionClosed, session.GetServer().ReceiveHeader(&header));
    }

    // データ受信時の切断がハンドリングできる
    {
        BasicSession session;

        NNT_EXPECT_RESULT_SUCCESS(session.GetClient().SendData(buffer, sizeof(buffer), sizeof(buffer)));

        session.GetServer().Finalize();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationSessionClosed, session.GetServer().ReceiveData(buffer, sizeof(buffer), sizeof(buffer)));
    }

    // データ分割受信中の切断がハンドリングできる
    {
        BasicSession session;

        static const size_t ContentLength = sizeof(buffer) * 4;

        auto header = nn::nim::srv::MakeLocalCommunicationDeliveryProtocolResponseHeader(nn::nim::srv::GetContentTag, ContentLength);
        auto sendContentFunc = [&](size_t writeSize, int64_t offset) -> nn::Result
        {
            NN_RESULT_SUCCESS;
        };
        NNT_EXPECT_RESULT_SUCCESS(session.GetServer().Send(header, &buffer, sizeof(buffer), sendContentFunc));

        auto receiveContent = [&](size_t recvSize, int64_t offset) -> nn::Result
        {
            NN_UNUSED(offset);

            if (offset >= ContentLength / 2)
            {
                session.GetClient().Finalize();
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }

            NN_RESULT_SUCCESS;
        };
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultLocalCommunicationSessionClosed, session.GetClient().ReceiveData(&buffer, sizeof(buffer), ContentLength, receiveContent));
    }
}
