﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/ncmutil/ncmutil_InstallUtils.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/socket.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ns/ns_Result.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ec/system/ec_DeviceAuthenticationApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>

#include "../Common/testLocalCommunication_Common.h"

namespace {

    class LocalCommunicationReceiveApplicationTaskTest : public testing::Test, public nnt::ncmutil::SubmissionPackageFileInstaller
    {
    protected:
        virtual void SetUp()
        {

#if defined( NN_BUILD_CONFIG_OS_WIN )
            SetupBisWorkingDirectory(nnt::GetHostArgv()[1]);
#endif
            nn::ncm::Initialize();
            m_NspDirectory = std::string(nnt::GetHostArgv()[2]);
#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::CreateContentMetaDatabase(nn::ncm::StorageId::BuiltInUser));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::CreateContentStorage(nn::ncm::StorageId::BuiltInUser));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::ActivateContentMetaDatabase(nn::ncm::StorageId::BuiltInUser));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::ActivateContentStorage(nn::ncm::StorageId::BuiltInUser));
#endif
        }

        virtual void TearDown()
        {
            nn::ncm::Finalize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ABORT_UNLESS_RESULT_SUCCESS(TearDownBisWorkingDirectory());
#endif
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::nifm::SubmitNetworkRequestAndWait();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ASSERT(nnt::GetHostArgc() > 2);
            nn::ovln::PrepareSenderAndReceiverForDevelop();
            nn::ovln::InitializeSenderLibraryForOverlay();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
            static NN_ALIGNAS(4096) uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(s_SocketMemoryPoolBuffer),
                                                                  nn::socket::DefaultSocketMemoryPoolSize,
                                                                  nn::socket::DefaultSocketAllocatorSize,
                                                                  nn::socket::DefaultConcurrencyLimit));
            auto result = curl_global_init(CURL_GLOBAL_ALL);
            NN_ASSERT_EQUAL(CURLE_OK, result);
#endif
        }

        static void TearDownTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            curl_global_cleanup();
            nn::socket::Finalize();
            nn::fs::UnmountHostRoot();
            nn::ovln::FinalizeSenderLibraryForOverlay();
            nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
#endif
            nn::nifm::CancelNetworkRequest();
        }

        nn::Result InstallAllNsp(nn::ncm::StorageId storage)
        {
            std::vector<std::string> outList;
            return InstallAll(&outList, GetNspDirectory(), storage);
        }

    private:
        const char* GetNspDirectory() const NN_NOEXCEPT
        {
            return m_NspDirectory.c_str();
        }

        std::string m_NspDirectory;
    };

    nn::Result DeleteContentByKey(const nn::ncm::ContentMetaKey& key, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
        nn::ncm::ContentMetaDatabase db;
        NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));
        nn::ncm::ContentStorage storage;
        NN_RESULT_DO(nn::ncm::OpenContentStorage(&storage, storageId));
        nn::ncm::ContentInfo contentInfoList[16];
        int offset = 0;
        while(NN_STATIC_CONDITION(true))
        {
            int infoCount;
            NN_RESULT_DO(db.ListContentInfo(&infoCount, contentInfoList, NN_ARRAY_SIZE(contentInfoList), key, offset));
            for (int j = 0; j < infoCount; j++)
            {
                storage.Delete(contentInfoList[j].GetId());
            }
            if (infoCount < NN_ARRAY_SIZE(contentInfoList))
            {
                break;
            }
            offset += infoCount;
        }

        NN_RESULT_SUCCESS;
    }
}

TEST_F(LocalCommunicationReceiveApplicationTaskTest, ReceivePatch)
{
    nn::nim::InitializeForNetworkInstallManager();

    //  Cleanup
    {
        nn::nim::NetworkInstallTaskId idListArray[64];
        int count = nn::nim::ListNetworkInstallTask(idListArray, sizeof(idListArray) / sizeof(idListArray[0]));
        for (int i = 0; i < count; i++)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::nim::DestroyNetworkInstallTask(idListArray[i]));
        }
    }

    {
        nn::nim::NetworkInstallTaskId listed;
        auto count = nn::nim::ListNetworkInstallTask(&listed, 1);
        ASSERT_EQ(0, count);
    }

    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuiltInUser;
    nn::ncm::ContentMetaDatabase db;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, storageId));
    nn::ncm::ContentStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, storageId));

    // ns の API を呼びたくないので、もし既にアプリケーションのコンテンツメタが存在していたらエラー
    {
        nn::ncm::ApplicationContentMetaKey appKey;
        auto count = db.ListApplication(&appKey, 1);
        ASSERT_EQ(count.total, 0);
        NN_UNUSED(appKey);
    }

    // ディレクトリ内の nsp をインストール
    NNT_ASSERT_RESULT_SUCCESS(InstallAllNsp(storageId));
    NN_UTIL_SCOPE_EXIT
    {
        nn::ncm::ApplicationContentMetaKey appKeyList[16];
        for (;;)
        {
            auto count = db.ListApplication(appKeyList, NN_ARRAY_SIZE(appKeyList));

            for (int i = 0; i < count.listed; i++)
            {
                // Result は無視
                DeleteContentByKey(appKeyList[i].key, storageId);
                db.Remove(appKeyList[i].key);
            }
            if (count.listed < NN_ARRAY_SIZE(appKeyList))
            {
                break;
            }

        }
    };

    for (;;)
    {
        const int numKey = 16;
        nn::ncm::ApplicationContentMetaKey appKeyList[16];

        auto count = db.ListApplication(appKeyList, numKey);
        for (int i = 0; i < count.listed; i++)
        {
            if (appKeyList[i].key.type == nn::ncm::ContentMetaType::Patch)
            {
                // 一旦消す
                nn::ncm::ContentInfo contentInfoList[16];
                int infoCount;
                NNT_EXPECT_RESULT_SUCCESS(db.ListContentInfo(&infoCount, contentInfoList, NN_ARRAY_SIZE(contentInfoList), appKeyList[i].key, 0));
                NNT_EXPECT_RESULT_SUCCESS(DeleteContentByKey(appKeyList[i].key, storageId));
                NNT_EXPECT_RESULT_SUCCESS(db.Remove(appKeyList[i].key));

                // 消したことの確認
                bool has;
                NNT_EXPECT_RESULT_SUCCESS(db.Has(&has, appKeyList[i].key));
                EXPECT_FALSE(has);
                for (int j = 0; j < infoCount; j++)
                {
                    NNT_EXPECT_RESULT_SUCCESS(storage.Has(&has, contentInfoList[j].GetId()));
                    EXPECT_FALSE(has);
                }

                // ダウンロード
                nn::nim::LocalCommunicationReceiveApplicationTaskId id;
                {
                    NNT_EXPECT_RESULT_SUCCESS(nn::nim::CreateLocalCommunicationReceiveApplicationTask(&id, 0, TestPort, appKeyList[i].applicationId, &appKeyList[i].key, 1, storageId));

                    nn::nim::LocalCommunicationReceiveApplicationTaskInfo info;
                    NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetLocalCommunicationReceiveApplicationTaskInfo(&info, id));
                    EXPECT_TRUE(appKeyList[i].applicationId == info.applicationId);
                    EXPECT_EQ(info.downloaded, 0LL);
                    EXPECT_FALSE(info.isRunning);
                    EXPECT_EQ(info.progress.state, nn::ncm::InstallProgressState::NotPrepared);

                    nn::nim::AsyncResult asyncResult;
                    NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestLocalCommunicationReceiveApplicationTaskRun(&asyncResult, id));

                    asyncResult.Wait();
                    NNT_EXPECT_RESULT_SUCCESS(asyncResult.Get());

                    NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetLocalCommunicationReceiveApplicationTaskInfo(&info, id));
                    EXPECT_TRUE(appKeyList[i].applicationId == info.applicationId);
                    EXPECT_GT(info.downloaded, 0LL);
                    EXPECT_TRUE(info.isRunning);
                    EXPECT_EQ(info.progress.state, nn::ncm::InstallProgressState::Downloaded);


                    // タスク実行中は呼び出し出来ない
                    nn::nim::AsyncResult testResult;
                    NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::RequestLocalCommunicationReceiveApplicationTaskRun(&testResult, id));
                    NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::CommitLocalCommunicationReceiveApplicationTask(id));
                    NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::DestroyLocalCommunicationReceiveApplicationTask(id));

                    NNT_EXPECT_RESULT_SUCCESS(asyncResult.Get());

                    // List
                    nn::nim::LocalCommunicationReceiveApplicationTaskId idList[2];
                    EXPECT_EQ( 1, nn::nim::ListLocalCommunicationReceiveApplicationTask(idList, NN_ARRAY_SIZE(idList)));
                    EXPECT_EQ(idList[0], id);

                    // ListContentMeta
                    nn::ncm::StorageContentMetaKey storageKey;
                    int keyCount;
                    NNT_EXPECT_RESULT_SUCCESS(nn::nim::ListLocalCommunicationReceiveApplicationTaskContentMeta(&keyCount, &storageKey, 1, 0, id));
                    EXPECT_TRUE(appKeyList[i].key == storageKey.key);
                    EXPECT_TRUE(storageId == storageKey.storageId);
                }

                // コミット
                NNT_EXPECT_RESULT_SUCCESS(nn::nim::CommitLocalCommunicationReceiveApplicationTask(id));

                // インストール出来たことの確認
                {

                    NNT_EXPECT_RESULT_SUCCESS(db.Has(&has, appKeyList[i].key));
                    EXPECT_TRUE(has);
                    for (int j = 0; j < infoCount; j++)
                    {
                        NNT_EXPECT_RESULT_SUCCESS(storage.Has(&has, contentInfoList[j].GetId()));
                        EXPECT_TRUE(has);
                    }
                }

                // 後処理の確認
                {
                    nn::nim::LocalCommunicationReceiveApplicationTaskInfo info;
                    NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetLocalCommunicationReceiveApplicationTaskInfo(&info, id));
                    EXPECT_TRUE(appKeyList[i].applicationId == info.applicationId);
                    EXPECT_GT(info.downloaded, 0LL);
                    EXPECT_FALSE(info.isRunning);
                    EXPECT_EQ(info.progress.state, nn::ncm::InstallProgressState::Commited);

                    NNT_EXPECT_RESULT_SUCCESS(nn::nim::DestroyLocalCommunicationReceiveApplicationTask(id));

                    nn::nim::LocalCommunicationReceiveApplicationTaskId idList[2];
                    EXPECT_EQ( 0, nn::nim::ListLocalCommunicationReceiveApplicationTask(idList, NN_ARRAY_SIZE(idList)));
                }
            }
        }
        if (count.listed < numKey)
        {
            break;
        }
    }

    nn::nim::FinalizeForNetworkInstallManager();
} // NOLINT(impl/function_size)
