﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/es/debug/es_ELicenseArchiveBuilder.h>
#include <nn/es/debug/es_ELicenseArchiveJsonUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Base64.h>

#include "es/es_DevicePublicKey.h"
#include "testNim_ELicenseArchivesUtil.h"

namespace nnt { namespace elicense {

class SignKey
{
public:
    const uint8_t* GetModulus() const NN_NOEXCEPT
    {
        return Modulus;
    }

    size_t GetModulusSize() const NN_NOEXCEPT
    {
        return sizeof(Modulus);
    }

    const uint8_t* GetPrivateExponent() const NN_NOEXCEPT
    {
        return PrivateExponent;
    }

    size_t GetPrivateExponentSize() const NN_NOEXCEPT
    {
        return sizeof(PrivateExponent);
    }

private:
    static const uint8_t Modulus[256];
    static const uint8_t PrivateExponent[256];
};

const uint8_t SignKey::Modulus[] =
{
    0xc9, 0xfd, 0xd4, 0x52, 0x39, 0x21, 0x8c, 0x8e, 0x45, 0xaf, 0xd2, 0x17, 0xb3, 0x72, 0x52,
    0x72, 0x59, 0xce, 0x77, 0x63, 0x59, 0x6b, 0x5a, 0xd4, 0xc1, 0xc9, 0xc2, 0xf9, 0xf5, 0x81,
    0xc1, 0xd5, 0x18, 0x20, 0x7f, 0x81, 0xd0, 0x37, 0xf5, 0x3d, 0x58, 0x89, 0xb0, 0x70, 0x3f,
    0xf3, 0x35, 0x2b, 0x16, 0x9e, 0x9f, 0xaf, 0x66, 0xd0, 0x4d, 0xa0, 0xf7, 0x7c, 0x26, 0xc0,
    0xa6, 0x70, 0xe5, 0xce, 0xee, 0x60, 0x13, 0x95, 0x0c, 0x0a, 0x6b, 0x35, 0xe6, 0xf5, 0x1e,
    0x13, 0xb1, 0xe8, 0x27, 0xab, 0xe6, 0x1b, 0x38, 0x7a, 0x28, 0xab, 0x2a, 0xef, 0xee, 0xd6,
    0x3f, 0xa9, 0xcb, 0x86, 0x4c, 0xe7, 0xbf, 0x3f, 0x71, 0x85, 0xfa, 0x5c, 0x1e, 0xc4, 0x8c,
    0xe1, 0x75, 0x3f, 0x6a, 0x6d, 0xa4, 0x66, 0x71, 0x91, 0x19, 0x44, 0x2f, 0xac, 0x6c, 0x35,
    0x35, 0x29, 0xcc, 0x61, 0x30, 0x51, 0xd5, 0x34, 0x87, 0xa9, 0x91, 0xe6, 0x9a, 0xb3, 0x64,
    0x7d, 0x91, 0x98, 0xe0, 0xfe, 0x6a, 0xaf, 0x7b, 0xc9, 0xd7, 0xd1, 0x50, 0xb2, 0x08, 0xad,
    0x3a, 0xc8, 0xb8, 0x3c, 0x9c, 0xce, 0xd7, 0x3f, 0x4a, 0x3b, 0x7c, 0xa0, 0xd5, 0xd9, 0xea,
    0x74, 0x58, 0x58, 0x5e, 0x4d, 0x75, 0x39, 0xdf, 0x56, 0xff, 0x86, 0xbb, 0x08, 0xdf, 0x48,
    0x58, 0x4d, 0x7b, 0x9b, 0x85, 0xc5, 0xb7, 0x46, 0x8e, 0x01, 0x9a, 0x15, 0x2f, 0x7c, 0x01,
    0xfd, 0xc5, 0x49, 0xdd, 0xfd, 0x6f, 0xbb, 0x0f, 0x0b, 0x5f, 0x29, 0xdf, 0x02, 0xcf, 0x3d,
    0x6a, 0xa0, 0x3e, 0xa7, 0xd3, 0x3a, 0x26, 0x24, 0x2a, 0xce, 0x7c, 0x28, 0xc6, 0x36, 0x2b,
    0x6b, 0xea, 0x9f, 0x56, 0x95, 0x2f, 0x73, 0x54, 0x87, 0x72, 0xe6, 0x7a, 0xe0, 0x58, 0x2d,
    0xdd, 0x1c, 0x12, 0x6c, 0xda, 0xa6, 0x66, 0x72, 0x16, 0x15, 0xc4, 0xf5, 0x54, 0xba, 0xea,
    0x33
};

const uint8_t SignKey::PrivateExponent[] =
{
    0x84, 0x0c, 0xa8, 0xad, 0x48, 0x24, 0x22, 0x9b, 0x2f, 0xfb, 0x8d, 0x41, 0x69, 0x52, 0xa7,
    0x8b, 0x92, 0x22, 0x63, 0xf2, 0x53, 0x16, 0xd7, 0xbe, 0x97, 0x21, 0xab, 0x91, 0x75, 0x6a,
    0x6a, 0xa9, 0x40, 0xa8, 0x9e, 0x32, 0xe3, 0x48, 0x71, 0xce, 0xb4, 0xfe, 0xee, 0x95, 0xb3,
    0x78, 0xef, 0x9d, 0x47, 0xa0, 0x53, 0x8c, 0x7b, 0x8e, 0x59, 0xf1, 0x8d, 0x59, 0x60, 0x8c,
    0x39, 0x43, 0x3d, 0x39, 0x56, 0xa5, 0x00, 0xa8, 0x43, 0xc2, 0xdb, 0xeb, 0xd5, 0x1d, 0x11,
    0x98, 0x96, 0x89, 0xca, 0xc5, 0x9a, 0x1c, 0xd9, 0x83, 0x22, 0xeb, 0x23, 0x9a, 0xcf, 0x6c,
    0xc9, 0x19, 0xdb, 0xaf, 0xe4, 0xbe, 0xec, 0x4c, 0x31, 0x8f, 0x00, 0xe9, 0x10, 0x44, 0x36,
    0x45, 0x7e, 0x3d, 0x02, 0xfa, 0xc2, 0xab, 0x26, 0x3b, 0x51, 0x65, 0x3e, 0x35, 0x5e, 0x58,
    0x3a, 0x75, 0x24, 0xe3, 0xb4, 0x8f, 0x0a, 0xed, 0xae, 0x31, 0xd8, 0x43, 0xfb, 0x5f, 0xda,
    0x87, 0x44, 0x9d, 0xdc, 0xaa, 0xf9, 0xb1, 0x61, 0x87, 0x87, 0x2c, 0x67, 0x67, 0x84, 0xd9,
    0x4b, 0xc1, 0x88, 0x35, 0x6c, 0x6c, 0x03, 0x04, 0x39, 0x9c, 0xb3, 0xd7, 0x22, 0x4e, 0x0c,
    0xc6, 0x0d, 0xcf, 0x6d, 0xf4, 0x75, 0xa9, 0x8f, 0x97, 0xf1, 0x1a, 0x37, 0x40, 0x76, 0xcb,
    0xe3, 0xca, 0x99, 0x81, 0x9c, 0x1c, 0x81, 0x65, 0x5f, 0xe9, 0xfc, 0x0f, 0x00, 0xe9, 0xeb,
    0xc8, 0x33, 0x3d, 0x13, 0xcc, 0x36, 0x62, 0x3a, 0x0e, 0xe1, 0xd6, 0x15, 0x37, 0x39, 0x1a,
    0xe7, 0x91, 0xc8, 0xfe, 0xc3, 0xf9, 0x9e, 0xd5, 0xf7, 0xa1, 0xda, 0xac, 0x8f, 0x72, 0xf0,
    0xd4, 0x7e, 0x92, 0x99, 0xa5, 0xb6, 0x5a, 0xfc, 0xd8, 0xdb, 0x70, 0x4b, 0x93, 0x85, 0x4d,
    0xa0, 0x41, 0xea, 0x7d, 0x83, 0x44, 0x77, 0xfd, 0x15, 0x9f, 0x81, 0x27, 0x2e, 0xb7, 0xba,
    0x61
};

//----------------------------------------------------------------------------
size_t ArchiveGenerator::CreateJson(char* outBuffer, size_t bufferSize
    , const nn::es::ELicenseArchiveId& eLicenseArchiveId
    , const nn::account::NintendoAccountId& naId
    , const nn::es::RightsId& titleId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
    };

    uint64_t challenge = 1234567890;
    nn::es::debug::ELicenseArchiveInfoForDebug archiveInfo = nn::es::debug::MakeELicenseArchiveInfo(challenge, naId, eLicenseArchiveId);

    const int ELicenseCount = 100;
    std::unique_ptr<nn::es::debug::ELicenseInfoForDebug[]> elicense(new nn::es::debug::ELicenseInfoForDebug[ELicenseCount]);
    for (int i = 0; i < ELicenseCount; i++)
    {
        nn::es::RightsId rightsId = titleId + i * 0x10000;
        nn::es::TicketId ticketId = 0x0100000000000000 + i * 0x10000;

        elicense.get()[i] = nn::es::debug::MakeDeviceLinkedPermanentELicense(rightsId, naId, ticketId, 1600000);
    }

    NN_ABORT_UNLESS(nn::es::debug::BuildELicenseArchiveJsonString(outBuffer, bufferSize, archiveInfo, elicense.get(), ELicenseCount));

    return strnlen(outBuffer, bufferSize);
}

//----------------------------------------------------------------------------
size_t ArchiveGenerator::Encrypt(void* outBuffer, size_t bufferSize, void* pSource, size_t sourceSize) NN_NOEXCEPT
{
    nn::es::debug::ELicenseArchiveBuilder builder;

    nn::es::Rsa2048DevicePublicKey key;
    builder.SetPayloadAndEncryptRsa2048OaepSha256(
        pSource, sourceSize,
        nn::es::ELicenseArchivePayloadDataFormatType::Json,
        key.GetModulus(), key.GetModulusSize(),
        key.GetPublicExponent(), key.GetPublicExponentSize()
    );

    SignKey signKey;
    builder.SignRsa2048Pkcs1Sha256(
        signKey.GetModulus(), signKey.GetModulusSize(),
        signKey.GetPrivateExponent(), signKey.GetPrivateExponentSize()
    );
    return builder.Get(outBuffer, bufferSize);
}

//----------------------------------------------------------------------------
size_t ArchiveGenerator::GenerateDummy(char* pOutBase64Buffer, size_t bufferSize
    , const nn::es::ELicenseArchiveId& eLicenseArchiveId
    , const nn::account::NintendoAccountId& naId
    , const nn::es::RightsId& titleId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutBase64Buffer);
    NN_SDK_ASSERT(bufferSize > 0u);

    char jsonBuffer[BufferCapacity] = {};
    size_t jsonSize = CreateJson(jsonBuffer, sizeof(jsonBuffer), eLicenseArchiveId, naId, titleId);
    m_ArchiveSize = Encrypt(m_Archive, sizeof(m_Archive), jsonBuffer, jsonSize);
    NN_ABORT_UNLESS(m_ArchiveSize > 0);

    const auto status = ::nn::util::Base64::ToBase64String(pOutBase64Buffer, bufferSize, m_Archive, m_ArchiveSize, ::nn::util::Base64::Mode_NormalNoLinefeed);
    return (status != ::nn::util::Base64::Status_Success) ? 0 : std::strlen(pOutBase64Buffer);
}

}} // ~namespace nnt::elicense
