﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <string>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/account.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/friends/friends_ApiAdmin.h>

#include <nn/ec.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/npns.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ssl.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/socket.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Uuid.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include <curl/curl.h>

#include <nn/ec/ec_ShopServiceAccessorForDebug.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/nim/nim_DynamicRightsApi.h>
#include "accounts/testNim_AccountUtil.h"
#include "testNim_AccountServiceUtil.h"
#include "testNim_ELicenseArchivesUtil.h"

#include "nimsrv/nim_DynamicRightsCommon.h"

// time ライブラリの GetCurrentTime が Windows のマクロと競合するため
#ifdef GetCurrentTime
#undef GetCurrentTime
#endif

#include <nn/time.h>

//!----------------------------------------------------------------------------
// テスト環境説明
//  - 強制的にユーザインデクス 0 に対して、テスト用ニンテンドーアカウントで NSA 連携を行います。
//  - ユーザの自動生成について
//    - ターゲットにユーザが存在する場合は、自動生成は行なわず既存ユーザを上書きします。
//    - ターゲットにユーザが存在しない場合は自動生成を行い、テスト終了時に NSA 連携解除、及びユーザの削除を行います。
//!----------------------------------------------------------------------------

//!----------------------------------------------------------------------------
#if defined( NN_SDK_BUILD_DEBUG ) || defined( NN_SDK_BUILD_DEVELOP )
#define PRIVATE_LOG(...) NN_LOG( "[DynamicRights::TEST] " __VA_ARGS__ )
#else
#define PRIVATE_LOG(...)            static_cast<void>(0)
#endif

#if defined( NN_BUILD_CONFIG_TOOLCHAIN_GCC )
#define PRIVATE_STR_TO_LL  strtoll
#define PRIVATE_STR_TO_ULL strtoull
#elif defined( NN_BUILD_CONFIG_OS_WIN )
#define PRIVATE_STR_TO_LL  _strtoi64
#define PRIVATE_STR_TO_ULL _strtoui64
#else
#define PRIVATE_STR_TO_LL  std::strtoll
#define PRIVATE_STR_TO_ULL std::strtoull
#endif

//!----------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_HORIZON )

using namespace nn;

namespace {

//!----------------------------------------------------------------------------
class StatefulInitializer
{
public:
    StatefulInitializer() NN_NOEXCEPT : m_IsInitialized(false) {}

    template<typename Predicate>
    void Initialize(const char* pLogPrefix, Predicate executor) NN_NOEXCEPT
    {
        PRIVATE_LOG("==== Initializing %s\n", pLogPrefix);
        executor();
        m_IsInitialized = true;
    }

    template<typename Predicate>
    void Finalize(const char* pLogPrefix, Predicate executor) NN_NOEXCEPT
    {
        PRIVATE_LOG("==== Cleaning up %s\n", pLogPrefix);
        if (m_IsInitialized)
        {
            m_IsInitialized = false;
            executor();
        }
    }

private:
    bool    m_IsInitialized;
};

//!----------------------------------------------------------------------------
class Environment
{
private:
    socket::ConfigDefaultWithMemory     m_SocketConfig;
    friends::DaemonSuspension           m_FriendSuspension;
    nnt::account::UserContext           m_Users[account::UserCountMax];

    StatefulInitializer m_InitialLibcurl;
    StatefulInitializer m_InitialSocket;
    StatefulInitializer m_InitialNifm;
    StatefulInitializer m_InitialSsl;

    bool    m_UsingAutoCreatedUser;

public:
    void Initialize() NN_NOEXCEPT
    {
        m_UsingAutoCreatedUser = false;

        NN_ABORT_UNLESS_RESULT_SUCCESS(npns::InitializeForSystem());
        NN_ABORT_UNLESS_RESULT_SUCCESS(friends::SuspendDaemon(&m_FriendSuspension));

#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        oe::Initialize();
#endif

        m_InitialNifm.Initialize("NIFM", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(nifm::Initialize());
            nifm::SubmitNetworkRequestAndWait();
        });
        m_InitialSsl.Initialize("SSL", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(ssl::Initialize());
        });
        m_InitialSocket.Initialize("socket", [this]() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(socket::Initialize(m_SocketConfig));
        });
        m_InitialLibcurl.Initialize("libcurl", []() -> void
        {
            ASSERT_EQ(CURLE_OK, curl_global_init(CURL_GLOBAL_ALL));
        });

        PrepareUsers(1);
    }

    void Finalize() NN_NOEXCEPT
    {
        FinishUsers();

        m_InitialLibcurl.Finalize("libcurl", []() -> void
        {
            curl_global_cleanup();
        });

        m_InitialSocket.Finalize("socket", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(socket::Finalize());
        });
        m_InitialSsl.Finalize("SSL", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(ssl::Finalize());
        });
        m_InitialNifm.Finalize("NIFM", []() -> void
        {
            nifm::CancelNetworkRequest();
        });
        npns::FinalizeForSystem();
    }

    Result AcquireNsaId(int index, account::NetworkServiceAccountId* pOutValue) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutValue);

        const auto& user = m_Users[index];
        NN_ASSERT(user.id);
        NN_ASSERT(user.isOpened);

        // ネットワークサービスアカウントの確定 ( NA連携 )
        NN_RESULT_DO(account::EnsureNetworkServiceAccountAvailable(user.open));
        PRIVATE_LOG("Success: EnsureNetworkServiceAccount.\n");

        // ネットワークサービスアカウントの情報
        NN_RESULT_DO(account::GetNetworkServiceAccountId(pOutValue, user.open));
        PRIVATE_LOG("Success: Acquire Network Service Account ID [ %016llx ].\n", pOutValue->id);

        NN_RESULT_SUCCESS;
    }

    const ::nn::account::Uid& GetUser(int index) const NN_NOEXCEPT
    {
        return m_Users[index].id;
    }

private:
    void RegisterNewUsers(int createdCount) NN_NOEXCEPT
    {
        account::Finalize();
        NN_UTIL_SCOPE_EXIT
        {
            account::Initialize();
        };

        // 古いユーザを削除。
        PRIVATE_LOG("Cleanup old users, And new users registration.\n");
        nnt::account::Cleanup();
        nnt::account::CreateUsers(m_Users, createdCount);
    }

    void UnregisterNewUsers() NN_NOEXCEPT
    {
        account::Finalize();
        nnt::account::Cleanup();
        account::Initialize();
    }

    void LinkNa(const account::Uid& uid) NN_NOEXCEPT
    {
        account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            account::Finalize();
        };
        account::NetworkServiceAccountAdministrator admin;
        NNT_ASSERT_RESULT_SUCCESS(::nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));
        if (!nnt::account::IsNetworkServiceAccountRegistered(admin))
        {
            NNT_ASSERT_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
        }
        ASSERT_TRUE(accounts::LinkNa(uid, "eshop-test+puxdd1_001@exmx.nintendo.co.jp", "puxdd1_001"));
    }

    void PrepareUsers(int createdCount) NN_NOEXCEPT
    {
        ASSERT_GT(createdCount, 0);

        int nUsers;
        account::Uid users[account::UserCountMax];
        NNT_ASSERT_RESULT_SUCCESS(account::ListAllUsers(&nUsers, users, NN_ARRAY_SIZE(users)));
        if (createdCount > nUsers)
        {
            RegisterNewUsers(createdCount);
            m_UsingAutoCreatedUser = true;
        }
        else
        {
            CreateUserContext(m_Users, users, nUsers);
        }
        for (int i = 0; i < createdCount; ++i)
        {
            auto& user = m_Users[i];
            account::Finalize();
            LinkNa(user.id);
            account::Initialize();
            user.Open();
        }
    }

    void FinishUsers() NN_NOEXCEPT
    {
        const auto n = NN_ARRAY_SIZE(m_Users);
        for (int i = 0; i < n; ++i)
        {
            m_Users[i].Close();
        }
        if (m_UsingAutoCreatedUser)
        {
            UnregisterNewUsers();
        }
    }
};

//!------------------------------------------------------------------------------------------------------
//! デバッグ機能支援
//!------------------------------------------------------------------------------------------------------
namespace DebugResponse {

    //! @brief  fwdbg 設定、および即時反映。
    //! @note   ec::DebugForShopServiceAccessor::Initialize() の呼び出しが必須です。
    void EnableFirmwareSettings(bool request) NN_NOEXCEPT
    {
        bool enabled = request;
        settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "enable_error_simulate", &enabled, sizeof(enabled));
        settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "enable_simulate_dynamic_rights", &enabled, sizeof(enabled));
        ec::DebugForShopServiceAccessor::RefreshAvailability();
    }

    //! @brief  全てのデバッグ設定をクリーンして支援機能を無効化します。
    void Cleanup() NN_NOEXCEPT
    {
        ec::DebugForShopServiceAccessor::ScopedSession session;
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::Initialize(&session));

        // NOTE: この関数が呼ばれるケースに「fwdbg 設定が無効時」も含まれ、その場合 nim::ResultNotSupported が返るので返値チェックしない。
        ec::DebugForShopServiceAccessor::ClearResponse();
        EnableFirmwareSettings(false);
    }

}   // ~DebugResponse

//!------------------------------------------------------------------------------------------------------
class TestForDynamicRights : public testing::Test
{
private:
    static Environment& GetEnvironment() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(Environment, s_Instance);
        return s_Instance;
    }

protected:
    CURL*   pCurlHandle;

    virtual void SetUp()
    {
        nim::InitializeForNetworkInstallManager();

        DebugResponse::Cleanup();
        pCurlHandle = curl_easy_init();
        NN_ABORT_UNLESS(nullptr != pCurlHandle);
    }

    virtual void TearDown()
    {
        if (nullptr != pCurlHandle)
        {
            curl_easy_cleanup(pCurlHandle);
        }

        nim::FinalizeForNetworkInstallManager();
    }

    static void SetUpTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        oe::Initialize();
#endif

        account::Initialize();
        GetEnvironment().Initialize();
    }

    static void TearDownTestCase()
    {
        DebugResponse::Cleanup();
        GetEnvironment().Finalize();
        account::Finalize();
        time::Finalize();
    }

public:
    TestForDynamicRights() : pCurlHandle(nullptr) {}

    Result AcquireNsaId(int index, account::NetworkServiceAccountId* pOutValue) NN_NOEXCEPT
    {
        return GetEnvironment().AcquireNsaId(index, pOutValue);
    }

    const ::nn::account::Uid& GetUser(int index) const NN_NOEXCEPT
    {
        return GetEnvironment().GetUser(index);
    }
};

//!------------------------------------------------------------------------------------------------------
#define DO_RESULT_CURLCODE(curlCode_)   \
{                                       \
    if (curlCode_ != CURLE_OK)          \
    {                                   \
        ToCurlErrorHandling(curlCode_); \
        return curlCode_;               \
    }                                   \
}

//!------------------------------------------------------------------------------------------------------
class CurlExecutor
{
private:
    struct SslCtxFunctionContext
    {
        bool isCalled;
        Result result;

        void Reset() NN_NOEXCEPT
        {
            isCalled = false;
            result = nn::ResultSuccess();
        }
    };

    void ToCurlErrorHandling(const CURLcode code) NN_NOEXCEPT
    {
        PRIVATE_LOG("CurlError: %u\n", static_cast<Bit32>(code));
    }

public:
    struct Response
    {
        char*   pBuffer;
        size_t  capacity;
        size_t  receivedSize;
        int     code;

        explicit Response(size_t initialCapacity) NN_NOEXCEPT
            : pBuffer(nullptr), capacity(initialCapacity), receivedSize(0), code(0)
        {
            pBuffer = reinterpret_cast<char*>(std::malloc(initialCapacity));
            NN_ABORT_UNLESS(nullptr != pBuffer);
            pBuffer[0] = '\0';
        }

        ~Response() NN_NOEXCEPT
        {
            if (nullptr != pBuffer)
            {
                capacity = 0;
                std::free(pBuffer);
            }
            Reset();
        }

        void Reset() NN_NOEXCEPT
        {
            code = 0;
            receivedSize = 0;
        }

        void Append(const char* pSource, size_t sourceSize) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS((receivedSize + sourceSize) < capacity);
            std::memcpy(&pBuffer[receivedSize], pSource, sourceSize);
            receivedSize += sourceSize;
            pBuffer[receivedSize] = '\0';
        }
    };

    const CURLcode Post(Response* pOutResponse, CURL* pCurlHandle, const char* pUrl, const char* pPostField, const char* pHeaders[], const int headerCount, long timeoutSeconds) NN_NOEXCEPT
    {
        NN_ASSERT(pOutResponse);
        NN_ASSERT(pCurlHandle);
        NN_ASSERT(pPostField);
        NN_ASSERT(pUrl);

        m_SslFunctionContext.Reset();
        curl_easy_reset(pCurlHandle);

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_XFERINFOFUNCTION, nullptr));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_XFERINFODATA, nullptr));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_NOPROGRESS, 1));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_URL, pUrl));

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_POST, 1));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_COPYPOSTFIELDS, pPostField));

        curl_slist* pSlist = nullptr;
        NN_UTIL_SCOPE_EXIT{curl_slist_free_all(pSlist);};

        for (int i = 0; i < headerCount; ++i)
        {
            pSlist = curl_slist_append(pSlist, pHeaders[i]);
            NN_ASSERT(nullptr != pSlist);
        }

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_HTTPHEADER, pSlist));

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_HEADERDATA, nullptr));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_HEADERFUNCTION, nullptr));

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_WRITEDATA, pOutResponse));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_WRITEFUNCTION, WriteCallback));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_NOBODY, 0));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_TIMEOUT, timeoutSeconds));

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_SSL_CTX_FUNCTION, DefaultSslCtxFunction));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_SSL_CTX_DATA, &m_SslFunctionContext));

        DO_RESULT_CURLCODE(curl_easy_perform(pCurlHandle));
        DO_RESULT_CURLCODE(curl_easy_getinfo(pCurlHandle, CURLINFO_RESPONSE_CODE, &pOutResponse->code));
        return CURLE_OK;
    }

protected:
    static size_t WriteCallback(char* pSource, size_t blockUnitSize, size_t blockCount, void* pUserData) NN_NOEXCEPT
    {
        const auto totalSize = blockUnitSize * blockCount;
        if (nullptr != pSource && nullptr != pUserData)
        {
            auto pHandle = static_cast<Response*>(pUserData);
            pHandle->Append(pSource, totalSize);
        }
        return totalSize;
    }

    static size_t DefaultSslCtxFunction(CURL* curlHandle, void* pSslContext, SslCtxFunctionContext* pFunctionContext) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFunctionContext);
        pFunctionContext->isCalled = true;
        if (!pFunctionContext->result.IsSuccess())
        {
            return 1;
        }
        auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
        auto r = context.Create(nn::ssl::Context::SslVersion_Auto);
        if (r.IsSuccess())
        {
            ssl::CertStoreId clientCertStoreId;
            r = context.RegisterInternalPki(&clientCertStoreId, ssl::Context::InternalPki_DeviceClientCertDefault);
        }
        if (!r.IsSuccess())
        {
            pFunctionContext->result = r;
            return 1;
        }
        return 0;
    }

private:
    SslCtxFunctionContext m_SslFunctionContext;
};

//!------------------------------------------------------------------------------------------------------
Result ParseErrorCodeResponse(Result* pOutResult, const char* pSource) NN_NOEXCEPT
{
    const Result result = ResultSuccess();
    nim::srv::DynamicRights::DragonsErrorResponseAdaptor adaptor;
    nim::srv::HttpJson::Stream::MemoryInputWithStaticOutput<1024> stream(pSource, std::strlen(pSource));
    NN_RESULT_DO(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<nim::srv::HttpJson::Canceler*>(nullptr)));
    *pOutResult = adaptor.GetResult(result);
    NN_RESULT_SUCCESS;
}

//!------------------------------------------------------------------------------------------------------
constexpr es::RightsId g_TtitleIds[] =
{
    0x01003ab001e32000,
    //0x01002ad001e32000
};

}   // ~::unnamed

//!------------------------------------------------------------------------------------------------------
//! JSONパーサーテスト
TEST_F(TestForDynamicRights, ErrorResponseParse)
{
    // ダミーエラーコードレスポンス
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=325012215 に基づく。
    const char ErrorCodeResponseBase[] =
    "{"
        "\"type\": \"%s\","
        "\"title\": \"%s\","
        "\"detail\": null,"
        "\"extension\": null"
    "}";

    Result outResult;
    char pSourceBuffer[4 * 1024];

    // 正常系
    ASSERT_TRUE(sizeof(pSourceBuffer) > util::SNPrintf(pSourceBuffer, sizeof(pSourceBuffer), ErrorCodeResponseBase, "https://dragons.nintendo.com/errors/v1/400/invalid_parameter", "Parameter is invalid"));
    NNT_ASSERT_RESULT_SUCCESS(ParseErrorCodeResponse(&outResult, pSourceBuffer));
    NNT_ASSERT_RESULT_FAILURE(nim::ResultDragonsInvalidParameter, outResult);

    ASSERT_TRUE(sizeof(pSourceBuffer) > util::SNPrintf(pSourceBuffer, sizeof(pSourceBuffer), ErrorCodeResponseBase, "https://dragons.nintendo.com/errors/v1/401/authentication_required", "Authentication is required"));
    NNT_ASSERT_RESULT_SUCCESS(ParseErrorCodeResponse(&outResult, pSourceBuffer));
    NNT_ASSERT_RESULT_FAILURE(nim::ResultDragonsAuthenticationRequired, outResult);

    ASSERT_TRUE(sizeof(pSourceBuffer) > util::SNPrintf(pSourceBuffer, sizeof(pSourceBuffer), ErrorCodeResponseBase, "https://dragons.nintendo.com/errors/v1/403/invalid_token", "Token is invalid"));
    NNT_ASSERT_RESULT_SUCCESS(ParseErrorCodeResponse(&outResult, pSourceBuffer));
    NNT_ASSERT_RESULT_FAILURE(nim::ResultDragonsInvalidToken, outResult);

    // 異常系( 前方不一致 )
    ASSERT_TRUE(sizeof(pSourceBuffer) > util::SNPrintf(pSourceBuffer, sizeof(pSourceBuffer), ErrorCodeResponseBase, "https://dragons.nintendo.co.jp/errors/v1/403/license_not_grantable", "License is not grantable"));
    NNT_ASSERT_RESULT_SUCCESS(ParseErrorCodeResponse(&outResult, pSourceBuffer));
    NNT_ASSERT_RESULT_FAILURE(nim::ResultDragonsUnknownError, outResult);

    // 異常系( 前方一致する存在しないエラーコードの検知ミス検証 )
    ASSERT_TRUE(sizeof(pSourceBuffer) > util::SNPrintf(pSourceBuffer, sizeof(pSourceBuffer), ErrorCodeResponseBase, "https://dragons.nintendo.com/errors/v1/403/license_not_grantable_dummy", "License is not grantable"));
    NNT_ASSERT_RESULT_SUCCESS(ParseErrorCodeResponse(&outResult, pSourceBuffer));
    NNT_ASSERT_RESULT_FAILURE(nim::ResultDragonsUnknownError, outResult);

    // 異常系( そもそもに期待するJSONじゃねぇ )
    ASSERT_TRUE(sizeof(pSourceBuffer) > util::SNPrintf(pSourceBuffer, sizeof(pSourceBuffer), "{\"notype\":\"%s\"}", "https://dragons.nintendo.com/errors/v1/403/license_not_grantable"));
    NNT_ASSERT_RESULT_SUCCESS(ParseErrorCodeResponse(&outResult, pSourceBuffer));
    NNT_ASSERT_RESULT_SUCCESS(outResult);
}

//!------------------------------------------------------------------------------------------------------
TEST_F(TestForDynamicRights, IpcCallViaDebugResponse)
{
    {
        NN_FUNCTION_LOCAL_STATIC(constexpr char, Response, [] =
        "{"
            "\"elicenses\": [{"
                "\"elicense_id\": \"387d31f4e62547dfa6b54043fa27b7fc\","
                "\"account_id\": \"fd3e99e841f985a2\","
                "\"rights_id\": \"01003ab001e32000\","
                "\"elicense_type\": \"temporary\""
            "}]"
        "}"
        );
        ec::DebugForShopServiceAccessor::ScopedSession session;
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::Initialize(&session));
        DebugResponse::EnableFirmwareSettings(true);

        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::ClearResponse());
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , "/v1/elicenses/extend"
            , Response
            , sizeof(Response) - 1
            , nn::ResultSuccess()
            , 10000
        ));
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , "/v1/elicenses/report"
            , Response
            , sizeof(Response) - 1
            , nn::ResultSuccess()
            , 10000
        ));
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , "/v1/elicenses/exercise"
            , Response
            , sizeof(Response) - 1
            , nn::ResultSuccess()
            , 10000
        ));
    }

    auto& user = GetUser(0);

    es::ELicenseId id;
    id.FromString("387d31f4e62547dfa6b54043fa27b7fc");

    account::NintendoAccountId naId;
    naId.id = 0xfd3e99e841f985a2;

    {
        nim::AsyncAssignedELicenses handle;
        NNT_ASSERT_RESULT_SUCCESS(nim::RequestExtendELicenses(&handle, user, &id, 1));

        int outCount = 0;
        nim::AssignedELicense licenses[2048];
        NNT_ASSERT_RESULT_SUCCESS(handle.Get(&outCount, licenses, NN_ARRAY_SIZE(licenses)));
    }
    {
        nim::AsyncResult result;
        NNT_ASSERT_RESULT_SUCCESS(nim::RequestReportActiveELicenses(&result, naId, &id, 1));
        NNT_ASSERT_RESULT_SUCCESS(result.Get());
    }
    {
        nim::AsyncResult result;
        NNT_ASSERT_RESULT_SUCCESS(nim::RequestReportActiveELicensesPassively(&result, naId, &id, 1));
        NNT_ASSERT_RESULT_SUCCESS(result.Get());
    }
}

//!------------------------------------------------------------------------------------------------------
//! @brief  デバッグレスポンス機能テスト( 期待しない Json レスポンス )
//! @note   "enable_simulate_dynamic_rights" 有効時の ServerType 指定は Dragons 固定扱いになるので冗長指定です.
TEST_F(TestForDynamicRights, DebugResponseWithUnexpectedDragonsResponse)
{
    {
        NN_FUNCTION_LOCAL_STATIC(constexpr char, UnexpectedReponse, [] = "This is debug response as unexpected!! Are you happy? I'm hungry...");

        ec::DebugForShopServiceAccessor::ScopedSession session;
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::Initialize(&session));
        DebugResponse::EnableFirmwareSettings(true);

        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::ClearResponse());
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , "/v1/rights/available_elicenses"
            , UnexpectedReponse
            , sizeof(UnexpectedReponse) - 1
            , nn::ResultSuccess()   // 通信自体は成功した扱いなので Success.
            , 10000
        ));
    }

    int outCount = 0;
    nim::AvailableELicense availableLicenses[2048];
    nim::AsyncAvailableELicenses handleAvailable;
    NNT_ASSERT_RESULT_SUCCESS(nim::RequestQueryAvailableELicenses(&handleAvailable, g_TtitleIds, NN_ARRAY_SIZE(g_TtitleIds)));

    // NOTE: JSONじゃないので JSONパーサーのエラーが返ります。
    NNT_EXPECT_RESULT_FAILURE(http::json::ResultJsonContentError, handleAvailable.Get(&outCount, availableLicenses, NN_ARRAY_SIZE(availableLicenses)));

    PRIVATE_LOG("outCount( %ld )\n", outCount);
    EXPECT_EQ(0, outCount);

    // 期待値( count == 0 )でない場合、何が返されたのかチェック用。
    for (int i = 0; i < outCount; ++i)
    {
        const auto& license = availableLicenses[i];
        PRIVATE_LOG("index[ %ld ] { rightsId(%llx), naId(%llx), type(%u), status(%u) }\n"
            , i
            , license.rightsId
            , license.naId.id
            , license.licenseType
            , license.licenseStatus
        );
        NN_UNUSED(license);
    }
}

//!------------------------------------------------------------------------------------------------------
//! @brief  デバッグレスポンス機能テスト( Problem details レスポンス )
//! @note   "enable_simulate_dynamic_rights" 有効時の ServerType 指定は Dragons 固定扱いになるので冗長指定です.
TEST_F(TestForDynamicRights, DebugResponseWithProblemDetailsResponse)
{
    {
        NN_FUNCTION_LOCAL_STATIC(constexpr char, ResponseBase, [] =
            "{"
                "\"type\": \"%s\","
                "\"title\": \"%s\","
                "\"detail\": null,"
                "\"extension\": null"
            "}"
        );
        char pResponse[4 * 1024];
        const size_t lengthOfResponse = util::SNPrintf(pResponse, sizeof(pResponse), ResponseBase, "https://dragons.nintendo.com/errors/v1/400/invalid_parameter", "Parameter is invalid");
        ASSERT_TRUE(sizeof(pResponse) > lengthOfResponse);

        ec::DebugForShopServiceAccessor::ScopedSession session;
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::Initialize(&session));
        DebugResponse::EnableFirmwareSettings(true);

        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::ClearResponse());
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , "/v1/rights/available_elicenses"
            , pResponse
            , lengthOfResponse
            , nim::ResultHttpStatus400BadRequest()
            , 10000
        ));
    }

    int outCount = 0;
    nim::AvailableELicense availableLicenses[2048];
    nim::AsyncAvailableELicenses handleAvailable;
    NNT_ASSERT_RESULT_SUCCESS(nim::RequestQueryAvailableELicenses(&handleAvailable, g_TtitleIds, NN_ARRAY_SIZE(g_TtitleIds)));
    NNT_ASSERT_RESULT_FAILURE(nim::ResultDragonsInvalidParameter, handleAvailable.Get(&outCount, availableLicenses, NN_ARRAY_SIZE(availableLicenses)));
}

//!------------------------------------------------------------------------------------------------------
//! @brief  デバッグレスポンス機能テスト( 正常系レスポンス )
//! @note   "enable_simulate_dynamic_rights" 有効時の ServerType 指定は Dragons 固定扱いになるので冗長指定です.
TEST_F(TestForDynamicRights, DebugResponseWithNormally)
{
    {
        NN_FUNCTION_LOCAL_STATIC(constexpr char, Response, [] =
            "{"
                "\"available_elicenses\": ["
                "{"
                    "\"account_id\": \"fd3e99e841f985a2\","
                    "\"rights_id\": \"0100000000000001\","
                    "\"is_available\": true,"
                    "\"elicense_type\": \"temporary\""
                "},{"
                    "\"account_id\": \"fd3e99e841f985a2\","
                    "\"rights_id\": \"010000000000000b\","
                    "\"is_available\": false,"
                    "\"elicense_type\": \"temporary\","
                    "\"reason\": \"limit_exceeded\""
                "}]"
            "}"
        );

        ec::DebugForShopServiceAccessor::ScopedSession session;
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::Initialize(&session));
        DebugResponse::EnableFirmwareSettings(true);

        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::ClearResponse());
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , "/v1/rights/available_elicenses"
            , Response
            , sizeof(Response) - 1
            , nn::ResultSuccess()
            , 10000
        ));
    }

    int outCount = 0;
    nim::AvailableELicense availableLicenses[2048];
    nim::AsyncAvailableELicenses handleAvailable;
    NNT_ASSERT_RESULT_SUCCESS(nim::RequestQueryAvailableELicenses(&handleAvailable, g_TtitleIds, NN_ARRAY_SIZE(g_TtitleIds)));
    NNT_ASSERT_RESULT_SUCCESS(handleAvailable.Get(&outCount, availableLicenses, NN_ARRAY_SIZE(availableLicenses)));
    PRIVATE_LOG("outCount( %ld )\n", outCount);
    EXPECT_EQ(2, outCount);
    {
        const auto& license = availableLicenses[0];
        EXPECT_EQ(es::RightsId(0x0100000000000001), license.rightsId);
        EXPECT_EQ(account::NintendoAccountId{0xfd3e99e841f985a2}, license.naId);
        EXPECT_EQ(nim::ELicenseType::Temporary, license.licenseType);
        EXPECT_EQ(nim::ELicenseStatus::Assignable, license.licenseStatus);
    }
    {
        const auto& license = availableLicenses[1];
        EXPECT_EQ(es::RightsId(0x010000000000000b), license.rightsId);
        EXPECT_EQ(account::NintendoAccountId{0xfd3e99e841f985a2}, license.naId);
        EXPECT_EQ(nim::ELicenseType::Temporary, license.licenseType);
        EXPECT_EQ(nim::ELicenseStatus::NotAssignableSinceLimitExceeded, license.licenseStatus);
    }

    // オフセット指定版テスト
    NNT_ASSERT_RESULT_SUCCESS(handleAvailable.Get(&outCount, availableLicenses, NN_ARRAY_SIZE(availableLicenses), 1));
    PRIVATE_LOG("offset(1), outCount( %ld )\n", outCount);
    EXPECT_EQ(1, outCount);
    {
        const auto& license = availableLicenses[0];
        EXPECT_EQ(es::RightsId(0x010000000000000b), license.rightsId);
        EXPECT_EQ(account::NintendoAccountId{0xfd3e99e841f985a2}, license.naId);
        EXPECT_EQ(nim::ELicenseType::Temporary, license.licenseType);
        EXPECT_EQ(nim::ELicenseStatus::NotAssignableSinceLimitExceeded, license.licenseStatus);
    }

    // オフセット指定版テスト( オーバーインデクス )
    NNT_ASSERT_RESULT_SUCCESS(handleAvailable.Get(&outCount, availableLicenses, NN_ARRAY_SIZE(availableLicenses), 2));
    PRIVATE_LOG("offset(2), outCount( %ld )\n", outCount);
    EXPECT_EQ(0, outCount);
}

//!------------------------------------------------------------------------------------------------------
//! @brief  デバッグレスポンス機能テスト( 異常構文レスポンス )
//! @note   "enable_simulate_dynamic_rights" 有効時の ServerType 指定は Dragons 固定扱いになるので冗長指定です.
TEST_F(TestForDynamicRights, DebugResponseWithJsonSyntaxError)
{
    {
        NN_FUNCTION_LOCAL_STATIC(constexpr char, Response, [] =
            "{"
                "\"available_elicenses\": ["
                "{"
                    "\"account_id\": \"fd3e99e841f985a2\","
                    "\"rights_id\": \"0100000000000001\","
                    "\"elicense_type\": \"temporary\""
                "},{"
                    "\"account_id\": \"fd3e99e841f985a2\","
                    "\"rights_id\": \"010000000000000b\","
                    "\"elicense_type\": \"unavailable\","
                    "\"reason\": \"limit_exceeded\""
                "}"     // <- ']' を外した。
            "}"
        );

        ec::DebugForShopServiceAccessor::ScopedSession session;
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::Initialize(&session));
        DebugResponse::EnableFirmwareSettings(true);

        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::ClearResponse());
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , "/v1/rights/available_elicenses"
            , Response
            , sizeof(Response) - 1
            , nn::ResultSuccess()
            , 10000
        ));
    }

    int outCount = 0;
    nim::AvailableELicense availableLicenses[2048];
    nim::AsyncAvailableELicenses handleAvailable;
    NNT_ASSERT_RESULT_SUCCESS(nim::RequestQueryAvailableELicenses(&handleAvailable, g_TtitleIds, NN_ARRAY_SIZE(g_TtitleIds)));
    // NOTE: JSON構文異常でJSONパーサーのエラーが返ります。
    NNT_EXPECT_RESULT_FAILURE(http::json::ResultJsonSyntaxError, handleAvailable.Get(&outCount, availableLicenses, NN_ARRAY_SIZE(availableLicenses)));

    PRIVATE_LOG("outCount( %ld )\n", outCount);
    EXPECT_EQ(0, outCount);

    // 期待値( count == 0 )でない場合、何が返されたのかチェック用。
    for (int i = 0; i < outCount; ++i)
    {
        const auto& license = availableLicenses[i];
        PRIVATE_LOG("index[ %ld ] { rightsId(%llx), naId(%llx), type(%u), status(%u) }\n"
            , i
            , license.rightsId
            , license.naId.id
            , license.licenseType
            , license.licenseStatus
        );
        NN_UNUSED(license);
    }
}

#if 0
//!------------------------------------------------------------------------------------------------------
// Challenge 無効化が外部から設定必要なのでデフォルトでは無効化しておきます。
TEST_F(TestForDynamicRights, DebugResponseForSyncELicense)
{
    // NOTE: Challenge 検証無効化をしておく必要があります。
    // DevMenuCommandSystem debug set-boolean-fwdbg --name "es.debug" --key "ignore_elicense_archive_challenge_verification_for_debug" true

    // naId(0xfd3e99e841f985a2) + RightsId(0x01003ab001e32000) 構成を ArchiveGenerator::GenerateDummy() で生成した archive_id は 8cb25c930b43432698d66986863b2087。
    const nn::es::ELicenseArchiveId eLicenseArchiveId = {0x8c, 0xb2, 0x5c, 0x93, 0x0b, 0x43, 0x43, 0x26, 0x98, 0xd6, 0x69, 0x86, 0x86, 0x3b, 0x20, 0x87};
    const nn::account::NintendoAccountId naId = {0xfd3e99e841f985a2};
    const nn::es::RightsId rightsId = g_TtitleIds[0];
    {
        constexpr char archiveResponsePrefix[] = "{\"elicense_archive\": \"";
        constexpr char archiveResponseSuffix[] = "\"}";
        constexpr size_t ArchiveResponseCapacity = 128 * 1024;
        std::unique_ptr<char> responseForArchive(new char[ArchiveResponseCapacity]);
        NN_STATIC_ASSERT(ArchiveResponseCapacity > (nnt::elicense::ArchiveGenerator::BufferCapacity * 4 / 3 + sizeof(archiveResponsePrefix) + sizeof(archiveResponseSuffix)));

        auto pResponse = responseForArchive.get();
        size_t availableSize = ArchiveResponseCapacity;

        ASSERT_TRUE(::nn::util::Strlcpy(pResponse, archiveResponsePrefix, availableSize) < availableSize);
        pResponse = &pResponse[sizeof(archiveResponsePrefix) - 1];
        availableSize -= sizeof(archiveResponsePrefix) - 1;

        nnt::elicense::ArchiveGenerator generator;
        const size_t lengthForBase64 = generator.GenerateDummy(pResponse, availableSize, eLicenseArchiveId, naId, rightsId);
        ASSERT_TRUE(lengthForBase64 > 0);
        pResponse = &pResponse[lengthForBase64];
        availableSize -= lengthForBase64;

        ASSERT_TRUE(::nn::util::Strlcpy(pResponse, archiveResponseSuffix, availableSize) < availableSize);
        availableSize -= sizeof(archiveResponseSuffix) - 1;

        ec::DebugForShopServiceAccessor::ScopedSession session;
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::Initialize(&session));
        DebugResponse::EnableFirmwareSettings(true);

        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::ClearResponse());

        // 同期発行。
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , "/v1/elicense_archives/publish"
            , responseForArchive.get()
            , ArchiveResponseCapacity - availableSize
            , nn::ResultSuccess()
            , 10000
        ));

        // 完了報告。
        char PathBuf[128];
        char idBuf[nn::es::ELicenseArchiveId::StringSize + 1];
        ASSERT_TRUE(sizeof(PathBuf) > util::SNPrintf(PathBuf, sizeof(PathBuf), "/v1/elicense_archives/%s/report", eLicenseArchiveId.ToString(idBuf, sizeof(idBuf))));
        NNT_ASSERT_RESULT_SUCCESS(ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , PathBuf
            , ""    // ゼロ長レスポンスはデバッグAPIがサポートしていないので最短の長さ 1 ( '\0' のみ配列 ) をデフォルト。
            , 1
            , nn::ResultSuccess()
            , 10000
        ));
    }

    nim::AsyncResult handle;
    NNT_ASSERT_RESULT_SUCCESS(nim::RequestSyncELicenses(&handle, naId));
    NNT_ASSERT_RESULT_SUCCESS(handle.Get());
}
#endif

//!------------------------------------------------------------------------------------------------------
TEST_F(TestForDynamicRights, NotificationToken)
{
    npns::NotificationToken token = {};
    nn::ApplicationId id = {0x010000000000001F};  // ns の ProgramId
    nn::account::Uid uid = {};
    // ユーザを指定しない通知なので 0 埋めの uid を渡す
    NNT_ASSERT_RESULT_SUCCESS(nn::npns::CreateToken(&token, uid, id));

    nim::AsyncResult handle;
    NNT_ASSERT_RESULT_SUCCESS(nim::RequestRegisterDynamicRightsNotificationToken(&handle, token));
    NNT_ASSERT_RESULT_SUCCESS(handle.Get());
}

//!------------------------------------------------------------------------------------------------------
//! RequestQueryAvailableELicenses ( 機器認証済権利 )
/*
    本テストは以下購入状態での動作を想定しています。
    DevMenuCommandSystem -- shop device-link-status 0
    DevMenuCommandSystem -- shop register-device-account AND shop shop-account-status 0
    DevMenuCommandSystem -- shop link-device 0
    DevMenuCommandSystem -- shop download-demo 0 --id 0x01003ab001e32000
*/
TEST_F(TestForDynamicRights, AvailableELicensesWithoutUser)
{
    {
        nim::AsyncAvailableELicenses handleAvailable;
        NNT_ASSERT_RESULT_SUCCESS(nim::RequestQueryAvailableELicenses(&handleAvailable, g_TtitleIds, NN_ARRAY_SIZE(g_TtitleIds)));

        int outCount = 0;
        nim::AvailableELicense availableLicenses[2048];
        NNT_ASSERT_RESULT_SUCCESS(handleAvailable.Get(&outCount, availableLicenses, NN_ARRAY_SIZE(availableLicenses)));

        PRIVATE_LOG("outCount( %ld )\n", outCount);
        {
            // 期待値( count == 0 )でない場合、何が返されたのかチェック用。
            for (int i = 0; i < outCount; ++i)
            {
                const auto& license = availableLicenses[i];
                PRIVATE_LOG("index[ %ld ] { rightsId(%llx), naId(%llx), type(%u), status(%u) }\n"
                    , i
                    , license.rightsId
                    , license.naId.id
                    , license.licenseType
                    , license.licenseStatus
                );
                NN_UNUSED(license);
            }
        }
        // Dragons は、対象 rights_id が対応している全 elicense_type を発行可能か調べて各々の発行可否を返す
        // 本テストは機器認証状態で行う想定なので ( device_linked_permanent + temporary )
        ASSERT_EQ(2, outCount);

        std::pair<nim::ELicenseType, bool> expectTypes[] =
        {
            {nim::ELicenseType::Temporary, false},
            {nim::ELicenseType::DeviceLinkedPermanent, false},
        };
        for (int i = 0; i < outCount; ++i)
        {
            const auto& license = availableLicenses[i];
            ASSERT_EQ(g_TtitleIds[0], license.rightsId);

            // レスポンスは順不同ぽいので期待する type が全てあるかチェック。
            const auto licenseType = license.licenseType;
            const auto pEnd = expectTypes + std::extent<decltype(expectTypes)>::value;  // Coverity 範囲外要素アクセス指摘対策.
            auto pFind = std::find_if(expectTypes, pEnd, [&licenseType](const std::pair<nim::ELicenseType, bool>& value) -> bool
            {
                return (licenseType == value.first && false == value.second);
            });
            ASSERT_NE(pFind, pEnd);
            pFind->second = true;   // pFind == pEnd なら、ASSERT_NE で止まる前提。

            if (licenseType == nim::ELicenseType::Temporary)
            {
                // temporary は NA 未指定なので権利がないと判断され、発行不可
                ASSERT_EQ(nim::ELicenseStatus::NotAssignableSinceNoRights, license.licenseStatus);
            }
            else if (licenseType == nim::ELicenseType::DeviceLinkedPermanent)
            {
                // NA 未指定だけれども機器認証しているので、device_linked_permanent は発行可
                ASSERT_EQ(nim::ELicenseStatus::Assignable, license.licenseStatus);
            }
        }
    }

    es::ELicenseId eLicenseId = es::InvalidELicenseId;
    account::NintendoAccountId naId = account::InvalidNintendoAccountId;
    {
        nim::AsyncAssignedELicenses handle;
        NNT_ASSERT_RESULT_SUCCESS(nim::RequestAssignELicenses(&handle, g_TtitleIds, NN_ARRAY_SIZE(g_TtitleIds), nim::ELicenseType::DeviceLinkedPermanent));

        int outCount = 0;
        nim::AssignedELicense licenses[2048];
        NNT_ASSERT_RESULT_SUCCESS(handle.Get(&outCount, licenses, NN_ARRAY_SIZE(licenses)));

        PRIVATE_LOG("outCount( %ld )\n", outCount);
        {
            // 期待値( count == 0 )でない場合、何が返されたのかチェック用。
            for (int i = 0; i < outCount; ++i)
            {
                char idBuf[es::ELicenseId::StringSize + 1];
                const auto& license = licenses[i];
                PRIVATE_LOG("index[ %ld ] { eLicenseId(%s), rightsId(%llx), naId(%llx), type(%u) }\n"
                    , i
                    , license.eLicenseId.ToString(idBuf, sizeof(idBuf))
                    , license.rightsId
                    , license.naId.id
                    , license.licenseType
                );
                NN_UNUSED(license);
                NN_UNUSED(idBuf);
            }
        }
        ASSERT_EQ(1, outCount);

        const auto& license = licenses[0];
        naId = license.naId;
        eLicenseId = license.eLicenseId;
        ASSERT_EQ(g_TtitleIds[0], license.rightsId);
        ASSERT_EQ(nim::ELicenseType::DeviceLinkedPermanent, license.licenseType);
    }
    if (es::InvalidELicenseId != eLicenseId && account::InvalidNintendoAccountId != naId)
    {
        nim::AsyncResult handle;
        NNT_ASSERT_RESULT_SUCCESS(nim::RequestReportActiveELicenses(&handle, naId, &eLicenseId, 1));
        NNT_ASSERT_RESULT_SUCCESS(handle.Get());
    }
}

//!------------------------------------------------------------------------------------------------------
//! RequestQueryAvailableELicenses ( 指定ユーザ権利 )
/*
    本テストは以下購入状態での動作を想定しています。
    DevMenuCommandSystem -- shop device-link-status 0
    DevMenuCommandSystem -- shop register-device-account AND shop shop-account-status 0
    DevMenuCommandSystem -- shop link-device 0
    DevMenuCommandSystem -- shop download-demo 0 --id 0x01003ab001e32000
*/
TEST_F(TestForDynamicRights, AvailableELicensesWithUser)
{
    auto& user = GetUser(0);
    {
        nim::AsyncAvailableELicenses handleAvailable;
        NNT_ASSERT_RESULT_SUCCESS(nim::RequestQueryAvailableELicenses(&handleAvailable, user, g_TtitleIds, NN_ARRAY_SIZE(g_TtitleIds)));

        int outCount = 0;
        nim::AvailableELicense availableLicenses[2048];
        NNT_ASSERT_RESULT_SUCCESS(handleAvailable.Get(&outCount, availableLicenses, NN_ARRAY_SIZE(availableLicenses)));

        PRIVATE_LOG("outCount( %ld )\n", outCount);
        {
            // 期待値( count == 0 )でない場合、何が返されたのかチェック用。
            for (int i = 0; i < outCount; ++i)
            {
                const auto& license = availableLicenses[i];
                PRIVATE_LOG("index[ %ld ] { rightsId(%llx), naId(%llx), type(%u), status(%u) }\n"
                    , i
                    , license.rightsId
                    , license.naId.id
                    , license.licenseType
                    , license.licenseStatus
                );
                NN_UNUSED(license);
            }
        }
        // Dragons は、対象 rights_id が対応している全 elicense_type を発行可能か調べて各々の発行可否を返す
        // 本テストは機器認証状態で行う想定なので ( device_linked_permanent + temporary )
        ASSERT_EQ(2, outCount);

        std::pair<nim::ELicenseType, bool> expectTypes[] =
        {
            {nim::ELicenseType::Temporary, false},
            {nim::ELicenseType::DeviceLinkedPermanent, false},
        };
        for (int i = 0; i < outCount; ++i)
        {
            const auto& license = availableLicenses[i];
            ASSERT_EQ(g_TtitleIds[0], license.rightsId);

            // 機器認証状態ユーザの指定での問い合わせなので、device_linked_permanent, temporary のどちらも割り当て可。
            ASSERT_EQ(nim::ELicenseStatus::Assignable, license.licenseStatus);

            // レスポンスは順不同ぽいので期待する type が全てあるかチェック。
            const auto licenseType = license.licenseType;
            const auto pEnd = expectTypes + std::extent<decltype(expectTypes)>::value;  // Coverity 範囲外要素アクセス指摘対策.
            auto pFind = std::find_if(expectTypes, pEnd, [&licenseType](const std::pair<nim::ELicenseType, bool>& value) -> bool
            {
                return (licenseType == value.first && false == value.second);
            });
            ASSERT_NE(pFind, pEnd);
            pFind->second = true;   // pFind == pEnd なら、ASSERT_NE で止まる前提。
        }
    }

    es::ELicenseId eLicenseId = es::InvalidELicenseId;
    account::NintendoAccountId naId = account::InvalidNintendoAccountId;
    {
        nim::AsyncAssignedELicenses handle;
        NNT_ASSERT_RESULT_SUCCESS(nim::RequestAssignELicenses(&handle, user, g_TtitleIds, NN_ARRAY_SIZE(g_TtitleIds), nim::ELicenseType::DeviceLinkedPermanent));

        int outCount = 0;
        nim::AssignedELicense licenses[2048];
        NNT_ASSERT_RESULT_SUCCESS(handle.Get(&outCount, licenses, NN_ARRAY_SIZE(licenses)));

        PRIVATE_LOG("outCount( %ld )\n", outCount);
        ASSERT_EQ(1, outCount);
        {
            const auto& license = licenses[0];
            naId = license.naId;
            eLicenseId = license.eLicenseId;
            EXPECT_EQ(g_TtitleIds[0], license.rightsId);
            EXPECT_EQ(nim::ELicenseType::DeviceLinkedPermanent, license.licenseType);

            // 期待値( count == 0 )でない場合、何が返されたのかチェック用。
            for (int i = 0; i < outCount; ++i)
            {
                char idBuf[es::ELicenseId::StringSize + 1];
                const auto& license = licenses[i];
                PRIVATE_LOG("index[ %ld ] { eLicenseId(%s), rightsId(%llx), naId(%llx), type(%u) }\n"
                    , i
                    , license.eLicenseId.ToString(idBuf, sizeof(idBuf))
                    , license.rightsId
                    , license.naId.id
                    , license.licenseType
                );
                NN_UNUSED(license);
                NN_UNUSED(idBuf);
            }
        }
    }
    if (es::InvalidELicenseId != eLicenseId && account::InvalidNintendoAccountId != naId)
    {
        // 同期
        {
            nim::AsyncResult handle;
            NNT_ASSERT_RESULT_SUCCESS(nim::RequestSyncELicenses(&handle, naId));
            NNT_ASSERT_RESULT_SUCCESS(handle.Get());
        }
        // 開始通知
        {
            nim::AsyncResult handle;
            NNT_ASSERT_RESULT_SUCCESS(nim::RequestReportActiveELicenses(&handle, naId, &eLicenseId, 1));
            NNT_ASSERT_RESULT_SUCCESS(handle.Get());
        }
        // 延長
        nim::AsyncAssignedELicenses extend;
        NNT_ASSERT_RESULT_SUCCESS(nim::RequestExtendELicenses(&extend, user, &eLicenseId, 1));

        int outCount = 0;
        nim::AssignedELicense licenses[2048];
        NNT_ASSERT_RESULT_SUCCESS(extend.Get(&outCount, licenses, NN_ARRAY_SIZE(licenses)));

        PRIVATE_LOG("outCount( %ld )\n", outCount);
        ASSERT_EQ(1, outCount);
        {
            const auto& license = licenses[0];
            EXPECT_EQ(naId, license.naId);
            EXPECT_EQ(eLicenseId, license.eLicenseId);

            // 期待値( count == 0 )でない場合、何が返されたのかチェック用。
            for (int i = 0; i < outCount; ++i)
            {
                char idBuf[es::ELicenseId::StringSize + 1];
                const auto& license = licenses[i];
                PRIVATE_LOG("index[ %ld ] { eLicenseId(%s), rightsId(%llx), naId(%llx), type(%u) }\n"
                    , i
                    , license.eLicenseId.ToString(idBuf, sizeof(idBuf))
                    , license.rightsId
                    , license.naId.id
                    , license.licenseType
                );
                NN_UNUSED(license);
                NN_UNUSED(idBuf);
            }
        }
    }
} // NOLINT(impl/function_size)

//!------------------------------------------------------------------------------------------------------
//! RequestAssignAllDeviceLinkedELicenses ( 機器認証済権利 )
/*
    本テストは以下購入状態での動作を想定しています。
    DevMenuCommandSystem -- shop device-link-status 0
    DevMenuCommandSystem -- shop register-device-account AND shop shop-account-status 0
    DevMenuCommandSystem -- shop link-device 0
    DevMenuCommandSystem -- shop download-demo 0 --id 0x01003ab001e32000
*/
TEST_F(TestForDynamicRights, RequestAssignAllDeviceLinkedELicenses)
{
    nim::AsyncAssignedELicenses handle;
    NNT_ASSERT_RESULT_SUCCESS(nim::RequestAssignAllDeviceLinkedELicenses(&handle));

    int outCount = 0;
    nim::AssignedELicense licenses[2048];
    NNT_ASSERT_RESULT_SUCCESS(handle.Get(&outCount, licenses, NN_ARRAY_SIZE(licenses)));

    PRIVATE_LOG("outCount( %ld )\n", outCount);
    {
        // 期待値( count == 0 )でない場合、何が返されたのかチェック用。
        for (int i = 0; i < outCount; ++i)
        {
            char idBuf[es::ELicenseId::StringSize + 1];
            const auto& license = licenses[i];
            PRIVATE_LOG("index[ %ld ] { eLicenseId(%s), rightsId(%llx), naId(%llx), type(%u) }\n"
                , i
                , license.eLicenseId.ToString(idBuf, sizeof(idBuf))
                , license.rightsId
                , license.naId.id
                , license.licenseType
            );
            NN_UNUSED(license);
            NN_UNUSED(idBuf);
        }
    }

    // テスト実施デバイスに機器認証しているアカウントと、各アカウントの持つ権利状態が不定のため、ここでは最低 1 以上を確認する事を是とする。
    ASSERT_GE(outCount, 1);

    // 事前条件である 0x01003ab001e32000 の購入権利が存在する事、及び、全ライセンスが DeviceLinkedPermanent である事を検証観点とする。
    bool hasExpectRights = false;
    for (int i = 0; i < outCount; ++i)
    {
        const auto& license = licenses[i];
        if (g_TtitleIds[0] == license.rightsId)
        {
            hasExpectRights = true;
        }
        ASSERT_EQ(nim::ELicenseType::DeviceLinkedPermanent, license.licenseType);
    }
    ASSERT_TRUE(hasExpectRights);
}

#endif // defined( NN_BUILD_CONFIG_OS_HORIZON )
