﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nim/srv/detail/nim_BufferAllocator.h>

using namespace nn::nim::srv::detail;

namespace {
    const size_t BufferSize = 256 * 1024;
    NN_ALIGNAS(BufferAlign) nn::Bit8 g_Buffer[BufferSize];

    typedef BufferFlagSet::Flag<0> Flag0;
    typedef BufferFlagSet::Flag<1> Flag1;
    typedef BufferFlagSet::Flag<2> Flag2;
    typedef BufferFlagSet::Flag<3> Flag3;
    typedef BufferFlagSet::Flag<4> Flag4;

    const BufferFlagSet Group0 = Flag0::Mask | Flag1::Mask;
    BufferInfo g_Info0(BufferAlign, Flag0::Mask, Group0);
    BufferInfo g_Info1(BufferAlign, Flag1::Mask, Group0);
    BufferInfo g_Info2(BufferAlign, Flag2::Mask, Flag2::Mask);

    const BufferInfo* InfoList[] =
    {
        &g_Info0,
        &g_Info1,
        &g_Info2,
    };

    const int InfoCount = static_cast<int>(NN_ARRAY_SIZE(InfoList));

}

TEST(BufferInfo, BaseTest)
{
    auto group = Flag0::Mask | Flag1::Mask;
    auto info = BufferInfo(BufferAlign, Flag0::Mask, group);

    EXPECT_EQ(BufferAlign, info.GetSize());
    EXPECT_EQ(Flag0::Mask, info.GetFlag());
    EXPECT_EQ(group, info.GetGroup());
}

TEST(BufferInfoDeathTest, ConstructorTest)
{
    // サイズのチェック
    EXPECT_DEATH_IF_SUPPORTED(new BufferInfo(0, Flag0::Mask, Flag0::Mask), "");
    EXPECT_DEATH_IF_SUPPORTED(new BufferInfo(1, Flag0::Mask, Flag0::Mask), "");
    EXPECT_DEATH_IF_SUPPORTED(new BufferInfo(BufferAlign - 1, Flag0::Mask, Flag0::Mask), "");
    EXPECT_DEATH_IF_SUPPORTED(new BufferInfo(BufferAlign + 1, Flag0::Mask, Flag0::Mask), "");

    // Flag が Group に含まれていない
    EXPECT_DEATH_IF_SUPPORTED(new BufferInfo(BufferAlign, Flag0::Mask, Flag1::Mask), "");

    // Flag に複数ビット立っている
    auto group = Flag0::Mask | Flag1::Mask;
    EXPECT_DEATH_IF_SUPPORTED(new BufferInfo(BufferAlign, group, Flag0::Mask), "");
}

TEST(BufferAllocatorTest, BasicTest)
{
    BufferAllocator allocator(g_Buffer, BufferSize, InfoList, InfoCount);

    size_t outSize;
    auto buffer0 = allocator.Allocate(&outSize, Flag0::Mask);
    EXPECT_TRUE(nullptr != buffer0);
    EXPECT_EQ(outSize, g_Info0.GetSize());
    auto buffer1 = allocator.Allocate(&outSize, Flag1::Mask);
    EXPECT_TRUE(nullptr != buffer1);
    EXPECT_EQ(outSize, g_Info1.GetSize());
    allocator.Free(buffer0, Flag0::Mask);
    allocator.Free(buffer1, Flag1::Mask);

    auto buffer2 = allocator.Allocate(&outSize, Flag2::Mask);
    EXPECT_TRUE(nullptr != buffer2);
    EXPECT_EQ(outSize, g_Info2.GetSize());
    allocator.Free(buffer2, Flag2::Mask);
}

TEST(BufferAllocatorDeathTest, ConstructorTest)
{
    EXPECT_DEATH_IF_SUPPORTED(new BufferAllocator(nullptr, BufferSize, InfoList, InfoCount), "");
    EXPECT_DEATH_IF_SUPPORTED(new BufferAllocator(g_Buffer, 0, InfoList, InfoCount), "");
    EXPECT_DEATH_IF_SUPPORTED(new BufferAllocator(g_Buffer, 1, InfoList, InfoCount), "");
    EXPECT_DEATH_IF_SUPPORTED(new BufferAllocator(g_Buffer, BufferAlign - 1, InfoList, InfoCount), "");
    EXPECT_DEATH_IF_SUPPORTED(new BufferAllocator(g_Buffer, BufferAlign + 1, InfoList, InfoCount), "");
    EXPECT_DEATH_IF_SUPPORTED(new BufferAllocator(g_Buffer, BufferSize, nullptr, InfoCount), "");
    EXPECT_DEATH_IF_SUPPORTED(new BufferAllocator(g_Buffer, BufferSize, InfoList, 0), "");

    // 容量不足
    BufferInfo tooBigInfo(BufferSize, Flag4::Mask, Flag4::Mask);
    const BufferInfo* infoList[] =
    {
        &tooBigInfo,
    };
    const int infoCount = static_cast<int>(NN_ARRAY_SIZE(infoList));
    EXPECT_DEATH_IF_SUPPORTED(new BufferAllocator(g_Buffer, BufferSize, infoList, infoCount), "");

}

TEST(BufferAllocatorDeathTest, AllocateTest)
{
    BufferAllocator allocator(g_Buffer, BufferSize, InfoList, InfoCount);

    size_t outSize;
    auto buffer0 = allocator.Allocate(&outSize, Flag0::Mask);
    EXPECT_TRUE(nullptr != buffer0);
    EXPECT_EQ(outSize, g_Info0.GetSize());
    auto buffer1 = allocator.Allocate(&outSize, Flag1::Mask);
    EXPECT_TRUE(nullptr != buffer1);
    EXPECT_EQ(outSize, g_Info1.GetSize());

    // 自分の所属しているメモリブロックのリスト以外のメモリブロックが確保中は確保できない
    EXPECT_DEATH_IF_SUPPORTED(allocator.Allocate(&outSize, Flag2::Mask), "");

    allocator.Free(buffer0, Flag0::Mask);

    EXPECT_DEATH_IF_SUPPORTED(allocator.Allocate(&outSize, Flag2::Mask), "");

    allocator.Free(buffer1, Flag1::Mask);

    auto buffer2 = allocator.Allocate(&outSize, Flag2::Mask);
    EXPECT_TRUE(nullptr != buffer2);
    EXPECT_EQ(outSize, g_Info2.GetSize());
    allocator.Free(buffer2, Flag2::Mask);

    // 未登録
    EXPECT_DEATH_IF_SUPPORTED(allocator.Allocate(&outSize, Flag3::Mask), "");
}

TEST(BufferAllocatorDeathTest, FreeTest)
{
    BufferAllocator allocator(g_Buffer, BufferSize, InfoList, InfoCount);
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(nullptr, Flag0::Mask), "");

    size_t outSize;
    auto buffer = allocator.Allocate(&outSize, Flag0::Mask);
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(buffer, Flag1::Mask), "");
    allocator.Free(buffer, Flag0::Mask);
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(buffer, Flag0::Mask), "");
}

