﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm/detail/core/nifm_AccessPointManager.h>
#include <nn/nifm/detail/core/networkInterface/nifm_WirelessInterface.h>
#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterface.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileManager.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_CharacterEncoding.h>

namespace
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }
#endif
}

namespace nn
{
namespace nifm
{
namespace detail
{

TEST(WirelessInterfaceCase, ConnectAndDisconnect )
{
    AccessPointManager apManager;
    NetworkProfileManager npManager;

    {
        const NetworkProfileBase* pNp = nullptr;

        for( int i=0; i<npManager.GetNetworkProfileCount(NetworkProfileBase::ProfileType::UserProfile); ++i )
        {
            pNp = npManager.GetNetworkProfile(NetworkProfileBase::ProfileType::UserProfile, i);
            ASSERT_NE(nullptr, pNp);

            char profileName[NetworkProfileBasicInfo::NameSize];
            pNp->GetName( profileName);

            if( nn::util::Strncmp( "nbd-plala1wpa2", profileName, NetworkProfileBasicInfo::NameSize ) == 0 )
            {
                break;
            }
        }

        WirelessInterface wirelessInterface;
        MacAddress macAddresses[1];
        int32_t outCount;

        WirelessInterface::GetAllMacAddresses( macAddresses, &outCount, 1 );
        if( outCount > 0 )
        {
            wirelessInterface.Initialize( macAddresses[0] );
        }
        else
        {
            NN_SDK_ASSERT( wirelessInterface.IsVirtualInterface() );
            NN_DETAIL_NIFM_LOG_DEBUG("add virtual wireless interface\n");
            wirelessInterface.Initialize();
        }

        NNT_ASSERT_RESULT_SUCCESS( wirelessInterface.Scan() );

        NNT_ASSERT_RESULT_SUCCESS( wirelessInterface.GetAccessPointList( &apManager ) );

        const AccessPoint* pAp = nullptr;
        bool isSuccessTestDone = false;
        bool isFailureTestDone = false;

        for( int i=0; i<apManager.GetAccessPointCount(); ++i )
        {
            pAp = apManager.GetAccessPoint(i);
            ASSERT_NE(nullptr, pAp);

            Ssid apSsid;
            pAp->GetSsid(&apSsid);

            Ssid profileSsid;
            pNp->GetWirelessSetting().GetSsid(&profileSsid);

            if( apSsid.length == profileSsid.length &&
                memcmp( apSsid.hex, profileSsid.hex, apSsid.length ) == 0 )
            {
                isSuccessTestDone = true;

                NNT_EXPECT_RESULT_SUCCESS( wirelessInterface.Connect( *pNp, *pAp ) );

                NNT_EXPECT_RESULT_FAILURE( nn::nifm::ResultInternalError, wirelessInterface.Connect( *pNp, *pAp ) );
                NNT_EXPECT_RESULT_FAILURE( nn::nifm::ResultInternalError, wirelessInterface.Release() );

                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1)); // TORIAEZU

                NNT_EXPECT_RESULT_SUCCESS( wirelessInterface.Disconnect() );

                NNT_EXPECT_RESULT_SUCCESS( wirelessInterface.Disconnect() );
                NNT_EXPECT_RESULT_FAILURE( nn::nifm::ResultInternalError, wirelessInterface.Connect( *pNp, *pAp ) );

                NNT_EXPECT_RESULT_SUCCESS( wirelessInterface.Release() );
            }
            else
            {
                isFailureTestDone = true;

                EXPECT_TRUE( wirelessInterface.Connect( *pNp, *pAp ).IsFailure() );

                NNT_EXPECT_RESULT_FAILURE( nn::nifm::ResultInternalError, wirelessInterface.Connect( *pNp, *pAp ) );
                NNT_EXPECT_RESULT_SUCCESS( wirelessInterface.Disconnect() );

                NNT_EXPECT_RESULT_SUCCESS( wirelessInterface.Release() );
            }
        }

        EXPECT_TRUE(isSuccessTestDone);
        EXPECT_TRUE(isFailureTestDone);

        wirelessInterface.Disconnect();
        NNT_ASSERT_RESULT_SUCCESS( wirelessInterface.Release() );
    }

} // NOLINT(readability/fn_size)

}
}
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
extern "C" void nninitStartup()
{
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    NN_ASSERT( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    NN_ASSERT( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );

    nn::fs::SetAllocator(Allocate, Deallocate);
}
#endif

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    WSADATA wsaData;
    if( ::WSAStartup(MAKEWORD(2,0), &wsaData) != ERROR_SUCCESS )
    {
        return;
    }

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
