﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Log.h>
#include <nn/os/os_TickApi.h>

#include "../Common/nifm_TestUtility.h"

namespace
{
}

class StatisticsCaseTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        // 本テストプロセスで独占
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(true);
    }

    static void TearDownTestCase()
    {
        // 本テストプロセスで独占解除
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(false);
    }
};

TEST_F(StatisticsCaseTest, Wpa2Aes)
{
    static const int testCount = 20;
    nn::TimeSpan notRequiredTimeSpan = nn::TimeSpan();
    nn::TimeSpan forcedTimeSpan = nn::TimeSpan();
    int notRequiredSuccessCount = 0;
    int forcedSuccessCount = 0;

    for(int i = 0; i < testCount; ++i)
    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  // authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        // ex.DHCP
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        // ConnectionConfirmationOption_NotRequired
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        auto start = nn::os::GetSystemTick();
        networkConnection.SubmitRequestAndWait();
        auto end = nn::os::GetSystemTick();

        notRequiredTimeSpan += (end - start).ToTimeSpan();
        if (networkConnection.GetResult().IsSuccess())
        {
            ++notRequiredSuccessCount;
        }

        networkConnection.CancelRequest();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        // ConnectionConfirmationOption_Forced
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

        start = nn::os::GetSystemTick();
        networkConnection.SubmitRequestAndWait();
        end = nn::os::GetSystemTick();

        forcedTimeSpan += (end - start).ToTimeSpan();
        if (networkConnection.GetResult().IsSuccess())
        {
            ++forcedSuccessCount;
        }

        networkConnection.CancelRequest();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_LOG("##teamcity[buildStatisticValue key='Wpa2Aes_NRrate' value='%f']\n", static_cast<double>(notRequiredSuccessCount) / testCount * 100);
    NN_LOG("##teamcity[buildStatisticValue key='Wpa2Aes_NRtime(ms)' value='%d']\n", notRequiredTimeSpan.GetMilliSeconds() / notRequiredSuccessCount);
    NN_LOG("##teamcity[buildStatisticValue key='Wpa2Aes_Frate' value='%f']\n", static_cast<double>(forcedSuccessCount) / testCount * 100);
    NN_LOG("##teamcity[buildStatisticValue key='Wpa2Aes_Ftime(ms)' value='%lld']\n", forcedTimeSpan.GetMilliSeconds() / forcedSuccessCount);
}

TEST_F(StatisticsCaseTest, ImatakeWep104)
{
    static const int testCount = 20;
    nn::TimeSpan notRequiredTimeSpan = nn::TimeSpan();
    int notRequiredSuccessCount = 0;

    for (int i = 0; i < testCount; ++i)
    {
        // imatake-openwep104
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    18,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x31,0x30,0x34 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  // authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_Wep  // encryption
                },
                {  // sharedKey
                    13,  // length
                    "Shi2iTaiZenTo"  // keyMaterial
                }
            }
        };
        // ex.DHCP
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        // ConnectionConfirmationOption_NotRequired
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        auto start = nn::os::GetSystemTick();
        networkConnection.SubmitRequestAndWait();
        auto end = nn::os::GetSystemTick();

        notRequiredTimeSpan += (end - start).ToTimeSpan();
        if (networkConnection.GetResult().IsSuccess())
        {
            ++notRequiredSuccessCount;
        }

        networkConnection.CancelRequest();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_LOG("##teamcity[buildStatisticValue key='Wep104_NRrate' value='%f']\n", static_cast<double>(notRequiredSuccessCount) / testCount * 100);
    NN_LOG("##teamcity[buildStatisticValue key='Wep104_NRtime(ms)' value='%d']\n", notRequiredTimeSpan.GetMilliSeconds() / notRequiredSuccessCount);
}

TEST_F(StatisticsCaseTest, Ethernet)
{
    static const int testCount = 20;
    nn::TimeSpan notRequiredTimeSpan = nn::TimeSpan();
    int notRequiredSuccessCount = 0;

    for (int i = 0; i < testCount; ++i)
    {
        // Ethernet
        const nn::nifm::WirelessSettingData wirelessSetting = nn::nifm::InvalidWirelessSettingData;
        const nn::nifm::IpSettingData ipSetting = { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 };
        nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_Temporary, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true,{ wirelessSetting }, ipSetting };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        // ConnectionConfirmationOption_NotRequired
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        auto start = nn::os::GetSystemTick();
        networkConnection.SubmitRequestAndWait();
        auto end = nn::os::GetSystemTick();

        notRequiredTimeSpan += (end - start).ToTimeSpan();
        if (networkConnection.GetResult().IsSuccess())
        {
            ++notRequiredSuccessCount;
        }

        networkConnection.CancelRequest();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    }

    NN_LOG("##teamcity[buildStatisticValue key='Eth_NRrate' value='%f']\n", static_cast<double>(notRequiredSuccessCount) / testCount * 100);
    NN_LOG("##teamcity[buildStatisticValue key='Eth_NRtime(ms)' value='%d']\n", notRequiredTimeSpan.GetMilliSeconds() / notRequiredSuccessCount);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
