﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Log.h>

#include <nn/util/util_Base64.h>
#include <nn/util/util_BitFlagSet.h>

#include <nn/nifm/detail/core/profile/nifm_WirelessUserSetting.h>
#include <nn/nifm/detail/core/profile/nifm_IpSetting.h>
#include <nn/nifm/detail/core/profile/nifm_UserNetworkProfile.h>
#include <nn/nifm/detail/core/profile/nifm_SsidListNetworkProfile.h>
#include <nn/nifm/detail/util/nifm_ArrayedFactory.h>

#include <nn/nifm/detail/util/nifm_SsidListUtility.h>

#include "../Common/nifm_TestUtility.h"

namespace
{
    class SsidListEntry
    {
    public:
        /// encoded
        char encodedId[nn::util::Uuid::StringSize];
        char encodedName[86];
        char encodedSsid[44];
        char encodedKey[86];
        int encodedAuth;
        int encodedEnc;
        int encodedIsAuto;
    public:
        /// decoded
        nn::util::Uuid id;
        char name[nn::nifm::NetworkProfileBasicInfo::NameSize];
        nn::nifm::WirelessSettingData setting;
        bool isAutoConnect;

    public:
        SsidListEntry() {};
        ~SsidListEntry() {};

        SsidListEntry(const char *pId, const char* pSsid, const char* pKey, int auth, int enc, const char* pName, int isAuto)
        {
            Set(pId, pSsid, pKey, auth, enc, pName, isAuto);
        }

        void Set(const char *pId, const char* pSsid, const char* pKey, int auth, int enc, const char* pName, int isAuto)
        {
            nn::util::Strlcpy(encodedId, pId, sizeof(encodedId));
            nn::util::Strlcpy(encodedName, pName, sizeof(encodedName));
            nn::util::Strlcpy(encodedSsid, pSsid, sizeof(encodedSsid));
            nn::util::Strlcpy(encodedKey, pKey, sizeof(encodedKey));
            encodedAuth = auth;
            encodedEnc = enc;
            encodedIsAuto = isAuto;

            // id
            id.FromString(encodedId);

            {
                // name 複号
                size_t num;
                nn::util::Base64::Status status = nn::util::Base64::FromBase64String(&num, name, nn::nifm::NetworkProfileBasicInfo::NameSize, encodedName, nn::util::Base64::Mode_NormalNoLinefeed);
                name[num] = '\0';

                // ssid 復号
                char decodedSsidStr[nn::nifm::Ssid::HexSize + 1];
                status = nn::util::Base64::FromBase64String(&num, decodedSsidStr, nn::nifm::Ssid::HexSize + 1, encodedSsid, nn::util::Base64::Mode_NormalNoLinefeed);
                std::memcpy(setting.ssidConfig.ssid.hex, decodedSsidStr, num);
                setting.ssidConfig.ssid.length = static_cast<uint8_t>(num);

                // auth
                switch (encodedAuth)
                {
                case 0: setting.security.authEncryption.authentication = nn::nifm::Authentication_Open;
                    break;
                case 1: setting.security.authEncryption.authentication = nn::nifm::Authentication_Shared;
                    break;
                case 2: setting.security.authEncryption.authentication = nn::nifm::Authentication_WpaPsk;
                    break;
                case 3: setting.security.authEncryption.authentication = nn::nifm::Authentication_Wpa2Psk;
                    break;
                default: setting.security.authEncryption.authentication = nn::nifm::Authentication_Invalid;
                    break;
                }

                // nec
                switch (encodedEnc)
                {
                case 0: setting.security.authEncryption.encryption = nn::nifm::Encryption_None;
                    break;
                case 1: setting.security.authEncryption.encryption = nn::nifm::Encryption_Wep;
                    break;
                case 2: setting.security.authEncryption.encryption = nn::nifm::Encryption_Aes;
                    break;
                default: setting.security.authEncryption.encryption = nn::nifm::Encryption_Invalid;
                    break;
                }

                // key
                status = nn::util::Base64::FromBase64String(&num, setting.security.sharedKey.keyMaterial, nn::nifm::SharedKey::KeyMaterialSize, encodedKey, nn::util::Base64::Mode_NormalNoLinefeed);
                setting.security.sharedKey.length = static_cast<uint8_t>(num);

                // isAuto
                isAutoConnect = (encodedIsAuto != 0);
            }
        }
    };

    struct SsidListStrFlag
    {
        typedef nn::util::BitFlagSet<32, SsidListStrFlag>::Flag<0>  LfCr;             // 改行コード
        typedef nn::util::BitFlagSet<32, SsidListStrFlag>::Flag<1>  DoubleQuote;      // ダブルクオート
        typedef nn::util::BitFlagSet<32, SsidListStrFlag>::Flag<2>  Space;            // 空白
        typedef nn::util::BitFlagSet<32, SsidListStrFlag>::Flag<3>  Tab;              // タブ

    };
    typedef nn::util::BitFlagSet<32, SsidListStrFlag> SsidListStrFlagSet;


    void ToStr(char* pOutBuffer, const char* pVersionStr, const SsidListEntry* pSsidEntryList, int entrySize, SsidListStrFlagSet flag)
    {
        char LfCode[sizeof("\r\n")] = "\n";
        char DoubleQuote[sizeof("\"")] = "";
        char Space[sizeof(" ")] = "";
        char Tab[sizeof("\t")] = "";

        if(flag.Test<SsidListStrFlag::LfCr>())
        {
            nn::util::Strlcpy(LfCode, "\r\n", sizeof(LfCode));
        }

        if (flag.Test<SsidListStrFlag::DoubleQuote>())
        {
            nn::util::Strlcpy(DoubleQuote, "\"", sizeof(DoubleQuote));
        }

        if (flag.Test<SsidListStrFlag::Space>())
        {
            nn::util::Strlcpy(Space, " ", sizeof(Space));
        }

        if (flag.Test<SsidListStrFlag::Tab>())
        {
            nn::util::Strlcpy(Tab, "\t", sizeof(Tab));
        }

        int p = 0;
        p = nn::util::SNPrintf(&pOutBuffer[p], 256, "%s%s", pVersionStr, LfCode);

        for (int i = 0; i < entrySize; ++i)
        {
            p += nn::util::SNPrintf(&pOutBuffer[p], 256, "%s%s%s%s%s,%s%s%s%s%s,%s%s%s%s%s,%s%s%u%s%s,%s%s%u%s%s,%s%s%s%s%s,%s%s%u%s%s\n",
                Tab, DoubleQuote, pSsidEntryList[i].encodedId, DoubleQuote, Space,
                Tab, DoubleQuote, pSsidEntryList[i].encodedSsid, DoubleQuote, Space,
                Tab, DoubleQuote, pSsidEntryList[i].encodedKey, DoubleQuote, Space,
                Space, DoubleQuote, pSsidEntryList[i].encodedAuth, DoubleQuote, Tab,
                Space, DoubleQuote, pSsidEntryList[i].encodedEnc, DoubleQuote, Tab,
                Space, DoubleQuote, pSsidEntryList[i].encodedName, DoubleQuote, Tab,
                Space, DoubleQuote, pSsidEntryList[i].encodedIsAuto, DoubleQuote, Tab);
        }
    }

    template<int I>
    void Compare(const char* expectedVersion, const SsidListEntry* expectedSsidList, SsidListStrFlagSet flag)
    {
        nn::nifm::detail::SsidListAccessor ssidListAccessor;
        char ssidListEntryStr[256 * I];
        ToStr(ssidListEntryStr, expectedVersion, expectedSsidList, I, flag);
        NN_LOG(">> %s\n", ssidListEntryStr);
        ssidListAccessor.Load(ssidListEntryStr);

        typedef nn::nifm::detail::ArrayedFactory<nn::nifm::detail::SsidListNetworkProfile, I> SsidListProfileList;

        SsidListProfileList ssidListNetworkProfileList;

        char version[sizeof("YYMMDD")];
        NNT_ASSERT_RESULT_SUCCESS(ssidListAccessor.GetVersion(version));
        ASSERT_EQ(0, nn::util::Strncmp(expectedVersion, version, sizeof("YYMMDD")));

        while (NN_STATIC_CONDITION(true))
        {
            nn::util::Uuid uuid;
            nn::nifm::Ssid ssid;
            nn::nifm::SharedKey sharedKey;
            nn::nifm::Authentication authentication;
            nn::nifm::Encryption encryption;
            char name[nn::nifm::NetworkProfileBasicInfo::NameSize];
            bool isAutoConnect;

            nn::Result result = ssidListAccessor.GetNextEntry(&uuid, &ssid, &sharedKey, &authentication, &encryption, name, &isAutoConnect);
            if (result <= nn::nifm::ResultEndOfStream())
            {
                break;
            }

            NNT_ASSERT_RESULT_SUCCESS(result);

            auto pNp = ssidListNetworkProfileList.Create(uuid);
            ASSERT_NE(nullptr, pNp);

            auto pWs = pNp->GetWirelessSettingPointer();
            ASSERT_NE(nullptr, pWs);

            pWs->SetNonBroadcastEnabled(false);
            pWs->SetSsidHex(ssid.hex, ssid.length);
            pWs->SetAuthentication(authentication);
            pWs->SetEncryption(encryption);
            pWs->SetKeyMaterial(sharedKey.keyMaterial, sharedKey.length);

            pNp->SetName(name);
            pNp->SetAutoConnect(isAutoConnect);
        }

        ASSERT_EQ(I, ssidListNetworkProfileList.GetCount());

        int count = 0;
        for (auto&& e : ssidListNetworkProfileList)
        {
            ASSERT_EQ(expectedSsidList[count].id, e->GetId());

            nn::nifm::WirelessSettingData setting;
            e->GetWirelessSettingPointer()->Export(&setting);
            ASSERT_EQ(expectedSsidList[count].setting, setting);

            char name[nn::nifm::NetworkProfileBasicInfo::NameSize];
            e->GetName(name);
            int nameLen = nn::util::Strnlen(expectedSsidList[count].name, sizeof(name));
            ASSERT_EQ(0, nn::util::Strncmp(expectedSsidList[count].name, name, nameLen));

            ASSERT_EQ(expectedSsidList[count].isAutoConnect, e->GetAutoConnect());

            ++count;
        }

    }
}

TEST(SsidListAccessor, Basic)
{
    static const char s_VersionStr[] = "170808";
    static const SsidListEntry s_SsidListEntries[] = {
        { "1a765924-cfb0-4833-a9da-aa81d83aea4a", "bW10NDcxODJoSmo=", "VEhtZ3NTMDFhQ0k=", 3, 2, "bW10NDcxODJoSmpOYW1l", 0 },
        { "1a765924-cfb0-4833-a9da-aa81d83aea4b", "bW10NDcxODJoSmo=", "VEhtZ3NTMDFhQ0k=", 0, 1, "bW10NDcxODJoSmpOYW1l", 1 },
    };

    SsidListStrFlagSet flag;

    {
        flag.Set<SsidListStrFlag::LfCr>(false);
        flag.Set<SsidListStrFlag::DoubleQuote>(false);
        flag.Set<SsidListStrFlag::Space>(false);
        flag.Set<SsidListStrFlag::Tab>(false);

        Compare<NN_ARRAY_SIZE(s_SsidListEntries)>(&s_VersionStr[0], &s_SsidListEntries[0], flag);
    }

    {
        // 改行コード LFCR
        flag.Set<SsidListStrFlag::LfCr>(false);
        flag.Set<SsidListStrFlag::DoubleQuote>(false);
        flag.Set<SsidListStrFlag::Space>(false);
        flag.Set<SsidListStrFlag::Tab>(false);

        Compare<NN_ARRAY_SIZE(s_SsidListEntries)>(&s_VersionStr[0], &s_SsidListEntries[0], flag);
    }

    {
        // ダブルクオート
        flag.Set<SsidListStrFlag::LfCr>(false);
        flag.Set<SsidListStrFlag::DoubleQuote>(true);
        flag.Set<SsidListStrFlag::Space>(false);
        flag.Set<SsidListStrFlag::Tab>(false);

        Compare<NN_ARRAY_SIZE(s_SsidListEntries)>(&s_VersionStr[0], &s_SsidListEntries[0], flag);
    }

    {
        // Space + Tab
        flag.Set<SsidListStrFlag::LfCr>(false);
        flag.Set<SsidListStrFlag::DoubleQuote>(false);
        flag.Set<SsidListStrFlag::Space>(true);
        flag.Set<SsidListStrFlag::Tab>(true);

        Compare<NN_ARRAY_SIZE(s_SsidListEntries)>(&s_VersionStr[0], &s_SsidListEntries[0], flag);
    }

    {
        // LFCR + ダブルクオート
        flag.Set<SsidListStrFlag::LfCr>(true);
        flag.Set<SsidListStrFlag::DoubleQuote>(true);
        flag.Set<SsidListStrFlag::Space>(false);
        flag.Set<SsidListStrFlag::Tab>(false);

        Compare<NN_ARRAY_SIZE(s_SsidListEntries)>(&s_VersionStr[0], &s_SsidListEntries[0], flag);
    }

    {
        // LFCR + ダブルクオート + Space + Tab
        flag.Set<SsidListStrFlag::LfCr>(true);
        flag.Set<SsidListStrFlag::DoubleQuote>(true);
        flag.Set<SsidListStrFlag::Space>(true);
        flag.Set<SsidListStrFlag::Tab>(true);

        Compare<NN_ARRAY_SIZE(s_SsidListEntries)>(&s_VersionStr[0], &s_SsidListEntries[0], flag);
    }
}

TEST(SsidListAccessor, MaxEntries)
{
    static const char s_VersionStr[] = "170808";
    static SsidListEntry s_SsidListEntries[300];

    for (int i = 0; i < NN_ARRAY_SIZE(s_SsidListEntries); ++i)
    {
        s_SsidListEntries[i].Set("1a765924-cfb0-4833-a9da-aa81d83aea4a", "bmludGVuZG9hcHBkZWJ1ZzE=", "M0YxNUVGNTU0OTdGRkEwMDdDQzNFQkEzRTZEMDM5MkU=", 3, 2, "TmludGVuZG8gV2ktRmkgMQ==", 0);
    }

    SsidListStrFlagSet flag;

    {
        flag.Set<SsidListStrFlag::LfCr>(false);
        flag.Set<SsidListStrFlag::DoubleQuote>(false);
        flag.Set<SsidListStrFlag::Space>(false);
        flag.Set<SsidListStrFlag::Tab>(false);

        Compare<NN_ARRAY_SIZE(s_SsidListEntries)>(&s_VersionStr[0], &s_SsidListEntries[0], flag);
    }
}
