﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>

#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include <algorithm>
#include <cstring>

#include "../Common/nifm_TestUtility.h"

namespace
{
    char** g_Argv;
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    int PrepareTcpSocket(const char* hostname, uint16_t port, bool isExempted)
    {
        auto sockfd = isExempted ?
            nn::socket::SocketExempt(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp) :
            nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        EXPECT_GE(sockfd, 0);

        auto* hostent = nn::socket::GetHostEntByName(hostname);
        EXPECT_NE(nullptr, hostent);

        nn::socket::SockAddrIn serverAddr;
        memset(&serverAddr, 0, sizeof(serverAddr));
        std::memcpy(&serverAddr.sin_addr, hostent->h_addr, hostent->h_length);
        serverAddr.sin_port = nn::socket::InetHtons(port);
        serverAddr.sin_family = nn::socket::Family::Af_Inet;

        auto ret = nn::socket::Connect(sockfd, reinterpret_cast<const nn::socket::SockAddr*>(&serverAddr), sizeof(serverAddr));
        EXPECT_EQ(0, ret);

        if (ret)
        {
            NN_LOG("Errorno:%d\n", nn::socket::GetLastError());
        }

        return sockfd;
    }

    // ブロッキングしている利用要求を取り下げる
    void CleanUpThreadFunc(void* p)
    {
        nn::nifm::NetworkConnection* pNetworkConnection = reinterpret_cast<nn::nifm::NetworkConnection*>(p);
        nn::os::SystemEvent& systemEvent = pNetworkConnection->GetSystemEvent();

        while (NN_STATIC_CONDITION(true))
        {
            systemEvent.Wait();
            // systemEvent を clear してはいけない

            nn::nifm::RequestState requestState = pNetworkConnection->GetRequestState();

            switch (requestState)
            {
            case nn::nifm::RequestState::RequestState_Blocking:
                // cleanup
                NN_LOG("Cancel blocking request\n");
                pNetworkConnection->CancelRequest();
                return;

            default:
                break;
            }
        }
    }
}

class ShutdownCaseTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        nn::nifm::test::ElapsedTime(nn::os::GetSystemTick().GetInt64Value());
    }

    static void TearDownTestCase()
    {
        NN_LOG("TotalTime [%s]\n", nn::nifm::test::ElapsedTime());
    }

    virtual void TearDown()
    {
        nn::nifm::Initialize();
        NN_UTIL_SCOPE_EXIT
        {
            nn::nifm::FinalizeForTest();
        };

        // 独占解除
        nn::nifm::test::SetExclusive<nn::nifm::Initialize, nn::nifm::FinalizeForTest>(false);
    }
};

TEST_F(ShutdownCaseTest, NormalEthernet)
{
    char testName[] = "NormalEthernet";
    if (nn::util::Strncmp(testName, g_Argv[1], sizeof(testName)) != 0)
    {
        return;
    }

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910964");  // ethernet
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // シャットダウンシーケンスに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Shutdown());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnectionBG.GetResult());

        // シャットダウンシーケンス中に出した要求は保留される

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        // シャットダウンシーケンス中は状態変更不可能

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::WakeUp());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::PutToSleep());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(ShutdownCaseTest, NormalWireless)
{
    char testName[] = "NormalWireless";
    if (nn::util::Strncmp(testName, g_Argv[1], sizeof(testName)) != 0)
    {
        return;
    }

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910960");  // imatake-wpa2aes
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // シャットダウンシーケンスに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Shutdown());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnectionBG.GetResult());

        // シャットダウンシーケンス中に出した要求は保留される

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        // シャットダウンシーケンス中は状態変更不可能

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::WakeUp());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::PutToSleep());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(ShutdownCaseTest, KeptInSleepEthernet)
{
    char testName[] = "KeptInSleepEthernet";
    if (nn::util::Strncmp(testName, g_Argv[1], sizeof(testName)) != 0)
    {
        return;
    }

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910964");  // ethernet
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnection.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnectionBG.GetRequestHandle(), true));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 有線接続はスリープ中に維持する設定でも、シャットダウンシーケンスに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Shutdown());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnectionBG.GetResult());

        // シャットダウンシーケンス中に出した要求は保留される

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        // シャットダウンシーケンス中は状態変更不可能

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::WakeUp());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::PutToSleep());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(ShutdownCaseTest, KeptInSleepWithoutSocketDescriptorWireless)
{
    char testName[] = "KeptInSleepWithoutSocketDescriptorWireless";
    if (nn::util::Strncmp(testName, g_Argv[1], sizeof(testName)) != 0)
    {
        return;
    }

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910960");  // imatake-wpa2aes
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnection.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnectionBG.GetRequestHandle(), true));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // ソケットディスクリプタを登録していない場合（していた場合も同様だが）
        // 無線接続でもシャットダウンシーケンスに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Shutdown());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        EXPECT_FALSE(networkConnection.GetResult().IsSuccess());
        EXPECT_FALSE(networkConnectionBG.GetResult().IsSuccess());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnectionBG.GetResult());

        // シャットダウンシーケンス中に出した要求は保留される

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        // シャットダウンシーケンス中は状態変更不可能

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::WakeUp());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::PutToSleep());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(ShutdownCaseTest, KeptInSleepWireless)
{
    char testName[] = "KeptInSleepWireless";
    if (nn::util::Strncmp(testName, g_Argv[1], sizeof(testName)) != 0)
    {
        return;
    }

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection1;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910960");  // imatake-wpa2aes
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnection1.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG1.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG1.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnectionBG1.GetRequestHandle(), true));

        networkConnection1.SubmitRequest();
        networkConnectionBG1.SubmitRequest();

        ASSERT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection1.IsAvailable());
        EXPECT_TRUE(networkConnectionBG1.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection1.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG1.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        nn::socket::Initialize(g_SocketConfigWithMemory);

        auto exemptSocketFd = PrepareTcpSocket("ctest.cdn.nintendo.net", 80, true);
        auto socketFd = PrepareTcpSocket("ctest.cdn.nintendo.net", 80, false);

        char buffer[10];
        EXPECT_EQ(-1, nn::socket::Recv(exemptSocketFd, buffer, sizeof(buffer), nn::socket::MsgFlag::Msg_DontWait));
        EXPECT_EQ(nn::socket::Errno::EAgain, nn::socket::GetLastError());
        EXPECT_EQ(-1, nn::socket::Recv(socketFd, buffer, sizeof(buffer), nn::socket::MsgFlag::Msg_DontWait));
        EXPECT_EQ(nn::socket::Errno::EAgain, nn::socket::GetLastError());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RegisterRequestSocketDescriptor(networkConnection1.GetRequestHandle(), exemptSocketFd));

        // 無線接続の場合も、シャットダウンシーケンスに入った際に要求が却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Shutdown());

        EXPECT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnection1.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnectionBG1.GetResult());

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

        // SocketExempt で生成されたソケットも shutdown されている
        EXPECT_EQ(-1, nn::socket::Recv(exemptSocketFd, buffer, sizeof(buffer), nn::socket::MsgFlag::Msg_DontWait));
        EXPECT_EQ(nn::socket::Errno::ENetDown, nn::socket::GetLastError());

        // Socket で生成されたソケットは shutdown されている
        EXPECT_EQ(-1, nn::socket::Recv(socketFd, buffer, sizeof(buffer), nn::socket::MsgFlag::Msg_DontWait));
        EXPECT_EQ(nn::socket::Errno::ENetDown, nn::socket::GetLastError());

        // シャットダウンシーケンス中に出した要求は保留される

        nn::nifm::NetworkConnection networkConnection2;
        nn::nifm::NetworkConnection networkConnectionBG2;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));

        networkConnection2.GetSystemEvent().Clear();
        networkConnectionBG2.GetSystemEvent().Clear();
        networkConnection2.SubmitRequest();
        networkConnectionBG2.SubmitRequest();

        EXPECT_FALSE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection2.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG2.IsRequestOnHold());
        EXPECT_FALSE(networkConnection2.IsAvailable());
        EXPECT_FALSE(networkConnectionBG2.IsAvailable());

        // シャットダウンシーケンス中は状態変更不可能

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::WakeUp());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::PutToSleep());

        EXPECT_EQ(0, nn::socket::Close(exemptSocketFd));
        EXPECT_EQ(0, nn::socket::Close(socketFd));

        NNT_EXPECT_RESULT_SUCCESS(nn::socket::Finalize());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(ShutdownCaseTest, KeptInSleepWirelessSsidList)
{
    char testName[] = "KeptInSleepWirelessSsidList";
    if (nn::util::Strncmp(testName, g_Argv[1], sizeof(testName)) != 0)
    {
        return;
    }

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("1a765924-cfb0-4833-a9da-aa81d83aea4a");  // mmt
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnection.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnectionBG.GetRequestHandle(), true));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // SSID リストの設定利用中でも、シャットダウンシーケンスに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Shutdown());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnectionBG.GetResult());

        // シャットダウンシーケンス中に出した要求は保留される

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        // シャットダウンシーケンス中は状態変更不可能

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::WakeUp());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::PutToSleep());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(ShutdownCaseTest, PersistentEthernet)
{
    char testName[] = "PersistentEthernet";
    if (nn::util::Strncmp(testName, g_Argv[1], sizeof(testName)) != 0)
    {
        return;
    }

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910964");  // ethernet
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(networkConnection.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // シャットダウンシーケンスに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Shutdown());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDeviceShutdown, networkConnectionBG.GetResult());

        // シャットダウンシーケンス中に出した要求は保留される

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        // シャットダウンシーケンス中は状態変更不可能

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::WakeUp());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::PutToSleep());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(ShutdownCaseTest, LocalConnection)
{
    char testName[] = "LocalConnection";
    if (nn::util::Strncmp(testName, g_Argv[1], sizeof(testName)) != 0)
    {
        return;
    }

    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalForApplet));

        networkConnection.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // ローカル通信を終了するためだけのスレッド
        NN_ALIGNAS(nn::os::ThreadStackAlignment) static char stackOfThread[4 * 1024];
        ASSERT_EQ(0, reinterpret_cast<uintptr_t>(&stackOfThread) % nn::os::ThreadStackAlignment);

        nn::os::ThreadType CleanUpThread;
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpThread, &CleanUpThreadFunc, &networkConnection, &stackOfThread, 4 * 1024, 24));
        nn::os::StartThread(&CleanUpThread);

        // シャットダウンシーケンスに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Shutdown());

        // ローカル通信終了
        nn::os::WaitThread(&CleanUpThread);
        nn::os::DestroyThread(&CleanUpThread);

        EXPECT_FALSE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());

        // シャットダウンシーケンス中に出した要求は保留される

        networkConnection.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());

        // シャットダウンシーケンス中は状態変更不可能

        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::WakeUp());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultInternalError, nn::nifm::PutToSleep());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    g_Argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, g_Argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
