﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_Request.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_Result.h>
#include <nn/nifm/nifm_ResultPrivate.h>
#include <nn/nifm/nifm_TypesRequest.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>
#include <nn/nifm/nifm_TypesScan.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_FormatString.h>

#include "../Common/nifm_TestUtility.h"

namespace
{
    // 利用要求の即座の受理または却下を期待した待ち時間
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;

    NN_ALIGNAS(nn::os::ThreadStackAlignment) static char stackOfThread[4 * 1024];

    struct ExpectedAccessPoint
    {
        nn::nifm::AccessPointData accessPointData;
        bool isFound;
    };

    static ExpectedAccessPoint s_ExpectedAccessPoints[] =
    {
        {
            { // imatake-wpa2aes
                { 15,{ 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 } },  // ssid
                { {} },                                                                                 // bssid（比較には未使用）
                -1,                                                                                     // rssi（比較には未使用）
                nn::nifm::LinkLevel_0,                                                                  // linkLevel（比較には未使用）
                0,                                                                                      // channel（比較には未使用）
                nn::nifm::Authentication_Wpa2Psk,                                                       // authentication
                nn::nifm::Encryption_Aes,                                                               // encryption
                nn::nifm::Encryption_Aes,                                                               // groupEncryption
                true,                                                                                   // isSupported
            }, false,
        },
        {
            { // imatake-mixed
                { 13,{ 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64 } },            // ssid
                { {} },                                                                                 // bssid（比較には未使用）
                -1,                                                                                     // rssi（比較には未使用）
                nn::nifm::LinkLevel_0,                                                                  // linkLevel（比較には未使用）
                0,                                                                                      // channel（比較には未使用）
                nn::nifm::Authentication_Wpa2Psk,                                                       // authentication
                nn::nifm::Encryption_Aes,                                                               // encryption
                nn::nifm::Encryption_Tkip,                                                              // groupEncryption
                true,                                                                                   // isSupported
            }, false,
        },
        {
            { // imatake-wpaaes
                { 14,{ 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x61,0x65,0x73 } },       // ssid
                { {} },                                                                                 // bssid（比較には未使用）
                -1,                                                                                     // rssi（比較には未使用）
                nn::nifm::LinkLevel_0,                                                                  // linkLevel（比較には未使用）
                0,                                                                                      // channel（比較には未使用）
                nn::nifm::Authentication_WpaPsk,                                                        // authentication
                nn::nifm::Encryption_Aes,                                                               // encryption
                nn::nifm::Encryption_Aes,                                                               // groupEncryption
                true,                                                                                   // isSupported
            }, false,
        },
        {
            { // imatake-openwep104
                { 18,{ 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,
                0x31,0x30,0x34 } },                                                              // ssid
                { {} },                                                                                 // bssid（比較には未使用）
                -1,                                                                                     // rssi（比較には未使用）
                nn::nifm::LinkLevel_0,                                                                  // linkLevel（比較には未使用）
                0,                                                                                      // channel（比較には未使用）
                nn::nifm::Authentication_Unknown,                                                       // authentication
                nn::nifm::Encryption_Wep,                                                               // encryption
                nn::nifm::Encryption_Wep,                                                               // groupEncryption
                true,                                                                                   // isSupported
            }, false,
        },
        {
            { // imatake-openwep40
                { 17,{ 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,
                0x34,0x30 } },                                                                   // ssid
                { {} },                                                                                 // bssid（比較には未使用）
                -1,                                                                                     // rssi（比較には未使用）
                nn::nifm::LinkLevel_0,                                                                  // linkLevel（比較には未使用）
                0,                                                                                      // channel（比較には未使用）
                nn::nifm::Authentication_Unknown,                                                       // authentication
                nn::nifm::Encryption_Wep,                                                               // encryption
                nn::nifm::Encryption_Wep,                                                               // groupEncryption
                true,                                                                                   // isSupported
            }, false,
        },
        {
            { // imatake-wpatkip
                { 15,{ 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x74,0x6b,0x69,0x70 } },  // ssid
                { {} },                                                                                 // bssid（比較には未使用）
                -1,                                                                                     // rssi（比較には未使用）
                nn::nifm::LinkLevel_0,                                                                  // linkLevel（比較には未使用）
                0,                                                                                      // channel（比較には未使用）
                nn::nifm::Authentication_WpaPsk,                                                        // authentication
                nn::nifm::Encryption_Tkip,                                                              // encryption
                nn::nifm::Encryption_Tkip,                                                              // groupEncryption
                false,                                                                                  // isSupported
            }, false,
        },
    };

    bool Compare(nn::nifm::AccessPointData ap1, nn::nifm::AccessPointData ap2)
    {
        if (ap1.ssid.length == ap2.ssid.length &&
            memcmp(ap1.ssid.hex, ap2.ssid.hex, ap1.ssid.length) == 0 &&
            ap1.authentication == ap2.authentication &&
            ap1.encryption == ap2.encryption &&
            ap1.groupEncryption == ap2.groupEncryption &&
            ap1.isSupported == ap2.isSupported)
        {
            return true;
        }

        if (ap1.ssid.length == ap2.ssid.length &&
            memcmp(ap1.ssid.hex, ap2.ssid.hex, ap1.ssid.length) == 0 &&
            memcmp(ap1.ssid.hex, "imatake-mixed", ap1.ssid.length) == 0 &&
            ap1.authentication == ap2.authentication &&
            ap1.encryption == ap2.encryption &&
            //ap1.groupEncryption == ap2.groupEncryption &&
            ap1.isSupported == ap2.isSupported)
        {
            // MIX モードの groupEncryption は AP 依存
            // WPA/WPA2-PSK(TKIP/AES), WPA/WPA2-PSK(AES)
            return true;
        }

        return false;
    }

    bool IsAllFound(const ExpectedAccessPoint* pExpectedAccessPoint, int size)
    {
        for (int i = 0; i < size; ++i)
        {
            if (!pExpectedAccessPoint[i].isFound)
            {
                return false;
            }
        }

        return true;
    }
}

class ScanTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        // 本テストプロセスで独占
        nn::nifm::test::SetExclusive<nn::nifm::InitializeSystem, nn::nifm::FinalizeSystemForTest>(true);
    }

    static void TearDownTestCase()
    {
        // 本テストプロセスで独占解除
        nn::nifm::test::SetExclusive<nn::nifm::InitializeSystem, nn::nifm::FinalizeSystemForTest>(false);
    }
};

TEST_F(ScanTest, ScanInNeighborDetection)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeSystem, nn::nifm::FinalizeSystemForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::os::ThreadType CleanUpThread;

    nn::nifm::NetworkConnection networkConnectionNeighbor;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionNeighbor.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetection));

    networkConnectionNeighbor.SubmitRequestAndWait();
    NNT_ASSERT_RESULT_SUCCESS(networkConnectionNeighbor.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpThread,
        [](void* p) {
        auto pNetworkConnection = reinterpret_cast<nn::nifm::NetworkConnection*>(p);
        nn::os::SystemEvent& systemEvent = pNetworkConnection->GetSystemEvent();
        while (NN_STATIC_CONDITION(true))
        {
            systemEvent.Wait();
            nn::nifm::RequestState requestState = pNetworkConnection->GetRequestState();
            switch (requestState)
            {
            case nn::nifm::RequestState::RequestState_Blocking:
                pNetworkConnection->CancelRequest();
                return;
            default:
                break;
            }
        }
    },
        &networkConnectionNeighbor, &stackOfThread, sizeof(stackOfThread), 24));
    nn::os::StartThread(&CleanUpThread);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(700));

    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::Scan());

    nn::os::WaitThread(&CleanUpThread);
    nn::os::DestroyThread(&CleanUpThread);
}

TEST_F(ScanTest, ScanPriority)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeSystem, nn::nifm::FinalizeSystemForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::NetworkConnection networkConnection;
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    nn::os::ThreadType CleanUpThread;
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpThread,
        [](void* p) {
        auto pNetworkConnection = reinterpret_cast<nn::nifm::NetworkConnection*>(p);
        nn::os::SystemEvent& systemEvent = pNetworkConnection->GetSystemEvent();
        while (NN_STATIC_CONDITION(true))
        {
            systemEvent.Wait();
            nn::nifm::RequestState requestState = pNetworkConnection->GetRequestState();
            switch (requestState)
            {
            case nn::nifm::RequestState::RequestState_Blocking:
                pNetworkConnection->CancelRequest();
                return;
            default:
                break;
            }
        }
    },
        &networkConnection, &stackOfThread, sizeof(stackOfThread), 24));
    nn::os::StartThread(&CleanUpThread);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(700));

    // 弱いローカル通信利用要求受理中はアプリはスキャン可能
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::Scan());
    // その際，弱いローカル通信は取り下げられる
    NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());

    nn::os::WaitThread(&CleanUpThread);
    nn::os::DestroyThread(&CleanUpThread);

    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForApplet));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    // 強いインターネット通信利用要求受理中でもアプリはスキャン可能
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::Scan());

    networkConnection.CancelRequest();
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_LocalForApplet));
    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpThread,
        [](void* p) {
        auto pNetworkConnection = reinterpret_cast<nn::nifm::NetworkConnection*>(p);
        nn::os::SystemEvent& systemEvent = pNetworkConnection->GetSystemEvent();
        while (NN_STATIC_CONDITION(true))
        {
            systemEvent.Wait();
            nn::nifm::RequestState requestState = pNetworkConnection->GetRequestState();
            switch (requestState)
            {
            case nn::nifm::RequestState::RequestState_Blocking:
                pNetworkConnection->CancelRequest();
                return;
            default:
                break;
            }
        }
    },
        &networkConnection, &stackOfThread, sizeof(stackOfThread), 24));
    nn::os::StartThread(&CleanUpThread);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(700));

    // 強いローカル通信利用要求受理中はアプリはスキャン不可能
    NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, nn::nifm::Scan());
    // アプレットはスキャン可能
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::Scan(nn::nifm::RequirementPreset_ScanForApplet));
    // その際，強いローカル通信は取り下げられる
    NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());

    nn::os::WaitThread(&CleanUpThread);
    nn::os::DestroyThread(&CleanUpThread);
}

TEST_F(ScanTest, ScanInvalidChannel)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::Initialize, nn::nifm::FinalizeForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // 自律スキャンに阻害されないように，一度スキャンを完了させる
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    const int16_t includeZero[] = { 0, 1, 2 };
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultInvalidArgument, nn::nifm::Scan(includeZero, NN_ARRAY_SIZE(includeZero)));

    const int16_t includeNegative[] = { -1, 10, 20 };
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultInvalidArgument, nn::nifm::Scan(includeNegative, NN_ARRAY_SIZE(includeNegative)));
}

TEST_F(ScanTest, GetAllowedChannels)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::Initialize, nn::nifm::FinalizeForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    int16_t channels1[1];
    int16_t channels2[nn::nifm::WirelessChannelsCountMax];
    int count1;
    int count2;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetAllowedChannels(channels1, &count1, NN_ARRAY_SIZE(channels1)));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetAllowedChannels(channels2, &count2, NN_ARRAY_SIZE(channels2)));
    ASSERT_EQ(count1, count2);
    ASSERT_EQ(channels1[0], channels2[0]);
}

TEST_F(ScanTest, Scan1Channel)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::Initialize, nn::nifm::FinalizeForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // 自律スキャンに阻害されないように，一度スキャンを完了させる
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    const int ScanApCount = nn::nifm::AccessPointCountMax;
    nn::nifm::AccessPointData accessPointList[ScanApCount] = {};
    int apCount = 0;
    int apCountPre = 0;

    int16_t channels[nn::nifm::WirelessChannelsCountMax];
    int channelCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetAllowedChannels(channels, &channelCount, NN_ARRAY_SIZE(channels)));

    // 1チャンネルスキャン
    for(int i = 0; i < channelCount; ++i)
    {
        NN_LOG("Scan Start: %d ch\n", channels[i]);
        auto t1 = nn::os::GetSystemTick();
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan(&channels[i], 1));
        auto scanTimeInMs = (nn::os::GetSystemTick() - t1).ToTimeSpan().GetMilliSeconds();
        EXPECT_LE(scanTimeInMs, 500);
        NN_LOG("Scan Finish: %d [ms]\n", scanTimeInMs);

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetScanData(accessPointList, &apCount, ScanApCount));
        EXPECT_GE(apCount, apCountPre);
        apCountPre = apCount;
        int count = std::min(apCount, ScanApCount);

        for (int j = 0; j < count; ++j)
        {
            nn::nifm::test::PrintAccessPoint(accessPointList[j]);
        }
    }
}

TEST_F(ScanTest, Scan2Channels)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::Initialize, nn::nifm::FinalizeForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // 自律スキャンに阻害されないように，一度スキャンを完了させる
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    const int ScanApCount = nn::nifm::AccessPointCountMax;
    nn::nifm::AccessPointData accessPointList[ScanApCount] = {};
    int apCount = 0;
    int apCountPre = 0;

    int16_t channels[nn::nifm::WirelessChannelsCountMax];
    int channelCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetAllowedChannels(channels, &channelCount, NN_ARRAY_SIZE(channels)));

    // 2チャンネルスキャン
    for (int i = 0; i < channelCount; i += 2)
    {
        NN_LOG("Scan Start: %d, %d ch\n", channels[i], channels[i + 1]);
        auto t1 = nn::os::GetSystemTick();
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan(&channels[i], channelCount - i >= 2 ? 2 : 1));
        auto scanTimeInMs = (nn::os::GetSystemTick() - t1).ToTimeSpan().GetMilliSeconds();
        EXPECT_LE(scanTimeInMs, 600);
        NN_LOG("Scan Finish: %d [ms]\n", scanTimeInMs);

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetScanData(accessPointList, &apCount, ScanApCount));
        EXPECT_GE(apCount, apCountPre);
        apCountPre = apCount;
        int count = std::min(apCount, ScanApCount);

        for (int j = 0; j < count; ++j)
        {
            nn::nifm::test::PrintAccessPoint(accessPointList[j]);
        }
    }
}

TEST_F(ScanTest, ScanWhileScanning)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::Initialize, nn::nifm::FinalizeForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());
    // 自律スキャンに阻害されないように，一度スキャンを完了させる
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    nn::os::ThreadType ScanThread;
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&ScanThread,
        [](void* p)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());
    },
        nullptr, &stackOfThread, 4 * 1024, 24));

    nn::os::StartThread(&ScanThread);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(800));

    // 同じチャンネル（ここでは全チャンネル）に対するスキャンは成功．実際は，完了待ちをするのみ
    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::Scan());

    nn::os::WaitThread(&ScanThread);
    nn::os::DestroyThread(&ScanThread);
}

TEST_F(ScanTest, BasicScan)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::Initialize, nn::nifm::FinalizeForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    const int ScanApCount = nn::nifm::AccessPointCountMax;
    nn::nifm::AccessPointData accessPointList[ScanApCount] = {};
    int apCount;

    // AP ビーコン取りこぼし対策で一定回数リトライ許可
    const int RetryMax = 3;
    for (int loop = 0; loop < RetryMax; ++loop)
    {
        NN_LOG("loop = %d\n", loop);
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetScanData(accessPointList, &apCount, ScanApCount));
        int count = std::min(apCount, ScanApCount);
        NN_LOG("count = %d (%d,%d)\n", count, apCount, ScanApCount);

        // ソートの確認
        nn::nifm::AccessPointData preAp;
        preAp.linkLevel = nn::nifm::LinkLevel_Count;
        for (int i = 0; i < count; ++i)
        {
            auto&& accessPoint = accessPointList[i];

            if (preAp.linkLevel != accessPoint.linkLevel)
            {
                ASSERT_GT(preAp.linkLevel, accessPoint.linkLevel);
            }
            else
            {
                ASSERT_GE(preAp.rssi, accessPoint.rssi);
            }

            preAp = accessPoint;
        }

        // テスト環境において必ず得られるべき内容の確認
        for (auto&& expAp : s_ExpectedAccessPoints)
        {
            for (int i = 0; i < count; ++i)
            {
                auto&& accessPoint = accessPointList[i];

                if (Compare(expAp.accessPointData, accessPoint))
                {
                    expAp.isFound = true;
                    break;
                }
            }

            if (!expAp.isFound)
            {
                NN_LOG("Not Found\n");
                nn::nifm::test::PrintAccessPoint(expAp.accessPointData);
            }
        }

        if (!IsAllFound(s_ExpectedAccessPoints, NN_ARRAY_SIZE(s_ExpectedAccessPoints)))
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            continue;
        }

        // もし EAP が存在した場合は isSupported=false を確認
        for (int i = 0; i < count; ++i)
        {
            auto&& accessPoint = accessPointList[i];

            if (accessPoint.authentication == nn::nifm::Authentication_Wpa2 ||
                accessPoint.authentication == nn::nifm::Authentication_Wpa)
            {
                ASSERT_EQ(false, accessPoint.isSupported);
                nn::nifm::test::PrintAccessPoint(accessPoint);
            }
        }

        break;
    }

    NN_ASSERT(IsAllFound(s_ExpectedAccessPoints, NN_ARRAY_SIZE(s_ExpectedAccessPoints)));
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
