﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Log.h>

#include <nn/nifm/detail/core/profile/nifm_NetworkProfileManager.h>

#include "../Common/nifm_TestUtility.h"


TEST(NetworkProfileManagerCase, Get)
{
    nn::nifm::detail::NetworkProfileManager networkProfileManager;

    nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo;
    nn::nifm::NetworkProfileData networkProfileData;

    auto profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::UserProfile);
    NN_LOG("UserProfile:%d\n", profileCount);
    for (int i = 0; i < profileCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::UserProfile, i));
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileData(&networkProfileData, networkProfileBasicInfo.id));
    }

    profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::SsidListProfile);
    NN_LOG("SsidListProfile:%d\n", profileCount);
    for (int i = 0; i < profileCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::SsidListProfile, i));
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileData(&networkProfileData, networkProfileBasicInfo.id));
    }

    profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::TemporaryProfile);
    NN_LOG("TemporaryProfile:%d\n", profileCount);
    for (int i = 0; i < profileCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::TemporaryProfile, i));
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileData(&networkProfileData, networkProfileBasicInfo.id));
    }

    profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::Any);
    NN_LOG("Any:%d\n", profileCount);
    for (int i = 0; i < profileCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::Any, i));
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileData(&networkProfileData, networkProfileBasicInfo.id));
    }

    profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::AnyIncludingDummy);
    NN_LOG("AnyIncludingDummy:%d\n", profileCount);
    for (int i = 0; i < profileCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::AnyIncludingDummy, i));
        if (networkProfileBasicInfo.id != nn::util::InvalidUuid)
        {
            NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileData(&networkProfileData, networkProfileBasicInfo.id));
        }
        else
        {
            // DummyProfile の ID は無効なので取得できない
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkProfileNotFound, networkProfileManager.GetNetworkProfileData(&networkProfileData, networkProfileBasicInfo.id));
        }
    }

    nn::nifm::SsidListVersion ssidListVersion;
    networkProfileManager.GetSsidListVersion(&ssidListVersion);
    NN_LOG("ssidListVersion:%s\n", ssidListVersion.data);

    nn::nifm::IpSettingData defaultIpSettingData;
    networkProfileManager.GetDefaultIpSetting(&defaultIpSettingData);

    nn::nifm::detail::IpSetting ipSetting;
    nn::nifm::IpSettingData ipSettingData;
    ipSetting.Export(&ipSettingData);

    EXPECT_EQ(ipSettingData.ip.isAuto, defaultIpSettingData.ip.isAuto);
    EXPECT_EQ(0, memcmp(ipSettingData.ip.ipAddress.data, defaultIpSettingData.ip.ipAddress.data, nn::nifm::IpV4Address::Size));
    EXPECT_EQ(0, memcmp(ipSettingData.ip.subnetMask.data, defaultIpSettingData.ip.subnetMask.data, nn::nifm::IpV4Address::Size));
    EXPECT_EQ(0, memcmp(ipSettingData.ip.defaultGateway.data, defaultIpSettingData.ip.defaultGateway.data, nn::nifm::IpV4Address::Size));
    EXPECT_EQ(ipSettingData.dns.isAuto, defaultIpSettingData.dns.isAuto);
    EXPECT_EQ(0, memcmp(ipSettingData.dns.preferredDns.data, defaultIpSettingData.dns.preferredDns.data, nn::nifm::IpV4Address::Size));
    EXPECT_EQ(0, memcmp(ipSettingData.dns.alternateDns.data, defaultIpSettingData.dns.alternateDns.data, nn::nifm::IpV4Address::Size));
    EXPECT_EQ(ipSettingData.proxy.isEnabled, defaultIpSettingData.proxy.isEnabled);
    EXPECT_EQ(ipSettingData.proxy.port, defaultIpSettingData.proxy.port);
    EXPECT_EQ(0, strncmp(ipSettingData.proxy.proxy, defaultIpSettingData.proxy.proxy, nn::nifm::ProxySetting::ProxyNameSize));
    EXPECT_EQ(ipSettingData.proxy.authentication.isEnabled, defaultIpSettingData.proxy.authentication.isEnabled);
    EXPECT_EQ(0, strncmp(ipSettingData.proxy.authentication.username, defaultIpSettingData.proxy.authentication.username, nn::nifm::AuthenticationSetting::UsernameSize));
    EXPECT_EQ(0, strncmp(ipSettingData.proxy.authentication.password, defaultIpSettingData.proxy.authentication.password, nn::nifm::AuthenticationSetting::PasswordSize));
    EXPECT_EQ(ipSettingData.mtu, defaultIpSettingData.mtu);

    // TODO
    //networkProfileManager.GetCopiedNetworkProfileCount();
    //networkProfileManager.GetCopiedNetworkProfile();
}

TEST(NetworkProfileManagerCase, Remove)
{
    nn::nifm::detail::NetworkProfileManager networkProfileManager;

    nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo;

    auto profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::UserProfile);
    for (int i = profileCount - 1; i >= 0; --i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::UserProfile, i));
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.RemoveNetworkProfile(nn::nifm::detail::NetworkProfileBase::ProfileType::UserProfile, networkProfileBasicInfo.id));
    }

    profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::SsidListProfile);
    for (int i = profileCount - 1; i >= 0; --i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::SsidListProfile, i));
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.RemoveNetworkProfile(nn::nifm::detail::NetworkProfileBase::ProfileType::SsidListProfile, networkProfileBasicInfo.id));
    }

    profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::TemporaryProfile);
    for (int i = profileCount - 1; i >= 0; --i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::TemporaryProfile, i));
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.RemoveNetworkProfile(nn::nifm::detail::NetworkProfileBase::ProfileType::TemporaryProfile, networkProfileBasicInfo.id));
    }

    profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::Any);
    for (int i = profileCount - 1; i >= 0; --i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::Any, i));
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.RemoveNetworkProfile(nn::nifm::detail::NetworkProfileBase::ProfileType::Any, networkProfileBasicInfo.id));
    }

    profileCount = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::AnyIncludingDummy);
    for (int i = profileCount - 1; i >= 0; --i)
    {
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, nn::nifm::detail::NetworkProfileBase::ProfileType::AnyIncludingDummy, i));
        if (networkProfileBasicInfo.id != nn::util::InvalidUuid)
        {
            NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.RemoveNetworkProfile(nn::nifm::detail::NetworkProfileBase::ProfileType::AnyIncludingDummy, networkProfileBasicInfo.id));
        }
        else
        {
            // DummyProfile の ID は無効なので削除できない
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkProfileNotFound, networkProfileManager.RemoveNetworkProfile(nn::nifm::detail::NetworkProfileBase::ProfileType::AnyIncludingDummy, networkProfileBasicInfo.id));
        }
    }
}


TEST(NetworkProfileManagerCase, Add)
{
    nn::nifm::detail::NetworkProfileManager networkProfileManager;

    {
        int profileCount1 = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::UserProfile);
        nn::nifm::NetworkProfileData networkProfileData = {};
        nn::util::Uuid id;
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.AddUserNetworkProfile(&id, networkProfileData));
        int profileCount2 = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::UserProfile);

        EXPECT_EQ(profileCount1 + 1, profileCount2);
        ASSERT_NE(nn::util::InvalidUuid, id);
        networkProfileManager.GetNetworkProfileData(&networkProfileData, id);
        EXPECT_EQ(nn::nifm::NetworkProfileType_User, networkProfileData.networkProfileType);    }

    {
        int profileCount1 = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::SsidListProfile);
        nn::nifm::NetworkProfileData networkProfileData = {};
        nn::util::Uuid id;
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.AddSsidListNetworkProfile(&id, networkProfileData));
        int profileCount2 = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::SsidListProfile);

        EXPECT_EQ(profileCount1 + 1, profileCount2);
        ASSERT_NE(nn::util::InvalidUuid, id);
        networkProfileManager.GetNetworkProfileData(&networkProfileData, id);
        EXPECT_EQ(nn::nifm::NetworkProfileType_SsidList, networkProfileData.networkProfileType);
    }

    {
        int profileCount1 = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::TemporaryProfile);
        nn::nifm::NetworkProfileData networkProfileData = {};
        nn::util::Uuid id;
        NNT_ASSERT_RESULT_SUCCESS(networkProfileManager.AddTemporaryNetworkProfile(&id, networkProfileData));
        int profileCount2 = networkProfileManager.GetNetworkProfileCount(nn::nifm::detail::NetworkProfileBase::ProfileType::TemporaryProfile);

        EXPECT_EQ(profileCount1 + 1, profileCount2);
        ASSERT_NE(nn::util::InvalidUuid, id);
        networkProfileManager.GetNetworkProfileData(&networkProfileData, id);
        EXPECT_EQ(nn::nifm::NetworkProfileType_Temporary, networkProfileData.networkProfileType);
    }

}
