﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>
#include <nn/nifm/detail/nifm_MacrosDetail.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/util/util_StringUtil.h>
#include <nn/nn_BitTypes.h>

#include <nn/erpt/erpt_TypesPrivate.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/erpt/erpt_Report.h>

#include <nn/msgpack.h>

#include <regex>

#include "../Common/nifm_TestUtility.h"

namespace
{
    static const int MacAddressStringSize = sizeof("AA-BB-CC-DD-EE-FF");
    static const int SsidListVersionSize = sizeof("YYMMDD");

    static char g_EthernetMacAddressString[MacAddressStringSize];
    static char g_WirelessMacAddressString[MacAddressStringSize];
    static char g_SsidListVersion[SsidListVersionSize];

    struct ExpectedReportItem
    {
        char key[32];
        char value[256];
    };

    /*******************************************/
    /* エラーレポートの期待値                  */
    /*******************************************/

    // TODO: IP, NetMask, Gateway，APMac アドレス等が存在することの確認

    static ExpectedReportItem g_Report14[] =
    {
        { "AccessPointSSID", "imatake-wpa2aes" },
        { "AccessPointSecurityType", "Wpa2-Psk(Aes)" },
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "UseStealthNetworkFlag", "False" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "0" },
        { "UseProxyFlag", "False" },
        { "MTU", "1001" },
        { "ConnectAutomaticallyFlag", "True" },
        { "PriorityDNSIPAddress", "1.2.3.4" },
        { "AlternateDNSIPAddress", "5.6.7.8" },
        { "ConnectionStatus", "Not Connected" },
        { "NifmErrorCode", "2110-3127" },
        { "ErrorCode", "2110-3127" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static ExpectedReportItem g_Report13[] =
    {
        { "AccessPointSSID", "imatake-openwep40" },
        { "AccessPointSecurityType", "Unknown(Wep)" },
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "UseStealthNetworkFlag", "False" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "1" },
        { "UseProxyFlag", "False" },
        { "MTU", "1410" },
        { "ConnectAutomaticallyFlag", "True" },
        { "ConnectionStatus", "Not Connected" },
        { "NifmErrorCode", "2110-2964" },
        { "ErrorCode", "2110-2964" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static ExpectedReportItem g_Report12[] =
    {
        { "AccessPointSSID", "imatake-wpa2aes" },
        { "AccessPointSecurityType", "Wpa2-Psk(Aes)" },
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "UseStealthNetworkFlag", "False" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "1" },
        { "UseProxyFlag", "True" },
        { "ProxyIPAddress", "proxy55.nintendowifi.net" },
        { "ProxyPort", "10001" },
        { "ProxyAutoAuthenticateFlag", "False" },
        { "MTU", "1220" },
        { "ConnectAutomaticallyFlag", "True" },
        { "ConnectionStatus", "Not Connected" },
        { "NifmErrorCode", "2110-3126" },
        { "ErrorCode", "2110-3126" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static ExpectedReportItem g_Report11[] =
    {
        { "AccessPointSSID", "imatake-wpa2aes" },
        { "AccessPointSecurityType", "Wpa2-Psk(Aes)" },
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "UseStealthNetworkFlag", "False" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "0" },
        { "UseProxyFlag", "False" },
        { "MTU", "1210" },
        { "ConnectAutomaticallyFlag", "True" },
        { "ConnectionStatus", "Not Connected" },
        { "NifmErrorCode", "2110-3127" },
        { "ErrorCode", "2110-3127" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static ExpectedReportItem g_Report10[] =
    {
        { "NintendoZoneSSIDListVersion", "180111" },
        { "NXMacAddress", "Replaced with actual address" },
        { "ConnectionStatus", "Not Connected" },
        { "NifmErrorCode", "2110-1100" },
        { "ErrorCode", "2110-1100" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static ExpectedReportItem g_Report9[] =
    {
        { "AccessPointSSID", "imatake-wpa2aes" },
        { "AccessPointSecurityType", "Wpa2-Psk(Aes)" },
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "UseStealthNetworkFlag", "False" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "1" },
        { "UseProxyFlag", "False" },
        { "MTU", "1320" },
        { "ConnectAutomaticallyFlag", "True" },
        { "ConnectionStatus", "Not Connected" },
        { "NifmErrorCode", "2110-22\\d{2}" },
        { "ErrorCode", "2110-2202" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static ExpectedReportItem g_Report8[] =
    {
        { "NintendoZoneSSIDListVersion", "180111" },
        { "NXMacAddress", "Replaced with actual address" },
        { "ConnectionStatus", "Not Connected" },
        { "NifmErrorCode", "2110-1100" },
        { "ErrorCode", "2110-1100" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static ExpectedReportItem g_Report7[] =
    {
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "1" },
        { "UseProxyFlag", "False" },
        { "MTU", "1500" },
        { "ConnectAutomaticallyFlag", "True" },
        { "ConnectionStatus", "Ethernet" },
        //{ "NifmErrorCode", "2000-0000" }, // 有線接続成功
        { "ErrorCode", "2110-0300" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static ExpectedReportItem g_Report6[] =
    {
        { "AccessPointSSID", "imatake-wpa2aes" },
        { "AccessPointSecurityType", "Wpa2-Psk(Aes)" },
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "UseStealthNetworkFlag", "False" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "0" },
        { "UseProxyFlag", "False" },
        { "MTU", "1400" },
        { "ConnectAutomaticallyFlag", "True" },
        { "ConnectionStatus", "Not Connected" },
        { "NifmErrorCode", "2110-3400" },
        { "ErrorCode", "2110-0300" },
        { "NifmConnectionTestRedirectUrl", "https://ssltest3.app.nintendowifi.net/login" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static ExpectedReportItem g_Report5[] =
    {
        { "AccessPointSSID", "imatake-wpa2aes" },
        { "AccessPointSecurityType", "Wpa2-Psk(Aes)" },
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "UseStealthNetworkFlag", "False" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "0" },
        { "UseProxyFlag", "False" },
        { "MTU", "1400" },
        { "ConnectAutomaticallyFlag", "True" },
        { "ConnectionStatus", "Not Connected" },
        { "NifmErrorCode", "2110-3400" },
        { "ErrorCode", "2110-0300" },
        { "NifmConnectionTestRedirectUrl", "https://ssltest3.app.nintendowifi.net/login" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static  ExpectedReportItem g_Report4[] =
    {
        { "AccessPointSSID", "imatake-wpa2aes" },
        { "AccessPointSecurityType", "Wpa2-Psk(Aes)" },
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "UseStealthNetworkFlag", "True" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "1" },
        { "UseProxyFlag", "False" },
        { "MTU", "1300" },
        { "ConnectAutomaticallyFlag", "True" },
        { "ConnectionStatus", "Wireless" },
        //{ "NifmErrorCode", "2000-0000" }, // FGによる接続成功更新
        { "ErrorCode", "2110-0300" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static  ExpectedReportItem g_Report3[] =
    {
        { "AccessPointSSID", "imatake-wpa2aes" },
        { "AccessPointSecurityType", "Wpa2-Psk(Aes)" },
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False" },
        { "NintendoZoneSSIDListVersion", "180111" },
        { "LimitHighCapacityFlag", "False" },
        { "UseStealthNetworkFlag", "True" },
        { "NXMacAddress", "Replaced with actual address" },
        { "IPAddressAcquisitionMethod", "1" },
        { "DNSType", "1" },
        { "UseProxyFlag", "False" },
        { "MTU", "1300" },
        { "ConnectAutomaticallyFlag", "True" },
        { "ConnectionStatus", "Wireless" },
        //{ "NifmErrorCode", "2000-0000" }, // ローカル通信による切断，更新されず
        { "ErrorCode", "2110-4001" },
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static  ExpectedReportItem g_Report2[] =
    {
        { "AccessPointSSID", "imatake-wpa2aes"},
        { "AccessPointSecurityType", "Wpa2-Psk(Aes)"},
        { "AccessPointChannel", "Range check" },
        { "RadioStrength", "Range check" },
        { "AccessPointRssi", "Range check" },
        { "NintendoZoneConnectedFlag", "False"},
        { "NintendoZoneSSIDListVersion", "180111"},
        { "LimitHighCapacityFlag", "False"},
        { "UseStealthNetworkFlag", "True"},
        { "NXMacAddress", "Replaced with actual address" },
        { "DNSType", "1"},
        { "UseProxyFlag", "False"},
        { "MTU", "1300"},
        { "ConnectAutomaticallyFlag", "True"},
        { "ConnectionStatus", "Wireless"},
        //{ "NifmErrorCode", "2000-0000" }, // 無線成功
        { "ErrorCode", "2110-0300"},
        //  存在しないかもしれない値は最後
        { "LANAdapterMacAddress", "Replaced with actual address" },
    };

    static  ExpectedReportItem g_Report1[] =
    {
        { "ConnectionStatus", "Not Connected" },
        //{ "NifmErrorCode", "2000-0000" }, // BG成功
        { "ErrorCode", "2110-0300" },
    };

    static  ExpectedReportItem g_Report0[] =
    {
        { "ConnectionStatus", "Not Connected" },
      //{ "NifmErrorCode", "2000-0000" }, // テスト開始状態の result は不明なので確認なし
        { "ErrorCode", "2110-0300" },
    };

    struct ExpectedReport
    {
        ExpectedReportItem* report;
        int size;
    };

#define EXPECTED_ERROR_REPORT_LIST_ITEM(r) { &r[0], NN_ARRAY_SIZE(r) }

    // 新しいエラーレポートから順に評価される
    ExpectedReport g_ExpectedReportList[] =
    {
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report14),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report13),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report12),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report11),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report10),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report9),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report8),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report7),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report6),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report5),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report4),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report3),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report2),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report1),
        EXPECTED_ERROR_REPORT_LIST_ITEM(g_Report0),
    };

    typedef void(*DumperFunction)(char* buffer, const char* format, ...);

    void DumpMessagePackValue(nn::msgpack::MpWalker& mpWalker, DumperFunction dumper, char* buffer)
    {
        const char* pRaw = reinterpret_cast<const char*>(mpWalker.GetPtr());
        char valueType = pRaw[0];

        if ((valueType & 0xe0) == nn::erpt::ValueTypeTag::Str ||
            valueType == nn::erpt::ValueTypeTag::Str256 ||
            valueType == nn::erpt::ValueTypeTag::Str16384)
        {
            const char* pStr;
            uint32_t size;
            mpWalker.GetString(&pStr, &size);
            std::unique_ptr<char[]> strBuffer(new char[size + 1]);
            std::memcpy(strBuffer.get(), pStr, size);
            strBuffer[size] = '\0';
            dumper(buffer, "%s", strBuffer.get());
        }
        else if (valueType == nn::erpt::ValueTypeTag::U8 || valueType == nn::erpt::ValueTypeTag::U16 || valueType == nn::erpt::ValueTypeTag::U32)
        {
            uint32_t v;
            mpWalker.GetUint(&v);
            dumper(buffer, "%u", v);
        }
        else if (valueType == nn::erpt::ValueTypeTag::I8 || valueType == nn::erpt::ValueTypeTag::I16 || valueType == nn::erpt::ValueTypeTag::I32)
        {
            int32_t  v;
            mpWalker.GetInt(&v);
            dumper(buffer, "%d", v);
        }
        else if (valueType == nn::erpt::ValueTypeTag::I64)
        {
            int64_t v;
            mpWalker.GetInt(&v);
            dumper(buffer, "%lld", v);
        }
        else if (valueType == nn::erpt::ValueTypeTag::U64)
        {
            uint64_t v;
            mpWalker.GetUint(&v);
            dumper(buffer, "0x%016llx", v);
        }
        else if (valueType == nn::erpt::ValueTypeTag::False)
        {
            dumper(buffer, "False");
        }
        else if (valueType == nn::erpt::ValueTypeTag::True)
        {
            dumper(buffer, "True");
        }
        else if ((valueType & 0xf0) == nn::erpt::ValueTypeTag::Array16)
        {
        }
        else if (valueType == nn::erpt::ValueTypeTag::Array16384)
        {
        }
        else if (valueType == nn::erpt::ValueTypeTag::Bin || valueType == nn::erpt::ValueTypeTag::Bin16384 )
        {
        }
        else
        {
            NN_LOG("* Unknown format * : valueType = 0x%02x\n", valueType);
            const char* p = reinterpret_cast<const char*>(mpWalker.GetPtr());
            for (size_t i = 0; i < mpWalker.GetSize(); i += 16)
            {
                for (size_t j = i; j < i + 16 && j < mpWalker.GetSize(); ++j)
                {
                    NN_LOG("%02x ", p[j]);
                }
                NN_LOG("\n");
            }
        }
    } // NOLINT(impl/function_size)

    void DefaultDumper(char* buffer, const char* format, ...)
    {
        std::va_list vaList;
        va_start(vaList, format);
        nn::util::VSNPrintf(buffer, 128, format, vaList);
        va_end(vaList);
    }

    nn::Result DumpReport(nn::erpt::ReportId& reportId, ExpectedReport& expectedReport)
    {
        nn::erpt::Report report;
        NN_RESULT_DO(report.Open(reportId));

        int64_t reportSize;
        NN_RESULT_DO(report.GetSize(&reportSize));

        std::unique_ptr<uint8_t[]> reportData(new uint8_t[static_cast<uint32_t>(reportSize)]);
        NN_RESULT_THROW_UNLESS(reportData != nullptr, nn::erpt::ResultOutOfMemory());

        uint32_t readCount;
        NN_RESULT_DO(report.Read(&readCount, reportData.get(), static_cast<uint32_t>(reportSize)));

        NN_LOG("ReportId: ");
        for (int i = 0; i < sizeof(reportId); ++i)
        {
            NN_LOG("%02x", reportId.u.id[i]);
        }
        NN_LOG("\n");
        NN_LOG("    <Key: Expected = Actual>\n");

        nn::msgpack::MpWalker mpWalker;
        mpWalker.Init(reportData.get(), readCount);
        uint32_t count;
        int comparedCount = 0;
        mpWalker.GetMapCount(&count);
        for (uint32_t i = 0; i < count; ++i)
        {
            const char* key;
            size_t keySize;
            mpWalker.At(i, &key, &keySize);
            std::unique_ptr<char[]> keyBuffer(new char[keySize + 1]);
            keyBuffer[keySize] = '\0';
            std::memcpy(keyBuffer.get(), key, keySize);
            auto value = mpWalker[keyBuffer.get()];
            char buffer[128];

            for (int i = 0; i < expectedReport.size; ++i)
            {
                if (nn::util::Strncmp(expectedReport.report[i].key, keyBuffer.get(), 32) == 0)
                {
                    DumpMessagePackValue(value, DefaultDumper, buffer);
                    NN_LOG("    %s: %s = %s\n", keyBuffer.get(), expectedReport.report[i].value, buffer);

                    if (nn::util::Strncmp("AccessPointChannel", keyBuffer.get(), sizeof("AccessPointChannel")) == 0)
                    {
                        // 1-14, 36-64, 100-140
                        char *pEndp;
                        auto channel = strtoul(buffer, &pEndp, 10);
                        EXPECT_EQ('\0', pEndp[0]);
                        EXPECT_TRUE((1 <= channel && channel <= 14) || (36 <= channel && channel <= 64) || (100 <= channel && channel <= 140));
                    }
                    else if (nn::util::Strncmp("AccessPointRssi", keyBuffer.get(), sizeof("AccessPointRssi")) == 0)
                    {
                        // [-128,0)
                        char *pEndp;
                        auto rssi = strtol(buffer, &pEndp, 10);
                        EXPECT_EQ('\0', pEndp[0]);
                        EXPECT_TRUE(-128 <= rssi && rssi < 0);
                    }
                    else if (nn::util::Strncmp("RadioStrength", keyBuffer.get(), sizeof("RadioStrength")) == 0)
                    {
                        // 0,1,2,3
                        char *pEndp;
                        auto linkLevel = strtoul(buffer, &pEndp, 10);
                        EXPECT_EQ('\0', pEndp[0]);
                        EXPECT_TRUE(linkLevel == 0 || linkLevel == 1 || linkLevel == 2 || linkLevel == 3);
                    }
                    else if (nn::util::Strncmp("NifmErrorCode", keyBuffer.get(), sizeof("NifmErrorCode")) == 0)
                    {
                        std::regex re(expectedReport.report[i].value);
                        EXPECT_TRUE(std::regex_match(buffer, re));
                    }
                    else
                    {
                        EXPECT_EQ(0, nn::util::Strncmp(expectedReport.report[i].value, buffer, 32));
                    }
                    ++comparedCount;
                }
            }
        }

        EXPECT_EQ(expectedReport.size, comparedCount);

        NN_RESULT_SUCCESS;
    }

    int Stringize(char* pOutString, int size, const nn::nifm::MacAddress& macAddress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(size >= MacAddressStringSize);

        return nn::util::SNPrintf(pOutString, size, "%02X-%02X-%02X-%02X-%02X-%02X",
            macAddress.data[0], macAddress.data[1], macAddress.data[2],
            macAddress.data[3], macAddress.data[4], macAddress.data[5]);
    }

    // エラーレポート期待値の有線・無線 Mac アドレスを現在のものに入れ替える
    void PrepareExpectedReport()
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

        nn::nifm::NetworkInterfaceInfo networkInterfaceInfo[8];
        int bufferSize;
        int outCount;

        // 利用対象の NIC を列挙
        bufferSize = nn::nifm::NetworkInterfaceCountMax;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::nifm::EnumerateNetworkInterfaces(
                networkInterfaceInfo, &outCount, bufferSize,
                nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet
            )
        );

        bool hasEthernetAdapter = false;

        for (int i = 0; i < outCount; ++i)
        {
            if (networkInterfaceInfo[i].type == nn::nifm::NetworkInterfaceType_Ethernet)
            {
                Stringize(g_EthernetMacAddressString, MacAddressStringSize, networkInterfaceInfo[i].macAddress);
                hasEthernetAdapter = true;
            }
            else if (networkInterfaceInfo[i].type == nn::nifm::NetworkInterfaceType_Ieee80211)
            {
                Stringize(g_WirelessMacAddressString, MacAddressStringSize, networkInterfaceInfo[i].macAddress);
            }
        }

        nn::nifm::SsidListVersion ssidListVersion;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetSsidListVersion(&ssidListVersion));
        nn::util::Strlcpy(g_SsidListVersion, ssidListVersion.data, SsidListVersionSize);

        for (int i = 0; i < NN_ARRAY_SIZE(g_ExpectedReportList); ++i)
        {
            for (int j = 0; j < g_ExpectedReportList[i].size; ++j)
            {
                if (nn::util::Strncmp("LANAdapterMacAddress", g_ExpectedReportList[i].report[j].key, sizeof("LANAdapterMacAddress")) == 0)
                {
                    nn::util::Strlcpy(g_ExpectedReportList[i].report[j].value, g_EthernetMacAddressString, MacAddressStringSize);
                }
                else if (nn::util::Strncmp("NXMacAddress", g_ExpectedReportList[i].report[j].key, sizeof("NXMacAddress")) == 0)
                {
                    nn::util::Strlcpy(g_ExpectedReportList[i].report[j].value, g_WirelessMacAddressString, MacAddressStringSize);
                }
            }

            // 有線 LAN が存在しない場合は比較対象から外す
            if (!hasEthernetAdapter)
            {
                --g_ExpectedReportList[i].size;
            }
        }
    }
}

class ErrorReportVerifyTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
    }

    static void TearDownTestCase()
    {
    }
};


TEST_F(ErrorReportVerifyTest, Basic)
{
    PrepareExpectedReport();

    nn::erpt::ReportList reportList;
    nn::erpt::Manager manager;

    manager.Initialize();
    manager.GetReportList(reportList);

    NN_LOG("report.count=%d\n", reportList.reportCount);

    for (uint32_t i = 0; i < NN_ARRAY_SIZE(g_ExpectedReportList); ++i)
    {
        auto reportInfo = reportList.Report[i];
        NN_LOG("[%02d]----------------------------------------------------------\n", NN_ARRAY_SIZE(g_ExpectedReportList) - i - 1);
        NN_ABORT_UNLESS_RESULT_SUCCESS(DumpReport(reportInfo.reportId, g_ExpectedReportList[i]));
    }

    manager.Finalize();
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
