﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>

#include <nn/erpt.h>

#include <nn/la/la_CommonArgumentsReader.h>
#include <nn/err/err_ErrorViewerAppletParam.h>

#include "../Common/nifm_TestUtility.h"


void LibraryAppletErrorMain(const nn::ae::LibraryAppletSelfInfo& info) NN_NOEXCEPT
{
    NN_LOG("===== DummyError Start =====\n");

    nn::la::CommonArgumentsReader commonArgumentsReader;

    ASSERT_TRUE(commonArgumentsReader.TryPopFromInChannel());

    NN_LOG("commonArguments\n");
    NN_LOG("\tlaVersion = %u\n", commonArgumentsReader.GetLaVersion());
    NN_LOG("\tcolorIndex = %d\n", commonArgumentsReader.GetColorIndex());
    NN_LOG("\tisPlayStartupSound = %s\n", commonArgumentsReader.IsPlayStartupSound() ? "true" : "false");

    //
    nn::err::ErrorViewerStartupParamForSystemData param;

    nn::applet::StorageHandle storageHandle;
    if (nn::ae::TryPopFromInChannel(&storageHandle))
    {
        ASSERT_GE(nn::applet::GetStorageSize(storageHandle), sizeof(param));
        NNT_ASSERT_RESULT_SUCCESS(nn::applet::ReadFromStorage(storageHandle, 0, &param, sizeof(param)));
    }
    nn::applet::ReleaseStorage(storageHandle);

    nn::err::ErrorCode errorCode = param.GetErrorCode();
    NN_LOG("\n");
    NN_LOG("ErrorViewerStartupParamForSystemData\n");
    NN_LOG("\tcommon.errorType = %d\n", param.common.errorType);
    NN_LOG("\tcommon.isJumpEnabled = %s\n", param.common.isJumpEnabled ? "true" : "false");
    NN_LOG("\tversion = %d\n", param.version);
    NN_LOG("\terrorCode = %04d-%04d\n", errorCode.category, errorCode.number);

    ASSERT_EQ(2110, errorCode.category);
    ASSERT_NE(nn::nifm::ResultCanceled().GetDescription(), errorCode.number);

    // エラーレポート書き出し
    nn::erpt::Context context(nn::erpt::ErrorInfo);
    char errorCodeString[nn::err::ErrorCode::StringLengthMax] = {};
    auto len = nn::util::SNPrintf(errorCodeString, sizeof(errorCodeString), "%04d-%04d", errorCode.category, errorCode.number);
    ASSERT_LT(len, sizeof(errorCodeString));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::ErrorCode, errorCodeString, len));
    NNT_ASSERT_RESULT_SUCCESS(context.CreateReport(nn::erpt::ReportType_Visible));

    NN_LOG("===== DummyError Finish =====\n");
}

extern "C" void nninitStartup()
{
    // 本テストはアプレット向け desc を利用しており、アプレット向けのリソース制限が適用されます。
    // ここでは、デフォルトの nninitStartup() のデフォルトメモリアロケータのサイズが
    // アプレットで利用できるサイズ上限を超えているため、自前で nninitStartup() を用意しています。
}

extern "C" void nnMain()
{
    nn::ae::InvokeLibraryAppletMain(LibraryAppletErrorMain);
}

