﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>

#include <nn/nn_Log.h>

#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_TypesAccessPoint.h>
#include <nn/nifm/nifm_TypesRequirement.h>

#include <nn/socket/socket_Types.h>

#include <algorithm>

// スレッド優先度調整に伴い、主に他の要求の影響で状態が変化することを確認する項目の前に必要
// 要求は優先度の高いものから状態が決定されるので、追い出し側の状態が決定した瞬間に追い出され側の状態は未決定
#define NNT_NIFM_WAIT_FOR_PROCESSING nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100))

namespace nn
{
namespace nifm
{
namespace test
{
    /*** 定数 ***/

    //
    static const nn::nifm::RequestParameters RequestParametersNone = { nn::nifm::RequirementPreset_None };
    static const nn::nifm::RequestParameters RequestParametersInternetGeneric = { nn::nifm::RequirementPreset_InternetGeneric };
    static const nn::nifm::RequestParameters RequestParametersLocalGeneric = { nn::nifm::RequirementPreset_LocalGeneric };

    /*** 表示・文字列変換 ***/

    // nn::socket::InAddr 表示
    void PrintInAddr(const nn::socket::InAddr& address) NN_NOEXCEPT;

    // ネットワークインタフェース表示
    void PrintNetworkInterfaces(const nn::nifm::NetworkInterfaceInfo* pNetworkInterfaceInfoList, int bufferCount, int outCount) NN_NOEXCEPT;

    // Requirement 内容表示
    void PrintRequirement(nn::nifm::Requirement& requirement) NN_NOEXCEPT;

    // AP 情報表示
    void PrintAccessPoint(const nn::nifm::AccessPointData& accessPoint) NN_NOEXCEPT;

    // 経過時間文字列取得
    char* ElapsedTime(int64_t initialTick = 0) NN_NOEXCEPT;


    /*** タイムアウト付き利用要求提出・待ち ***/

    bool WaitRequestCompletion(nn::nifm::NetworkConnection* pNetworkConnection, int seconds) NN_NOEXCEPT;

    bool WaitRequestCompletion(nn::nifm::Request* pInRequest, int seconds) NN_NOEXCEPT;

    bool SubmitRequestAndWait(nn::nifm::NetworkConnection* pInNetworkConnection, int seconds) NN_NOEXCEPT;

    bool SubmitRequestAndWait(nn::nifm::Request* pInRequest, int seconds) NN_NOEXCEPT;

    bool WaitDisconnection(int seconds) NN_NOEXCEPT;


    /*** 各種設定・取得 ***/

    // 接続制御許可設定取得
    bool IsCommunicationControlEnabled() NN_NOEXCEPT;

    // 接続制御許可設定
    void SetCommunicationControlEnabled(bool isEnabled) NN_NOEXCEPT;

    // 接続制御許可取得
    bool IsCommunicationControlEnabled() NN_NOEXCEPT;

    // 発売日 NUP 設定取得
    bool IsNetworkServiceEnabled() NN_NOEXCEPT;

    // 発売日 NUP 設定
    void SetNetworkServiceEnabled(bool isEnabled) NN_NOEXCEPT;



    /*** その他 ***/

    //
    void SetExclusive(bool isExclusive) NN_NOEXCEPT;

    // 高優先度のローカル通信利用要求を出すことによりインターネット接続を確実に切断する
    bool Disconnect() NN_NOEXCEPT;

    // 乱数
    uint32_t Random() NN_NOEXCEPT;

    //
    template<typename nn::Result(*Initialize)(), typename nn::Result(*Finalize)()>
    void SetExclusive(bool isExclusive) NN_NOEXCEPT
    {
        if (isExclusive)
        {
            Initialize();

            nn::nifm::ClientId clientId = nn::nifm::GetClientId();
            auto result = nn::nifm::SetExclusiveClient(clientId);
            NN_LOG("SetExclusive(%d), %s\n", clientId.value, result.IsSuccess() ? "Success" : "Failed");

            nn::nifm::InternetConnectionStatus internetConnectionStatus;

            // プロセス独占後，全インターネット接続が切断されるまで最長で30秒待つ
            nn::Result connectionResult = nn::ResultSuccess();
            for (int i = 0; i < 30; ++i)
            {
                connectionResult = nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus);
                if (connectionResult.IsFailure())
                {
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
        }
        else
        {
            Initialize();
            NN_UTIL_SCOPE_EXIT
            {
                Finalize();
            };

            nn::nifm::ClientId clientId = { nn::nifm::InvalidClientIdValue };
            auto result = nn::nifm::SetExclusiveClient(clientId);
            NN_LOG("SetExclusive(%d), %s\n", clientId.value, result.IsSuccess() ? "Success" : "Failed");
        }
    }

    // 初期化
    template<typename nn::Result(*Initialize)(), typename nn::Result(*Finalize)()>
    class ScopedInitializer
    {
    private:
        nn::Result m_Result;

    public:
        explicit ScopedInitializer(bool isLocalNetworkIncluded = true) NN_NOEXCEPT
            : m_Result(Initialize())
        {
            if (m_Result.IsSuccess())
            {
                // 初期化で ClientId が変化するので、新しい ID で独占
                nn::nifm::ClientId clientId = nn::nifm::GetClientId();
                auto result = nn::nifm::SetExclusiveClient(clientId);
                NN_LOG("SetExclusive(%d), %s\n", clientId, result.IsSuccess() ? "Success" : "Failed");

                if (isLocalNetworkIncluded)
                {
                    auto* pNetworkConnection = nn::nifm::GetGlobalNetworkConnectionPointer();
                    NN_ASSERT_NOT_NULL(pNetworkConnection);
                    if (pNetworkConnection != nullptr)
                    {
                        result = nn::nifm::SetRequestConnectionConfirmationOption(
                            pNetworkConnection->GetRequestHandle(),
                            nn::nifm::ConnectionConfirmationOption_NotRequired
                        );
                        NN_UNUSED(result);
                        NN_ASSERT(result.IsSuccess());
                    }
                }
            }
        }

        ~ScopedInitializer() NN_NOEXCEPT
        {
            // 独占は解除しない
            NNT_EXPECT_RESULT_SUCCESS(Finalize());
        }

        nn::Result GetResult() const NN_NOEXCEPT
        {
            return m_Result;
        }
    };

}
}
}
