﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/nn_Log.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiEthernetCommunicationControl.h>
#include <nn/nifm/nifm_ApiInternetConnectionStatus.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ResultPrivate.h>
#include <nn/nifm/nifm_TypesIpAddress.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>
#include <nn/nifm/nifm_TypesRequestPrivate.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include <nn/util/util_StringUtil.h>

#include "nifm_TestUtility.h"

namespace nn
{
namespace nifm
{
namespace test
{

void PrintInAddr(const nn::socket::InAddr& address) NN_NOEXCEPT
{
    unsigned long addr = address.S_addr;
    NN_LOG("%u.%u.%u.%u\n", (addr >> 0) & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff, (addr >> 24) & 0xff);
}

bool WaitRequestCompletion(nn::nifm::NetworkConnection* pNetworkConnection, int seconds) NN_NOEXCEPT
{
    int milliseconds = seconds * 1000;
    bool isCompleted = pNetworkConnection->GetRequestState() != nn::nifm::RequestState_OnHold;

    while (milliseconds > 0 && !isCompleted)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        milliseconds -= 16;
        isCompleted = pNetworkConnection->GetRequestState() != nn::nifm::RequestState_OnHold;
    }

    if (!isCompleted)
    {
        NN_LOG("WaitRequestCompletion failed: %x\n", pNetworkConnection->GetResult().GetInnerValueForDebug());
        pNetworkConnection->CancelRequest();
    }

    return isCompleted;
}

bool WaitRequestCompletion(nn::nifm::Request* pInRequest, int seconds) NN_NOEXCEPT
{
    int milliseconds = seconds * 1000;
    bool isCompleted = pInRequest->GetRequestState() != nn::nifm::RequestState_OnHold;

    while (milliseconds > 0 && !isCompleted)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        milliseconds -= 16;
        isCompleted = pInRequest->GetRequestState() != nn::nifm::RequestState_OnHold;
    }

    if (!isCompleted)
    {
        NN_LOG("WaitRequestCompletion failed: %x\n", pInRequest->GetResult().GetInnerValueForDebug());
        pInRequest->Cancel();
    }

    return isCompleted;
}

bool WaitDisconnection(int seconds) NN_NOEXCEPT
{
    int milliseconds = seconds * 1000;

    nn::nifm::InternetConnectionStatus status;
    bool isCompleted = nn::nifm::GetInternetConnectionStatus(&status) <= nn::nifm::ResultNotConnected();

    while (milliseconds > 0 && !isCompleted)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        milliseconds -= 16;
        isCompleted = nn::nifm::GetInternetConnectionStatus(&status) <= nn::nifm::ResultNotConnected();
    }

    return isCompleted;
}

bool SubmitRequestAndWait(nn::nifm::NetworkConnection* pInNetworkConnection, int seconds) NN_NOEXCEPT
{
    pInNetworkConnection->SubmitRequest();

    return WaitRequestCompletion(pInNetworkConnection, seconds);
}

bool SubmitRequestAndWait(nn::nifm::Request* pInRequest, int seconds) NN_NOEXCEPT
{
    pInRequest->Submit();

    return WaitRequestCompletion(pInRequest, seconds);
}

void PrintNetworkInterfaces(const nn::nifm::NetworkInterfaceInfo* pNetworkInterfaceInfoList, int bufferCount, int outCount) NN_NOEXCEPT
{
    NN_LOG("networkInterfaceInfoList.count : %d\n", outCount);
    const int count = std::min(bufferCount, outCount);
    for (int i = 0; i<count; ++i)
    {
        const auto &info = pNetworkInterfaceInfoList[i];
        NN_LOG(
            "%s (%s)\n",
            info.type == nn::nifm::NetworkInterfaceType_Invalid ? "Invalid" : info.type == nn::nifm::NetworkInterfaceType_Ieee80211 ? "IEEE802.11" : "Ethernet",
            info.isAvailable ? "available" : "not available"
        );
        NN_LOG("    mac address:%02x-%02x-%02x-%02x-%02x-%02x\n", info.macAddress.data[0], info.macAddress.data[1], info.macAddress.data[2], info.macAddress.data[3], info.macAddress.data[4], info.macAddress.data[5]);
    }
}

// Requirement 内容表示
void PrintRequirement(nn::nifm::Requirement& requirement) NN_NOEXCEPT
{
    nn::util::Uuid profileId(requirement._profileId);

    char uuidString[nn::util::Uuid::StringSize];
    NN_UNUSED(uuidString);

    NN_LOG("Requirement = { %u, %s, %s }\n",
        requirement._priority,
        requirement._networkType == nn::nifm::NetworkType_None ? "None" :
        requirement._networkType == nn::nifm::NetworkType_Infrastructure ? "Infrastructure" :
        requirement._networkType == nn::nifm::NetworkType_Local ? "Local" : "Others",
        profileId.ToString(uuidString, nn::util::Uuid::StringSize));
}

// 高優先度のローカル通信利用要求を出すことによりインターネット接続を確実に切断する
bool Disconnect() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::nifm::Request request(nn::nifm::test::RequestParametersLocalGeneric);
    nn::nifm::SetRequestRawPriority(request.GetHandle(), 0);   // highest priority
    nn::nifm::SetRequestRejectable(request.GetHandle(), true);

    if (!SubmitRequestAndWait(&request, 60))
    {
        return false;
    }

    if (request.GetResult().IsFailure())
    {
        return false;
    }

    request.Cancel();

    if (!request.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)))
    {
        return false;
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    return nn::nifm::ResultCanceled().Includes(request.GetResult()) && !nn::nifm::IsAnyForegroundRequestAccepted();
#else
    // CI マシンの無線対応後は Horizon と同様の切断が可能
    nn::nifm::SetEthernetCommunicationEnabled(false);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
    nn::nifm::SetEthernetCommunicationEnabled(true);
    return true;
#endif
}

// 経過時間文字列取得
char* ElapsedTime(int64_t initialTick) NN_NOEXCEPT
{
    using timeStr_t = std::aligned_storage<sizeof("00:00:00"), NN_ALIGNOF(char)>::type;
    NN_FUNCTION_LOCAL_STATIC(timeStr_t, timeStr);
    NN_FUNCTION_LOCAL_STATIC(int64_t, startTick, =0);
    NN_FUNCTION_LOCAL_STATIC(int64_t, tickFreq, =nn::os::GetSystemTickFrequency());

    if (initialTick)
    {
        startTick = initialTick;
        return nullptr;
    }

    int64_t t = (nn::os::GetSystemTick().GetInt64Value() - startTick) / tickFreq;
    int s = static_cast<int>(t % 60);
    int m = static_cast<int>((t / 60) % 60);
    int h = static_cast<int>(t / (60 * 60));
    nn::util::SNPrintf(reinterpret_cast<char*>(&timeStr), sizeof(timeStr), "%02d:%02d:%02d", h, m, s);

    return reinterpret_cast<char*>(&timeStr);
}

// 接続制御許可取得
bool IsCommunicationControlEnabled() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    bool isEnabled = false;
    auto outSize = nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "nifm", "is_communication_control_enabled_for_test");
    EXPECT_EQ(outSize, sizeof(isEnabled));
    return isEnabled;
#else
    return false;
#endif
}

// 接続制御許可設定
void SetCommunicationControlEnabled(bool isEnabled) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
#else
    NN_UNUSED(isEnabled);
#endif
}

// 発売日 NUP 設定
bool IsNetworkServiceEnabled() NN_NOEXCEPT
{
    return true;
}

// 発売日 NUP 設定
void SetNetworkServiceEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_UNUSED(isEnabled);
}

void PrintAccessPoint(const nn::nifm::AccessPointData& accessPoint) NN_NOEXCEPT
{
    const nn::nifm::Ssid& ssid = accessPoint.ssid;
    bool hasNonAscii = false;

    for (int i = 0; i < ssid.length; ++i)
    {
        if (ssid.hex[i] < 0x20 || 0x7e < ssid.hex[i])
        {
            hasNonAscii = true;
            break;
        }
    }

    NN_LOG("----- SSID(%d) : ", ssid.length);
    if (hasNonAscii)
    {
        for (int i = 0; i < ssid.length; ++i)
        {
            NN_LOG("%02x", ssid.hex[i]);
        }
    }
    else
    {
        for (int i = 0; i < ssid.length; ++i)
        {
            NN_LOG("%c", ssid.hex[i]);
        }
    }
    NN_LOG(" -----\n");

    switch (accessPoint.authentication)
    {
    case nn::nifm::Authentication_Open:
        NN_LOG("\tAuthentication : Open\n");
        break;
    case nn::nifm::Authentication_Shared:
        NN_LOG("\tAuthentication : Shared\n");
        break;
    case nn::nifm::Authentication_WpaPsk:
        NN_LOG("\tAuthentication : WPA-PSK\n");
        break;
    case nn::nifm::Authentication_Wpa2Psk:
        NN_LOG("\tAuthentication : WPA2-PSK\n");
        break;
    case nn::nifm::Authentication_Wpa:
        NN_LOG("\tAuthentication : WPA-EAP\n");
        break;
    case nn::nifm::Authentication_Wpa2:
        NN_LOG("\tAuthentication : WPA2-EAP\n");
        break;
    default:
        NN_LOG("\tAuthentication : UNKNOWN(%d)\n", accessPoint.authentication);
        break;
    }

    switch (accessPoint.encryption)
    {
    case nn::nifm::Encryption_None:
        NN_LOG("\tEncryption : None\n");
        break;
    case nn::nifm::Encryption_Wep:
        NN_LOG("\tEncryption : WEP\n");
        break;
    case nn::nifm::Encryption_Tkip:
        NN_LOG("\tEncryption : TKIP\n");
        break;
    case nn::nifm::Encryption_Aes:
        NN_LOG("\tEncryption : AES(CCMP)\n");
        break;
    default:
        NN_LOG("\tEncryption : UNKNOWN(%d)\n", accessPoint.encryption);
        break;
    }

    switch (accessPoint.groupEncryption)
    {
    case nn::nifm::Encryption_None:
        NN_LOG("\tgroupEncryption : None\n");
        break;
    case nn::nifm::Encryption_Wep:
        NN_LOG("\tgroupEncryption : WEP\n");
        break;
    case nn::nifm::Encryption_Tkip:
        NN_LOG("\tgroupEncryption : TKIP\n");
        break;
    case nn::nifm::Encryption_Aes:
        NN_LOG("\tgroupEncryption : AES(CCMP)\n");
        break;
    default:
        NN_LOG("\tgroupEncryption : UNKNOWN(%d)\n", accessPoint.groupEncryption);
        break;
    }


    NN_LOG("\tBSSID : ");
    for (int i = 0; i < accessPoint.bssid.Size; ++i)
    {
        NN_LOG("%02x", accessPoint.bssid.data[i]);
    }
    NN_LOG("\n");

    NN_LOG("\tChannel : %d\n", accessPoint.channel);
    NN_LOG("\tRSSI : %d\n", accessPoint.rssi);
    NN_LOG("\tLink-Level : %d\n", accessPoint.linkLevel);
    NN_LOG("-----------------------\n");
} // NOLINT(impl/function_size)

uint32_t Random() NN_NOEXCEPT
{
    static uint32_t prev = 0;
    return (prev = (prev ^ (uint32_t)(nn::os::GetSystemTick().GetInt64Value())));
}



}
}
}
