﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>

#include <nn/util/util_BitUtil.h>

#include <algorithm>
#include <cstring>

#include "../Common/nifm_TestUtility.h"

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;
}

class AutoNetworkSwitchTest : public ::testing::Test
{
protected:
    virtual void SetUp()
    {
    }
    virtual void TearDown()
    {
        nn::nifm::FinalizeAdminForTest();
    }
};


TEST_F(AutoNetworkSwitchTest, ForegroundRequestDisappearance)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // 既存の接続設定を全消去

    for (;;)
    {
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo;
        int count = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(&networkProfileBasicInfo, &count, 1, nn::nifm::NetworkProfileType_User));

        if (count == 0)
        {
            break;
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile(networkProfileBasicInfo.id));
    }

    {
        // 自動接続対象の有線接続設定
        const nn::nifm::WirelessSettingData wirelessSetting = nn::nifm::InvalidWirelessSettingData;
        const nn::nifm::IpSettingData ipSetting = { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 };
        nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true,{ wirelessSetting }, ipSetting };

        nn::util::Uuid id;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, networkProfile));
    }

    nn::util::Uuid idWirelessProfile;

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };

        nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211, true, true,{ wirelessSetting }, ipSetting };

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&idWirelessProfile, networkProfile));
    }

    nn::nifm::InternetConnectionStatus internetConnectionStatus;

    // 自動接続対象の有線接続設定がある場合

    // Profile ID 指定の FG 要求（⊂有線への切り替えを抑制する要求）で無線に切り替える

    nn::nifm::NetworkConnection networkConnection1;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), idWirelessProfile));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
    ASSERT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, internetConnectionStatus.networkInterfaceType);

    // ID 指定のない FG 要求で相乗りする

    nn::nifm::NetworkConnection networkConnection2;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), nn::util::InvalidUuid));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
    ASSERT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, internetConnectionStatus.networkInterfaceType);

    // ID 指定のない BG 要求で相乗りする

    nn::nifm::NetworkConnection networkConnection3;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection3.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection3.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection3.GetRequestHandle(), nn::util::InvalidUuid));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection3, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection3.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
    ASSERT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, internetConnectionStatus.networkInterfaceType);

    // ID 指定の要求をキャンセルしても FG 要求が残るので無線のまま

    networkConnection1.CancelRequest();
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
    ASSERT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, internetConnectionStatus.networkInterfaceType);

    EXPECT_TRUE(nn::nifm::IsAnyForegroundRequestAccepted());

    // FG 要求を全部キャンセルすると、 BG の要求が残っていても切断される

    networkConnection3.GetSystemEvent().Clear();

    networkConnection2.CancelRequest();
    EXPECT_FALSE(nn::nifm::IsAnyForegroundRequestAccepted());

    ASSERT_TRUE(networkConnection3.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
} // NOLINT(impl/function_size)

TEST_F(AutoNetworkSwitchTest, SustainedRequest)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // 既存の接続設定を全消去

    for (;;)
    {
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo;
        int count = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(&networkProfileBasicInfo, &count, 1, nn::nifm::NetworkProfileType_User));

        if (count == 0)
        {
            break;
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile(networkProfileBasicInfo.id));
    }

    {
        // 自動接続対象の有線接続設定
        const nn::nifm::WirelessSettingData wirelessSetting = nn::nifm::InvalidWirelessSettingData;
        const nn::nifm::IpSettingData ipSetting = { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 };
        nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true,{ wirelessSetting }, ipSetting };

        nn::util::Uuid id;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, networkProfile));
    }

    nn::util::Uuid idWirelessProfile;

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };

        nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211, true, true,{ wirelessSetting }, ipSetting };

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&idWirelessProfile, networkProfile));
    }

    nn::nifm::InternetConnectionStatus internetConnectionStatus;

    // 自動接続対象の有線接続設定がある場合

    // Profile ID 指定の FG かつ Sustainable な要求で無線に切り替える

    nn::nifm::NetworkConnection networkConnection1;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForNetConnectApplet));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), idWirelessProfile));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
    ASSERT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, internetConnectionStatus.networkInterfaceType);

    // ID 指定のない BG 要求で相乗りする

    nn::nifm::NetworkConnection networkConnection2;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), nn::util::InvalidUuid));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection2.GetResult());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
    ASSERT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, internetConnectionStatus.networkInterfaceType);

    // FG 要求をキャンセルしても有線切換抑制属性を持った Sustained な要求が残るので無線のまま

    networkConnection1.CancelRequest();
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
    ASSERT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, internetConnectionStatus.networkInterfaceType);

    EXPECT_FALSE(nn::nifm::IsAnyForegroundRequestAccepted());   // FG 属性の要求はもうない

    // BG 要求のキャンセルも影響を与えない

    networkConnection2.CancelRequest();
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
    ASSERT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, internetConnectionStatus.networkInterfaceType);

    EXPECT_FALSE(nn::nifm::IsAnyForegroundRequestAccepted());   // FG 属性の要求はとっくにない

    // ローカル通信の要求を出して Sustained な要求を取り下げさせる
    // （Sustained な要求が残った状態で有線に切り替わるのは有線がリンクアップしたときだけで、自動テストではその状態を作れない）

    nn::nifm::NetworkConnection networkConnection3;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection3.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection3, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection3.GetResult());
} // NOLINT(impl/function_size)

TEST_F(AutoNetworkSwitchTest, EthernetProfileAppearance)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    // 既存の接続設定を全消去

    for (;;)
    {
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo;
        int count = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(&networkProfileBasicInfo, &count, 1, nn::nifm::NetworkProfileType_User));

        if (count == 0)
        {
            break;
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile(networkProfileBasicInfo.id));
    }

    {
        // 非自動接続対象の有線接続設定
        const nn::nifm::WirelessSettingData wirelessSetting = nn::nifm::InvalidWirelessSettingData;
        const nn::nifm::IpSettingData ipSetting = { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 };
        nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, false, true,{ wirelessSetting }, ipSetting };

        nn::util::Uuid id;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, networkProfile));
    }

    nn::util::Uuid idWirelessProfile;

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };

        nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211, true, true,{ wirelessSetting }, ipSetting };

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&idWirelessProfile, networkProfile));
    }

    nn::nifm::InternetConnectionStatus internetConnectionStatus;

    // 自動接続対象の有線接続設定がない場合

    // Profile ID 指定の BG 要求で無線に切り替える

    nn::nifm::NetworkConnection networkConnection1;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), idWirelessProfile));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

    // 自動接続対象の有線設定がないので BG 要求のみでも無線が維持される

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));  // 切替判定を通すために要求変化のイベントを発生させる
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
    ASSERT_EQ(nn::nifm::NetworkInterfaceType_Ieee80211, internetConnectionStatus.networkInterfaceType);

    // 自動接続対象の有効な有線接続設定の出現で無線が切断される

    networkConnection1.GetSystemEvent().Clear();

    {
        // 自動接続対象の有線接続設定
        const nn::nifm::WirelessSettingData wirelessSetting = nn::nifm::InvalidWirelessSettingData;
        const nn::nifm::IpSettingData ipSetting = { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 };
        nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true,{ wirelessSetting }, ipSetting };

        nn::util::Uuid id;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, networkProfile));
    }

    ASSERT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(10)));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));
} // NOLINT(impl/function_size)

// TODO: Ethrenet リンクアップ時の挙動が確認できない

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
