﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
#pragma execution_character_set("utf-8")
#endif

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>

#include <nn/nifm/nifm_Request.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_TypesRequest.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>

#include <nn/nifm/nifm_ResultPrivate.h>

#include <algorithm>

#include "../Common/nifm_TestUtility.h"

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;
}

class AutoConnectCaseTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        // 本テストプロセスで独占
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(true);
    }

    static void TearDownTestCase()
    {
        // 本テストプロセスで独占解除
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(false);
    }
};


TEST_F(AutoConnectCaseTest, NetworkInterfaceList)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    nn::nifm::NetworkInterfaceInfo networkInterfaceInfo[8];
    int outCount = 0;

    // 最初の scan 終了後の NIC 登録を待つ workaround
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(15));

    // 利用対象の有線 NIC を列挙
    int bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet
        )
    );

    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);
}

TEST_F(AutoConnectCaseTest, Basic)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[1];
    int outCount;
    nn::nifm::EnumerateNetworkProfiles( networkProfileBasicInfo, &outCount, 1, nn::nifm::NetworkProfileType_User );
    ASSERT_EQ(1, outCount);

    nn::nifm::NetworkProfileData networkProfileData;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile( &networkProfileData, networkProfileBasicInfo[0].id ));

    // 自動接続対象ではない接続設定しかない
    ASSERT_FALSE(networkProfileData.isAutoConnect);

    {
        nn::nifm::NetworkConnection networkConnection;
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        ASSERT_FALSE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());
    }

    // ID 指定であれば接続可能
    {
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), networkProfileBasicInfo[0].id));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, 60));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());
        networkConnection.CancelRequest();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());
    }

    // 自動接続対象に変更すれば接続可能
    {
        networkProfileData.isAutoConnect = true;

        nn::util::Uuid outId;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &outId, networkProfileData ));
        networkProfileData.id = outId;

        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        networkConnection.CancelRequest();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());
    }

    // 元に戻す（自動接続対象ではなくする）
    {
        networkProfileData.isAutoConnect = false;

        nn::util::Uuid outId;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &outId, networkProfileData ));
        networkProfileData.id = outId;
    }
}

TEST_F(AutoConnectCaseTest, ProfileOrder)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    nn::util::Uuid ethernetId;

    //
    {
        nn::nifm::NetworkProfileBasicInfo basicInfo[1];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles( basicInfo, &outCount, 1, nn::nifm::NetworkProfileType_User));
        ASSERT_EQ(1, outCount);
        ethernetId = basicInfo[0].id;

        // 有線の自動接続を有効に
        nn::nifm::NetworkProfileData networkProfileData;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile( &networkProfileData, ethernetId ));
        networkProfileData.isAutoConnect = true;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &ethernetId, networkProfileData ));
    }

    //
    nn::util::Uuid openApId;
    {
        // imatake-openwep40
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    17,  // length
                    {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x34,0x30}  // hex
                },
                false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_Wep  // encryption
                },
                {  // sharedKey
                    5,  // length
                    "Shi2i"  // keyMaterial
                    }
                }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
        nn::nifm::NetworkProfileData userNetworkProfile = {
            nn::util::InvalidUuid,  // id
            "オープンWepAP",  // name
            nn::nifm::NetworkProfileType_User, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        NNT_EXPECT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &openApId, userNetworkProfile ) );
        EXPECT_NE( openApId, nn::util::InvalidUuid );

        // 自動接続
        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // 有線につながる
        nn::nifm::NetworkProfileData networkProfile;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile( &networkProfile ));
        EXPECT_EQ(ethernetId, networkProfile.id);

        // 切断
        networkConnection.CancelRequest();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());

        // ID 指定で，無線に接続
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), openApId));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // 無線につながる
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile( &networkProfile ));
        EXPECT_EQ(openApId, networkProfile.id);

        // 切断
        networkConnection.CancelRequest();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

        // ID 指定せずに、再度自動接続
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), nn::util::InvalidUuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // 直前につながった無線よりも有線が優先なので、再度有線につながる
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile( &networkProfile ));
        EXPECT_EQ(ethernetId, networkProfile.id);

        // 切断
        networkConnection.CancelRequest();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );
    }

    //
    {
        // 有線の自動接続を無効に
        nn::nifm::NetworkProfileData networkProfileData;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile( &networkProfileData, ethernetId ));
        networkProfileData.isAutoConnect = false;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &ethernetId, networkProfileData ));
    }

    //
    nn::util::Uuid stealthApId;
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

        // imatake-opennone-stealth
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    24,  // length
                    {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x6e,0x6f,0x6e,0x65,0x2d,0x73,0x74,0x65,0x61,0x6c,0x74,0x68}  // hex
                },
                true  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_None  // encryption
                },
                {  // sharedKey
                    7,  // length
                    "NotUsed"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        nn::nifm::NetworkProfileData userNetworkProfile = {
            nn::util::InvalidUuid,  // id
            "ステルスAP",  // name
            nn::nifm::NetworkProfileType_User, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType,
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        NNT_EXPECT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &stealthApId, userNetworkProfile ) );
        EXPECT_NE( stealthApId, nn::util::InvalidUuid );

        // 自動接続
        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // // 直近にスキャンしているのでアクティブスキャンが行われずに、優先度の高い追加したばかりの設定でステルス AP につながる
        nn::nifm::NetworkProfileData networkProfile;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile( &networkProfile ));
        EXPECT_EQ(stealthApId, networkProfile.id);

        char stealthApIdStr[nn::util::Uuid::StringSize];
        stealthApId.ToString( stealthApIdStr, nn::util::Uuid::StringSize );
        NN_LOG("==> StealthApId:%s\n", stealthApIdStr);

        // 切断
        networkConnection.CancelRequest();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());

        // ID 指定で，オープン AP に接続
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), openApId));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        char openApIdStr[nn::util::Uuid::StringSize];
        openApId.ToString( openApIdStr, nn::util::Uuid::StringSize );
        NN_LOG("==> OpenApId:%s\n", openApIdStr);

        // 切断
        networkConnection.CancelRequest();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

        // ID 指定せずに、再度自動接続
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), nn::util::InvalidUuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // 直前につながったオープン AP に再度つながる
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile( &networkProfile ));
        EXPECT_EQ(openApId, networkProfile.id);

        // 切断
        networkConnection.CancelRequest();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection.GetRequestState());
        EXPECT_FALSE(networkConnection.IsAvailable());
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );
    }

    //
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

        {
            // imatake-wpa2aes
            const nn::nifm::WirelessSettingData wirelessSetting = {
                {  // ssidConfig
                    {  // ssid
                        15,  // length
                        {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73}  // hex
                    },
                    false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Wpa2Psk,  // authentication
                        nn::nifm::Encryption_Aes  // encryption
                    },
                    {  // sharedKey
                        11,  // length
                        "Shi2iTaiZen"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfileData = {
                nn::util::InvalidUuid,  // id
                "テンポラリ設定",  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true,  // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting
                },
                ipSetting
            };
            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfileData);
            nn::util::Uuid tempId = temporaryNetworkProfile.GetId();

            // 自動接続
            nn::nifm::NetworkConnection networkConnection;
            NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            EXPECT_TRUE(networkConnection.IsAvailable());
            NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

            // 直近にスキャンしているのでアクティブスキャンが行われずに、優先度の高いテンポラリ接続設定が利用される
            nn::nifm::NetworkProfileData networkProfile;
            NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile( &networkProfile ));
            EXPECT_EQ(tempId, networkProfile.id);

            char tempIdStr[nn::util::Uuid::StringSize];
            tempId.ToString( tempIdStr, nn::util::Uuid::StringSize );
            NN_LOG("==> TempApId:%s\n", tempIdStr);
        }

        // 自動接続
        nn::nifm::NetworkConnection networkConnection;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        // テンポラリ接続設定が削除されたので、直前につながったオープン AP に再度つながっているはず
        nn::nifm::NetworkProfileData networkProfile;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile( &networkProfile ));
        EXPECT_EQ(openApId, networkProfile.id);
    }

    // 元に戻す
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile( openApId ));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile( stealthApId ));

        nn::nifm::NetworkProfileData networkProfile;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile( &networkProfile, ethernetId ));
        networkProfile.isAutoConnect = false;

        nn::util::Uuid outId;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &outId, networkProfile ));
    }
} // NOLINT(impl/function_size)


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
