﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// テスト用のヘッダ
#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_ErrorResult.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

// テスト対象のヘッダ
#include <../../../../../Programs/Eris/Sources/Libraries/ngc/succinct/ngc_Bp.h>

namespace {

class TreeNode;

/**
 * @brief   TreeNode をつなぐための単方向リストです
 */
struct LinkedList
{
    LinkedList* pNext;  // 次の要素へのポインタ
    TreeNode* pNode;    // 保管しているノード
};

/**
 * @brief   木構造の各ノードを表す構造体です。
 */
class TreeNode
{
public:
    TreeNode() NN_NOEXCEPT : m_pParent(NULL), m_pChildrenHead(NULL),
                             m_pChildrenTail(NULL), m_pNextChild(NULL),
                             m_ChildSize(0), m_Num(0)
    {
    }
    explicit TreeNode(TreeNode* pParent) NN_NOEXCEPT : m_pParent(NULL), m_pChildrenHead(NULL),
                                              m_pChildrenTail(NULL), m_pNextChild(NULL),
                                              m_ChildSize(0), m_Num(0)
    {
        Init(pParent);
    }
    ~TreeNode() NN_NOEXCEPT
    {
        Finalize();
    }

    /**
     * @brief   親のポインタを設定し初期化します。
     */
    void Init(TreeNode* pParent) NN_NOEXCEPT
    {
        m_pParent = pParent;
        m_pChildrenHead = NULL;
        m_pChildrenTail = NULL;
        m_pNextChild = NULL;
        m_ChildSize = 0;
        m_Num = 0;
    }

    /**
     * @brief   自分の子を芋づる式に解放します。
     */
    void Finalize() NN_NOEXCEPT
    {
        m_pParent = NULL;
        LinkedList* pCurrent = m_pChildrenHead;
        while (pCurrent)
        {
            if (pCurrent->pNode)
            {
                delete pCurrent->pNode;
            }
            LinkedList* pTmp = pCurrent;
            pCurrent = pCurrent->pNext;
            delete pTmp;
        }
        m_pChildrenHead = NULL;
        m_pChildrenTail = NULL;
        m_pNextChild = NULL;
    }

    /**
     * @brief   子を一人追加します
     */
    void AddChild() NN_NOEXCEPT
    {
        TreeNode* pChild = new TreeNode();
        LinkedList* pList = new LinkedList();
        pChild->Init(this);
        pList->pNext = NULL;
        pList->pNode = pChild;
        if (m_ChildSize == 0)
        {
            m_pChildrenHead = pList;
            m_pChildrenTail = pList;
            m_pNextChild = pList;
        }
        else
        {
            m_pChildrenTail->pNext = pList;
            m_pChildrenTail = m_pChildrenTail->pNext;
        }
        m_ChildSize++;
    }

    /**
     * @brief   指定インデックスの子供ノードを返します。(基数は 0 )
     *          なければ NULL を返します。
     */
    TreeNode* GetChildren(size_t idx) NN_NOEXCEPT
    {
        if (idx >= m_ChildSize)
        {
            return NULL;
        }
        LinkedList* pCurrent = m_pChildrenHead;
        for (size_t i = 0; i < idx && pCurrent != NULL; ++i)
        {
            pCurrent = pCurrent->pNext;
        }
        if (pCurrent)
        {
            return pCurrent->pNode;
        }
        return NULL;
    }

    /**
     * @brief   自分の子供を順にたどります。
     *          子供がいるなら子供のノードを、いないなら NULL を返します。
     */
    TreeNode* GetNextChildren() NN_NOEXCEPT
    {
        if (!m_pNextChild)
        {
            return NULL;
        }
        TreeNode* ptr = m_pNextChild->pNode;
        m_pNextChild = m_pNextChild->pNext;
        return ptr;
    }

    /**
     * @brief   深さ優先で使うフラグをクリアします。
     *          芋づる式に以下の子すべてのフラグもクリアされます。
     */
    void ClearNext() NN_NOEXCEPT
    {
        m_pNextChild = m_pChildrenHead;
        LinkedList* pCurrent = m_pChildrenHead;
        while (pCurrent)
        {
            pCurrent->pNode->ClearNext();
            pCurrent = pCurrent->pNext;
        }
    }

    // ノード番号へのアクセサ
    void SetNodeNum(int n) NN_NOEXCEPT { m_Num = n; }
    int GetNodeNum() NN_NOEXCEPT { return m_Num; }

    TreeNode* GetParent() NN_NOEXCEPT { return m_pParent; }

private:
    TreeNode* m_pParent;                // 親のアドレス
    LinkedList* m_pChildrenHead;        // 子のノードの先頭
    LinkedList* m_pChildrenTail;        // 子のノードの末尾
    LinkedList* m_pNextChild;           // 次に見る子供
    size_t m_ChildSize;                 // 持っている子供の数
    int m_Num;                          // 自由に指定できる任意のノード番号

};

/**
 * @brief   Bp で作成する簡潔木の元となる木構造を作成するためのクラスです。
 */
class TreeIter
{
public:
    TreeIter() NN_NOEXCEPT : m_pRoot(NULL), m_pCurrent(NULL) {}
    explicit TreeIter(TreeNode* pRoot) NN_NOEXCEPT
    {
        Init(pRoot);
    }
    ~TreeIter() NN_NOEXCEPT
    {
        m_pRoot = NULL;
        m_pCurrent = NULL;
    }

    /**
     * @brief   根ノードを設定します
     */
    void Init(TreeNode* pRoot) NN_NOEXCEPT
    {
        m_pCurrent = m_pRoot = pRoot;
    }

    /**
     * @brief   現在見ているノードの子ノードを指定数分作成します。
     */
    void MakeChildren(size_t num) NN_NOEXCEPT
    {
        for (size_t i = 0; i < num; ++i)
        {
            m_pCurrent->AddChild();
        }
    }

    /**
     * @brief   指定インデックスの子に移動します。
     *          移動に成功したら true が返ります。
     */
    bool GotoChild(size_t idx) NN_NOEXCEPT
    {
        TreeNode* pNext = m_pCurrent->GetChildren(idx);
        if (pNext)
        {
            m_pCurrent = pNext;
            return true;
        }
        return false;
    }

    // ノード番号へのアクセサ
    void SetNodeNum(int n) NN_NOEXCEPT { m_pCurrent->SetNodeNum(n); }
    int GetNodeNum() NN_NOEXCEPT { return m_pCurrent->GetNodeNum(); }

    /**
     * @brief   親に移動します。
     *          自分が根ノードの場合は false を返します。
     */
    bool GotoParent() NN_NOEXCEPT
    {
        if (m_pCurrent == m_pRoot) { return false; }
        NN_ASSERT(m_pCurrent->GetParent());
        m_pCurrent = m_pCurrent->GetParent();
        return true;
    }

    /**
     * @brief   根に戻ります。
     *          各ノードの内部の探索状態もクリアします
     */
    void BackToRoot() NN_NOEXCEPT
    {
        m_pCurrent = m_pRoot;
        m_pCurrent->ClearNext();
    }

    /**
     * @brief   木構造を BP(BalancedParentheses) で出力します。
     */
    void Disp() NN_NOEXCEPT
    {
        TreeNode* pNext = m_pRoot;
        NN_LOG("(");
        while(pNext)
        {
            TreeNode* Tmp = pNext->GetNextChildren();
            if (Tmp)
            {
                NN_LOG("(");
                pNext = Tmp;
            }
            else
            {
                if (pNext == m_pRoot)
                {
                    break;
                }
                NN_LOG(")");
                pNext = pNext->GetParent();
            }
        }
        NN_LOG(")\n");
        BackToRoot();
    }

    // ----- 以下 Bp の作成に必要な関数 ----- //
    /**
     * @brief   ノードのポインタを返します。
     */
    const void* NodePtr() NN_NOEXCEPT
    {
        return m_pCurrent;
    }

    /**
     * @brief   親ノードのポインタを返します。
     */
    const void* ParentNodePtr() NN_NOEXCEPT
    {
        return m_pCurrent->GetParent();
    }

    /**
     * @brief   深さ優先で次のノードに移動します。
     *          移動に成功したら true が返ります。
     */
    bool MoveNext() NN_NOEXCEPT
    {
        do
        {
            TreeNode* pNext = m_pCurrent->GetNextChildren();
            if (pNext)
            {
                m_pCurrent = pNext;
                return true;
            }
        } while (GotoParent()); // すべての子供をたどっているなら、親に戻る
        return false;
    }

private:
    TreeNode* m_pRoot;          // 根ノードのポインタ
    TreeNode* m_pCurrent;       // 今見ているノードのポインタ
};

/**
 * @brief   テストに与えるパラメータです。
 */
enum BpTestParam
{
    BpTestParam_UseMalloc               = 0,    //!< デフォルトのアロケータ(std::malloc)を利用する
    BpTestParam_UseWorkBufAllocator             //!< nn::ngc::detail::WorkBufAllocator を利用する
};

/**
 * @brief   テストで利用するテストフィクスチャです。
 */
class SuccinctBpTest : public ::testing::TestWithParam<int>
{
protected:
    SuccinctBpTest() NN_NOEXCEPT : m_HeapSize(128 * 1024), m_HeapAddr(NULL), m_Bp(),
        m_pIter(NULL), m_pRoot(NULL), m_Allocator()
    {
        m_pIter = nullptr;
        m_pRoot = nullptr;
    }
    ~SuccinctBpTest() NN_NOEXCEPT
    {
        if (m_pIter)
        {
            delete m_pIter;
        }
        if (m_pRoot)
        {
            delete m_pRoot;
        }
    }

    /**
     * @brief   テスト用木構造を作成します
     */
    void MakeTestTree(int arraySize, int* pChildArray) NN_NOEXCEPT
    {
        if (m_pIter)
        {
            delete m_pIter;
        }
        m_pIter = new TreeIter;
        if (m_pRoot)
        {
            delete m_pRoot;
        }
        m_pRoot = new TreeNode;
        m_pRoot->Init(NULL);

        m_pIter->Init(m_pRoot);
        for (int i = 0; i < arraySize; ++i)
        {
            m_pIter->SetNodeNum(i);
            m_pIter->MakeChildren(pChildArray[i]);
            m_pIter->MoveNext();
        }
        m_pIter->BackToRoot();
    }

    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        if (GetParam() == BpTestParam_UseWorkBufAllocator)
        {
            m_HeapAddr = std::malloc(m_HeapSize);
            NN_ABORT_UNLESS(m_HeapAddr);
            m_Allocator.Initialize(m_HeapAddr, m_HeapSize);
            m_Bp.SetAllocator(&m_Allocator);
        }
    }
    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        if (GetParam() == BpTestParam_UseWorkBufAllocator)
        {
            m_Bp.ReleaseAllocator();
            m_Allocator.Finalize();
            std::free(m_HeapAddr);
        }
    }

protected:
    const size_t m_HeapSize;
    void* m_HeapAddr;                               // ワークバッファ用メモリ
    nn::ngc::detail::Bp m_Bp;
    TreeIter* m_pIter;
    TreeNode* m_pRoot;                              // ツリー構造の根
    nn::ngc::detail::WorkBufAllocator m_Allocator;  // ワークバッファ用アロケータ
};

}   // namespace

TEST_P(SuccinctBpTest, Simple)
{
    const int arraySize = 4;
    // 何番のノードがどれだけの子供を持つかの配列
    // 要素 : index 番目のノードが持つ子(基数は 0 )
    // 以下のような Bp ができる
    //((())())
    int pChildArray[arraySize] = { 2, 1, 0, 0 };

    const int nodeIdAns[arraySize * 2 + 1] = { 0, 1, 2, 2, 1, 3, 3, 0, -1 };
    const int posAns[arraySize + 1] = { 0, 1, 2, 5, -1};
    const int openAns[arraySize * 2 + 1] = { 0, 1, 2, 2, 1, 5, 5, 0, -1};
    const int closeAns[arraySize * 2 + 1] = { 7, 4, 3, 3, 4, 6, 6, 7, -1};
    const int encloseAns[arraySize * 2 + 1] = { -1, 0, 1, 1, 0, 0, 0, -1, -1};
    const int firstAns[arraySize * 2 + 1] = { 1, 2, -1, -1, 2, -1, -1, 1, -1};
    const int lastAns[arraySize * 2 + 1] = { 6, 3, -1, -1, 3, -1, -1, 6, -1 };
    const int nextAns[arraySize * 2 + 1] = { -1, 5, -1, -1, 5, -1, -1, -1, -1 };
    const int prevAns[arraySize * 2 + 1] = { -1, -1, -1, -1, -1, 4, 4, -1, -1 };

    MakeTestTree(arraySize, pChildArray);
    NN_LOG("BP: ");
    m_pIter->Disp();

    m_Bp.Init(*m_pIter, arraySize);

    for (int i = 0; i < arraySize * 2 + 1; ++i)
    {
        if (i < arraySize + 1)
        {
            EXPECT_EQ(m_Bp.ToPos(i), posAns[i]);
        }
        EXPECT_EQ(m_Bp.ToNodeId(i), nodeIdAns[i]);
        EXPECT_EQ(m_Bp.FindOpen(i), openAns[i]);
        EXPECT_EQ(m_Bp.FindClose(i), closeAns[i]);
        EXPECT_EQ(m_Bp.Enclose(i), encloseAns[i]);
        EXPECT_EQ(m_Bp.FirstChild(i), firstAns[i]);
        EXPECT_EQ(m_Bp.LastChild(i), lastAns[i]);
        EXPECT_EQ(m_Bp.NextSibling(i), nextAns[i]);
        EXPECT_EQ(m_Bp.PrevSibling(i), prevAns[i]);
    }
}

TEST_P(SuccinctBpTest, Simple2)
{
    const int arraySize = 32;
    // 以下のような Bp ができる
    // ((((()))(()((()())())))((()(())()))(()()((())(()(())()))(())()))
    int pChildArray[arraySize] = { 3, 2, 1, 1, 0, 2, 0, 2, 2, 0,
                                   0, 0, 1, 3, 0, 1, 0, 0, 5, 0,
                                   0, 2, 1, 0, 3, 0, 1, 0, 0, 1,
                                   0, 0 };

    MakeTestTree(arraySize, pChildArray);
    NN_LOG("BP: ");
    m_pIter->Disp();

    m_Bp.Init(*m_pIter, arraySize);

    auto DoSmokeTest = [&](nn::ngc::detail::Bp& bp)
    {
        EXPECT_EQ(bp.ToNodeId(9), 6);
        EXPECT_EQ(bp.ToPos(12), 23);
        EXPECT_EQ(bp.FindOpen(17), 12);
        EXPECT_EQ(bp.FindClose(0), 63);
        EXPECT_EQ(bp.Enclose(32), 24);
        EXPECT_EQ(bp.FirstChild(8), 9);
        EXPECT_EQ(bp.LastChild(62), 61);
        EXPECT_EQ(bp.NextSibling(5), -1);
        EXPECT_EQ(bp.PrevSibling(31), 30);
    };

    DoSmokeTest(m_Bp);

    // ムーブの確認
    nn::ngc::detail::Bp bp2(std::move(m_Bp));
    DoSmokeTest(bp2);
    m_Bp = std::move(bp2);
    DoSmokeTest(m_Bp);

    // Reset() の確認
    int pArray[3] = { 2, 0, 0 };
    MakeTestTree(3, pArray);
    m_Bp.Reset();
    m_Bp.Init(*m_pIter, 3);
    EXPECT_EQ(m_Bp.FindOpen(2), 1);
}

INSTANTIATE_TEST_CASE_P(SwitchAllocator,
                        SuccinctBpTest,
                        testing::Values(BpTestParam_UseMalloc,
                                        BpTestParam_UseWorkBufAllocator));
