﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
  ■閲覧上の注意■

  このファイルには試験目的のため好ましくない文字列が含まれています。
  あらかじめご了承ください。
 */

// テスト用のヘッダ
#include <cstdlib>
#include <memory>

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_ErrorResult.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_CharacterEncoding.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/system/settings_Language.h>

// テスト対象のヘッダ
#include <nn/ngc.h>

#define RoundUp(x, align)   (((u32)(x) + (align) - 1) & ~((align) - 1))

//#define DISABLE_TESTS

namespace {
    char g_TestMemory[nn::ngc::ProfanityFilter::WorkMemorySize];
    void* g_pTestMemoryAddr = g_TestMemory;

    // テストフィクスチャ
    template<class T>
    class NgcApiTest : public ::testing::Test
    {
    protected:
        NgcApiTest() NN_NOEXCEPT {}

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE {
            //テスト用に言語設定を日本語に変更
            nn::settings::GetLanguageCode(&m_Language);
            const auto code = nn::settings::LanguageCode::Make(nn::settings::Language_Japanese);
            nn::settings::system::SetLanguageCode(code);
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE {
            //言語設定を戻す
            nn::settings::system::SetLanguageCode(m_Language);
        }
    private:
        nn::settings::LanguageCode m_Language;
    };

    // 型リストの定義と紐づけ
    typedef ::testing::Types<uint16_t, char16_t> Utf16Types;
    TYPED_TEST_CASE(NgcApiTest, Utf16Types);

}   // unnamed namespace

#if !defined(DISABLE_TESTS)
/*!--------------------------------------------------------------------------*
  Name:         InitializeTest

  @brief        Initialize を繰り返すテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, InitializeTest)
{
    const int InstanceNum = 256;
    nn::ngc::ProfanityFilter filter[InstanceNum];

    // 単一インスタンスのチェック
    for(int i = 0; i < 256; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(
                filter[0].Initialize(g_pTestMemoryAddr,nn::ngc::ProfanityFilter::WorkMemorySize));
        filter[0].Finalize();
    }

    // 複数インスタンスのチェック
    for(int i = 0; i < InstanceNum; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(
                filter[i].Initialize(g_pTestMemoryAddr,nn::ngc::ProfanityFilter::WorkMemorySize));
    }

    // FSチェック
    for(int i = 0; i < InstanceNum; i++)
    {
        NN_ASSERT(filter[i].GetContentVersion() != 0);
        filter[i].Finalize();
    }

}

/*!--------------------------------------------------------------------------*
  Name:         NotInitializedWordTest

  @brief        初期化しないで呼び出したら怒られるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, NotInitializedWordTest)
{
    nn::ngc::ProfanityFilter filter;

    const char* ppWords[] =
    {
        "mario"
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    nn::Result result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_FAILURE( nn::ngc::ResultNotInitialized ,result );
}

/*!--------------------------------------------------------------------------*
  Name:         InvalidArgumentWordTest1

  @brief        引数がおかしい状態で呼び出すテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, InvalidArgument_Word_Test1)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
        "mario"
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);

    result = filter.CheckProfanityWords( nullptr, 0, ppWords, WordCount );
    NNT_ASSERT_RESULT_FAILURE( nn::ngc::ResultInvalidPointer,result );
}

/*!--------------------------------------------------------------------------*
  Name:         InvalidArgumentWordTest2

  @brief        引数がおかしい状態で呼び出すテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, InvalidArgumentWordTest2)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    nn::Bit32 results[ 1 ];
    const char** ppWords = nullptr;

    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, 1 );
    NNT_ASSERT_RESULT_FAILURE( nn::ngc::ResultInvalidPointer,result );
}

/*!--------------------------------------------------------------------------*
  Name:         InvalidArgumentWordTest3

  @brief        引数がおかしい状態で呼び出すテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, InvalidArgumentWordTest3)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
        "mario00",
        "mario01",
        "mario02",
        "mario03",
        "mario04",
        "mario05",
        "mario06",
        "mario07",
        "mario08",
        "mario09",
        "mario10",
        "mario11",
        "mario12",
        "mario13",
        "mario14",
        "mario15",
        "mario16",
        "mario17",
        "mario18",
        "mario19",
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];
    NN_STATIC_ASSERT(nn::ngc::WordCountMax < WordCount);

    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_FAILURE( nn::ngc::ResultInvalidSize,result );
}

/*!--------------------------------------------------------------------------*
  Name:         SingleWordTest

  @brief        １つの単語でテストを行います。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, SingleWordTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
        "mario"
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         SingleWordMaxLengthTest

  @brief        １つの単語の最大文字数でテストを行います。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, SingleWordMaxLengthTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
        "123456789012345678901234567890123456789012345678901234warumoshi"
    };
    ASSERT_TRUE( nn::ngc::WordLengthMax == std::strlen(ppWords[0]) + 1 );
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         MultiwordTest

  @brief        複数の単語を同時にチェックするテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, MultiwordTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
        "mario",
        "luigi",
        "badword",
// windows版でビルドするとShift_JIS文字扱いとなるためUtf-8のバイト配列で定義する。
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "クッパ",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\xaf\xe3\x83\x83\xe3\x83\x91"      //  "クッパ",
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] == 0 );
    ASSERT_TRUE( results[ 2 ] != 0 );
    ASSERT_TRUE( results[ 3 ] == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         PatternListTest1

  @brief        特定のリージョンのビットが立っているテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, PatternListTest1)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "ちんこ"
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x81\xa1\xe3\x82\x93\xe3\x81\x93"  // "ちんこ"
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_Japanese ) )
                 == ( nn::ngc::ProfanityFilterPatternList_Japanese ) );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_AmericanEnglish ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_CanadianFrench ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_LatinAmericanSpanish ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_TraditionalChinese) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_Dutch ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_BritishEnglish ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_French ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_German ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_Italian ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_Spanish ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_Korean ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_Portuguese ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_Russian ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_SouthAmericanPortuguese ) ) == 0 );
    ASSERT_TRUE( ( results[ 0 ] & ( nn::ngc::ProfanityFilterPatternList_SimplifiedChinese) ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         PatternListTest2

  @brief        特定のパターンのみに対してテストをかけるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, PatternListTest2)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "ちんこ"
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x81\xa1\xe3\x82\x93\xe3\x81\x93"  // "ちんこ"
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results,
                                         nn::ngc::ProfanityFilterPatternList_Japanese,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         PatternListTest3

  @brief        共通パターンに不正文字が含まれる場合、
                本体言語に対応するパターンと入力したパターンが設定されるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, PatternListTest3)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
            "badworda",
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];
    nn::Bit32 testPattern;
    nn::Bit32 expectPattern;

    // 追加指定パターンなし(本体設定に対応するパターンのみ設定される)
    testPattern = 0;
    expectPattern = nn::ngc::ProfanityFilterPatternList_Japanese;
    result = filter.CheckProfanityWords( results,
                                         testPattern,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_EQ( results[ 0 ] , expectPattern );

    // 追加指定パターンあり(本体設定に対応するパターンと入力パターンが設定される)
    testPattern = 0x55555555;
    expectPattern = nn::ngc::ProfanityFilterPatternList_Japanese | testPattern;
    expectPattern &= (1 << nn::ngc::ProfanityFilterPatternList_Max) - 1;

    result = filter.CheckProfanityWords( results,
                                         testPattern,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_EQ( results[ 0 ] , expectPattern );

    // 追加指定パターンあり(本体設定に対応するパターンと入力パターンが設定される)
    testPattern = 0xaaaaaaaa;
    expectPattern = nn::ngc::ProfanityFilterPatternList_Japanese | testPattern;
    expectPattern &= (1 << nn::ngc::ProfanityFilterPatternList_Max) - 1;

    result = filter.CheckProfanityWords( results,
                                         testPattern,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_EQ( results[ 0 ] , expectPattern );

    // 全指定
    testPattern = 0xffffffff;
    expectPattern = testPattern;
    expectPattern &= (1 << nn::ngc::ProfanityFilterPatternList_Max) - 1;
    result = filter.CheckProfanityWords( results,
                                         testPattern,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_EQ( results[ 0 ] , expectPattern );
}

/*!--------------------------------------------------------------------------*
  Name:         PatternListTest4

  @brief        特定のパターンのみに対してテストをかけるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, PatternListTest4)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
        nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 10.txt (Korean) にある ^18년.* が正しくマッチすることをテスト
    const char* ppWords1[] = {
        "\x31\x38\xEB\x85\x84",                  // 18년    … マッチする
        "\x31\x38\xEB\x85\x84\x61\x61\x61",      // 18년aaa … マッチする
        "\x61\x61\x61\x31\x38\xEB\x85\x84",      // aaa18년 … マッチしない
    };
    nn::Bit32 results;
    nn::Bit32 testPattern;

    auto DoTest = [&](const char** ppWords, nn::Bit32 expect)
    {
        result = filter.CheckProfanityWords(&results,
            testPattern,
            ppWords,
            1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(results, expect);
        results = 0;
    };

    testPattern = nn::ngc::ProfanityFilterPatternList_Korean;

    DoTest(&ppWords1[0], nn::ngc::ProfanityFilterPatternList_Korean);
    DoTest(&ppWords1[1], nn::ngc::ProfanityFilterPatternList_Korean);
    DoTest(&ppWords1[2], 0);
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertTargetWordTest1

  @brief        文字の変換テストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, ConvertTargetWordTest1)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "チンコ",
        "ﾁﾝｺ",
        "チんｺ",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x83\x81\xe3\x83\xb3\xe3\x82\xb3", //  "チンコ",
        "\xef\xbe\x81\xef\xbe\x9d\xef\xbd\xba", //  "ﾁﾝｺ",
        "\xe3\x83\x81\xe3\x82\x93\xef\xbd\xba"  //  "チんｺ",
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results,
                                         nn::ngc::ProfanityFilterPatternList_Japanese,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] != 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
    ASSERT_TRUE( results[ 2 ] != 0 );
}


/*!--------------------------------------------------------------------------*
  Name:         ConvertTargetWordTest2

  @brief        文字の変換テストです(半角カタカナ＋濁音)
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, ConvertTargetWordTest2)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "ｺﾝﾄﾞｰﾑ",
        "ﾀﾞｯﾁﾜｲﾌ",
        "ｴﾌｪﾄﾞﾘﾝ"
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xef\xbd\xba\xef\xbe\x9d\xef\xbe\x84\xef\xbe\x9e\xef\xbd\xb0\xef\xbe\x91",              //  "ｺﾝﾄﾞｰﾑ",
        "\xef\xbe\x80\xef\xbe\x9e\xef\xbd\xaf\xef\xbe\x81\xef\xbe\x9c\xef\xbd\xb2\xef\xbe\x8c",  //  "ﾀﾞｯﾁﾜｲﾌ",
        "\xef\xbd\xb4\xef\xbe\x8c\xef\xbd\xaa\xef\xbe\x84\xef\xbe\x9e\xef\xbe\x98\xef\xbe\x9d"   //  "ｴﾌｪﾄﾞﾘﾝ"
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results,
                                         nn::ngc::ProfanityFilterPatternList_Japanese,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] != 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
    ASSERT_TRUE( results[ 2 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertTargetWordTest3

  @brief        文字の変換テストです(半角カタカナ＋半濁音)
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, ConvertTargetWordTest3)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "ﾌﾟｯｼｰ",
        "ﾚｲﾌﾟ",
        "ﾏｯﾄﾌﾟﾚｲ"
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xef\xbe\x8c\xef\xbe\x9f\xef\xbd\xaf\xef\xbd\xbc\xef\xbd\xb0",                          //  "ﾌﾟｯｼｰ",
        "\xef\xbe\x9a\xef\xbd\xb2\xef\xbe\x8c\xef\xbe\x9f",                                      //  "ﾚｲﾌﾟ",
        "\xef\xbe\x8f\xef\xbd\xaf\xef\xbe\x84\xef\xbe\x8c\xef\xbe\x9f\xef\xbe\x9a\xef\xbd\xb2"   //  "ﾏｯﾄﾌﾟﾚｲ"
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results,
                                         nn::ngc::ProfanityFilterPatternList_Japanese,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] != 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
    ASSERT_TRUE( results[ 2 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertTargetWordTest4

  @brief        文字の変換テストです(小さな文字を変換できるかテスト)
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, ConvertTargetWordTest4)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "フェラチオ",
        "ﾌｪﾗﾁｵ",
        "フエラチオ",
        "ふぇらちお",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x83\x95\xe3\x82\xa7\xe3\x83\xa9\xe3\x83\x81\xe3\x82\xaa",  //  "フェラチオ",
        "\xef\xbe\x8c\xef\xbd\xaa\xef\xbe\x97\xef\xbe\x81\xef\xbd\xb5",  //  "ﾌｪﾗﾁｵ",
        "\xe3\x83\x95\xe3\x82\xa8\xe3\x83\xa9\xe3\x83\x81\xe3\x82\xaa",  //  "フエラチオ",
        "\xe3\x81\xb5\xe3\x81\x87\xe3\x82\x89\xe3\x81\xa1\xe3\x81\x8a"   //  "ふぇらちお",
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results,
                                         nn::ngc::ProfanityFilterPatternList_Japanese,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] != 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
    ASSERT_TRUE( results[ 2 ] != 0 );
    ASSERT_TRUE( results[ 3 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertTargetWordTest5

  @brief        文字の変換テストです(全角英数を半角英数に変換)
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, ConvertTargetWordTest5)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
        "badwordj",
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "ｂａｄｗｏｒｄｊ",
        "BADWORDJ",
        "ＢＡＤＷＯＲＤＪ",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a", // "ｂａｄｗｏｒｄｊ",
        "BADWORDJ",
        "\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa"  // "ＢＡＤＷＯＲＤＪ",
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results,
                                         nn::ngc::ProfanityFilterPatternList_Japanese,
                                         ppWords,
                                         WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] != 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
    ASSERT_TRUE( results[ 2 ] != 0 );
    ASSERT_TRUE( results[ 3 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertTargetWordTest6

  @brief        文字の変換テストです(半角スペースを削除)
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, ConvertTargetWordTest6)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "ちんこ",
        "ち んこ",
        "ちん こ",
        "ち ん こ",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x81\xa1\xe3\x82\x93\xe3\x81\x93",            //  "ちんこ",
        "\xe3\x81\xa1\x20\xe3\x82\x93\xe3\x81\x93",        //  "ち んこ",
        "\xe3\x81\xa1\xe3\x82\x93\x20\xe3\x81\x93",        //  "ちん こ",
        "\xe3\x81\xa1\x20\xe3\x82\x93\x20\xe3\x81\x93"     //  "ち ん こ",
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppWords,
                                        WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] != 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
    ASSERT_TRUE( results[ 2 ] != 0 );
    ASSERT_TRUE( results[ 3 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertTargetWordTest7

  @brief        文字の変換テストです(英語以外のアルファベットを含む)
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, ConvertTargetWordTest7)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    // ロシア語
    const char* ppWords1[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "хуй",
        "хуЙ",
        "хУй",
        "хУЙ",
        "Хуй",
        "ХуЙ",
        "ХУй",
        "ХУЙ",
        "х уй",
        "ху Й",
        "х У й",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xd1\x85\xd1\x83\xd0\xb9",         //  "хуй",
        "\xd1\x85\xd1\x83\xd0\x99",         //  "хуЙ",
        "\xd1\x85\xd0\xA3\xd0\xb9",         //  "хУй",
        "\xd1\x85\xd0\xA3\xd0\x99",         //  "хУЙ",
        "\xd0\xa5\xd1\x83\xd0\xb9",         //  "Хуй",
        "\xd0\xa5\xd1\x83\xd0\x99",         //  "ХуЙ",
        "\xd0\xa5\xd0\xA3\xd0\xb9",         //  "ХУй",
        "\xd0\xa5\xd0\xA3\xd0\x99",         //  "ХУЙ",
        "\xd1\x85\x20\xd1\x83\xd0\xb9",     //  "х уй",
        "\xd1\x85\xd1\x83\x20\xd0\x99",     //  "ху Й",
        "\xd1\x85\x20\xd0\xA3\x20\xd0\xb9", //  "х У й",
#else
#error "unsupported os"
#endif
    };

    // フランス語
    const char* ppWords2[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "nègre",
        "nÈgre",
        "NèGRE",
        "NÈGRE",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x6e\xc3\xa8\x67\x72\x65",         //  "nègre",
        "\x6e\xc3\x88\x67\x72\x65",         //  "nÈgre",
        "\x4e\xc3\xa8\x47\x52\x45",         //  "NèGRE",
        "\x4e\xc3\x88\x47\x52\x45",         //  "NÈGRE",
#else
#error "unsupported os"
#endif
    };

    // スペイン語
    const char* ppWords3[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "coño",
        "coÑo",
        "COñO",
        "COÑO",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x63\x6f\xc3\xb1\x6f", //  "coño",
        "\x63\x6f\xc3\x91\x6f", //  "coÑo",
        "\x43\x4f\xc3\xb1\x4f", //  "COñO",
        "\x43\x4f\xc3\x91\x4f", //  "COÑO",
#else
#error "unsupported os"
#endif
    };

    // ドイツ語
    const char* ppWords4[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "säufer",
        "sÄufer",
        "SäUFER",
        "SÄUFER",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x73\xc3\xa4\x75\x66\x65\x72", //  "säufer",
        "\x73\xc3\x84\x75\x66\x65\x72", //  "sÄufer",
        "\x53\xc3\xa4\x55\x46\x45\x52", //  "SäUFER",
        "\x53\xc3\x84\x55\x46\x45\x52", //  "SÄUFER",
#else
#error "unsupported os"
#endif
    };

    // イタリア語
    const char* ppWords5[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "gesùzozzo",
        "gesÙzozzo",
        "GESùZOZZO",
        "GESÙZOZZO",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x67\x65\x73\xc3\xb9\x7a\x6f\x7a\x7a\x6f", //  "gesùzozzo",
        "\x67\x65\x73\xc3\x99\x7a\x6f\x7a\x7a\x6f", //  "gesÙzozzo",
        "\x47\x45\x53\xc3\xb9\x5a\x4f\x5a\x5a\x4f", //  "GESùZOZZO",
        "\x47\x45\x53\xc3\x99\x5a\x4f\x5a\x5a\x4f", //  "GESÙZOZZO",
#else
#error "unsupported os"
#endif
    };

    // ポルトガル語
    const char* ppWords6[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "piço",
        "piÇo",
        "PIçO",
        "PIÇO",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x70\x69\xc3\xa7\x6f", //  "piço",
        "\x70\x69\xc3\x87\x6f", //  "piÇo",
        "\x50\x49\xc3\xa7\x4f", //  "PIçO",
        "\x50\x49\xc3\x87\x4f", //  "PIÇO",
#else
#error "unsupported os"
#endif
    };

    auto DoTest = [&](nn::Bit32 patterns, const char** ppWords, const int wordCount)
    {
        std::unique_ptr<nn::Bit32> results(new nn::Bit32[wordCount]);

        result = filter.CheckProfanityWords(results.get(),
                                            patterns,
                                            ppWords,
                                            wordCount);
        NNT_ASSERT_RESULT_SUCCESS(result);

        for (int i = 0; i < wordCount; ++i)
        {
            EXPECT_NE(results.get()[i], 0u) << "i: " << i;
            results.get()[i] = 0;
        }
    };

    DoTest(nn::ngc::ProfanityFilterPatternList_Russian,
           ppWords1, sizeof(ppWords1) / sizeof(ppWords1[0]));
    DoTest(nn::ngc::ProfanityFilterPatternList_CanadianFrench,
           ppWords2, sizeof(ppWords2) / sizeof(ppWords2[0]));
    DoTest(nn::ngc::ProfanityFilterPatternList_French,
           ppWords2, sizeof(ppWords2) / sizeof(ppWords2[0]));
    DoTest(nn::ngc::ProfanityFilterPatternList_LatinAmericanSpanish,
           ppWords3, sizeof(ppWords3) / sizeof(ppWords3[0]));
    DoTest(nn::ngc::ProfanityFilterPatternList_Spanish,
           ppWords3, sizeof(ppWords3) / sizeof(ppWords3[0]));
    DoTest(nn::ngc::ProfanityFilterPatternList_German,
           ppWords4, sizeof(ppWords4) / sizeof(ppWords4[0]));
    DoTest(nn::ngc::ProfanityFilterPatternList_Italian,
           ppWords5, sizeof(ppWords5) / sizeof(ppWords5[0]));
    DoTest(nn::ngc::ProfanityFilterPatternList_Portuguese,
           ppWords6, sizeof(ppWords6) / sizeof(ppWords6[0]));
}   // NOLINT(impl/function_size)

/*!--------------------------------------------------------------------------*
  Name:         AtSignNoSkipTest()

  @brief        ＠記号が含まれている場合は常に問題のある語句とするテストです
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, AtSignNoSkipTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "@aaa",
        "＠あいう",
        "hoge@hogehoge.jp",
        "hoge",
        "あいう@"
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "@aaa",
        "\xef\xbc\xa0\xe3\x81\x82\xe3\x81\x84\xe3\x81\x86", // "＠あいう",
        "hoge@hogehoge.jp",
        "hoge",
        "\xe3\x81\x82\xe3\x81\x84\xe3\x81\x86\x40"          //  "あいう@"
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppWords,
                                        WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] != 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
    ASSERT_TRUE( results[ 2 ] != 0 );
    ASSERT_TRUE( results[ 3 ] == 0 );   // アットマークは含まれない
    ASSERT_TRUE( results[ 4 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         AtSignSkipTest()

  @brief        ＠記号が含まれていても、スキップ設定するとすべて問題ないことになるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, AtSignSkipTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    filter.SkipAtSignCheck( nn::ngc::ProfanityFilter::SkipMode_SkipAtSign );
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppWords[] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "@aaa",
        "＠あいう",
        "hoge@hogehoge.jp",
        "hoge",
        "あいう@",
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "@aaa",
        "\xef\xbc\xa0\xe3\x81\x82\xe3\x81\x84\xe3\x81\x86", //  "＠あいう",
        "hoge@hogehoge.jp",
        "hoge",
        "\xe3\x81\x82\xe3\x81\x84\xe3\x81\x86\x40"          //  "あいう@",
#else
#error "unsupported os"
#endif
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppWords,
                                        WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    // 答え合わせ(スキップ設定がオンなので、すべて問題ない語句となります)
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] == 0 );
    ASSERT_TRUE( results[ 2 ] == 0 );
    ASSERT_TRUE( results[ 3 ] == 0 );
    ASSERT_TRUE( results[ 4 ] == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         NumberCountTest

  @brief        数字の計上テスト
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, NumberCountTest)
{
    ASSERT_TRUE( 0 == nn::ngc::CountNumbers( "" ) );
    ASSERT_TRUE( 0 == nn::ngc::CountNumbers( "aaa" ) );
    ASSERT_TRUE( 0 == nn::ngc::CountNumbers( "abc" ) );
    ASSERT_TRUE( 1 == nn::ngc::CountNumbers( "1" ) );
    ASSERT_TRUE( 1 == nn::ngc::CountNumbers( "1aaa" ) );
    ASSERT_TRUE( 1 == nn::ngc::CountNumbers( "aaa1" ) );
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    ASSERT_TRUE( 2 == nn::ngc::CountNumbers( "８aaa１" ) );
    ASSERT_TRUE( 3 == nn::ngc::CountNumbers( "８a⑦a１" ) );
    ASSERT_TRUE( 4 == nn::ngc::CountNumbers( "８a⑦a１ⅴ" ) );
    ASSERT_TRUE( 5 == nn::ngc::CountNumbers( "ⅱa⑦a１ⅴⅣ" ) );
    ASSERT_TRUE( 2 == nn::ngc::CountNumbers( "ⅻⅫ" ) );
    ASSERT_TRUE( 3 == nn::ngc::CountNumbers( "➄➍➒" ) );
    ASSERT_TRUE( 5 == nn::ngc::CountNumbers( "❶❷❻❽❿" ) );
    ASSERT_TRUE( 4 == nn::ngc::CountNumbers( "⒛⒕⒏⒈" ) );
    ASSERT_TRUE(10 == nn::ngc::CountNumbers( "⒇⒆⒄⒀⑸⑽⑺⑿⑴⑵" ) );
    ASSERT_TRUE(10 == nn::ngc::CountNumbers( "⑳①⑱④⑤⑥⑩⑫⑭⑯" ) );
    ASSERT_TRUE(10 == nn::ngc::CountNumbers( "0000000000" ) );
    ASSERT_TRUE( 5 == nn::ngc::CountNumbers( "¹aaa²b³c⓿⓪" ) );
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    ASSERT_TRUE( 2 == nn::ngc::CountNumbers( "\xef\xbc\x98\x61\x61\x61\xef\xbc\x91" ) );                                    //  "８aaa１"
    ASSERT_TRUE( 3 == nn::ngc::CountNumbers( "\xef\xbc\x98\x61\xe2\x91\xa6\x61\xef\xbc\x91" ) );                            //  "８a⑦a１"
    ASSERT_TRUE( 4 == nn::ngc::CountNumbers( "\xef\xbc\x98\x61\xe2\x91\xa6\x61\xef\xbc\x91\xe2\x85\xb4" ) );                //  "８a⑦a１ⅴ"
    ASSERT_TRUE( 5 == nn::ngc::CountNumbers( "\xe2\x85\xb1\x61\xe2\x91\xa6\x61\xef\xbc\x91\xe2\x85\xb4\xe2\x85\xa3" ) );    //  "ⅱa⑦a１ⅴⅣ"
    ASSERT_TRUE( 2 == nn::ngc::CountNumbers( "\xe2\x85\xbb\xe2\x85\xab" ) );                                                //  "ⅻⅫ"
    ASSERT_TRUE( 3 == nn::ngc::CountNumbers( "\xe2\x9e\x84\xe2\x9e\x8d\xe2\x9e\x92" ) );                                    //  "➄➍➒"
    ASSERT_TRUE( 5 == nn::ngc::CountNumbers( "\xe2\x9d\xb6\xe2\x9d\xb7\xe2\x9d\xbb\xe2\x9d\xbd\xe2\x9d\xbf" ) );            //  "❶❷❻❽❿"
    ASSERT_TRUE( 4 == nn::ngc::CountNumbers( "\xe2\x92\x9b\xe2\x92\x95\xe2\x92\x8f\xe2\x92\x88" ) );                        //  "⒛⒕⒏⒈"
    ASSERT_TRUE(10 == nn::ngc::CountNumbers( "\xe2\x92\x87\xe2\x92\x86\xe2\x92\x84\xe2\x92\x80\xe2\x91\xb8\xe2\x91\xbd\xe2\x91\xba\xe2\x91\xbf\xe2\x91\xb4\xe2\x91\xb5" ) );   //  "⒇⒆⒄⒀⑸⑽⑺⑿⑴⑵"
    ASSERT_TRUE(10 == nn::ngc::CountNumbers( "\xe2\x91\xb3\xe2\x91\xa0\xe2\x91\xb1\xe2\x91\xa3\xe2\x91\xa4\xe2\x91\xa5\xe2\x91\xa9\xe2\x91\xab\xe2\x91\xad\xe2\x91\xaf" ) );   //  "⑳①⑱④⑤⑥⑩⑫⑭⑯"
    ASSERT_TRUE(10 == nn::ngc::CountNumbers( "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30" ) );                               //  "0000000000"
    ASSERT_TRUE( 5 == nn::ngc::CountNumbers( "\xc2\xb9\x61\x61\x61\xc2\xb2\x62\xc2\xb3\x63\xe2\x93\xbf\xe2\x93\xaa" ) );    //  "¹aaa²b³c⓿⓪"
#else
#error "unsupported os"
#endif

}

/*!--------------------------------------------------------------------------*
   Name:         GetVersionTest

  @brief        NGワードフィルタのファイルバージョンを取得するテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, GetVersionTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    uint32_t version = filter.GetContentVersion();
    NN_LOG("Version = %d\n", version);
    ASSERT_TRUE( version != 0 );
}


/*!--------------------------------------------------------------------------*
  Name:         FukudaTest()

  @brief        リストの問題でFUKUDAがNGワードとして認識されてしまうのが修正されているか確認
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, FukudaTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppFukuda[]   = { "FUKUDA" };
    const char* ppFuk[]      = { "FUK" };
    nn::Bit32 results[ 1 ];

    // 日本語のリスト
    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppFukuda,
                                        sizeof(ppFukuda) / sizeof(ppFukuda[0]) );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] == 0 );  // FUKUDAは問題ない

    // 北米英語のリスト
    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_AmericanEnglish,
                                        ppFukuda,
                                        sizeof(ppFukuda) / sizeof(ppFukuda[0]) );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] == 0 );  // FUKUDAは問題ない
    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_AmericanEnglish,
                                        ppFuk,
                                        sizeof(ppFuk) / sizeof(ppFuk[0]) );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] != 0 );  // FUKは問題あり
}

/*!--------------------------------------------------------------------------*
  Name:         FellaTest()

  @brief        リストの問題で"カフェラテ"等が NG になっていたものが修正されているか確認
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, FellaTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const char* ppCaffeLatte[]       = { "カフェラテ" };
    const char* ppFerragamo[]        = { "フェラガモ" };
    const char* ppFerrari[]          = { "フェラーリ" };
    const char* ppFella[]            = { "フェラチオ" };
    const char* ppTestFellatioTest[] = { "テストフェラチオテスト" };
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    const char* ppCaffeLatte[]       = { "\xe3\x82\xab\xe3\x83\x95\xe3\x82\xa7\xe3\x83\xa9\xe3\x83\x86" }; //  "カフェラテ"
    const char* ppFerragamo[]        = { "\xe3\x83\x95\xe3\x82\xa7\xe3\x83\xa9\xe3\x82\xac\xe3\x83\xa2" }; //  "フェラガモ"
    const char* ppFerrari[]          = { "\xe3\x83\x95\xe3\x82\xa7\xe3\x83\xa9\xe3\x83\xbc\xe3\x83\xaa" }; //  "フェラーリ"
    const char* ppFella[]            = { "\xe3\x83\x95\xe3\x82\xa7\xe3\x83\xa9\xe3\x83\x81\xe3\x82\xaa" }; //  "フェラチオ"
    const char* ppTestFellatioTest[] = { "\xe3\x83\x86\xe3\x82\xb9\xe3\x83\x88\xe3\x83\x95\xe3\x82\xa7\xe3\x83\xa9\xe3\x83\x81\xe3\x82\xaa\xe3\x83\x86\xe3\x82\xb9\xe3\x83\x88" }; //  "テストフェラチオテスト"
#else
#error "unsupported os"
#endif
    nn::Bit32 results[1];

    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppCaffeLatte,
                                        sizeof(ppCaffeLatte) / sizeof(ppCaffeLatte[0]));
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] == 0 );  // "カフェラテ"は問題ない
    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppFerragamo,
                                        sizeof(ppFerragamo) / sizeof(ppFerragamo[0]));
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] == 0 );  // "フェラガモ"は問題ない
    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppFerrari,
                                        sizeof(ppFerrari) / sizeof(ppFerrari[0]));
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] == 0 );  // "フェラーリ"は問題ない
    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppFella,
                                        sizeof(ppFella) / sizeof(ppFella[0]));
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] != 0 );  // "フェラ"は問題あり
    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppTestFellatioTest,
                                        sizeof(ppTestFellatioTest) / sizeof(ppTestFellatioTest[0]));
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] != 0 );  // "テストフェラチオテスト"は問題あり
}

/*!--------------------------------------------------------------------------*
  Name:         MelonTest()

  @brief        リストの問題で Melon がNGワードとして認識されてしまうのが修正されているか確認
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, MelonTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppMelon[]   = { "melon" };
    const int WordCount = sizeof(ppMelon) / sizeof(ppMelon[0]);
    nn::Bit32 results[ WordCount ];

    // 日本語のリスト
    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppMelon,
                                        WordCount);
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] == 0 );  // "Melon"は問題ない

    // 北米英語のリスト
    result = filter.CheckProfanityWords( results,
                                        nn::ngc::ProfanityFilterPatternList_AmericanEnglish,
                                        ppMelon,
                                        WordCount);
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] == 0 );  // "Melon"は問題ない
}

/*!--------------------------------------------------------------------------*
  Name:         BadWordTest()

  @brief        badwordはすべてのリストで禁止されているかどうかの確認です。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, BadWordTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const char* ppBadword[] = { "badword" };
    const int WordCount = sizeof(ppBadword) / sizeof(ppBadword[0]);
    nn::Bit32 results[ WordCount ];

    for ( int i = 0; i < nn::ngc::ProfanityFilterPatternList_Max; i++ )
    {
        result = filter.CheckProfanityWords
        (
            results,
            0x01 << i,
            ppBadword,
            WordCount
        );
        NNT_ASSERT_RESULT_SUCCESS( result );
        ASSERT_TRUE( results[ 0 ] != 0 );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         WarumoshiTest()

  @brief        ワルモシはすべてのリストで禁止されているかどうかの確認です。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, WarumoshiTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const char* ppBadword[] = { "ワルモシ" };
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    const char* ppBadword[] = { "\xe3\x83\xaf\xe3\x83\xab\xe3\x83\xa2\xe3\x82\xb7" };   //  "ワルモシ"
#else
#error "unsupported os"
#endif
    const int WordCount = sizeof(ppBadword) / sizeof(ppBadword[0]);
    nn::Bit32 results[ WordCount ];

    for ( int i = 0; i < nn::ngc::ProfanityFilterPatternList_Max; i++ )
    {
        result = filter.CheckProfanityWords
        (
            results,
            0x01 << i,
            ppBadword,
            WordCount
        );
        NNT_ASSERT_RESULT_SUCCESS( result );
        ASSERT_TRUE( results[ 0 ] != 0 );
    }
}


/*!
 * @brief ロットチェック確認用NGワードと、そのリストをペアにした構造体
 */
typedef struct LotCheckBadword
{
    //! ロットチェック確認用NGワード
    const char* pBadword;

    //! NGワードリスト
    nn::ngc::ProfanityFilterPatternList nPatternList;

    NN_PADDING3;
} LotCheckBadword;

/*!--------------------------------------------------------------------------*
  Name:         LotCheckRegionTest()

  @brief        新規追加されたロットチェック用文字列が正しくNGワードとして
                認識されるかを確認するためのテストです(リージョンごと)。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, LotCheckRegionTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    LotCheckBadword lotcheckList[] =
    {
        { "badwordj", nn::ngc::ProfanityFilterPatternList_Japanese },
        { "badworde", nn::ngc::ProfanityFilterPatternList_AmericanEnglish },
        { "badworde", nn::ngc::ProfanityFilterPatternList_CanadianFrench },
        { "badworde", nn::ngc::ProfanityFilterPatternList_LatinAmericanSpanish },
        { "badwordp", nn::ngc::ProfanityFilterPatternList_BritishEnglish },
        { "badwordp", nn::ngc::ProfanityFilterPatternList_French },
        { "badwordp", nn::ngc::ProfanityFilterPatternList_German },
        { "badwordp", nn::ngc::ProfanityFilterPatternList_Italian },
        { "badwordp", nn::ngc::ProfanityFilterPatternList_Spanish },
        { "badwordp", nn::ngc::ProfanityFilterPatternList_Dutch },
        { "badwordp", nn::ngc::ProfanityFilterPatternList_Portuguese },
        { "badwordp", nn::ngc::ProfanityFilterPatternList_Russian },
        { "badwordk", nn::ngc::ProfanityFilterPatternList_Korean },
        { "badwordc", nn::ngc::ProfanityFilterPatternList_SimplifiedChinese },
        { "badworde", nn::ngc::ProfanityFilterPatternList_SouthAmericanPortuguese },
        { "badwordt", nn::ngc::ProfanityFilterPatternList_TraditionalChinese },
    };
    NN_STATIC_ASSERT( sizeof( lotcheckList ) / sizeof( LotCheckBadword )
                      == nn::ngc::ProfanityFilterPatternList_Max );

    for ( int i = 0; i < nn::ngc::ProfanityFilterPatternList_Max; i++ )
    {
        nn::Bit32 results[ 1 ];
        const char* ppBadwords[] = { lotcheckList[ i ].pBadword };
        result = filter.CheckProfanityWords( results,
                                            lotcheckList[ i ].nPatternList,
                                            ppBadwords,
                                            sizeof(ppBadwords) / sizeof(ppBadwords[0]) );
        NNT_ASSERT_RESULT_SUCCESS( result );
        ASSERT_TRUE( results[ 0 ] != 0 );
    }

    LotCheckBadword lotcheckCommon =
            { "badworda", nn::ngc::ProfanityFilterPatternList_Japanese };//共通パターンは本体設定言語扱い

    nn::Bit32 results[ 1 ];
    const char* ppBadwords[] = { lotcheckCommon.pBadword };
    result = filter.CheckProfanityWords( results,
                                         lotcheckCommon.nPatternList,
                                        ppBadwords,
                                        sizeof(ppBadwords) / sizeof(ppBadwords[0]) );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         LotCheckWordTest()

  @brief        新規追加されたロットチェック用文字列が正しくNGワードとして
                認識されるかを確認するためのテストです(リストごと)。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, LotCheckWordTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    LotCheckBadword lotcheckList[] =
    {
        { "bbwjja", nn::ngc::ProfanityFilterPatternList_Japanese },
        { "bbween", nn::ngc::ProfanityFilterPatternList_AmericanEnglish },
        { "bbwefr", nn::ngc::ProfanityFilterPatternList_CanadianFrench },
        { "bbwees", nn::ngc::ProfanityFilterPatternList_LatinAmericanSpanish },
        { "bbwpen", nn::ngc::ProfanityFilterPatternList_BritishEnglish },
        { "bbwpfr", nn::ngc::ProfanityFilterPatternList_French },
        { "bbwpde", nn::ngc::ProfanityFilterPatternList_German },
        { "bbwpit", nn::ngc::ProfanityFilterPatternList_Italian },
        { "bbwpes", nn::ngc::ProfanityFilterPatternList_Spanish },
        { "bbwpnl", nn::ngc::ProfanityFilterPatternList_Dutch },
        { "bbwppt", nn::ngc::ProfanityFilterPatternList_Portuguese },
        { "bbwpru", nn::ngc::ProfanityFilterPatternList_Russian },
        { "bbwkko", nn::ngc::ProfanityFilterPatternList_Korean },
        { "bbwczh", nn::ngc::ProfanityFilterPatternList_SimplifiedChinese },
        { "bbwept", nn::ngc::ProfanityFilterPatternList_SouthAmericanPortuguese },
        { "bbwtzh", nn::ngc::ProfanityFilterPatternList_TraditionalChinese },
    };
    NN_STATIC_ASSERT( sizeof( lotcheckList ) / sizeof( LotCheckBadword )
                      == nn::ngc::ProfanityFilterPatternList_Max );

    for ( int i = 0; i < nn::ngc::ProfanityFilterPatternList_Max; i++ )
    {
        nn::Bit32 results[ 1 ];
        const char* ppBadwords[] = { lotcheckList[ i ].pBadword };
        result = filter.CheckProfanityWords( results,
                                            lotcheckList[ i ].nPatternList,
                                            ppBadwords,
                                            sizeof(ppBadwords) / sizeof(ppBadwords[0]) );
        NNT_ASSERT_RESULT_SUCCESS( result );
        ASSERT_TRUE( results[ 0 ] != 0 );
    }

    LotCheckBadword lotcheckCommon =
            { "badworda", nn::ngc::ProfanityFilterPatternList_Japanese };//共通パターンは本体設定言語扱い

    nn::Bit32 results[ 1 ];
    const char* ppBadwords[] = { lotcheckCommon.pBadword };
    result = filter.CheckProfanityWords( results,
                                         lotcheckCommon.nPatternList,
                                        ppBadwords,
                                        sizeof(ppBadwords) / sizeof(ppBadwords[0]) );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( results[ 0 ] != 0 );

}

/*!--------------------------------------------------------------------------*
  Name:         FullwidthWordMaxCountLengthTest

  @brief        マルチバイト文字で最大文字数、WordCountMaxでのNGワードチェックを実施する試験です。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, FullwidthWordMaxCountLengthTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    //  最大文字数(WordLengthMax = 64)、最大ワード数(WordCountMax = 16)、
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const char* ppWords[] =
    {
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ｗａｒｕｍｏｓｈｉ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ＷＡＲＵＭＯＳＨＩ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ｗａｒｕｍｏｓｈｉ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ＷＡＲＵＭＯＳＨＩ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ｗａｒｕｍｏｓｈｉ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ＷＡＲＵＭＯＳＨＩ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ｗａｒｕｍｏｓｈｉ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ＷＡＲＵＭＯＳＨＩ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ｗａｒｕｍｏｓｈｉ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ＷＡＲＵＭＯＳＨＩ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ｗａｒｕｍｏｓｈｉ",
    };
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    const char* ppWords[] =
    {
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\xb7\xef\xbc\xa1\xef\xbc\xb2\xef\xbc\xb5\xef\xbc\xad\xef\xbc\xaf\xef\xbc\xb3\xef\xbc\xa8\xef\xbc\xa9",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\xb7\xef\xbc\xa1\xef\xbc\xb2\xef\xbc\xb5\xef\xbc\xad\xef\xbc\xaf\xef\xbc\xb3\xef\xbc\xa8\xef\xbc\xa9",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\xb7\xef\xbc\xa1\xef\xbc\xb2\xef\xbc\xb5\xef\xbc\xad\xef\xbc\xaf\xef\xbc\xb3\xef\xbc\xa8\xef\xbc\xa9",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\xb7\xef\xbc\xa1\xef\xbc\xb2\xef\xbc\xb5\xef\xbc\xad\xef\xbc\xaf\xef\xbc\xb3\xef\xbc\xa8\xef\xbc\xa9",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\xb7\xef\xbc\xa1\xef\xbc\xb2\xef\xbc\xb5\xef\xbc\xad\xef\xbc\xaf\xef\xbc\xb3\xef\xbc\xa8\xef\xbc\xa9",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
    };
#else
#error "unsupported os"
#endif

    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];
    NN_STATIC_ASSERT(nn::ngc::WordCountMax == WordCount);

    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    ASSERT_TRUE( results[0]  != 0 );
    ASSERT_TRUE( results[1]  != 0 );
    ASSERT_TRUE( results[2]  == 0 );
    ASSERT_TRUE( results[3]  != 0 );
    ASSERT_TRUE( results[4]  != 0 );
    ASSERT_TRUE( results[5]  == 0 );
    ASSERT_TRUE( results[6]  != 0 );
    ASSERT_TRUE( results[7]  != 0 );
    ASSERT_TRUE( results[8]  == 0 );
    ASSERT_TRUE( results[9]  != 0 );
    ASSERT_TRUE( results[10] != 0 );
    ASSERT_TRUE( results[11] == 0 );
    ASSERT_TRUE( results[12] != 0 );
    ASSERT_TRUE( results[13] != 0 );
    ASSERT_TRUE( results[14] == 0 );
    ASSERT_TRUE( results[15] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         MixedWordMaxCountLengthTest

  @brief        全角・半角混在でWordLengthMax * WordCountMaxでのNGワードチェックを実施する試験です。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, MixedWordMaxCountLengthTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    //  最大文字数(WordLengthMax = 64)、最大ワード数(WordCountMax = 16)、
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const char* ppWords[] =
    {
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ｗａｒｕｍｏｓｈｉ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ＷＡＲＵＭＯＳＨＩ",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３warumoshi",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３WARUMOSHI",
       "0123456789０１２３４５６７８９0123456789０１２３４５６７８９0123456789０１２３ｗａｒｕｍｏｓｈｉ",
       "0123456789０１２３４５６７８９0123456789０１２３４５６７８９0123456789０１２３ＷＡＲＵＭＯＳＨＩ",
       "012345678901234567890123456789012345678901234567890123456789012",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "0123456789０１２３４５６７８９0123456789０１２３４５６７８９0123456789０１２３warumoshi",
       "0123456789０１２３４５６７８９0123456789０１２３４５６７８９0123456789０１２３WARUMOSHI"
    };
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    const char* ppWords[] =
    {
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\xb7\xef\xbc\xa1\xef\xbc\xb2\xef\xbc\xb5\xef\xbc\xad\xef\xbc\xaf\xef\xbc\xb3\xef\xbc\xa8\xef\xbc\xa9",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\x77\x61\x72\x75\x6d\x6f\x73\x68\x69",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\x57\x41\x52\x55\x4d\x4f\x53\x48\x49",
       "\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
       "\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\xb7\xef\xbc\xa1\xef\xbc\xb2\xef\xbc\xb5\xef\xbc\xad\xef\xbc\xaf\xef\xbc\xb3\xef\xbc\xa8\xef\xbc\xa9",
       "\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\x77\x61\x72\x75\x6d\x6f\x73\x68\x69",
       "\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\x57\x41\x52\x55\x4d\x4f\x53\x48\x49",
    };
#else
#error "unsupported os"
#endif

    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];
    NN_STATIC_ASSERT(nn::ngc::WordCountMax == WordCount);

    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    ASSERT_TRUE( results[0]  != 0 );
    ASSERT_TRUE( results[1]  != 0 );
    ASSERT_TRUE( results[2]  != 0 );
    ASSERT_TRUE( results[3]  != 0 );
    ASSERT_TRUE( results[4]  != 0 );
    ASSERT_TRUE( results[5]  != 0 );
    ASSERT_TRUE( results[6]  == 0 );
    ASSERT_TRUE( results[7]  == 0 );
    ASSERT_TRUE( results[8]  == 0 );
    ASSERT_TRUE( results[9]  == 0 );
    ASSERT_TRUE( results[10] == 0 );
    ASSERT_TRUE( results[11] == 0 );
    ASSERT_TRUE( results[12] == 0 );
    ASSERT_TRUE( results[13] == 0 );
    ASSERT_TRUE( results[14] != 0 );
    ASSERT_TRUE( results[15] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         MixedWordMaxOverLengthTest

  @brief        WordLengthMax * WordCountMax +1文字でのNGワードチェックを実施する試験です。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, MixedWordMaxOverLengthTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    //  最大文字数(WordLengthMax = 64)+1※、最大ワード数(WordCountMax = 16)
    //  ※最大文字・ワード数テスト(MixedWordMaxCountLengthTest)データの先頭に'X'追加
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const char* ppWords[] =
    {
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ｗａｒｕｍｏｓｈｉ",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３ＷＡＲＵＭＯＳＨＩ",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３warumoshi",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３WARUMOSHI",
       "X0123456789０１２３４５６７８９0123456789０１２３４５６７８９0123456789０１２３ｗａｒｕｍｏｓｈｉ",
       "X0123456789０１２３４５６７８９0123456789０１２３４５６７８９0123456789０１２３ＷＡＲＵＭＯＳＨＩ",
       "X012345678901234567890123456789012345678901234567890123456789012",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "X０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２３４５６７８９０１２",
       "X0123456789０１２３４５６７８９0123456789０１２３４５６７８９0123456789０１２３warumoshi",
       "X0123456789０１２３４５６７８９0123456789０１２３４５６７８９0123456789０１２３WARUMOSHI"
    };
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    const char* ppWords[] =
    {
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\xb7\xef\xbc\xa1\xef\xbc\xb2\xef\xbc\xb5\xef\xbc\xad\xef\xbc\xaf\xef\xbc\xb3\xef\xbc\xa8\xef\xbc\xa9",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\x77\x61\x72\x75\x6d\x6f\x73\x68\x69",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\x57\x41\x52\x55\x4d\x4f\x53\x48\x49",
       "\x58\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbd\x97\xef\xbd\x81\xef\xbd\x92\xef\xbd\x95\xef\xbd\x8d\xef\xbd\x8f\xef\xbd\x93\xef\xbd\x88\xef\xbd\x89",
       "\x58\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\xb7\xef\xbc\xa1\xef\xbc\xb2\xef\xbc\xb5\xef\xbc\xad\xef\xbc\xaf\xef\xbc\xb3\xef\xbc\xa8\xef\xbc\xa9",
       "\x58\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x30\x31\x32",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\x58\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92",
       "\x58\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\x77\x61\x72\x75\x6d\x6f\x73\x68\x69",
       "\x58\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\xef\xbc\x94\xef\xbc\x95\xef\xbc\x96\xef\xbc\x97\xef\xbc\x98\xef\xbc\x99\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\xef\xbc\x90\xef\xbc\x91\xef\xbc\x92\xef\xbc\x93\x57\x41\x52\x55\x4d\x4f\x53\x48\x49",
    };
#else
#error "unsupported os"
#endif

    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];
    NN_STATIC_ASSERT(nn::ngc::WordCountMax == WordCount);

    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );

    ASSERT_TRUE( results[0]  == 0 );
    ASSERT_TRUE( results[1]  == 0 );
    ASSERT_TRUE( results[2]  == 0 );
    ASSERT_TRUE( results[3]  == 0 );
    ASSERT_TRUE( results[4]  == 0 );
    ASSERT_TRUE( results[5]  == 0 );
    ASSERT_TRUE( results[6]  == 0 );
    ASSERT_TRUE( results[7]  == 0 );
    ASSERT_TRUE( results[8]  == 0 );
    ASSERT_TRUE( results[9]  == 0 );
    ASSERT_TRUE( results[10] == 0 );
    ASSERT_TRUE( results[11] == 0 );
    ASSERT_TRUE( results[12] == 0 );
    ASSERT_TRUE( results[13] == 0 );
    ASSERT_TRUE( results[14] == 0 );
    ASSERT_TRUE( results[15] == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         NotInitializedTextTest()

  @brief        初期化しないまま文章のマスク処理を行うとエラーになるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, NotInitializedTextTest)
{
    nn::ngc::ProfanityFilter filter;

    char szHoge[] = "hogehoge";
    nn::Result result =
            filter.MaskProfanityWordsInText( nullptr,
                                             szHoge,
                                             nn::ngc::ProfanityFilterPatternList_Japanese);
    NNT_ASSERT_RESULT_FAILURE( nn::ngc::ResultNotInitialized ,result );
}

/*!--------------------------------------------------------------------------*
  Name:         InvalidArgument_Text_Test

  @brief        引数がおかしい状態でマスク処理を呼び出すテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, InvalidArgument_Text_Test)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    char* pText = nullptr;

    result = filter.MaskProfanityWordsInText(nullptr,
                                             pText,
                                             nn::ngc::ProfanityFilterPatternList_Japanese);
    NNT_ASSERT_RESULT_FAILURE( nn::ngc::ResultInvalidPointer ,result );
}

/*!--------------------------------------------------------------------------*
  Name:         NoProfanityTextTest()

  @brief        不正文字列が現れないテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, NoProfanityTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] = "We love mario.";

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 0 );
    ASSERT_TRUE( std::strcmp( "We love mario.", szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         SimpleMaskTextTest()

  @brief        テキストのマスキング処理の単純なテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, SimpleMaskTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] = "このロリコンめ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] = "\xe3\x81\x93\xe3\x81\xae\xe3\x83\xad\xe3\x83\xaa\xe3\x82\xb3\xe3\x83\xb3\xe3\x82\x81";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 1 );
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    ASSERT_TRUE( std::strcmp( "この****め", szText ) == 0 );
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    ASSERT_TRUE( std::strcmp( "\xe3\x81\x93\xe3\x81\xae\x2a\x2a\x2a\x2a\xe3\x82\x81", szText ) == 0 );  //  "この****め"
#else
#error "unsupported os"
#endif
}

/*!--------------------------------------------------------------------------*
  Name:         CapitalLetterMaskTextTest()

  @brief        ラテン 1 補助大文字を含むテキストのマスキング処理テストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, LatinCapitalLetterMaskTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "scheiße\n"
        "scheiẞe\n"
        "SCHEIßE\n"
        "SCHEIẞE\n"
        "s c h e i ß e\n"
        "S C H E I ẞ E\n"
        "rülps\n"
        "rÜlps\n"
        "RüLPS\n"
        "RÜLPS\n"
        "r ü l p s\n"
        "R Ü L P S";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x73\x63\x68\x65\x69\xc3\x9f\x65\x0a"
        "\x73\x63\x68\x65\x69\xe1\xba\x9e\x65\x0a"
        "\x53\x43\x48\x45\x49\xc3\x9f\x45\x0a"
        "\x53\x43\x48\x45\x49\xe1\xba\x9e\x45\x0a"
        "\x73\x20\x63\x20\x68\x20\x65\x20\x69\x20\xc3\x9f\x20\x65\x0a"
        "\x53\x20\x43\x20\x48\x20\x45\x20\x49\x20\xe1\xba\x9e\x20\x45\x0a"
        "\x72\xC3\xBC\x6C\x70\x73\x0a"
        "\x72\xC3\x9C\x6C\x70\x73\x0a"
        "\x52\xC3\xBC\x4C\x50\x53\x0a"
        "\x52\xC3\x9C\x4C\x50\x53\x0a"
        "\x72\x20\xC3\xBC\x20\x6C\x20\x70\x20\x73\x0a"
        "\x52\x20\xC3\x9C\x20\x4C\x20\x50\x20\x53";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "*******\n"
        "*******\n"
        "*******\n"
        "*******\n"
        "*************\n"
        "*************\n"
        "*****\n"
        "*****\n"
        "*****\n"
        "*****\n"
        "*********\n"
        "*********";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x0a"
        "\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_German
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    EXPECT_EQ( nWordCount, 12 );
    ASSERT_TRUE(std::strcmp(szExpect, szText) == 0);
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingSpaceTextTest()

  @brief        スペースを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingSpaceTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "わ る も し \n"
        "ああわるもしああ"
        "ああ わ る も し ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe3\x82\x8f\x20\xe3\x82\x8b\x20\xe3\x82\x82\x20\xe3\x81\x97\x20\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x20\xe3\x82\x8f\x20\xe3\x82\x8b\x20\xe3\x82\x82\x20\xe3\x81\x97\x20\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "******* \n"               // 間にスペースが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ ******* ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x20\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x20\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x20\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthSpaceTextTest()

  @brief        全角スペースを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthSpaceTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "　わ　る　も　し　\n"
        "ああわるもしああ"
        "ああ　わ　る　も　し　ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe3\x80\x80\xe3\x82\x8f\xe3\x80\x80\xe3\x82\x8b\xe3\x80\x80\xe3\x82\x82\xe3\x80\x80\xe3\x81\x97\xe3\x80\x80\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x80\xe3\x82\x8f\xe3\x80\x80\xe3\x82\x8b\xe3\x80\x80\xe3\x82\x82\xe3\x80\x80\xe3\x81\x97\xe3\x80\x80\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "　*******　\n"            // 間にスペースが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ　*******　ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xe3\x80\x80\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe3\x80\x80\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x80\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe3\x80\x80\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingJapanesePeriodTextTest()

  @brief        半角句点を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingJapanesePeriodTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "｡わ｡る｡も｡し｡\n"
        "ああわるもしああ"
        "ああ｡わ｡る｡も｡し｡ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbd\xa1\xe3\x82\x8f\xef\xbd\xa1\xe3\x82\x8b\xef\xbd\xa1\xe3\x82\x82\xef\xbd\xa1\xe3\x81\x97\xef\xbd\xa1\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\xa1\xe3\x82\x8f\xef\xbd\xa1\xe3\x82\x8b\xef\xbd\xa1\xe3\x82\x82\xef\xbd\xa1\xe3\x81\x97\xef\xbd\xa1\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "｡*******｡\n"              // 間に半角句読点が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ｡*******｡ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbd\xa1\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\xa1\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\xa1\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\xa1\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthJapanesePeriodTextTest()

  @brief        全角句点を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthJapanesePeriodTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "。わ。る。も。し。\n"
        "ああわるもしああ"
        "ああ。わ。る。も。し。ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe3\x80\x82\xe3\x82\x8f\xe3\x80\x82\xe3\x82\x8b\xe3\x80\x82\xe3\x82\x82\xe3\x80\x82\xe3\x81\x97\xe3\x80\x82\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x82\xe3\x82\x8f\xe3\x80\x82\xe3\x82\x8b\xe3\x80\x82\xe3\x82\x82\xe3\x80\x82\xe3\x81\x97\xe3\x80\x82\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "。*******。\n"            // 間に全角句読点が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ。*******。ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xe3\x80\x82\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe3\x80\x82\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x82\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe3\x80\x82\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingPunctuationTextTest()

  @brief        半角読点を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingJapaneseCommaTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "､わ､る､も､し､\n"
        "ああわるもしああ"
        "ああ､わ､る､も､し､ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbd\xa4\xe3\x82\x8f\xef\xbd\xa4\xe3\x82\x8b\xef\xbd\xa4\xe3\x82\x82\xef\xbd\xa4\xe3\x81\x97\xef\xbd\xa4\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\xa4\xe3\x82\x8f\xef\xbd\xa4\xe3\x82\x8b\xef\xbd\xa4\xe3\x82\x82\xef\xbd\xa4\xe3\x81\x97\xef\xbd\xa4\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "､*******､\n"              // 間に半角句読点が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ､*******､ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbd\xa4\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\xa4\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\xa4\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\xa4\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthPunctuationTextTest()

  @brief        全角読点を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthJapaneseCommaTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "、わ、る、も、し、\n"
        "ああわるもしああ"
        "ああ、わ、る、も、し、ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe3\x80\x81\xe3\x82\x8f\xe3\x80\x81\xe3\x82\x8b\xe3\x80\x81\xe3\x82\x82\xe3\x80\x81\xe3\x81\x97\xe3\x80\x81\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x81\xe3\x82\x8f\xe3\x80\x81\xe3\x82\x8b\xe3\x80\x81\xe3\x82\x82\xe3\x80\x81\xe3\x81\x97\xe3\x80\x81\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "、*******、\n"            // 間に全角句読点が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ、*******、ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xe3\x80\x81\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe3\x80\x81\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x81\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe3\x80\x81\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingPeriodTextTest()

  @brief        半角ピリオドを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingPeriodTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        ".わ.る.も.し.\n"
        "ああわるもしああ"
        "ああ.わ.る.も.し.ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x2e\xe3\x82\x8f\x2e\xe3\x82\x8b\x2e\xe3\x82\x82\x2e\xe3\x81\x97\x2e\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2e\xe3\x82\x8f\x2e\xe3\x82\x8b\x2e\xe3\x82\x82\x2e\xe3\x81\x97\x2e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        ".*******.\n"              // 間に半角ピリオドが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ.*******.ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x2e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2e\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthPeriodTextTest()

  @brief        全角ピリオドを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthPeriodTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "．わ．る．も．し．\n"
        "ああわるもしああ"
        "ああ．わ．る．も．し．ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x8e\xe3\x82\x8f\xef\xbc\x8e\xe3\x82\x8b\xef\xbc\x8e\xe3\x82\x82\xef\xbc\x8e\xe3\x81\x97\xef\xbc\x8e\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8e\xe3\x82\x8f\xef\xbc\x8e\xe3\x82\x8b\xef\xbc\x8e\xe3\x82\x82\xef\xbc\x8e\xe3\x81\x97\xef\xbc\x8e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "．*******．\n"            // 間に全角ピリオドが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ．*******．ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x8e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8e\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingCommaTextTest()

  @brief        半角カンマを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingCommaTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        ",わ,る,も,し,\n"
        "ああわるもしああ"
        "ああ,わ,る,も,し,ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x2c\xe3\x82\x8f\x2c\xe3\x82\x8b\x2c\xe3\x82\x82\x2c\xe3\x81\x97\x2c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2c\xe3\x82\x8f\x2c\xe3\x82\x8b\x2c\xe3\x82\x82\x2c\xe3\x81\x97\x2c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        ",*******,\n"              // 間に半角カンマが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ,*******,ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x2c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthCommaTextTest()

  @brief        全角カンマを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthCommaTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "，わ，る，も，し，\n"
        "ああわるもしああ"
        "ああ，わ，る，も，し，ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x8c\xe3\x82\x8f\xef\xbc\x8c\xe3\x82\x8b\xef\xbc\x8c\xe3\x82\x82\xef\xbc\x8c\xe3\x81\x97\xef\xbc\x8c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8c\xe3\x82\x8f\xef\xbc\x8c\xe3\x82\x8b\xef\xbc\x8c\xe3\x82\x82\xef\xbc\x8c\xe3\x81\x97\xef\xbc\x8c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "，*******，\n"            // 間に全角カンマが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ，*******，ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x8c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingMixedBlankCodeTextTest()

  @brief        半角スペース、半角句読点、半角ピリオド、半角カンマ、
                全角スペース、全角句読点、全角ピリオド、全角カンマを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingMixedBlankCodeTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        " ｡､.,　。、．，わ ｡､.,　。、．，る ｡､.,　。、．，も ｡､.,　。、．，し ｡､.,　。、．，\n"
        "ああわるもしああ"
        "ああ ｡､.,　。、．，わ ｡､.,　。、．，る ｡､.,　。、．，も ｡､.,　。、．，し ｡､.,　。、．，ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x8f\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x8b\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x81\x97\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x8f\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x8b\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x81\x97\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                                                               // 単純にマッチングするテスト
        " ｡､.,　。、．，********************************** ｡､.,　。、．，\n"   // 間にブランクコードが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ ｡､.,　。、．，********************************** ｡､.,　。、．，ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         LotCheckRegionTextTest()

  @brief        ロットチェック用の文字列がマスクされるかどうかの確認テストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, LotCheckRegionTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
        "bbwjja\n"
        " bbwjja \n";

    char szExpect[] =
        "******\n"
        " ****** \n";

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 2 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         HalfWidthTextTest()

  @brief        半角スペースを含む文字列が正しくマスク出来るかどうかを調べるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, HalfWidthTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] = "あいうえおﾚｲﾌﾟ。";
    char szExpect[] = "あいうえお****。";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] = // "あいうえおﾚｲﾌﾟ。";
                    "\xe3\x81\x82\xe3\x81\x84\xe3\x81\x86\xe3\x81\x88\xe3\x81\x8a\xef\xbe\x9a\xef\xbd\xb2\xef\xbe\x8c\xef\xbe\x9f\xe3\x80\x82";
    char szExpect[] = // "あいうえお****。";
                    "\xe3\x81\x82\xe3\x81\x84\xe3\x81\x86\xe3\x81\x88\xe3\x81\x8a\x2a\x2a\x2a\x2a\xe3\x80\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 1 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         CheckAllListTextTest()

  @brief        すべてのNGワードパターンについて、チェックを行うテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckAllListTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    char szText[] =
            "Japanese => badwordj bbwjja\n"
            "AmericanEnglish => badworde bbween\n"
            "CanadianFrench => badworde bbwefr\n"
            "LatinAmericanSpanish => badworde bbwees\n"
            "BritishEnglish => badwordp bbwpen\n"
            "French => badwordp bbwpfr\n"
            "German => badwordp bbwpde\n"
            "Italian => badwordp bbwpit\n"
            "Spanish => badwordp bbwpes\n"
            "Dutch => badwordp bbwpnl\n"
            "Korean => badwordk bbwkko\n"
            "Chinese => badwordc bbwczh\n"
            "Portuguese => badwordp bbwppt\n"
            "Russian => badwordp bbwpru\n"
            "Taiwanese => badwordt bbwtzh\n"
            "SouthAmericanPortuguese => badworde bbwept\n"
            "Common => badworda";

    char szExpect[] =
            "Japanese => ******** ******\n"
            "AmericanEnglish => ******** ******\n"
            "CanadianFrench => ******** ******\n"
            "LatinAmericanSpanish => ******** ******\n"
            "BritishEnglish => ******** ******\n"
            "French => ******** ******\n"
            "German => ******** ******\n"
            "Italian => ******** ******\n"
            "Spanish => ******** ******\n"
            "Dutch => ******** ******\n"
            "Korean => ******** ******\n"
            "Chinese => ******** ******\n"
            "Portuguese => ******** ******\n"
            "Russian => ******** ******\n"
            "Taiwanese => ******** ******\n"
            "SouthAmericanPortuguese => ******** ******\n"
            "Common => ********";

    result = filter.MaskProfanityWordsInText( nullptr, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         CheckMaxSizeTextMaskTest()

  @brief        最大文字でのマスクを実施する試験です。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckMaxSizeTextMaskTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    char szText[] =
            "badwordj bbwjja\n"
            "badworde bbween\n"
            "badworde bbwefr\n"
            "badworde bbwees\n"
            "badwordp bbwpen\n"
            "badwordp bbwpfr\n"
            "badwordp bbwpde\n"
            "badwordp bbwpit\n"
            "badwordp bbwpes\n"
            "badwordp bbwpnl\n"
            "badwordk bbwkko\n"
            "badwordc bbwczh\n"
            "badwordp bbwppt\n"
            "badwordp bbwpru\n"
            "badwordt bbwtzh\n"
            "badworde bbwept\n"
            "badworda\n"
            "warumoshi\n"
            "badwordj bbwjja\n"
            "badworde bbween\n"
            "badworde bbwefr\n"
            "badworde bbwees\n"
            "badwordp bbwpen\n"
            "badwordp bbwpfr\n"
            "badwordp bbwpde\n"
            "badwordp bbwpit\n"
            "badwordp bbwpes\n"
            "badwordp bbwpnl\n"
            "badwordk bbwkko\n"
            "badwordc bbwczh\n"
            "badwordp bbwppt\n"
            "badwordp bbwpru\n"
            "   warumoshi";

    char szExpect[] =
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "********\n"
            "*********\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "******** ******\n"
            "   *********";

    ASSERT_TRUE( nn::ngc::TextLengthMax == std::strlen(szText) + 1 );

    result = filter.MaskProfanityWordsInText( nullptr, szText, 0xFFFFFFFF );

    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         CheckFullwidthWordTextLengthMaxTest()

  @brief        全角最大文字でのマスクを実施する試験です。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckFullwidthWordTextLengthMaxTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
       // 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9  (20文字/行 * 25行) + 11文字/最終行 + \0 = 512文字
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　";
    char szExpect[] =
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80";
    char szExpect[] =
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80";
#else
#error "unsupported os"
#endif
    result = filter.MaskProfanityWordsInText( nullptr, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
} //NOLINT(impl/function_size)

/*!--------------------------------------------------------------------------*
  Name:         CheckMixedWordTextLengthMaxTest()

  @brief        全角・半角混在 最大文字でのマスクを実施する試験です。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckMixedWordTextLengthMaxTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
    //    0 1 2 3 4 5 6 7 8 9 0123456789    (20文字/行 * 25行) + 11文字/最終行 + \0 = 512文字
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "　　　badwordj";
    char szExpect[] =
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "　　　********";
    char szText2[] =
    //    0 1 2 3 4 5 6 7 8 9 012345678 9    (20文字/行 * 25行) + 12文字/最終行 + \0 = 512文字 + 1文字
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "　　warumoshi";
    char szExpect2[] =
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "　　*********";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x62\x61\x64\x77\x6f\x72\x64\x6a";
    char szExpect[] =
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********";
    char szText2[] =
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\xe3\x80\x80\xe3\x80\x80\x77\x61\x72\x75\x6d\x6f\x73\x68\x69";
    char szExpect2[] =
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "\xe3\x80\x80\xe3\x80\x80*********";

#else
#error "unsupported os"
#endif

    // "　　　badwordj" ⇒ "　　　********"
    result = filter.MaskProfanityWordsInText( nullptr, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );

    // "　　warumoshi" ⇒ "　　*********"
    result = filter.MaskProfanityWordsInText( nullptr, szText2, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( std::strcmp( szExpect2, szText2 ) == 0 );

} //NOLINT(impl/function_size)

/*!--------------------------------------------------------------------------*
  Name:         CheckWordTextLengthMaxOverTest()

  @brief        最大文字+1でのマスクを実施する試験です。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckWordTextLengthMaxOverTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
    //    0 1 2 3 4 5 6 7 8 9 012345678 9    (20文字/行 * 25行) + 12文字/最終行 + \0 = 512文字 + 1文字
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        " 　　　badwordj";
    char szExpect[] =
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        " 　　　*******j";          // 'j'を除く'badword'がNGワードに定義されているため'badword'部がマスクされます。
    char szText2[] =
    //    0 1 2 3 4 5 6 7 8 9 012345678 9    (20文字/行 * 25行) + 12文字/最終行 + \0 = 512文字 + 1文字
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        " 　　warumoshi";
    char szExpect2[] =
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        "********　　　********\n"
        " 　　warumoshi";           // 'i'を除く'warumosh'で判定されるためマスクされません。
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        " \xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x62\x61\x64\x77\x6f\x72\x64\x6a";
    char szExpect[] =
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        " \xe3\x80\x80\xe3\x80\x80\xe3\x80\x80*******\x6a";
    char szText2[] =
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        " \xe3\x80\x80\xe3\x80\x80\x77\x61\x72\x75\x6d\x6f\x73\x68\x69";
    char szExpect2[] =
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        "********\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80********\n"
        " \xe3\x80\x80\xe3\x80\x80warumoshi";
#else
#error "unsupported os"
#endif

    // " 　　　badwordj" ⇒ " 　　　*******j"
    result = filter.MaskProfanityWordsInText( nullptr, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );

    // " 　　warumoshi" ⇒ " 　　warumoshi"
    result = filter.MaskProfanityWordsInText( nullptr, szText2, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( std::strcmp( szExpect2, szText2 ) == 0 );

} //NOLINT(impl/function_size)

/*!--------------------------------------------------------------------------*
  Name:         CheckMyLanguageListTextTest()

  @brief        現在のリージョン設定に基づいてチェックを行うテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckMyLanguageListTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] = "this is badword";

    char szExpect[] = "this is *******";

    result = filter.MaskProfanityWordsInText( &nWordCount, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 1 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         CheckMailAddressTextTest()

  @brief        メールアドレスのようなものが現れたらマスキングされるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckMailAddressTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "こんにちは。はじめまして。\n"
        "一緒にゲームをしていて あなたのことが(以下略)\n"
        "もしよかったらhoge@foobar.netまでメールをくれませんか？\n"
        "偽物のアドレスhoge@foobarはマスクされてはいけません！";

    char szExpect[] =
        "こんにちは。はじめまして。\n"
        "一緒にゲームをしていて あなたのことが(以下略)\n"
        "もしよかったら***************までメールをくれませんか？\n"
        "偽物のアドレスhoge@foobarはマスクされてはいけません！";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x81\x93\xe3\x82\x93\xe3\x81\xab\xe3\x81\xa1\xe3\x81\xaf\xe3\x80\x82\xe3\x81\xaf\xe3\x81\x98\xe3\x82\x81\xe3\x81\xbe\xe3\x81\x97\xe3\x81\xa6\xe3\x80\x82\x0a\xe4\xb8\x80\xe7\xb7\x92\xe3\x81\xab\xe3\x82\xb2\xe3\x83\xbc\xe3\x83\xa0\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\xa6\x20\xe3\x81\x82\xe3\x81\xaa\xe3\x81\x9f\xe3\x81\xae\xe3\x81\x93\xe3\x81\xa8\xe3\x81\x8c\x28\xe4\xbb\xa5\xe4\xb8\x8b\xe7\x95\xa5\x29\x0a\xe3\x82\x82\xe3\x81\x97\xe3\x82\x88\xe3\x81\x8b\xe3\x81\xa3\xe3\x81\x9f\xe3\x82\x89\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\x2e\x6e\x65\x74\xe3\x81\xbe\xe3\x81\xa7\xe3\x83\xa1\xe3\x83\xbc\xe3\x83\xab\xe3\x82\x92\xe3\x81\x8f\xe3\x82\x8c\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xe3\x81\x8b\xef\xbc\x9f\x0a\xe5\x81\xbd\xe7\x89\xa9\xe3\x81\xae\xe3\x82\xa2\xe3\x83\x89\xe3\x83\xac\xe3\x82\xb9\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\xe3\x81\xaf\xe3\x83\x9e\xe3\x82\xb9\xe3\x82\xaf\xe3\x81\x95\xe3\x82\x8c\xe3\x81\xa6\xe3\x81\xaf\xe3\x81\x84\xe3\x81\x91\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xef\xbc\x81";

    char szExpect[] =
        "\xe3\x81\x93\xe3\x82\x93\xe3\x81\xab\xe3\x81\xa1\xe3\x81\xaf\xe3\x80\x82\xe3\x81\xaf\xe3\x81\x98\xe3\x82\x81\xe3\x81\xbe\xe3\x81\x97\xe3\x81\xa6\xe3\x80\x82\x0a\xe4\xb8\x80\xe7\xb7\x92\xe3\x81\xab\xe3\x82\xb2\xe3\x83\xbc\xe3\x83\xa0\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\xa6\x20\xe3\x81\x82\xe3\x81\xaa\xe3\x81\x9f\xe3\x81\xae\xe3\x81\x93\xe3\x81\xa8\xe3\x81\x8c\x28\xe4\xbb\xa5\xe4\xb8\x8b\xe7\x95\xa5\x29\x0a\xe3\x82\x82\xe3\x81\x97\xe3\x82\x88\xe3\x81\x8b\xe3\x81\xa3\xe3\x81\x9f\xe3\x82\x89\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe3\x81\xbe\xe3\x81\xa7\xe3\x83\xa1\xe3\x83\xbc\xe3\x83\xab\xe3\x82\x92\xe3\x81\x8f\xe3\x82\x8c\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xe3\x81\x8b\xef\xbc\x9f\x0a\xe5\x81\xbd\xe7\x89\xa9\xe3\x81\xae\xe3\x82\xa2\xe3\x83\x89\xe3\x83\xac\xe3\x82\xb9\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\xe3\x81\xaf\xe3\x83\x9e\xe3\x82\xb9\xe3\x82\xaf\xe3\x81\x95\xe3\x82\x8c\xe3\x81\xa6\xe3\x81\xaf\xe3\x81\x84\xe3\x81\x91\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xef\xbc\x81";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText( &nWordCount, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 1 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         SkipMailAddressTextTest()

  @brief        メールアドレスのマッチングをスキップするテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, SkipMailAddressTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SkipAtSignCheck( nn::ngc::ProfanityFilter::SkipMode_SkipAtSign );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "こんにちは。はじめまして。\n"
        "一緒にゲームをしていて あなたのことが(以下略)\n"
        "もしよかったらhoge@foobar.netまでメールをくれませんか？\n"
        "偽物のアドレスhoge@foobarはマスクされてはいけません！";

    char szExpect[] =
        "こんにちは。はじめまして。\n"
        "一緒にゲームをしていて あなたのことが(以下略)\n"
        "もしよかったらhoge@foobar.netまでメールをくれませんか？\n"
        "偽物のアドレスhoge@foobarはマスクされてはいけません！";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x81\x93\xe3\x82\x93\xe3\x81\xab\xe3\x81\xa1\xe3\x81\xaf\xe3\x80\x82\xe3\x81\xaf\xe3\x81\x98\xe3\x82\x81\xe3\x81\xbe\xe3\x81\x97\xe3\x81\xa6\xe3\x80\x82\x0a\xe4\xb8\x80\xe7\xb7\x92\xe3\x81\xab\xe3\x82\xb2\xe3\x83\xbc\xe3\x83\xa0\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\xa6\x20\xe3\x81\x82\xe3\x81\xaa\xe3\x81\x9f\xe3\x81\xae\xe3\x81\x93\xe3\x81\xa8\xe3\x81\x8c\x28\xe4\xbb\xa5\xe4\xb8\x8b\xe7\x95\xa5\x29\x0a\xe3\x82\x82\xe3\x81\x97\xe3\x82\x88\xe3\x81\x8b\xe3\x81\xa3\xe3\x81\x9f\xe3\x82\x89\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\x2e\x6e\x65\x74\xe3\x81\xbe\xe3\x81\xa7\xe3\x83\xa1\xe3\x83\xbc\xe3\x83\xab\xe3\x82\x92\xe3\x81\x8f\xe3\x82\x8c\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xe3\x81\x8b\xef\xbc\x9f\x0a\xe5\x81\xbd\xe7\x89\xa9\xe3\x81\xae\xe3\x82\xa2\xe3\x83\x89\xe3\x83\xac\xe3\x82\xb9\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\xe3\x81\xaf\xe3\x83\x9e\xe3\x82\xb9\xe3\x82\xaf\xe3\x81\x95\xe3\x82\x8c\xe3\x81\xa6\xe3\x81\xaf\xe3\x81\x84\xe3\x81\x91\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xef\xbc\x81";

    char szExpect[] =
        "\xe3\x81\x93\xe3\x82\x93\xe3\x81\xab\xe3\x81\xa1\xe3\x81\xaf\xe3\x80\x82\xe3\x81\xaf\xe3\x81\x98\xe3\x82\x81\xe3\x81\xbe\xe3\x81\x97\xe3\x81\xa6\xe3\x80\x82\x0a\xe4\xb8\x80\xe7\xb7\x92\xe3\x81\xab\xe3\x82\xb2\xe3\x83\xbc\xe3\x83\xa0\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\xa6\x20\xe3\x81\x82\xe3\x81\xaa\xe3\x81\x9f\xe3\x81\xae\xe3\x81\x93\xe3\x81\xa8\xe3\x81\x8c\x28\xe4\xbb\xa5\xe4\xb8\x8b\xe7\x95\xa5\x29\x0a\xe3\x82\x82\xe3\x81\x97\xe3\x82\x88\xe3\x81\x8b\xe3\x81\xa3\xe3\x81\x9f\xe3\x82\x89\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\x2e\x6e\x65\x74\xe3\x81\xbe\xe3\x81\xa7\xe3\x83\xa1\xe3\x83\xbc\xe3\x83\xab\xe3\x82\x92\xe3\x81\x8f\xe3\x82\x8c\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xe3\x81\x8b\xef\xbc\x9f\x0a\xe5\x81\xbd\xe7\x89\xa9\xe3\x81\xae\xe3\x82\xa2\xe3\x83\x89\xe3\x83\xac\xe3\x82\xb9\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\xe3\x81\xaf\xe3\x83\x9e\xe3\x82\xb9\xe3\x82\xaf\xe3\x81\x95\xe3\x82\x8c\xe3\x81\xa6\xe3\x81\xaf\xe3\x81\x84\xe3\x81\x91\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xef\xbc\x81";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText( &nWordCount, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 0 );      // スキップフラグがオンなので検出されません。
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         NoProfanityTextNoOverwriteTest()

  @brief        不正文字列が現れないテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, NoProfanityTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
    char szText[] = "We love mario.";

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 0 );
    ASSERT_TRUE( std::strcmp( "We love mario.", szText ) == 0 );
}


/*!--------------------------------------------------------------------------*
  Name:         SimpleMaskTextNoOverwriteTest()

  @brief        テキストのマスキング処理の単純なテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, SimpleMaskTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] = "このロリコンめ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] = "\xe3\x81\x93\xe3\x81\xae\xe3\x83\xad\xe3\x83\xaa\xe3\x82\xb3\xe3\x83\xb3\xe3\x82\x81";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 1 );
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    ASSERT_TRUE( std::strcmp( "この*め", szText ) == 0 );
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    ASSERT_TRUE( std::strcmp( "\xe3\x81\x93\xe3\x81\xae\x2a\xe3\x82\x81", szText ) == 0 );   // "この*め"
#else
#error "unsupported os"
#endif
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingSpaceTextNoOverwriteTest()

  @brief        スペースを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingSpaceTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        "わ る も し \n"
        "ああわるもしああ"
        "ああ わ る も し ああ";

    char szExpect[] =
        "*\n"                  // 単純にマッチングするテスト
        "* \n"                 // 間にスペースが含まれていても、そこは無視してマスクされます。
        "ああ*ああ"
        "ああ * ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] = "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe3\x82\x8f\x20\xe3\x82\x8b\x20\xe3\x82\x82\x20\xe3\x81\x97\x20\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x20\xe3\x82\x8f\x20\xe3\x82\x8b\x20\xe3\x82\x82\x20\xe3\x81\x97\x20\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] ="\x2a\x0a\x2a\x20\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x20\x2a\x20\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthSpaceTextNoOverwriteTest()

  @brief        全角スペースを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthSpaceTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        "　わ　る　も　し　\n"
        "ああわるもしああ"
        "ああ　わ　る　も　し　ああ";

    char szExpect[] =
        "*\n"                  // 単純にマッチングするテスト
        "　*　\n"              // 間にスペースが含まれていても、そこは無視してマスクされます。
        "ああ*ああ"
        "ああ　*　ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe3\x80\x80\xe3\x82\x8f\xe3\x80\x80\xe3\x82\x8b\xe3\x80\x80\xe3\x82\x82\xe3\x80\x80\xe3\x81\x97\xe3\x80\x80\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x80\xe3\x82\x8f\xe3\x80\x80\xe3\x82\x8b\xe3\x80\x80\xe3\x82\x82\xe3\x80\x80\xe3\x81\x97\xe3\x80\x80\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] =
        "\x2a\x0a\xe3\x80\x80\x2a\xe3\x80\x80\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x80\x2a\xe3\x80\x80\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingJapanesePeriodTextNoOverwriteTest()

  @brief        半角句点を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingJapanesePeriodTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        "｡わ｡る｡も｡し｡\n"
        "ああわるもしああ"
        "ああ｡わ｡る｡も｡し｡ああ";

    char szExpect[] =
        "*\n"                  // 単純にマッチングするテスト
        "｡*｡\n"                // 間に半角句読点が含まれていても、そこは無視してマスクされます。
        "ああ*ああ"
        "ああ｡*｡ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbd\xa1\xe3\x82\x8f\xef\xbd\xa1\xe3\x82\x8b\xef\xbd\xa1\xe3\x82\x82\xef\xbd\xa1\xe3\x81\x97\xef\xbd\xa1\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\xa1\xe3\x82\x8f\xef\xbd\xa1\xe3\x82\x8b\xef\xbd\xa1\xe3\x82\x82\xef\xbd\xa1\xe3\x81\x97\xef\xbd\xa1\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] =
        "\x2a\x0a\xef\xbd\xa1\x2a\xef\xbd\xa1\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\xa1\x2a\xef\xbd\xa1\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthJapanesePeriodTextNoOverwriteTest()

  @brief        全角句点を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthJapanesePeriodTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        "。わ。る。も。し。\n"
        "ああわるもしああ"
        "ああ。わ。る。も。し。ああ";

    char szExpect[] =
        "*\n"                  // 単純にマッチングするテスト
        "。*。\n"              // 間に全角句読点が含まれていても、そこは無視してマスクされます。
        "ああ*ああ"
         "ああ。*。ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe3\x80\x82\xe3\x82\x8f\xe3\x80\x82\xe3\x82\x8b\xe3\x80\x82\xe3\x82\x82\xe3\x80\x82\xe3\x81\x97\xe3\x80\x82\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x82\xe3\x82\x8f\xe3\x80\x82\xe3\x82\x8b\xe3\x80\x82\xe3\x82\x82\xe3\x80\x82\xe3\x81\x97\xe3\x80\x82\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] =
        "\x2a\x0a\xe3\x80\x82\x2a\xe3\x80\x82\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x82\x2a\xe3\x80\x82\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingJapaneseCommaTextNoOverwriteTest()

  @brief        半角読点を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingJapaneseCommaTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
         "わるもし\n"
         "､わ､る､も､し､\n"
         "ああわるもしああ"
         "ああ､わ､る､も､し､ああ";

    char szExpect[] =
         "*\n"                  // 単純にマッチングするテスト
         "､*､\n"                // 間に半角句読点が含まれていても、そこは無視してマスクされます。
         "ああ*ああ"
         "ああ､*､ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbd\xa4\xe3\x82\x8f\xef\xbd\xa4\xe3\x82\x8b\xef\xbd\xa4\xe3\x82\x82\xef\xbd\xa4\xe3\x81\x97\xef\xbd\xa4\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\xa4\xe3\x82\x8f\xef\xbd\xa4\xe3\x82\x8b\xef\xbd\xa4\xe3\x82\x82\xef\xbd\xa4\xe3\x81\x97\xef\xbd\xa4\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] =
        "\x2a\x0a\xef\xbd\xa4\x2a\xef\xbd\xa4\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\xa4\x2a\xef\xbd\xa4\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthJapaneseCommaTextNoOverwriteTest()

  @brief        全角読点を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthJapaneseCommaTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        "、わ、る、も、し、\n"
        "ああわるもしああ"
        "ああ、わ、る、も、し、ああ";

    char szExpect[] =
        "*\n"                  // 単純にマッチングするテスト
        "、*、\n"              // 間に全角句読点が含まれていても、そこは無視してマスクされます。
        "ああ*ああ"
        "ああ、*、ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe3\x80\x81\xe3\x82\x8f\xe3\x80\x81\xe3\x82\x8b\xe3\x80\x81\xe3\x82\x82\xe3\x80\x81\xe3\x81\x97\xe3\x80\x81\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x81\xe3\x82\x8f\xe3\x80\x81\xe3\x82\x8b\xe3\x80\x81\xe3\x82\x82\xe3\x80\x81\xe3\x81\x97\xe3\x80\x81\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] =
        "\x2a\x0a\xe3\x80\x81\x2a\xe3\x80\x81\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x80\x81\x2a\xe3\x80\x81\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingPeriodTextNoOverwriteTest()

  @brief        半角ピリオドを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingPeriodTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        ".わ.る.も.し.\n"
        "ああわるもしああ"
        "ああ.わ.る.も.し.ああ";

    char szExpect[] =
        "*\n"                  // 単純にマッチングするテスト
        ".*.\n"                // 間に半角ピリオドが含まれていても、そこは無視してマスクされます。
        "ああ*ああ"
        "ああ.*.ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x2e\xe3\x82\x8f\x2e\xe3\x82\x8b\x2e\xe3\x82\x82\x2e\xe3\x81\x97\x2e\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2e\xe3\x82\x8f\x2e\xe3\x82\x8b\x2e\xe3\x82\x82\x2e\xe3\x81\x97\x2e\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] =
        "\x2a\x0a\x2e\x2a\x2e\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2e\x2a\x2e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthPeriodTextNoOverwriteTest()

  @brief        全角ピリオドを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthPeriodTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        "．わ．る．も．し．\n"
        "ああわるもしああ"
        "ああ．わ．る．も．し．ああ";

    char szExpect[] =
        "*\n"                  // 単純にマッチングするテスト
        "．*．\n"              // 間に全角ピリオドが含まれていても、そこは無視してマスクされます。
        "ああ*ああ"
        "ああ．*．ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x8e\xe3\x82\x8f\xef\xbc\x8e\xe3\x82\x8b\xef\xbc\x8e\xe3\x82\x82\xef\xbc\x8e\xe3\x81\x97\xef\xbc\x8e\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8e\xe3\x82\x8f\xef\xbc\x8e\xe3\x82\x8b\xef\xbc\x8e\xe3\x82\x82\xef\xbc\x8e\xe3\x81\x97\xef\xbc\x8e\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] =
        "\x2a\x0a\xef\xbc\x8e\x2a\xef\xbc\x8e\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8e\x2a\xef\xbc\x8e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingCommaTextNoOverwriteTest()

  @brief        半角カンマを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingCommaTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        ",わ,る,も,し,\n"
        "ああわるもしああ"
        "ああ,わ,る,も,し,ああ";

    char szExpect[] =
        "*\n"                  // 単純にマッチングするテスト
        ",*,\n"                // 間に半角カンマが含まれていても、そこは無視してマスクされます。
        "ああ*ああ"
        "ああ,*,ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x2c\xe3\x82\x8f\x2c\xe3\x82\x8b\x2c\xe3\x82\x82\x2c\xe3\x81\x97\x2c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2c\xe3\x82\x8f\x2c\xe3\x82\x8b\x2c\xe3\x82\x82\x2c\xe3\x81\x97\x2c\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] =
        "\x2a\x0a\x2c\x2a\x2c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2c\x2a\x2c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthCommaTextNoOverwriteTest()

  @brief        全角カンマを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthCommaTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        "，わ，る，も，し，\n"
        "ああわるもしああ"
        "ああ，わ，る，も，し，ああ";

    char szExpect[] =
        "*\n"                  // 単純にマッチングするテスト
        "，*，\n"              // 間に全角カンマが含まれていても、そこは無視してマスクされます。
        "ああ*ああ"
        "ああ，*，ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x8c\xe3\x82\x8f\xef\xbc\x8c\xe3\x82\x8b\xef\xbc\x8c\xe3\x82\x82\xef\xbc\x8c\xe3\x81\x97\xef\xbc\x8c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8c\xe3\x82\x8f\xef\xbc\x8c\xe3\x82\x8b\xef\xbc\x8c\xe3\x82\x82\xef\xbc\x8c\xe3\x81\x97\xef\xbc\x8c\xe3\x81\x82\xe3\x81\x82";

    char szExpect[] =
        "\x2a\x0a\xef\xbc\x8c\x2a\xef\xbc\x8c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8c\x2a\xef\xbc\x8c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif
    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingOpenKakuKatsukoTextTest()

  @brief        半角角カッコ(開く)を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingOpenKakuKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "[わ[る[も[し[\n"
        "ああわるもしああ"
        "ああ[わ[る[も[し[ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x5b\xe3\x82\x8f\x5b\xe3\x82\x8b\x5b\xe3\x82\x82\x5b\xe3\x81\x97\x5b\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5b\xe3\x82\x8f\x5b\xe3\x82\x8b\x5b\xe3\x82\x82\x5b\xe3\x81\x97\x5b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "[*******[\n"              // 間に半角角カッコ(開く)が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ[*******[ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x5b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5b\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthOpenKakuKatsukoTextTest()

  @brief        全角角カッコ(開く)を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthOpenKakuKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "［わ［る［も［し［\n"
        "ああわるもしああ"
        "ああ［わ［る［も［し［ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\xbb\xe3\x82\x8f\xef\xbc\xbb\xe3\x82\x8b\xef\xbc\xbb\xe3\x82\x82\xef\xbc\xbb\xe3\x81\x97\xef\xbc\xbb\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xbb\xe3\x82\x8f\xef\xbc\xbb\xe3\x82\x8b\xef\xbc\xbb\xe3\x82\x82\xef\xbc\xbb\xe3\x81\x97\xef\xbc\xbb\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "［*******［\n"            // 間に角カッコ(開く)が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ［*******［ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\xbb\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xbb\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xbb\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xbb\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingExclamationTextTest()

  @brief        半角エクスクラメーションを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingExclamationTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "!わ!る!も!し!\n"
        "ああわるもしああ"
        "ああ!わ!る!も!し!ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x21\xe3\x82\x8f\x21\xe3\x82\x8b\x21\xe3\x82\x82\x21\xe3\x81\x97\x21\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x21\xe3\x82\x8f\x21\xe3\x82\x8b\x21\xe3\x82\x82\x21\xe3\x81\x97\x21\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "!*******!\n"              // 間に半角エクスクラメーションが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ!*******!ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x21\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x21\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x21\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x21\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthExclamationTextTest()

  @brief        全角エクスクラメーションを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthExclamationTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "！わ！る！も！し！\n"
        "ああわるもしああ"
        "ああ！わ！る！も！し！ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x81\xe3\x82\x8f\xef\xbc\x81\xe3\x82\x8b\xef\xbc\x81\xe3\x82\x82\xef\xbc\x81\xe3\x81\x97\xef\xbc\x81\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x81\xe3\x82\x8f\xef\xbc\x81\xe3\x82\x8b\xef\xbc\x81\xe3\x82\x82\xef\xbc\x81\xe3\x81\x97\xef\xbc\x81\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "！*******！\n"            // 間に全角エクスクラメーションが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ！*******！ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x81\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x81\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x81\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x81\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingDoubleQuotationTextTest()

  @brief        半角ダブルクオテーションを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingDoubleQuotationTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "\"わ\"る\"も\"し\"\n"
        "ああわるもしああ"
        "ああ\"わ\"る\"も\"し\"ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x22\xe3\x82\x8f\x22\xe3\x82\x8b\x22\xe3\x82\x82\x22\xe3\x81\x97\x22\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x22\xe3\x82\x8f\x22\xe3\x82\x8b\x22\xe3\x82\x82\x22\xe3\x81\x97\x22\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "\"*******\"\n"            // 間に半角ダブルクオテーションが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ\"*******\"ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x22\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x22\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x22\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x22\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthOpenDoubleQuotationTextTest()

  @brief        全角ダブルクオテーション（開く）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthOpenDoubleQuotationTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "“わ“る“も“し“\n"
        "ああわるもしああ"
        "ああ“わ“る“も“し“ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe2\x80\x9c\xe3\x82\x8f\xe2\x80\x9c\xe3\x82\x8b\xe2\x80\x9c\xe3\x82\x82\xe2\x80\x9c\xe3\x81\x97\xe2\x80\x9c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe2\x80\x9c\xe3\x82\x8f\xe2\x80\x9c\xe3\x82\x8b\xe2\x80\x9c\xe3\x82\x82\xe2\x80\x9c\xe3\x81\x97\xe2\x80\x9c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "“*******“\n"            // 間に全角ダブルクオテーション（開く）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ“*******“ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xe2\x80\x9c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe2\x80\x9c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe2\x80\x9c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe2\x80\x9c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthCloseDoubleQuotationTextTest()

  @brief        全角ダブルクオテーション（閉じる）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthCloseDoubleQuotationTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "”わ”る”も”し”\n"
        "ああわるもしああ"
        "ああ”わ”る”も”し”ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe2\x80\x9d\xe3\x82\x8f\xe2\x80\x9d\xe3\x82\x8b\xe2\x80\x9d\xe3\x82\x82\xe2\x80\x9d\xe3\x81\x97\xe2\x80\x9d\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe2\x80\x9d\xe3\x82\x8f\xe2\x80\x9d\xe3\x82\x8b\xe2\x80\x9d\xe3\x82\x82\xe2\x80\x9d\xe3\x81\x97\xe2\x80\x9d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "”*******”\n"            // 間に全角ダブルクオテーション（閉じる）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ”*******”ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xe2\x80\x9d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe2\x80\x9d\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe2\x80\x9d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe2\x80\x9d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingSharpTextTest()

  @brief        半角シャープを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingSharpTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "#わ#る#も#し#\n"
        "ああわるもしああ"
        "ああ#わ#る#も#し#ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x23\xe3\x82\x8f\x23\xe3\x82\x8b\x23\xe3\x82\x82\x23\xe3\x81\x97\x23\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x23\xe3\x82\x8f\x23\xe3\x82\x8b\x23\xe3\x82\x82\x23\xe3\x81\x97\x23\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "#*******#\n"              // 間に半角シャープが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ#*******#ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x23\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x23\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x23\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x23\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthSharpTextTest()

  @brief        全角シャープを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthSharpTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＃わ＃る＃も＃し＃\n"
        "ああわるもしああ"
        "ああ＃わ＃る＃も＃し＃ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x83\xe3\x82\x8f\xef\xbc\x83\xe3\x82\x8b\xef\xbc\x83\xe3\x82\x82\xef\xbc\x83\xe3\x81\x97\xef\xbc\x83\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x83\xe3\x82\x8f\xef\xbc\x83\xe3\x82\x8b\xef\xbc\x83\xe3\x82\x82\xef\xbc\x83\xe3\x81\x97\xef\xbc\x83\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＃*******＃\n"            // 間に全角シャープが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＃*******＃ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x83\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x83\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x83\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x83\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingDollarTextTest()

  @brief        半角ダラーを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingDollarTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "$わ$る$も$し$\n"
        "ああわるもしああ"
        "ああ$わ$る$も$し$ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x24\xe3\x82\x8f\x24\xe3\x82\x8b\x24\xe3\x82\x82\x24\xe3\x81\x97\x24\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x24\xe3\x82\x8f\x24\xe3\x82\x8b\x24\xe3\x82\x82\x24\xe3\x81\x97\x24\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "$*******$\n"              // 間に半角ダラーが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ$*******$ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x24\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x24\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x24\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x24\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthDollarTextTest()

  @brief        全角ダラーを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthDollarTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＄わ＄る＄も＄し＄\n"
        "ああわるもしああ"
        "ああ＄わ＄る＄も＄し＄ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x84\xe3\x82\x8f\xef\xbc\x84\xe3\x82\x8b\xef\xbc\x84\xe3\x82\x82\xef\xbc\x84\xe3\x81\x97\xef\xbc\x84\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x84\xe3\x82\x8f\xef\xbc\x84\xe3\x82\x8b\xef\xbc\x84\xe3\x82\x82\xef\xbc\x84\xe3\x81\x97\xef\xbc\x84\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＄*******＄\n"            // 間に全角ダラーが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＄*******＄ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x84\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x84\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x84\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x84\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingPercentTextTest()

  @brief        半角パーセントを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingPercentTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "%わ%る%も%し%\n"
        "ああわるもしああ"
        "ああ%わ%る%も%し%ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x25\xe3\x82\x8f\x25\xe3\x82\x8b\x25\xe3\x82\x82\x25\xe3\x81\x97\x25\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x25\xe3\x82\x8f\x25\xe3\x82\x8b\x25\xe3\x82\x82\x25\xe3\x81\x97\x25\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "%*******%\n"              // 間に半角パーセントが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ%*******%ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x25\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x25\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x25\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x25\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthPercentTextTest()

  @brief        全角パーセントを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthPercentTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "％わ％る％も％し％\n"
        "ああわるもしああ"
        "ああ％わ％る％も％し％ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x85\xe3\x82\x8f\xef\xbc\x85\xe3\x82\x8b\xef\xbc\x85\xe3\x82\x82\xef\xbc\x85\xe3\x81\x97\xef\xbc\x85\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x85\xe3\x82\x8f\xef\xbc\x85\xe3\x82\x8b\xef\xbc\x85\xe3\x82\x82\xef\xbc\x85\xe3\x81\x97\xef\xbc\x85\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "％*******％\n"            // 間に全角パーセントが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ％*******％ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x85\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x85\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x85\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x85\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingAmpersandTextTest()

  @brief        半角アンパサンドを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingAmpersandTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "&わ&る&も&し&\n"
        "ああわるもしああ"
        "ああ&わ&る&も&し&ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x26\xe3\x82\x8f\x26\xe3\x82\x8b\x26\xe3\x82\x82\x26\xe3\x81\x97\x26\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x26\xe3\x82\x8f\x26\xe3\x82\x8b\x26\xe3\x82\x82\x26\xe3\x81\x97\x26\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "&*******&\n"              // 間に半角アンパサンドが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ&*******&ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x26\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x26\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x26\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x26\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthAmpersandTextTest()

  @brief        全角アンパサンドを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthAmpersandTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＆わ＆る＆も＆し＆\n"
        "ああわるもしああ"
        "ああ＆わ＆る＆も＆し＆ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x86\xe3\x82\x8f\xef\xbc\x86\xe3\x82\x8b\xef\xbc\x86\xe3\x82\x82\xef\xbc\x86\xe3\x81\x97\xef\xbc\x86\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x86\xe3\x82\x8f\xef\xbc\x86\xe3\x82\x8b\xef\xbc\x86\xe3\x82\x82\xef\xbc\x86\xe3\x81\x97\xef\xbc\x86\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＆*******＆\n"            // 間に全角アンパサンドが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＆*******＆ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x86\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x86\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x86\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x86\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingSingleSingleQuotationTextTest()

  @brief        半角シングルクオテーションを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingSingleSingleQuotationTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "'わ'る'も'し'\n"
        "ああわるもしああ"
        "ああ'わ'る'も'し'ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x27\xe3\x82\x8f\x27\xe3\x82\x8b\x27\xe3\x82\x82\x27\xe3\x81\x97\x27\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x27\xe3\x82\x8f\x27\xe3\x82\x8b\x27\xe3\x82\x82\x27\xe3\x81\x97\x27\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "'*******'\n"              // 間に半角シングルクオテーションが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ'*******'ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x27\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x27\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x27\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x27\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}


/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthOpenSingleQuotationTextTest()

  @brief        全角シングルクオテーション(開く)を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthOpenSingleQuotationTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "‘わ‘る‘も‘し‘\n"
        "ああわるもしああ"
        "ああ‘わ‘る‘も‘し‘ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe2\x80\x98\xe3\x82\x8f\xe2\x80\x98\xe3\x82\x8b\xe2\x80\x98\xe3\x82\x82\xe2\x80\x98\xe3\x81\x97\xe2\x80\x98\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe2\x80\x98\xe3\x82\x8f\xe2\x80\x98\xe3\x82\x8b\xe2\x80\x98\xe3\x82\x82\xe2\x80\x98\xe3\x81\x97\xe2\x80\x98\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "‘*******‘\n"            // 間に全角シングルクオテーション(開く)が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ‘*******‘ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xe2\x80\x98\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe2\x80\x98\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe2\x80\x98\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe2\x80\x98\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthCloseSingleQuotationTextTest()

@brief        全角シングルクオテーション(閉じる)を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthCloseSingleQuotationTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "’わ’る’も’し’\n"
        "ああわるもしああ"
        "ああ’わ’る’も’し’ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xe2\x80\x99\xe3\x82\x8f\xe2\x80\x99\xe3\x82\x8b\xe2\x80\x99\xe3\x82\x82\xe2\x80\x99\xe3\x81\x97\xe2\x80\x99\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe2\x80\x99\xe3\x82\x8f\xe2\x80\x99\xe3\x82\x8b\xe2\x80\x99\xe3\x82\x82\xe2\x80\x99\xe3\x81\x97\xe2\x80\x99\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "’*******’\n"            // 間に全角シングルクオテーション(閉じる)が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ’*******’ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xe2\x80\x99\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe2\x80\x99\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe2\x80\x99\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe2\x80\x99\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingOpenMaruKatsukoTextTest()

  @brief        半角丸カッコ（開く）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingOpenMaruKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "(わ(る(も(し(\n"
        "ああわるもしああ"
        "ああ(わ(る(も(し(ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x28\xe3\x82\x8f\x28\xe3\x82\x8b\x28\xe3\x82\x82\x28\xe3\x81\x97\x28\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x28\xe3\x82\x8f\x28\xe3\x82\x8b\x28\xe3\x82\x82\x28\xe3\x81\x97\x28\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "(*******(\n"              // 間に半角丸カッコ（開く）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ(*******(ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x28\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x28\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x28\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x28\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthOpenMaruKatsukoTextTest()

  @brief        全角丸カッコ（開く）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthOpenMaruKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "（わ（る（も（し（\n"
        "ああわるもしああ"
        "ああ（わ（る（も（し（ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x88\xe3\x82\x8f\xef\xbc\x88\xe3\x82\x8b\xef\xbc\x88\xe3\x82\x82\xef\xbc\x88\xe3\x81\x97\xef\xbc\x88\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x88\xe3\x82\x8f\xef\xbc\x88\xe3\x82\x8b\xef\xbc\x88\xe3\x82\x82\xef\xbc\x88\xe3\x81\x97\xef\xbc\x88\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "（*******（\n"            // 間に全角丸カッコ（開く）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ（*******（ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x88\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x88\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x88\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x88\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingCloseMaruKatsukoTextTest()

  @brief        半角丸カッコ（閉じる）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingCloseMaruKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        ")わ)る)も)し)\n"
        "ああわるもしああ"
        "ああ)わ)る)も)し)ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x29\xe3\x82\x8f\x29\xe3\x82\x8b\x29\xe3\x82\x82\x29\xe3\x81\x97\x29\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x29\xe3\x82\x8f\x29\xe3\x82\x8b\x29\xe3\x82\x82\x29\xe3\x81\x97\x29\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        ")*******)\n"              // 間に半角丸カッコ（閉じる）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ)*******)ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x29\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x29\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x29\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x29\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthCloseMaruKatsukoTextTest()

  @brief        全角半角丸カッコ（閉じる）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthCloseMaruKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "）わ）る）も）し）\n"
        "ああわるもしああ"
        "ああ）わ）る）も）し）ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x89\xe3\x82\x8f\xef\xbc\x89\xe3\x82\x8b\xef\xbc\x89\xe3\x82\x82\xef\xbc\x89\xe3\x81\x97\xef\xbc\x89\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x89\xe3\x82\x8f\xef\xbc\x89\xe3\x82\x8b\xef\xbc\x89\xe3\x82\x82\xef\xbc\x89\xe3\x81\x97\xef\xbc\x89\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "）*******）\n"            // 間に全角丸カッコ（閉じる）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ）*******）ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x89\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x89\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x89\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x89\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingAsteriskTextTest()

  @brief        半角アスタリスクを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingAsteriskTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "*わ*る*も*し*\n"
        "ああわるもしああ"
        "ああ*わ*る*も*し*ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x2a\xe3\x82\x8f\x2a\xe3\x82\x8b\x2a\xe3\x82\x82\x2a\xe3\x81\x97\x2a\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x82\x8f\x2a\xe3\x82\x8b\x2a\xe3\x82\x82\x2a\xe3\x81\x97\x2a\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "*********\n"              // 間に半角アスタリスクが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ*********ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthAsteriskTextTest()

  @brief        全角アスタリスクを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthAsteriskTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＊わ＊る＊も＊し＊\n"
        "ああわるもしああ"
        "ああ＊わ＊る＊も＊し＊ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x8a\xe3\x82\x8f\xef\xbc\x8a\xe3\x82\x8b\xef\xbc\x8a\xe3\x82\x82\xef\xbc\x8a\xe3\x81\x97\xef\xbc\x8a\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8a\xe3\x82\x8f\xef\xbc\x8a\xe3\x82\x8b\xef\xbc\x8a\xe3\x82\x82\xef\xbc\x8a\xe3\x81\x97\xef\xbc\x8a\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＊*******＊\n"            // 間に全角アスタリスクが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＊*******＊ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x8a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8a\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8a\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingPlusTextTest()

  @brief        半角プラスを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingPlusTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "+わ+る+も+し+\n"
        "ああわるもしああ"
        "ああ+わ+る+も+し+ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x2b\xe3\x82\x8f\x2b\xe3\x82\x8b\x2b\xe3\x82\x82\x2b\xe3\x81\x97\x2b\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2b\xe3\x82\x8f\x2b\xe3\x82\x8b\x2b\xe3\x82\x82\x2b\xe3\x81\x97\x2b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "+*******+\n"              // 間に半角プラスが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ+*******+ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x2b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2b\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthPlusTextTest()

  @brief        全角プラスを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthPlusTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＋わ＋る＋も＋し＋\n"
        "ああわるもしああ"
        "ああ＋わ＋る＋も＋し＋ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x8b\xe3\x82\x8f\xef\xbc\x8b\xe3\x82\x8b\xef\xbc\x8b\xe3\x82\x82\xef\xbc\x8b\xe3\x81\x97\xef\xbc\x8b\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8b\xe3\x82\x8f\xef\xbc\x8b\xe3\x82\x8b\xef\xbc\x8b\xe3\x82\x82\xef\xbc\x8b\xe3\x81\x97\xef\xbc\x8b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＋*******＋\n"            // 間に全角プラスが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＋*******＋ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x8b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8b\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingSlashTextTest()

  @brief        半角スラッシュを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingSlashTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "/わ/る/も/し/\n"
        "ああわるもしああ"
        "ああ/わ/る/も/し/ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x2f\xe3\x82\x8f\x2f\xe3\x82\x8b\x2f\xe3\x82\x82\x2f\xe3\x81\x97\x2f\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2f\xe3\x82\x8f\x2f\xe3\x82\x8b\x2f\xe3\x82\x82\x2f\xe3\x81\x97\x2f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "/*******/\n"              // 間に半角スラッシュが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ/*******/ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x2f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2f\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthSlashTextTest()

  @brief        全角スラッシュを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthSlashTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "／わ／る／も／し／\n"
        "ああわるもしああ"
        "ああ／わ／る／も／し／ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x8f\xe3\x82\x8f\xef\xbc\x8f\xe3\x82\x8b\xef\xbc\x8f\xe3\x82\x82\xef\xbc\x8f\xe3\x81\x97\xef\xbc\x8f\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8f\xe3\x82\x8f\xef\xbc\x8f\xe3\x82\x8b\xef\xbc\x8f\xe3\x82\x82\xef\xbc\x8f\xe3\x81\x97\xef\xbc\x8f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "／*******／\n"            // 間に全角スラッシュが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ／*******／ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x8f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8f\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingColonTextTest()

  @brief        半角コロンを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingColonTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        ":わ:る:も:し:\n"
        "ああわるもしああ"
        "ああ:わ:る:も:し:ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x3a\xe3\x82\x8f\x3a\xe3\x82\x8b\x3a\xe3\x82\x82\x3a\xe3\x81\x97\x3a\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3a\xe3\x82\x8f\x3a\xe3\x82\x8b\x3a\xe3\x82\x82\x3a\xe3\x81\x97\x3a\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        ":*******:\n"              // 間に半角コロンが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ:*******:ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x3a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3a\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3a\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthColonTextTest()

  @brief        全角コロンを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthColonTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "：わ：る：も：し：\n"
        "ああわるもしああ"
        "ああ：わ：る：も：し：ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x9a\xe3\x82\x8f\xef\xbc\x9a\xe3\x82\x8b\xef\xbc\x9a\xe3\x82\x82\xef\xbc\x9a\xe3\x81\x97\xef\xbc\x9a\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9a\xe3\x82\x8f\xef\xbc\x9a\xe3\x82\x8b\xef\xbc\x9a\xe3\x82\x82\xef\xbc\x9a\xe3\x81\x97\xef\xbc\x9a\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "：*******：\n"            // 間に全角コロンが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ：*******：ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x9a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9a\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9a\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9a\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingSemicolonTextTest()

  @brief        半角セミコロンを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingSemicolonTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        ";わ;る;も;し;\n"
        "ああわるもしああ"
        "ああ;わ;る;も;し;ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x3b\xe3\x82\x8f\x3b\xe3\x82\x8b\x3b\xe3\x82\x82\x3b\xe3\x81\x97\x3b\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3b\xe3\x82\x8f\x3b\xe3\x82\x8b\x3b\xe3\x82\x82\x3b\xe3\x81\x97\x3b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        ";*******;\n"              // 間に半角セミコロンが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ;*******;ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x3b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3b\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthSemicolonTextTest()

  @brief        全角セミコロンを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthSemicolonTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "；わ；る；も；し；\n"
        "ああわるもしああ"
        "ああ；わ；る；も；し；ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x9b\xe3\x82\x8f\xef\xbc\x9b\xe3\x82\x8b\xef\xbc\x9b\xe3\x82\x82\xef\xbc\x9b\xe3\x81\x97\xef\xbc\x9b\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9b\xe3\x82\x8f\xef\xbc\x9b\xe3\x82\x8b\xef\xbc\x9b\xe3\x82\x82\xef\xbc\x9b\xe3\x81\x97\xef\xbc\x9b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "；*******；\n"            // 間に全角セミコロンが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ；*******；ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x9b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9b\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingOpenKagiKatsukoTextTest()

  @brief        半角カギカッコ（開く）カンマを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingOpenKagiKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "<わ<る<も<し<\n"
        "ああわるもしああ"
        "ああ<わ<る<も<し<ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x3c\xe3\x82\x8f\x3c\xe3\x82\x8b\x3c\xe3\x82\x82\x3c\xe3\x81\x97\x3c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3c\xe3\x82\x8f\x3c\xe3\x82\x8b\x3c\xe3\x82\x82\x3c\xe3\x81\x97\x3c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "<*******<\n"              // 間に半角カギカッコ（開く）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ<*******<ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x3c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthOpenKagiKatsukoTextTest()

  @brief        全角カギカッコ（開く）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthOpenKagiKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＜わ＜る＜も＜し＜\n"
        "ああわるもしああ"
        "ああ＜わ＜る＜も＜し＜ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x9c\xe3\x82\x8f\xef\xbc\x9c\xe3\x82\x8b\xef\xbc\x9c\xe3\x82\x82\xef\xbc\x9c\xe3\x81\x97\xef\xbc\x9c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9c\xe3\x82\x8f\xef\xbc\x9c\xe3\x82\x8b\xef\xbc\x9c\xe3\x82\x82\xef\xbc\x9c\xe3\x81\x97\xef\xbc\x9c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＜*******＜\n"            // 間に全角カギカッコ（開く）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＜*******＜ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x9c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingEqualTextTest()

  @brief        半角イコールを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingEqualTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "=わ=る=も=し=\n"
        "ああわるもしああ"
        "ああ=わ=る=も=し=ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x3d\xe3\x82\x8f\x3d\xe3\x82\x8b\x3d\xe3\x82\x82\x3d\xe3\x81\x97\x3d\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3d\xe3\x82\x8f\x3d\xe3\x82\x8b\x3d\xe3\x82\x82\x3d\xe3\x81\x97\x3d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "=*******=\n"              // 間に半角イコールが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ=*******=ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x3d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3d\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthEqualTextTest()

  @brief        全角イコールを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthEqualTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＝わ＝る＝も＝し＝\n"
        "ああわるもしああ"
        "ああ＝わ＝る＝も＝し＝ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x9d\xe3\x82\x8f\xef\xbc\x9d\xe3\x82\x8b\xef\xbc\x9d\xe3\x82\x82\xef\xbc\x9d\xe3\x81\x97\xef\xbc\x9d\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9d\xe3\x82\x8f\xef\xbc\x9d\xe3\x82\x8b\xef\xbc\x9d\xe3\x82\x82\xef\xbc\x9d\xe3\x81\x97\xef\xbc\x9d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＝*******＝\n"            // 間に全角イコールが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＝*******＝ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x9d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9d\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingCloseKagiKatsukoTextTest()

  @brief        半角カギカッコ（閉じる）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingCloseKagiKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        ">わ>る>も>し>\n"
        "ああわるもしああ"
        "ああ>わ>る>も>し>ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x3e\xe3\x82\x8f\x3e\xe3\x82\x8b\x3e\xe3\x82\x82\x3e\xe3\x81\x97\x3e\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3e\xe3\x82\x8f\x3e\xe3\x82\x8b\x3e\xe3\x82\x82\x3e\xe3\x81\x97\x3e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        ">*******>\n"              // 間に半角カギカッコ（閉じる）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ>*******>ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x3e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3e\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthCloseKagiKatsukoTextTest()

  @brief        全角カギカッコ（閉じる）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthCloseKagiKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＞わ＞る＞も＞し＞\n"
        "ああわるもしああ"
        "ああ＞わ＞る＞も＞し＞ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x9e\xe3\x82\x8f\xef\xbc\x9e\xe3\x82\x8b\xef\xbc\x9e\xe3\x82\x82\xef\xbc\x9e\xe3\x81\x97\xef\xbc\x9e\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9e\xe3\x82\x8f\xef\xbc\x9e\xe3\x82\x8b\xef\xbc\x9e\xe3\x82\x82\xef\xbc\x9e\xe3\x81\x97\xef\xbc\x9e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＞*******＞\n"            // 間に全角カギカッコ（閉じる）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＞*******＞ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x9e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9e\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingQuestionTextTest()

  @brief        半角クエスチョンを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingQuestionTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "?わ?る?も?し?\n"
        "ああわるもしああ"
        "ああ?わ?る?も?し?ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x3f\xe3\x82\x8f\x3f\xe3\x82\x8b\x3f\xe3\x82\x82\x3f\xe3\x81\x97\x3f\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3f\xe3\x82\x8f\x3f\xe3\x82\x8b\x3f\xe3\x82\x82\x3f\xe3\x81\x97\x3f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "?*******?\n"              // 間に半角クエスチョンが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ?*******?ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x3f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3f\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x3f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x3f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthQuestionTextTest()

  @brief        全角クエスチョンを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthQuestionTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "？わ？る？も？し？\n"
        "ああわるもしああ"
        "ああ？わ？る？も？し？ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x9f\xe3\x82\x8f\xef\xbc\x9f\xe3\x82\x8b\xef\xbc\x9f\xe3\x82\x82\xef\xbc\x9f\xe3\x81\x97\xef\xbc\x9f\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9f\xe3\x82\x8f\xef\xbc\x9f\xe3\x82\x8b\xef\xbc\x9f\xe3\x82\x82\xef\xbc\x9f\xe3\x81\x97\xef\xbc\x9f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "？*******？\n"            // 間に全角クエスチョンが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ？*******？ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x9f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9f\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x9f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x9f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingAtMarkTextTest()

  @brief        半角アットマークを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingAtMarkTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "@わ@る@も@し@\n"
        "ああわるもしああ"
        "ああ@わ@る@も@し@ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x40\xe3\x82\x8f\x40\xe3\x82\x8b\x40\xe3\x82\x82\x40\xe3\x81\x97\x40\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x40\xe3\x82\x8f\x40\xe3\x82\x8b\x40\xe3\x82\x82\x40\xe3\x81\x97\x40\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "@*******@\n"              // 間に半角アットマークが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ@*******@ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x40\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x40\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x40\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x40\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthAtMarkTextTest()

  @brief        全角アットマークを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthAtMarkTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＠わ＠る＠も＠し＠\n"
        "ああわるもしああ"
        "ああ＠わ＠る＠も＠し＠ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\xa0\xe3\x82\x8f\xef\xbc\xa0\xe3\x82\x8b\xef\xbc\xa0\xe3\x82\x82\xef\xbc\xa0\xe3\x81\x97\xef\xbc\xa0\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xa0\xe3\x82\x8f\xef\xbc\xa0\xe3\x82\x8b\xef\xbc\xa0\xe3\x82\x82\xef\xbc\xa0\xe3\x81\x97\xef\xbc\xa0\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＠*******＠\n"            // 間に全角アットマークが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＠*******＠ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\xa0\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xa0\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xa0\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xa0\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingYenTextTest()

  @brief        半角円を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingYenTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "\\わ\\る\\も\\し\\\n"
        "ああわるもしああ"
        "ああ\\わ\\る\\も\\し\\ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x5c\xe3\x82\x8f\x5c\xe3\x82\x8b\x5c\xe3\x82\x82\x5c\xe3\x81\x97\x5c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5c\xe3\x82\x8f\x5c\xe3\x82\x8b\x5c\xe3\x82\x82\x5c\xe3\x81\x97\x5c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "\\*******\\\n"            // 間に半角円が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ\\*******\\ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x5c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthYenTextTest()

  @brief        全角円を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthYenTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "￥わ￥る￥も￥し￥\n"
        "ああわるもしああ"
        "ああ￥わ￥る￥も￥し￥ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbf\xa5\xe3\x82\x8f\xef\xbf\xa5\xe3\x82\x8b\xef\xbf\xa5\xe3\x82\x82\xef\xbf\xa5\xe3\x81\x97\xef\xbf\xa5\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbf\xa5\xe3\x82\x8f\xef\xbf\xa5\xe3\x82\x8b\xef\xbf\xa5\xe3\x82\x82\xef\xbf\xa5\xe3\x81\x97\xef\xbf\xa5\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "￥*******￥\n"            // 間に全角円が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ￥*******￥ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbf\xa5\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbf\xa5\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbf\xa5\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbf\xa5\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingCulletTextTest()

  @brief        半角カレットを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingCulletTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "^わ^る^も^し^\n"
        "ああわるもしああ"
        "ああ^わ^る^も^し^ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x5e\xe3\x82\x8f\x5e\xe3\x82\x8b\x5e\xe3\x82\x82\x5e\xe3\x81\x97\x5e\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5e\xe3\x82\x8f\x5e\xe3\x82\x8b\x5e\xe3\x82\x82\x5e\xe3\x81\x97\x5e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "^*******^\n"              // 間に半角カレットが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ^*******^ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x5e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5e\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthCulletTextTest()

  @brief        全角カレットを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthCulletTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＾わ＾る＾も＾し＾\n"
        "ああわるもしああ"
        "ああ＾わ＾る＾も＾し＾ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\xbe\xe3\x82\x8f\xef\xbc\xbe\xe3\x82\x8b\xef\xbc\xbe\xe3\x82\x82\xef\xbc\xbe\xe3\x81\x97\xef\xbc\xbe\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xbe\xe3\x82\x8f\xef\xbc\xbe\xe3\x82\x8b\xef\xbc\xbe\xe3\x82\x82\xef\xbc\xbe\xe3\x81\x97\xef\xbc\xbe\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＾*******＾\n"            // 間に全角カレットが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＾*******＾ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\xbe\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xbe\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xbe\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xbe\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingUnderScoreTextTest()

  @brief        半角アンダースコアを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingUnderScoreTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "_わ_る_も_し_\n"
        "ああわるもしああ"
        "ああ_わ_る_も_し_ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x5f\xe3\x82\x8f\x5f\xe3\x82\x8b\x5f\xe3\x82\x82\x5f\xe3\x81\x97\x5f\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5f\xe3\x82\x8f\x5f\xe3\x82\x8b\x5f\xe3\x82\x82\x5f\xe3\x81\x97\x5f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "_*******_\n"              // 間に半角アンダースコアが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ_*******_ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x5f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5f\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5f\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5f\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthUnderScoreTextTest()

  @brief        全角アンダースコアを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthUnderScoreTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "＿わ＿る＿も＿し＿\n"
        "ああわるもしああ"
        "ああ＿わ＿る＿も＿し＿ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\xbf\xe3\x82\x8f\xef\xbc\xbf\xe3\x82\x8b\xef\xbc\xbf\xe3\x82\x82\xef\xbc\xbf\xe3\x81\x97\xef\xbc\xbf\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xbf\xe3\x82\x8f\xef\xbc\xbf\xe3\x82\x8b\xef\xbc\xbf\xe3\x82\x82\xef\xbc\xbf\xe3\x81\x97\xef\xbc\xbf\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "＿*******＿\n"            // 間に全角アンダースコアが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ＿*******＿ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\xbf\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xbf\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xbf\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xbf\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingBackAutoTextTest()

  @brief        半角バッククォートを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingBackquoteTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "`わ`る`も`し`\n"
        "ああわるもしああ"
        "ああ`わ`る`も`し`ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x60\xe3\x82\x8f\x60\xe3\x82\x8b\x60\xe3\x82\x82\x60\xe3\x81\x97\x60\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x60\xe3\x82\x8f\x60\xe3\x82\x8b\x60\xe3\x82\x82\x60\xe3\x81\x97\x60\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "`*******`\n"              // 間に半角バッククォートが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ`*******`ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x60\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x60\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x60\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x60\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthBackquoteTextTest()

  @brief        全角バッククォートを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthBackquoteTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "｀わ｀る｀も｀し｀\n"
        "ああわるもしああ"
        "ああ｀わ｀る｀も｀し｀ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbd\x80\xe3\x82\x8f\xef\xbd\x80\xe3\x82\x8b\xef\xbd\x80\xe3\x82\x82\xef\xbd\x80\xe3\x81\x97\xef\xbd\x80\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x80\xe3\x82\x8f\xef\xbd\x80\xe3\x82\x8b\xef\xbd\x80\xe3\x82\x82\xef\xbd\x80\xe3\x81\x97\xef\xbd\x80\xe3\x81\x82\xe3\x81\x82\x00";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "｀*******｀\n"            // 間に全角バッククォートが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ｀*******｀ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbd\x80\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x80\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x80\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x80\xe3\x81\x82\xe3\x81\x82\x00";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingOpenMiddleKatsukoTextTest()

  @brief        半角中カッコ（開く）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingOpenMiddleKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "{わ{る{も{し{\n"
        "ああわるもしああ"
        "ああ{わ{る{も{し{ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x7b\xe3\x82\x8f\x7b\xe3\x82\x8b\x7b\xe3\x82\x82\x7b\xe3\x81\x97\x7b\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x7b\xe3\x82\x8f\x7b\xe3\x82\x8b\x7b\xe3\x82\x82\x7b\xe3\x81\x97\x7b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "{*******{\n"              // 間に半角中カッコ（開く）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ{*******{ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x7b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x7b\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x7b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x7b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthOpenMiddleKatsukoTextTest()

  @brief        全角中カッコ（開く）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthOpenMiddleKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "｛わ｛る｛も｛し｛\n"
        "ああわるもしああ"
        "ああ｛わ｛る｛も｛し｛ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbd\x9b\xe3\x82\x8f\xef\xbd\x9b\xe3\x82\x8b\xef\xbd\x9b\xe3\x82\x82\xef\xbd\x9b\xe3\x81\x97\xef\xbd\x9b\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x9b\xe3\x82\x8f\xef\xbd\x9b\xe3\x82\x8b\xef\xbd\x9b\xe3\x82\x82\xef\xbd\x9b\xe3\x81\x97\xef\xbd\x9b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "｛*******｛\n"            // 間に全角中カッコ（開く）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ｛*******｛ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbd\x9b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x9b\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x9b\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x9b\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingPipeLineTextTest()

  @brief        半角パイプラインを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingPipeLineTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "|わ|る|も|し|\n"
        "ああわるもしああ"
        "ああ|わ|る|も|し|ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x7c\xe3\x82\x8f\x7c\xe3\x82\x8b\x7c\xe3\x82\x82\x7c\xe3\x81\x97\x7c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x7c\xe3\x82\x8f\x7c\xe3\x82\x8b\x7c\xe3\x82\x82\x7c\xe3\x81\x97\x7c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "|*******|\n"              // 間に半角パイプラインが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ|*******|ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x7c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x7c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x7c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x7c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthPipeLineTextTest()

  @brief        全角パイプラインを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthPipeLineTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "｜わ｜る｜も｜し｜\n"
        "ああわるもしああ"
        "ああ｜わ｜る｜も｜し｜ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbd\x9c\xe3\x82\x8f\xef\xbd\x9c\xe3\x82\x8b\xef\xbd\x9c\xe3\x82\x82\xef\xbd\x9c\xe3\x81\x97\xef\xbd\x9c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x9c\xe3\x82\x8f\xef\xbd\x9c\xe3\x82\x8b\xef\xbd\x9c\xe3\x82\x82\xef\xbd\x9c\xe3\x81\x97\xef\xbd\x9c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "｜*******｜\n"            // 間に全角パイプラインが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ｜*******｜ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbd\x9c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x9c\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x9c\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x9c\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingCloseMiddleKatsukoTextTest()

  @brief        半角中カッコ（閉じる）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingCloseMiddleKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "}わ}る}も}し}\n"
        "ああわるもしああ"
        "ああ}わ}る}も}し}ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x7d\xe3\x82\x8f\x7d\xe3\x82\x8b\x7d\xe3\x82\x82\x7d\xe3\x81\x97\x7d\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x7d\xe3\x82\x8f\x7d\xe3\x82\x8b\x7d\xe3\x82\x82\x7d\xe3\x81\x97\x7d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "}*******}\n"              // 間に半角中カッコ（閉じる）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ}*******}ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x7d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x7d\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x7d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x7d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthCloseMiddleKatsukoTextTest()

  @brief        全角中カッコ（閉じる）を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthCloseMiddleKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "｝わ｝る｝も｝し｝\n"
        "ああわるもしああ"
        "ああ｝わ｝る｝も｝し｝ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbd\x9d\xe3\x82\x8f\xef\xbd\x9d\xe3\x82\x8b\xef\xbd\x9d\xe3\x82\x82\xef\xbd\x9d\xe3\x81\x97\xef\xbd\x9d\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x9d\xe3\x82\x8f\xef\xbd\x9d\xe3\x82\x8b\xef\xbd\x9d\xe3\x82\x82\xef\xbd\x9d\xe3\x81\x97\xef\xbd\x9d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "｝*******｝\n"            // 間に全角中カッコ（閉じる）が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ｝*******｝ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbd\x9d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x9d\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x9d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x9d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingTildeTextTest()

  @brief        半角チルダを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingTildeTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "~わ~る~も~し~\n"
        "ああわるもしああ"
        "ああ~わ~る~も~し~ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x7e\xe3\x82\x8f\x7e\xe3\x82\x8b\x7e\xe3\x82\x82\x7e\xe3\x81\x97\x7e\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x7e\xe3\x82\x8f\x7e\xe3\x82\x8b\x7e\xe3\x82\x82\x7e\xe3\x81\x97\x7e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "~*******~\n"              // 間に半角チルダが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ~*******~ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x7e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x7e\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x7e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x7e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthTildeTextTest()

  @brief        全角チルダを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthTildeTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "～わ～る～も～し～\n"
        "ああわるもしああ"
        "ああ～わ～る～も～し～ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbd\x9e\xe3\x82\x8f\xef\xbd\x9e\xe3\x82\x8b\xef\xbd\x9e\xe3\x82\x82\xef\xbd\x9e\xe3\x81\x97\xef\xbd\x9e\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x9e\xe3\x82\x8f\xef\xbd\x9e\xe3\x82\x8b\xef\xbd\x9e\xe3\x82\x82\xef\xbd\x9e\xe3\x81\x97\xef\xbd\x9e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "～*******～\n"            // 間に全角チルダが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ～*******～ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbd\x9e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x9e\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbd\x9e\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbd\x9e\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingMinusTextTest()

  @brief        半角マイナスを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingMinusTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "-わ-る-も-し-\n"
        "ああわるもしああ"
        "ああ-わ-る-も-し-ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x2d\xe3\x82\x8f\x2d\xe3\x82\x8b\x2d\xe3\x82\x82\x2d\xe3\x81\x97\x2d\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2d\xe3\x82\x8f\x2d\xe3\x82\x8b\x2d\xe3\x82\x82\x2d\xe3\x81\x97\x2d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "-*******-\n"              // 間に半角マイナスが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ-*******-ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x2d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2d\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x2d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x2d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthMinusTextTest()

  @brief        全角マイナスを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthMinusTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "－わ－る－も－し－\n"
        "ああわるもしああ"
        "ああ－わ－る－も－し－ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\x8d\xe3\x82\x8f\xef\xbc\x8d\xe3\x82\x8b\xef\xbc\x8d\xe3\x82\x82\xef\xbc\x8d\xe3\x81\x97\xef\xbc\x8d\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8d\xe3\x82\x8f\xef\xbc\x8d\xe3\x82\x8b\xef\xbc\x8d\xe3\x82\x82\xef\xbc\x8d\xe3\x81\x97\xef\xbc\x8d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "－*******－\n"            // 間に全角マイナスが含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ－*******－ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\x8d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8d\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\x8d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\x8d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingCloseKakuKatsukoTextTest()

  @brief        半角角カッコ(閉じる)を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingCloseKakuKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "]わ]る]も]し]\n"
        "ああわるもしああ"
        "ああ]わ]る]も]し]ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x5d\xe3\x82\x8f\x5d\xe3\x82\x8b\x5d\xe3\x82\x82\x5d\xe3\x81\x97\x5d\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5d\xe3\x82\x8f\x5d\xe3\x82\x8b\x5d\xe3\x82\x82\x5d\xe3\x81\x97\x5d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "]*******]\n"              // 間に半角角カッコ(閉じる)が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ]*******]ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\x5d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5d\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x5d\x2a\x2a\x2a\x2a\x2a\x2a\x2a\x5d\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingFullwidthCloseKakuKatsukoTextTest()

  @brief        全角角カッコ(閉じる)を含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingFullwidthCloseKakuKatsukoTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "わるもし\n"
        "］わ］る］も］し］\n"
        "ああわるもしああ"
        "ああ］わ］る］も］し］ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\xef\xbc\xbd\xe3\x82\x8f\xef\xbc\xbd\xe3\x82\x8b\xef\xbc\xbd\xe3\x82\x82\xef\xbc\xbd\xe3\x81\x97\xef\xbc\xbd\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xbd\xe3\x82\x8f\xef\xbc\xbd\xe3\x82\x8b\xef\xbc\xbd\xe3\x82\x82\xef\xbc\xbd\xe3\x81\x97\xef\xbc\xbd\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    char szExpect[] =
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "****\n"                   // 単純にマッチングするテスト
        "］*******］\n"            // 間に全角角カッコ(閉じる)が含まれていても、そこは無視してマスクされます。
        "ああ****ああ"
        "ああ］*******］ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\x2a\x2a\x2a\x2a\x0a\xef\xbc\xbd\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xbd\x0a\xe3\x81\x82\xe3\x81\x82\x2a\x2a\x2a\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xef\xbc\xbd\x2a\x2a\x2a\x2a\x2a\x2a\x2a\xef\xbc\xbd\xe3\x81\x82\xe3\x81\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         IncludingMixedBlankCodeTextNoOverwriteTest()

  @brief        全空白対象コードを含むテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, IncludingMixedBlankCodeTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        " ｡､.,[!\"#$%&'()*+/:;<=>?@\\^_`{|}~-]　。、．，［！“”＃＄％＆‘’（）＊＋／：；＜＝＞？＠￥＾＿｀｛｜｝～－］わ ｡､.,[!\"#$%&'()*+/:;<=>?@\\^_`{|}~-]　。、．，［！“”＃＄％＆‘’（）＊＋／：；＜＝＞？＠￥＾＿｀｛｜｝～－］る ｡､.,[!\"#$%&'()*+/:;<=>?@\\^_`{|}~-]　。、．，［！“”＃＄％＆‘’（）＊＋／：；＜＝＞？＠￥＾＿｀｛｜｝～－］も ｡､.,[!\"#$%&'()*+/:;<=>?@\\^_`{|}~-]　。、．，［！“”＃＄％＆‘’（）＊＋／：；＜＝＞？＠￥＾＿｀｛｜｝～－］し ｡､.,[!\"#$%&'()*+/:;<=>?@\\^_`{|}~-]　。、．，［！“”＃＄％＆‘’（）＊＋／：；＜＝＞？＠￥＾＿｀｛｜｝～－］\n"
        "ああわるもしああ"
        "ああ ｡､.,[!\"#わ$%&'()*+/:;<=>?@る\\^_`{|}~-]も　。、．，［！“”＃＄％＆‘’（）し＊＋／：；＜＝＞？＠￥＾＿｀｛｜｝～－］ああ";
    char szExpect[] =
        "*\n"                                 // 単純にマッチングするテスト
        " ｡､.,[!\"#$%&'()*+/:;<=>?@\\^_`{|}~-]　。、．，［！“”＃＄％＆‘’（）＊＋／：；＜＝＞？＠￥＾＿｀｛｜｝～－］* ｡､.,[!\"#$%&'()*+/:;<=>?@\\^_`{|}~-]　。、．，［！“”＃＄％＆‘’（）＊＋／：；＜＝＞？＠￥＾＿｀｛｜｝～－］\n"
        "ああ*ああ"
        "ああ ｡､.,[!\"#*＊＋／：；＜＝＞？＠￥＾＿｀｛｜｝～－］ああ";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\x5b\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2f\x3a\x3b\x3c\x3d\x3e\x3f\x40\x5c\x5e\x5f\x60\x7b\x7c\x7d\x7e\x2d\x5d\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xef\xbc\xbb\xef\xbc\x81\xe2\x80\x9c\xe2\x80\x9d\xef\xbc\x83\xef\xbc\x84\xef\xbc\x85\xef\xbc\x86\xe2\x80\x98\xe2\x80\x99\xef\xbc\x88\xef\xbc\x89\xef\xbc\x8a\xef\xbc\x8b\xef\xbc\x8f\xef\xbc\x9a\xef\xbc\x9b\xef\xbc\x9c\xef\xbc\x9d\xef\xbc\x9e\xef\xbc\x9f\xef\xbc\xa0\xef\xbf\xa5\xef\xbc\xbe\xef\xbc\xbf\xef\xbd\x80\xef\xbd\x9b\xef\xbd\x9c\xef\xbd\x9d\xef\xbd\x9e\xef\xbc\x8d\xef\xbc\xbd\xe3\x82\x8f\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\x5b\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2f\x3a\x3b\x3c\x3d\x3e\x3f\x40\x5c\x5e\x5f\x60\x7b\x7c\x7d\x7e\x2d\x5d\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xef\xbc\xbb\xef\xbc\x81\xe2\x80\x9c\xe2\x80\x9d\xef\xbc\x83\xef\xbc\x84\xef\xbc\x85\xef\xbc\x86\xe2\x80\x98\xe2\x80\x99\xef\xbc\x88\xef\xbc\x89\xef\xbc\x8a\xef\xbc\x8b\xef\xbc\x8f\xef\xbc\x9a\xef\xbc\x9b\xef\xbc\x9c\xef\xbc\x9d\xef\xbc\x9e\xef\xbc\x9f\xef\xbc\xa0\xef\xbf\xa5\xef\xbc\xbe\xef\xbc\xbf\xef\xbd\x80\xef\xbd\x9b\xef\xbd\x9c\xef\xbd\x9d\xef\xbd\x9e\xef\xbc\x8d\xef\xbc\xbd\xe3\x82\x8b\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\x5b\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2f\x3a\x3b\x3c\x3d\x3e\x3f\x40\x5c\x5e\x5f\x60\x7b\x7c\x7d\x7e\x2d\x5d\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xef\xbc\xbb\xef\xbc\x81\xe2\x80\x9c\xe2\x80\x9d\xef\xbc\x83\xef\xbc\x84\xef\xbc\x85\xef\xbc\x86\xe2\x80\x98\xe2\x80\x99\xef\xbc\x88\xef\xbc\x89\xef\xbc\x8a\xef\xbc\x8b\xef\xbc\x8f\xef\xbc\x9a\xef\xbc\x9b\xef\xbc\x9c\xef\xbc\x9d\xef\xbc\x9e\xef\xbc\x9f\xef\xbc\xa0\xef\xbf\xa5\xef\xbc\xbe\xef\xbc\xbf\xef\xbd\x80\xef\xbd\x9b\xef\xbd\x9c\xef\xbd\x9d\xef\xbd\x9e\xef\xbc\x8d\xef\xbc\xbd\xe3\x82\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\x5b\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2f\x3a\x3b\x3c\x3d\x3e\x3f\x40\x5c\x5e\x5f\x60\x7b\x7c\x7d\x7e\x2d\x5d\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xef\xbc\xbb\xef\xbc\x81\xe2\x80\x9c\xe2\x80\x9d\xef\xbc\x83\xef\xbc\x84\xef\xbc\x85\xef\xbc\x86\xe2\x80\x98\xe2\x80\x99\xef\xbc\x88\xef\xbc\x89\xef\xbc\x8a\xef\xbc\x8b\xef\xbc\x8f\xef\xbc\x9a\xef\xbc\x9b\xef\xbc\x9c\xef\xbc\x9d\xef\xbc\x9e\xef\xbc\x9f\xef\xbc\xa0\xef\xbf\xa5\xef\xbc\xbe\xef\xbc\xbf\xef\xbd\x80\xef\xbd\x9b\xef\xbd\x9c\xef\xbd\x9d\xef\xbd\x9e\xef\xbc\x8d\xef\xbc\xbd\xe3\x81\x97\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\x5b\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2f\x3a\x3b\x3c\x3d\x3e\x3f\x40\x5c\x5e\x5f\x60\x7b\x7c\x7d\x7e\x2d\x5d\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xef\xbc\xbb\xef\xbc\x81\xe2\x80\x9c\xe2\x80\x9d\xef\xbc\x83\xef\xbc\x84\xef\xbc\x85\xef\xbc\x86\xe2\x80\x98\xe2\x80\x99\xef\xbc\x88\xef\xbc\x89\xef\xbc\x8a\xef\xbc\x8b\xef\xbc\x8f\xef\xbc\x9a\xef\xbc\x9b\xef\xbc\x9c\xef\xbc\x9d\xef\xbc\x9e\xef\xbc\x9f\xef\xbc\xa0\xef\xbf\xa5\xef\xbc\xbe\xef\xbc\xbf\xef\xbd\x80\xef\xbd\x9b\xef\xbd\x9c\xef\xbd\x9d\xef\xbd\x9e\xef\xbc\x8d\xef\xbc\xbd\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\x5b\x21\x22\x23\xe3\x82\x8f\x24\x25\x26\x27\x28\x29\x2a\x2b\x2f\x3a\x3b\x3c\x3d\x3e\x3f\x40\xe3\x82\x8b\x5c\x5e\x5f\x60\x7b\x7c\x7d\x7e\x2d\x5d\xe3\x82\x82\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xef\xbc\xbb\xef\xbc\x81\xe2\x80\x9c\xe2\x80\x9d\xef\xbc\x83\xef\xbc\x84\xef\xbc\x85\xef\xbc\x86\xe2\x80\x98\xe2\x80\x99\xef\xbc\x88\xef\xbc\x89\xe3\x81\x97\xef\xbc\x8a\xef\xbc\x8b\xef\xbc\x8f\xef\xbc\x9a\xef\xbc\x9b\xef\xbc\x9c\xef\xbc\x9d\xef\xbc\x9e\xef\xbc\x9f\xef\xbc\xa0\xef\xbf\xa5\xef\xbc\xbe\xef\xbc\xbf\xef\xbd\x80\xef\xbd\x9b\xef\xbd\x9c\xef\xbd\x9d\xef\xbd\x9e\xef\xbc\x8d\xef\xbc\xbd\xe3\x81\x82\xe3\x81\x82";
    char szExpect[] =
        "\x2a\x0a\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\x5b\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2f\x3a\x3b\x3c\x3d\x3e\x3f\x40\x5c\x5e\x5f\x60\x7b\x7c\x7d\x7e\x2d\x5d\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xef\xbc\xbb\xef\xbc\x81\xe2\x80\x9c\xe2\x80\x9d\xef\xbc\x83\xef\xbc\x84\xef\xbc\x85\xef\xbc\x86\xe2\x80\x98\xe2\x80\x99\xef\xbc\x88\xef\xbc\x89\xef\xbc\x8a\xef\xbc\x8b\xef\xbc\x8f\xef\xbc\x9a\xef\xbc\x9b\xef\xbc\x9c\xef\xbc\x9d\xef\xbc\x9e\xef\xbc\x9f\xef\xbc\xa0\xef\xbf\xa5\xef\xbc\xbe\xef\xbc\xbf\xef\xbd\x80\xef\xbd\x9b\xef\xbd\x9c\xef\xbd\x9d\xef\xbd\x9e\xef\xbc\x8d\xef\xbc\xbd\x2a\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\x5b\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2f\x3a\x3b\x3c\x3d\x3e\x3f\x40\x5c\x5e\x5f\x60\x7b\x7c\x7d\x7e\x2d\x5d\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xef\xbc\xbb\xef\xbc\x81\xe2\x80\x9c\xe2\x80\x9d\xef\xbc\x83\xef\xbc\x84\xef\xbc\x85\xef\xbc\x86\xe2\x80\x98\xe2\x80\x99\xef\xbc\x88\xef\xbc\x89\xef\xbc\x8a\xef\xbc\x8b\xef\xbc\x8f\xef\xbc\x9a\xef\xbc\x9b\xef\xbc\x9c\xef\xbc\x9d\xef\xbc\x9e\xef\xbc\x9f\xef\xbc\xa0\xef\xbf\xa5\xef\xbc\xbe\xef\xbc\xbf\xef\xbd\x80\xef\xbd\x9b\xef\xbd\x9c\xef\xbd\x9d\xef\xbd\x9e\xef\xbc\x8d\xef\xbc\xbd\x0a\xe3\x81\x82\xe3\x81\x82\x2a\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\x5b\x21\x22\x23\x2a\xef\xbc\x8a\xef\xbc\x8b\xef\xbc\x8f\xef\xbc\x9a\xef\xbc\x9b\xef\xbc\x9c\xef\xbc\x9d\xef\xbc\x9e\xef\xbc\x9f\xef\xbc\xa0\xef\xbf\xa5\xef\xbc\xbe\xef\xbc\xbf\xef\xbd\x80\xef\xbd\x9b\xef\xbd\x9c\xef\xbd\x9d\xef\xbd\x9e\xef\xbc\x8d\xef\xbc\xbd\xe3\x81\x82\xe3\x81\x82";

#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 4 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         LotCheckRegionTextNoOverwriteTest()

  @brief        ロットチェック用の文字列がマスクされるかどうかの確認テストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, LotCheckRegionTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
    char szText[] =
        "bbwjja\n"
        " bbwjja \n";

    char szExpect[] =
        "*\n"
        " * \n";

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 2 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         HalfWidthTextNoOverwriteTest()

  @brief        半角カナを含む文字列が正しくマスク出来るかどうかを調べるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, HalfWidthTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] = "あいうえおﾚｲﾌﾟ。ﾔﾘﾏﾝ。";

    char szExpect[] = "あいうえお*。*。";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] = "\xe3\x81\x82\xe3\x81\x84\xe3\x81\x86\xe3\x81\x88\xe3\x81\x8a\xef\xbe\x9a\xef\xbd\xb2\xef\xbe\x8c\xef\xbe\x9f\xe3\x80\x82\xef\xbe\x94\xef\xbe\x98\xef\xbe\x8f\xef\xbe\x9d\xe3\x80\x82";

    char szExpect[] = "\xe3\x81\x82\xe3\x81\x84\xe3\x81\x86\xe3\x81\x88\xe3\x81\x8a\x2a\xe3\x80\x82\x2a\xe3\x80\x82";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 2 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         CheckAllListTextNoOverwriteTest()

  @brief        すべてのNGワードパターンについて、チェックを行うテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckAllListTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    char szText[] =
            "Japanese => badwordj bbwjja\n"
            "AmericanEnglish => badworde bbween\n"
            "CanadianFrench => badworde bbwefr\n"
            "LatinAmericanSpanish => badworde bbwees\n"
            "BritishEnglish => badwordp bbwpen\n"
            "French => badwordp bbwpfr\n"
            "German => badwordp bbwpde\n"
            "Italian => badwordp bbwpit\n"
            "Spanish => badwordp bbwpes\n"
            "Dutch => badwordp bbwpnl\n"
            "Korean => badwordk bbwkko\n"
            "Chinese => badwordc bbwczh\n"
            "Portuguese => badwordp bbwppt\n"
            "Russian => badwordp bbwpru\n"
            "Taiwanese => badwordt bbwtzh\n"
            "SouthAmericanPortuguese => badworde bbwept\n"
            "Common => badworda";

    char szExpect[] =
            "Japanese => * *\n"
            "AmericanEnglish => * *\n"
            "CanadianFrench => * *\n"
            "LatinAmericanSpanish => * *\n"
            "BritishEnglish => * *\n"
            "French => * *\n"
            "German => * *\n"
            "Italian => * *\n"
            "Spanish => * *\n"
            "Dutch => * *\n"
            "Korean => * *\n"
            "Chinese => * *\n"
            "Portuguese => * *\n"
            "Russian => * *\n"
            "Taiwanese => * *\n"
            "SouthAmericanPortuguese => * *\n"
            "Common => *";

    result = filter.MaskProfanityWordsInText( nullptr, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         CheckMyLanguageListTextNoOverwriteTest()

  @brief        現在のリージョン設定に基づいてチェックを行うテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckMyLanguageListTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
    char szText[] = "this is badword";

    char szExpect[] = "this is *";

    result = filter.MaskProfanityWordsInText( &nWordCount, szText, 0 );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 1 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}


/*!--------------------------------------------------------------------------*
  Name:         CheckMailAddressTextNoOverwriteTest()

  @brief        メールアドレスのようなものが現れたらマスキングされるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckMailAddressTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
"こんにちは。はじめまして。\n"
"一緒にゲームをしていて あなたのことが(以下略)\n"
"もしよかったらhoge@foobar.netまでメールをくれませんか？\n"
"偽物のアドレスhoge@foobarはマスクされてはいけません！";

    char szExpect[] =
"こんにちは。はじめまして。\n"
"一緒にゲームをしていて あなたのことが(以下略)\n"
"もしよかったら*までメールをくれませんか？\n"
"偽物のアドレスhoge@foobarはマスクされてはいけません！";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] = "\xe3\x81\x93\xe3\x82\x93\xe3\x81\xab\xe3\x81\xa1\xe3\x81\xaf\xe3\x80\x82\xe3\x81\xaf\xe3\x81\x98\xe3\x82\x81\xe3\x81\xbe\xe3\x81\x97\xe3\x81\xa6\xe3\x80\x82\x0a\xe4\xb8\x80\xe7\xb7\x92\xe3\x81\xab\xe3\x82\xb2\xe3\x83\xbc\xe3\x83\xa0\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\xa6\x20\xe3\x81\x82\xe3\x81\xaa\xe3\x81\x9f\xe3\x81\xae\xe3\x81\x93\xe3\x81\xa8\xe3\x81\x8c\x28\xe4\xbb\xa5\xe4\xb8\x8b\xe7\x95\xa5\x29\x0a\xe3\x82\x82\xe3\x81\x97\xe3\x82\x88\xe3\x81\x8b\xe3\x81\xa3\xe3\x81\x9f\xe3\x82\x89\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\x2e\x6e\x65\x74\xe3\x81\xbe\xe3\x81\xa7\xe3\x83\xa1\xe3\x83\xbc\xe3\x83\xab\xe3\x82\x92\xe3\x81\x8f\xe3\x82\x8c\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xe3\x81\x8b\xef\xbc\x9f\x0a\xe5\x81\xbd\xe7\x89\xa9\xe3\x81\xae\xe3\x82\xa2\xe3\x83\x89\xe3\x83\xac\xe3\x82\xb9\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\xe3\x81\xaf\xe3\x83\x9e\xe3\x82\xb9\xe3\x82\xaf\xe3\x81\x95\xe3\x82\x8c\xe3\x81\xa6\xe3\x81\xaf\xe3\x81\x84\xe3\x81\x91\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xef\xbc\x81";

    char szExpect[] = "\xe3\x81\x93\xe3\x82\x93\xe3\x81\xab\xe3\x81\xa1\xe3\x81\xaf\xe3\x80\x82\xe3\x81\xaf\xe3\x81\x98\xe3\x82\x81\xe3\x81\xbe\xe3\x81\x97\xe3\x81\xa6\xe3\x80\x82\x0a\xe4\xb8\x80\xe7\xb7\x92\xe3\x81\xab\xe3\x82\xb2\xe3\x83\xbc\xe3\x83\xa0\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\xa6\x20\xe3\x81\x82\xe3\x81\xaa\xe3\x81\x9f\xe3\x81\xae\xe3\x81\x93\xe3\x81\xa8\xe3\x81\x8c\x28\xe4\xbb\xa5\xe4\xb8\x8b\xe7\x95\xa5\x29\x0a\xe3\x82\x82\xe3\x81\x97\xe3\x82\x88\xe3\x81\x8b\xe3\x81\xa3\xe3\x81\x9f\xe3\x82\x89\x2a\xe3\x81\xbe\xe3\x81\xa7\xe3\x83\xa1\xe3\x83\xbc\xe3\x83\xab\xe3\x82\x92\xe3\x81\x8f\xe3\x82\x8c\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xe3\x81\x8b\xef\xbc\x9f\x0a\xe5\x81\xbd\xe7\x89\xa9\xe3\x81\xae\xe3\x82\xa2\xe3\x83\x89\xe3\x83\xac\xe3\x82\xb9\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\xe3\x81\xaf\xe3\x83\x9e\xe3\x82\xb9\xe3\x82\xaf\xe3\x81\x95\xe3\x82\x8c\xe3\x81\xa6\xe3\x81\xaf\xe3\x81\x84\xe3\x81\x91\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xef\xbc\x81";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText( &nWordCount, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 1 );
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         SkipMailAddressTextNoOverwriteTest()

  @brief        メールアドレスのマッチングをスキップするテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, SkipMailAddressTextNoOverwriteTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    filter.SkipAtSignCheck( nn::ngc::ProfanityFilter::SkipMode_SkipAtSign );
    filter.SetMaskMode( nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter );

    int nWordCount;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
"こんにちは。はじめまして。\n"
"一緒にゲームをしていて あなたのことが(以下略)\n"
"もしよかったらhoge@foobar.netまでメールをくれませんか？\n"
"偽物のアドレスhoge@foobarはマスクされてはいけません！";

    char szExpect[] =
"こんにちは。はじめまして。\n"
"一緒にゲームをしていて あなたのことが(以下略)\n"
"もしよかったらhoge@foobar.netまでメールをくれませんか？\n"
"偽物のアドレスhoge@foobarはマスクされてはいけません！";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] = "\xe3\x81\x93\xe3\x82\x93\xe3\x81\xab\xe3\x81\xa1\xe3\x81\xaf\xe3\x80\x82\xe3\x81\xaf\xe3\x81\x98\xe3\x82\x81\xe3\x81\xbe\xe3\x81\x97\xe3\x81\xa6\xe3\x80\x82\x0a\xe4\xb8\x80\xe7\xb7\x92\xe3\x81\xab\xe3\x82\xb2\xe3\x83\xbc\xe3\x83\xa0\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\xa6\x20\xe3\x81\x82\xe3\x81\xaa\xe3\x81\x9f\xe3\x81\xae\xe3\x81\x93\xe3\x81\xa8\xe3\x81\x8c\x28\xe4\xbb\xa5\xe4\xb8\x8b\xe7\x95\xa5\x29\x0a\xe3\x82\x82\xe3\x81\x97\xe3\x82\x88\xe3\x81\x8b\xe3\x81\xa3\xe3\x81\x9f\xe3\x82\x89\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\x2e\x6e\x65\x74\xe3\x81\xbe\xe3\x81\xa7\xe3\x83\xa1\xe3\x83\xbc\xe3\x83\xab\xe3\x82\x92\xe3\x81\x8f\xe3\x82\x8c\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xe3\x81\x8b\xef\xbc\x9f\x0a\xe5\x81\xbd\xe7\x89\xa9\xe3\x81\xae\xe3\x82\xa2\xe3\x83\x89\xe3\x83\xac\xe3\x82\xb9\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\xe3\x81\xaf\xe3\x83\x9e\xe3\x82\xb9\xe3\x82\xaf\xe3\x81\x95\xe3\x82\x8c\xe3\x81\xa6\xe3\x81\xaf\xe3\x81\x84\xe3\x81\x91\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xef\xbc\x81";

    char szExpect[] = "\xe3\x81\x93\xe3\x82\x93\xe3\x81\xab\xe3\x81\xa1\xe3\x81\xaf\xe3\x80\x82\xe3\x81\xaf\xe3\x81\x98\xe3\x82\x81\xe3\x81\xbe\xe3\x81\x97\xe3\x81\xa6\xe3\x80\x82\x0a\xe4\xb8\x80\xe7\xb7\x92\xe3\x81\xab\xe3\x82\xb2\xe3\x83\xbc\xe3\x83\xa0\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\xa6\x20\xe3\x81\x82\xe3\x81\xaa\xe3\x81\x9f\xe3\x81\xae\xe3\x81\x93\xe3\x81\xa8\xe3\x81\x8c\x28\xe4\xbb\xa5\xe4\xb8\x8b\xe7\x95\xa5\x29\x0a\xe3\x82\x82\xe3\x81\x97\xe3\x82\x88\xe3\x81\x8b\xe3\x81\xa3\xe3\x81\x9f\xe3\x82\x89\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\x2e\x6e\x65\x74\xe3\x81\xbe\xe3\x81\xa7\xe3\x83\xa1\xe3\x83\xbc\xe3\x83\xab\xe3\x82\x92\xe3\x81\x8f\xe3\x82\x8c\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xe3\x81\x8b\xef\xbc\x9f\x0a\xe5\x81\xbd\xe7\x89\xa9\xe3\x81\xae\xe3\x82\xa2\xe3\x83\x89\xe3\x83\xac\xe3\x82\xb9\x68\x6f\x67\x65\x40\x66\x6f\x6f\x62\x61\x72\xe3\x81\xaf\xe3\x83\x9e\xe3\x82\xb9\xe3\x82\xaf\xe3\x81\x95\xe3\x82\x8c\xe3\x81\xa6\xe3\x81\xaf\xe3\x81\x84\xe3\x81\x91\xe3\x81\xbe\xe3\x81\x9b\xe3\x82\x93\xef\xbc\x81";
#else
#error "unsupported os"
#endif

    result = filter.MaskProfanityWordsInText( &nWordCount, szText, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 0 );      // スキップフラグがオンなので検出されません。
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         SkipMailAddressTextNoOverwriteTest()

  @brief        長いメールアドレスのマスキングのテストです。
  @details      メールアドレスの最大長はローカル部で最大 64 文字、ドメイン部で 255 文字、
                メールアドレス全体で 256 文字までという制限がある
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, LongMailAddressTextTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    int nWordCount;
    char szText1[] =
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
"@aa.jp\n";
    char szText2[] =
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
"aaaaaaaaaaaaaaaaaaaaa@aaaaaaaaaaaaaaaaaaaaaaaaaaaa"
"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
"aaaaaaaaaaaaaaaaaaaaaa.aaaaaaaaaaaaaaaaaaaaaaaaaaa"
"aaa.jp\n";

    char szExpect1[] =
"**************************************************"
"**************************************************"
"**************************************************"
"**************************************************"
"**************************************************"
"******\n";
    char szExpect2[] = "*\n";

    result = filter.MaskProfanityWordsInText( &nWordCount, szText1, 0xFFFFFFFF );
    NNT_ASSERT_RESULT_SUCCESS( result );
    EXPECT_TRUE( nWordCount == 1 );
    EXPECT_TRUE( std::strcmp( szExpect1, szText1 ) == 0 );

    filter.SetMaskMode(nn::ngc::ProfanityFilter::MaskMode_ReplaceByOneCharacter);
    result = filter.MaskProfanityWordsInText(&nWordCount, szText2, 0xFFFFFFFF);
    NNT_ASSERT_RESULT_SUCCESS(result);
    EXPECT_TRUE( nWordCount == 1 );
    EXPECT_TRUE( std::strcmp( szExpect2, szText2 ) == 0);
}

/*!--------------------------------------------------------------------------*
  Name:         SpecialCharacterTest()

  @brief        特殊文字のチェックのテスト
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, SpecialCharacterTest)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    ASSERT_TRUE( result.IsSuccess() );

    const char* ppConstWords[] =
    {
        "hard-on",
        "hard - on"
    };

    char szText[] =
        "hard-on\n"
        "hard - on";

    char szExpect[] =
        "*******\n"
        "*********";

    const size_t word_num = 2;

    // 答え合わせ
    for(int i = 0; i < word_num; i++)
    {
        // 単語チェック
        nn::Bit32 results [ 1 ];
        result = filter.CheckProfanityWords( results,
                                            nn::ngc::ProfanityFilterPatternList_AmericanEnglish,
                                            &ppConstWords[i],
                                            1 );
        NNT_ASSERT_RESULT_SUCCESS( result );
        ASSERT_TRUE( results[ 0 ] != 0);
    }

    int nWordCount = 0;
    result = filter.MaskProfanityWordsInText( &nWordCount,
                                              szText,
                                              nn::ngc::ProfanityFilterPatternList_AmericanEnglish);
    NNT_ASSERT_RESULT_SUCCESS( result );
    ASSERT_TRUE( nWordCount == 2);
    ASSERT_TRUE( std::strcmp( szExpect, szText ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         WorkMemoryTest

  @brief        ワークメモリテストを行います。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, WorkMemoryTest)
{
    const int InstanceCount = 10;
    static char s_WorkMemory[InstanceCount][nn::ngc::ProfanityFilter::WorkMemorySize];
    void* pWorkMemoryAddr[InstanceCount];
    nn::ngc::ProfanityFilter filter[InstanceCount];
    nn::Result result;

    const char* ppWords[] =
    {
        "mario",
        "badword"
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    for(int i=0; i < InstanceCount; ++i)
    {
        pWorkMemoryAddr[i] = s_WorkMemory[i];
        result = filter[i].Initialize(pWorkMemoryAddr[i], nn::ngc::ProfanityFilter::WorkMemorySize);
        NNT_ASSERT_RESULT_SUCCESS( result );
        result = filter[i].CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
        NNT_ASSERT_RESULT_SUCCESS( result );
        // 答え合わせ
        ASSERT_TRUE( results[ 0 ] == 0 );
        ASSERT_TRUE( results[ 1 ] != 0 );
    }

    // 最後の一つだけ残し、正常に動作するか確認
    for(int i=0; i < InstanceCount - 1; ++i)
    {
        filter[i].Finalize();
        memset(s_WorkMemory[i], 0x00, nn::ngc::ProfanityFilter::WorkMemorySize);
    }

    result = filter[InstanceCount - 1].CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         AlreadyInitializedConstructor1Test

  @brief        ProfanityFilter::ProfanityFilter()
                同インスタンスで2回以上Initialize()をコールするテストを行います。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, AlreadyInitializedConstructor1Test)
{
    nn::ngc::ProfanityFilter filter;
    const char* ppWords[] =
    {
        "mario",
        "badword"
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );

    // Finalizeを実施した場合
    filter.Finalize();
    result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );

    // Finalizeを実施しなかった場合
    result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_FAILURE( nn::ngc::ResultAlreadyInitialized ,result );
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         AlreadyInitializedConstructor2Test

  @brief        ProfanityFilter::ProfanityFilter( void* pWorkMemory, size_t workMemorySize, bool checkDesiredLanguage )
                同インスタンスで2回以上Initialize()をコールするテストを行います。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, AlreadyInitializedConstructor2Test)
{
    const char* ppWords[] =
    {
        "mario",
        "badword"
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];
    nn::ngc::ProfanityFilter filter(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize, true);
    nn::Result result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );

    // Finalizeを実施しなかった場合
    result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_FAILURE( nn::ngc::ResultAlreadyInitialized ,result );
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );

    // Finalizeを実施した場合
    filter.Finalize();
    result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         AlreadyInitializedConstructor3Test

  @brief        ProfanityFilter::ProfanityFilter( void* pWorkMemory, size_t workMemorySize )
                同インスタンスで2回以上Initialize()をコールするテストを行います。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, AlreadyInitializedConstructor3Test)
{
    const char* ppWords[] =
    {
        "mario",
        "badword"
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];
    nn::ngc::ProfanityFilter filter(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    nn::Result result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );

    // Finalizeを実施しなかった場合
    result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_FAILURE( nn::ngc::ResultAlreadyInitialized ,result );
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );

    // Finalizeを実施した場合
    filter.Finalize();
    result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         FinalizeContinueCallTest

  @brief        Finalize()を2回以上コールするテストを行います。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, FinalizeContinueCallTest)
{
    nn::ngc::ProfanityFilter filter;
    const char* ppWords[] =
    {
        "mario",
        "badword"
    };
    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];

    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );

    // Finalizeを2回以上コール
    for(int i=0; i < 10; ++i){
        filter.Finalize();
    }
    result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    NNT_ASSERT_RESULT_SUCCESS( result );
    // 答え合わせ
    ASSERT_TRUE( results[ 0 ] == 0 );
    ASSERT_TRUE( results[ 1 ] != 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         MeasureProfanityCheckTimeTest()

  @brief        NG ワードチェックにかかる時間の測定用テスト
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, MeasureProfanityCheckTimeTest)
{

    char* pBuffer = nullptr;
    nn::TimeSpan startTime;
    nn::TimeSpan endTime;

    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    const int BufferSize = 64;
    pBuffer = reinterpret_cast<char*>(std::malloc(BufferSize));
    ASSERT_TRUE(pBuffer != nullptr);

    const char* ppConstWords[] =
        {
            "test",
            "test words",
            "badword",
            "abcdefghijklmnopqrstuvwxyz"
        };
    const size_t WordNum = sizeof(ppConstWords) / sizeof(ppConstWords[0]);

    NN_LOG("Pattern list 0 : ProfanityFilterPatternList_Japanese\n");
    NN_LOG("Pattern list 1 : ProfanityFilterPatternList_AmericanEnglish\n");
    NN_LOG("Pattern list 2 : ProfanityFilterPatternList_CanadianFrench\n");
    NN_LOG("Pattern list 3 : ProfanityFilterPatternList_LatinAmericanSpanish\n");
    NN_LOG("Pattern list 4 : ProfanityFilterPatternList_BritishEnglish\n");
    NN_LOG("Pattern list 5 : ProfanityFilterPatternList_French\n");
    NN_LOG("Pattern list 6 : ProfanityFilterPatternList_German\n");
    NN_LOG("Pattern list 7 : ProfanityFilterPatternList_Italian\n");
    NN_LOG("Pattern list 8 : ProfanityFilterPatternList_Spanish\n");
    NN_LOG("Pattern list 9 : ProfanityFilterPatternList_Dutch\n");
    NN_LOG("Pattern list 10 :ProfanityFilterPatternList_Korean \n");
    NN_LOG("Pattern list 11 :ProfanityFilterPatternList_TraditionalChinese \n");
    NN_LOG("Pattern list 12 :ProfanityFilterPatternList_Portuguese \n");
    NN_LOG("Pattern list 13 :ProfanityFilterPatternList_Russian \n");
    NN_LOG("Pattern list 14 :ProfanityFilterPatternList_SouthAmericanPortuguese \n");
    NN_LOG("Pattern list 15 :ProfanityFilterPatternList_Taiwanese \n");

    for(int i = 0; i < WordNum; i++)
    {
        std::sprintf(pBuffer, "Check word %d : %s\n", i, ppConstWords[i]);
        NN_LOG(pBuffer);
    }

    for(int j = nn::ngc::ProfanityFilterPatternList_Japanese;
            j < nn::ngc::ProfanityFilterPatternList_Max; j++)
    {
        for(int i = 0; i < WordNum; i++)
        {
            startTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

            // 単語チェック
            nn::Bit32 results [ 1 ];
            result = filter.CheckProfanityWords( results, 1 << j, &ppConstWords[i], 1 );
            NNT_ASSERT_RESULT_SUCCESS( result );

            endTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            NN_LOG("Pattern list : %d, Check word : %d, Check time : %ld [msec](word)\n",
                   j, i, endTime.GetMilliSeconds() - startTime.GetMilliSeconds());
        }
    }

    char *pText = reinterpret_cast<char*>(std::malloc(BufferSize));
    ASSERT_TRUE(pText != nullptr);

    int nWordCount = 0;

    for(int j = nn::ngc::ProfanityFilterPatternList_Japanese;
            j < nn::ngc::ProfanityFilterPatternList_Max; j++)
    {
        for(int i = 0; i < WordNum; i++)
        {
            std::memset(pText,0x00,BufferSize);
            std::strcpy(pText, ppConstWords[i]);
            startTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

            // 長文チェック
            result = filter.MaskProfanityWordsInText( &nWordCount, pText, 1 << j );
            NNT_ASSERT_RESULT_SUCCESS( result );

            endTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            NN_LOG("Pattern list : %d, Check word : %d, Check time : %ld [msec](text)\n",
                    j, i, endTime.GetMilliSeconds() - startTime.GetMilliSeconds());
        }
    }

    std::free(pText);
    std::free(pBuffer);
}

/*!--------------------------------------------------------------------------*
  Name:         MeasureProfanityCheckWorstTimeTest

  @brief        NG ワードチェックにかかるワースト時間測定用テスト
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, MeasureProfanityCheckWorstTimeTest)
{
    nn::TimeSpan startTime;
    nn::TimeSpan endTime;
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS( result );

    //  最大文字数(WordLengthMax = 64)、最大ワード数(WordCountMax = 16)、
    //  全言語で空振り（全NGワード照合）した場合を計測する。
    const char* ppWords[] =
    {
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012",
       "012345678901234567890123456789012345678901234567890123456789012"
    };

    const int WordCount = sizeof(ppWords) / sizeof(ppWords[0]);
    nn::Bit32 results[ WordCount ];
    NN_STATIC_ASSERT(nn::ngc::WordCountMax == WordCount);

    startTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
    result = filter.CheckProfanityWords( results, 0xFFFFFFFF, ppWords, WordCount );
    endTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
    NNT_ASSERT_RESULT_SUCCESS( result );
    for(int i=0; i < nn::ngc::WordCountMax; ++i)
    {
        ASSERT_TRUE( results[i] == 0 );
    }

    NN_LOG("CheckProfanityWords() Run Time %d [msec]\n", endTime.GetMilliSeconds() - startTime.GetMilliSeconds());
}

/*!--------------------------------------------------------------------------*
  Name:         CheckCountConcordance()

  @brief        入力に UTF-8 と UTF-16 のどちらを与えても
              CountNumbers の実行結果が変わらないことを確かめるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckCountConcordance)
{
    const char* pWords[] = { "", "aaa", "abc", "1", "1aaa", "aaa1",
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
                       "８aaa１",
                       "８a⑦a１",
                       "８a⑦a１ⅴ",
                       "ⅱa⑦a１ⅴⅣ",
                       "ⅱa⑦a１ⅴⅣ",
                       "ⅻⅫ",
                       "➄➍➒",
                       "❶❷❻❽❿",
                       "⒛⒕⒏⒈",
                       "⒇⒆⒄⒀⑸⑽⑺⑿⑴⑵",
                       "⑳①⑱④⑤⑥⑩⑫⑭⑯",
                       "0000000000",
                       "¹aaa²b³c⓿⓪"
#elif defined( NN_BUILD_CONFIG_OS_WIN )
                       "\xef\xbc\x98\x61\x61\x61\xef\xbc\x91",                                    //  "８aaa１"
                       "\xef\xbc\x98\x61\xe2\x91\xa6\x61\xef\xbc\x91",                            //  "８a⑦a１"
                       "\xef\xbc\x98\x61\xe2\x91\xa6\x61\xef\xbc\x91\xe2\x85\xb4",                //  "８a⑦a１ⅴ"
                       "\xe2\x85\xb1\x61\xe2\x91\xa6\x61\xef\xbc\x91\xe2\x85\xb4\xe2\x85\xa3",    //  "ⅱa⑦a１ⅴⅣ"
                       "\xe2\x85\xbb\xe2\x85\xab",                                                //  "ⅻⅫ"
                       "\xe2\x9e\x84\xe2\x9e\x8d\xe2\x9e\x92",                                    //  "➄➍➒"
                       "\xe2\x9d\xb6\xe2\x9d\xb7\xe2\x9d\xbb\xe2\x9d\xbd\xe2\x9d\xbf",            //  "❶❷❻❽❿"
                       "\xe2\x92\x9b\xe2\x92\x95\xe2\x92\x8f\xe2\x92\x88",                        //  "⒛⒕⒏⒈"
                       "\xe2\x92\x87\xe2\x92\x86\xe2\x92\x84\xe2\x92\x80\xe2\x91\xb8\xe2\x91\xbd\xe2\x91\xba\xe2\x91\xbf\xe2\x91\xb4\xe2\x91\xb5",   //  "⒇⒆⒄⒀⑸⑽⑺⑿⑴⑵"
                       "\xe2\x91\xb3\xe2\x91\xa0\xe2\x91\xb1\xe2\x91\xa3\xe2\x91\xa4\xe2\x91\xa5\xe2\x91\xa9\xe2\x91\xab\xe2\x91\xad\xe2\x91\xaf",   //  "⑳①⑱④⑤⑥⑩⑫⑭⑯"
                       "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30",                               //  "0000000000"
                       "\xc2\xb9\x61\x61\x61\xc2\xb2\x62\xc2\xb3\x63\xe2\x93\xbf\xe2\x93\xaa"    //  "¹aaa²b³c⓿⓪"
#else
#error "unsupported os"
#endif
                      };

    for (auto word : pWords)
    {
        int utf16Length = 0;
        auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&utf16Length, word, static_cast<int>(strlen(word)));
        ASSERT_TRUE(nn::util::CharacterEncodingResult_Success == result);
        utf16Length += 1;
        TypeParam* pStr = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * static_cast<size_t>(utf16Length)));
        ASSERT_TRUE(pStr != nullptr);
        result = nn::util::ConvertStringUtf8ToUtf16Native(pStr, utf16Length, word);
        ASSERT_TRUE(nn::util::CharacterEncodingResult_Success == result);
        ASSERT_EQ(nn::ngc::CountNumbers(word), nn::ngc::CountNumbers(pStr));
        std::free(pStr);
    }
}

/*!--------------------------------------------------------------------------*
  Name:         CheckWordsConcordance()

  @brief        入力に UTF-8 と UTF-16 のどちらを与えても
                CheckProfanityWords の実行結果が変わらないことを確かめるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckWordsConcordance)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS(result);

    const int WordCount = 15;
    const char* ppWordsUtf8[WordCount] =
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        "ｺﾝﾄﾞｰﾑ",
        "ﾀﾞｯﾁﾜｲﾌ",
        "ｴﾌｪﾄﾞﾘﾝ",
        "チンコ",
        "ﾁﾝｺ",
        "チんｺ",
        "123456789012345678901234567890123456789012345678901234warumoshi",                      // 最大文字数
        "mario",
        "luigi",
        "badword",
        "クッパ",
        "ち ん こ",
        "@aaa",
        "＠あいう",
        "hoge@hogehoge.jp"
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        "\xef\xbd\xba\xef\xbe\x9d\xef\xbe\x84\xef\xbe\x9e\xef\xbd\xb0\xef\xbe\x91",             // "ｺﾝﾄﾞｰﾑ"
        "\xef\xbe\x80\xef\xbe\x9e\xef\xbd\xaf\xef\xbe\x81\xef\xbe\x9c\xef\xbd\xb2\xef\xbe\x8c", // "ﾀﾞｯﾁﾜｲﾌ"
        "\xef\xbd\xb4\xef\xbe\x8c\xef\xbd\xaa\xef\xbe\x84\xef\xbe\x9e\xef\xbe\x98\xef\xbe\x9d", // "ｴﾌｪﾄﾞﾘﾝ"
        "\xe3\x83\x81\xe3\x83\xb3\xe3\x82\xb3",                                                 // "チンコ"
        "\xef\xbe\x81\xef\xbe\x9d\xef\xbd\xba",                                                 // "ﾁﾝｺ"
        "\xe3\x83\x81\xe3\x82\x93\xef\xbd\xba",                                                 // "チんｺ"
        "123456789012345678901234567890123456789012345678901234warumoshi",                      // 最大文字数
        "mario",
        "luigi",
        "badword",
        "\xe3\x82\xaf\xe3\x83\x83\xe3\x83\x91",                                                 // "クッパ"
        "\xe3\x81\xa1\x20\xe3\x82\x93\x20\xe3\x81\x93",                                         // "ち ん こ"
        "@aaa",
        "\xef\xbc\xa0\xe3\x81\x82\xe3\x81\x84\xe3\x81\x86",                                     // "＠あいう"
        "hoge@hogehoge.jp"
#else
#error "unsupported os"
#endif
    };

    TypeParam* ppWordsUtf16[WordCount];

    for (int i = 0; i < WordCount; ++i)
    {
        int utf16Length = 0;
        auto encodingResult = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&utf16Length, ppWordsUtf8[i], static_cast<int>(strlen(ppWordsUtf8[i])));
        ASSERT_TRUE(nn::util::CharacterEncodingResult_Success == encodingResult);
        utf16Length += 1;
        TypeParam* pStr = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * static_cast<size_t>(utf16Length)));
        ASSERT_TRUE(pStr != nullptr);
        encodingResult = nn::util::ConvertStringUtf8ToUtf16Native(pStr, utf16Length, ppWordsUtf8[i]);
        ASSERT_TRUE(nn::util::CharacterEncodingResult_Success == encodingResult);
        ppWordsUtf16[i] = pStr;
    }

    nn::Bit32 resultsUtf8[WordCount], resultsUtf16[WordCount];

    result = filter.CheckProfanityWords(resultsUtf8,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        ppWordsUtf8,
                                        WordCount);
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = filter.CheckProfanityWords(resultsUtf16,
                                        nn::ngc::ProfanityFilterPatternList_Japanese,
                                        const_cast<const TypeParam**>(ppWordsUtf16),
                                        WordCount);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 答え合わせ
    for (int i = 0; i < WordCount; ++i)
    {
        ASSERT_EQ(resultsUtf8[i], resultsUtf16[i]);
        std::free(ppWordsUtf16[i]);
    }
}

/*!--------------------------------------------------------------------------*
  Name:         CheckTextConcordance()

  @brief        入力に UTF-8 と UTF-16 のどちらを与えても
                MaskProfanityWordsInText の実行結果が変わらないことを確かめるテストです。
 *---------------------------------------------------------------------------*/
TYPED_TEST(NgcApiTest, CheckTextConcordance)
{
    nn::ngc::ProfanityFilter filter;
    nn::Result result = filter.Initialize(g_pTestMemoryAddr,
                                          nn::ngc::ProfanityFilter::WorkMemorySize);
    NNT_ASSERT_RESULT_SUCCESS(result);

    int nWordCount = 0;
    int nWordCount2 = 0;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char szText[] =
        "わるもし\n"
        " ｡､.,　。、．，わ ｡､.,　。、．，る ｡､.,　。、．，も ｡､.,　。、．，し ｡､.,　。、．，\n"
        "ああわるもしああ"
        "ああ ｡､.,　。、．，わ ｡､.,　。、．，る ｡､.,　。、．，も ｡､.,　。、．，し ｡､.,　。、．，ああ";
    char szText2[] =
        //    0 1 2 3 4 5 6 7 8 9 012345678 9    (20文字/行 * 25行) + 12文字/最終行 + \0 = 512文字 + 1文字
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "badwordj　　　ＢＡＤＷＯＲＤＪ\n"
        "ｂａｄｗｏｒｄｊ　　　BADWORDJ\n"
        "　　warumoshi";
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    char szText[] =
        "\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\x0a\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x8f\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x8b\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x81\x97\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\x0a\xe3\x81\x82\xe3\x81\x82\xe3\x82\x8f\xe3\x82\x8b\xe3\x82\x82\xe3\x81\x97\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\xe3\x81\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x8f\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x8b\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x82\x82\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x81\x97\x20\xef\xbd\xa1\xef\xbd\xa4\x2e\x2c\xe3\x80\x80\xe3\x80\x82\xe3\x80\x81\xef\xbc\x8e\xef\xbc\x8c\xe3\x81\x82\xe3\x81\x82";
    char szText2[] =
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\x62\x61\x64\x77\x6f\x72\x64\x6a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\xef\xbc\xa2\xef\xbc\xa1\xef\xbc\xa4\xef\xbc\xb7\xef\xbc\xaf\xef\xbc\xb2\xef\xbc\xa4\xef\xbc\xaa\x0a"
        "\xef\xbd\x82\xef\xbd\x81\xef\xbd\x84\xef\xbd\x97\xef\xbd\x8f\xef\xbd\x92\xef\xbd\x84\xef\xbd\x8a\xe3\x80\x80\xe3\x80\x80\xe3\x80\x80\x42\x41\x44\x57\x4f\x52\x44\x4a\x0a"
        "\xe3\x80\x80\xe3\x80\x80\x77\x61\x72\x75\x6d\x6f\x73\x68\x69";
#else
#error "unsupported os"
#endif

    int szTextUtf16Length = 0;
    int szText2Utf16Length = 0;

    // UTF-16 文字列の作成
    auto encodingResult = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&szTextUtf16Length, szText, static_cast<int>(strlen(szText)));
    ASSERT_TRUE(nn::util::CharacterEncodingResult_Success == encodingResult);
    encodingResult = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&szText2Utf16Length, szText2, static_cast<int>(strlen(szText2)));
    ASSERT_TRUE(nn::util::CharacterEncodingResult_Success == encodingResult);

    szTextUtf16Length += 1;
    szText2Utf16Length += 1;

    TypeParam* szTextUtf16 = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * static_cast<size_t>(szTextUtf16Length)));
    ASSERT_TRUE(szTextUtf16 != nullptr);
    TypeParam* szText2Utf16 = reinterpret_cast<TypeParam*>(std::malloc(sizeof(TypeParam) * static_cast<size_t>(szText2Utf16Length)));
    ASSERT_TRUE(szText2Utf16 != nullptr);

    encodingResult = nn::util::ConvertStringUtf8ToUtf16Native(szTextUtf16, szTextUtf16Length, szText);
    ASSERT_TRUE(nn::util::CharacterEncodingResult_Success == encodingResult);
    encodingResult = nn::util::ConvertStringUtf8ToUtf16Native(szText2Utf16, szText2Utf16Length, szText2);
    ASSERT_TRUE(nn::util::CharacterEncodingResult_Success == encodingResult);

    // NG ワードチェックの実行
    result = filter.MaskProfanityWordsInText
    (
        &nWordCount,
        szText,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = filter.MaskProfanityWordsInText
    (
        &nWordCount2,
        szTextUtf16,
        nn::ngc::ProfanityFilterPatternList_Japanese
    );
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = filter.MaskProfanityWordsInText(nullptr, szText2, 0xFFFFFFFF);
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = filter.MaskProfanityWordsInText(nullptr, szText2Utf16, 0xFFFFFFFF);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // UTF-16 の結果を UTF-8 に戻す
    int szTextUtf8Length = static_cast<int>(strlen(szText)) + 1;    // 終端文字ぶん加算
    int szText2Utf8Length = static_cast<int>(strlen(szText2)) + 1;
    char* szTextUtf8 = reinterpret_cast<char*>(std::malloc(sizeof(char) * static_cast<size_t>(szTextUtf8Length)));
    ASSERT_TRUE(szTextUtf8 != nullptr);
    char* szText2Utf8 = reinterpret_cast<char*>(std::malloc(sizeof(char) * static_cast<size_t>(szText2Utf8Length)));
    ASSERT_TRUE(szText2Utf8 != nullptr);

    encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(szTextUtf8, szTextUtf8Length, szTextUtf16);
    ASSERT_EQ(nn::util::CharacterEncodingResult_Success, encodingResult);
    encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(szText2Utf8, szText2Utf8Length, szText2Utf16);
    ASSERT_TRUE(nn::util::CharacterEncodingResult_Success == encodingResult);

    // 答え合わせ
    ASSERT_EQ(nWordCount, nWordCount2);
    ASSERT_TRUE(std::strcmp(szTextUtf8, szText) == 0);
    ASSERT_TRUE(std::strcmp(szText2Utf8, szText2) == 0);

    std::free(szTextUtf16);
    std::free(szText2Utf16);
    std::free(szTextUtf8);
    std::free(szText2Utf8);
} //NOLINT(impl/function_size)

#endif
