﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/hid.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNpt_Common.h>


namespace
{
//================================================================================
// このテストで使用する定義です。
//================================================================================
    enum WaitTime
    {
        WaitTime_TagSwitch              = 3000,
        WaitTime_CountDown              = 1000,
        WaitTime_AttachTag              = 10000,
        WaitTime_NextStep               = 1000,
        WaitTime_BluetoothSignalRestore = 10000,
    };

    // スレッド関連の定義
    const size_t StackSize = 32 * 1024;//4096;
    nn::os::ThreadType g_RunApiThread;
    NN_ALIGNAS( 4096 ) char g_Stack[ StackSize ];
    nn::os::EventType g_BootUpEvent;
    nn::os::EventType g_RunApiEvent;
    nn::Result g_RunApiThreadResult;

    // 要求API設定
    enum RunNptApi
    {
        RunNptApi_Read = 0,
        RunNptApi_Write,
    };
    // 初期値はFlushの試験に設定する
    RunNptApi g_RunNptApi = RunNptApi_Read;

    // マルチデバイス関連定義
    // Nptタグは2枚で試験を実施する
#if defined(NNT_NPT_MULTI_2DEVICES)
    const int DeviceCountMax = 2;
#else // defined(NNT_NPT_MULTI_2DEVICES)
    const int DeviceCountMax = 4;
#endif // defined(NNT_NPT_MULTI_2DEVICES)
    const int MaxCore = 3;
    struct TagData
    {
        nn::Bit8 *NormalTagData;
        unsigned char *NormalTagNickName;
    };
    struct threadinfo
    {
        int deviceNum;
    };
#if defined(NNT_NPT_MULTI_2DEVICES)
    threadinfo g_threadParam[DeviceCountMax] = {{0},{1}};
#else // defined(NNT_NPT_MULTI_2DEVICES)
    threadinfo g_threadParam[DeviceCountMax] = {{0},{1},{2},{3}};
#endif // defined(NNT_NPT_MULTI_2DEVICES)
    void DeviceTestThread(void* pArg) NN_NOEXCEPT;
    const size_t DeviceThreadStackSize = 32 * 1024;
    nn::os::ThreadType g_DeviceThread[DeviceCountMax];
    NN_OS_ALIGNAS_THREAD_STACK char g_DeviceThreadStack[DeviceCountMax][DeviceThreadStackSize];
    nn::nfc::DeviceHandle g_CurrentDeviceHandleMulti[DeviceCountMax];
    const int WaitTimeout = 5000;
    const int TestLoop = 10;
    bool g_isStateKeep;

//================================================================================
// テストスイート全体で共通の処理です。
//================================================================================

    void ThreadRunNptApi(void* pPtr)
    {
        NN_UNUSED(pPtr);

        g_RunApiThreadResult = nn::nfc::ResultNfcDeviceError();

        NN_LOG("### %s Activated !!\n", __FUNCTION__);

        if(nn::os::TryWaitEvent(&g_BootUpEvent) == true)
        {
            NN_LOG("### Illeagal Event State !!\n");
            nn::os::SignalEvent(&g_BootUpEvent);
            nn::os::ClearEvent(&g_BootUpEvent);
            EXPECT_TRUE(false);
            return;
        }

        NN_LOG("### Indicate Thread Boot!!\n");
        nn::os::SignalEvent(&g_BootUpEvent);

        NN_LOG("### Waiting for RunNptApi Signal...\n");
        nn::os::WaitEvent(&g_RunApiEvent);

        switch (g_RunNptApi)
        {
        case RunNptApi_Write:
            {
                NN_LOG("### Run Write\n");
                // 送信データ(WRITE コマンド)
                nn::Bit8 sendWriteCommandData[6];
                // WRITE コマンドに対するレスポンスデータ
                size_t sendWriteCommandResponseDataSize;
                nn::Bit8 sendWriteCommandResponseData[1];
                nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);
                nnt::npt::PrepareSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
                nnt::npt::PrintSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
                g_RunApiThreadResult = nnt::npt::wrapper::SendCommandByPassThrough(sendWriteCommandResponseData, &sendWriteCommandResponseDataSize,
                                                                      sendWriteCommandData, sizeof(sendWriteCommandData),
                                                                      sizeof(sendWriteCommandResponseData), timeout);
            }
            break;
        case RunNptApi_Read:
            {
                NN_LOG("### Run Read\n");
                // 送信データ(READ コマンド)
                nn::Bit8 sendReadCommandData[2];
                // READ コマンドに対するレスポンスデータ
                size_t sendReadCommandResponseDataSize;
                nn::Bit8 sendReadCommandResponseData[16];
                nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);
                nnt::npt::PrepareSendReadCommandData(sendReadCommandData, sizeof(sendReadCommandData));
                g_RunApiThreadResult = nnt::npt::wrapper::SendCommandByPassThrough(sendReadCommandResponseData, &sendReadCommandResponseDataSize,
                                                                      sendReadCommandData, sizeof(sendReadCommandData),
                                                                      sizeof(sendReadCommandResponseData), timeout);
                if(g_RunApiThreadResult.IsSuccess())
                {
                    nnt::npt::PrintSendReadCommandResponseData(sendReadCommandResponseData, sendReadCommandResponseDataSize);
                }
            }
            break;
        default:
            {
                NN_LOG("### Nothing\n");
            }
            break;
        }
        NN_LOG("### Indicate End RunApi Signal RunApi: %d, Result: %s !!\n",
                g_RunNptApi, nnt::npt::GetNptResultTypeString(g_RunApiThreadResult));
        nn::os::ClearEvent(&g_RunApiEvent);

        if(nn::os::TryWaitEvent(&g_BootUpEvent) == true)
        {
            nn::os::SignalEvent(&g_BootUpEvent);
        }

        nn::os::ClearEvent(&g_BootUpEvent);
        return;
    }

    inline void CheckDeactive(bool isDeactive)
    {
        EXPECT_EQ(nnt::npt::WaitForDeactivate(),isDeactive);
    }

    void TestInitRunApiThreadWait()
    {
        // スタック準備
        nn::os::InitializeEvent(&g_BootUpEvent, false,  nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&g_RunApiEvent, false,  nn::os::EventClearMode_ManualClear);

        nn::os::ClearEvent(&g_BootUpEvent);
        nn::os::ClearEvent(&g_RunApiEvent);

        // RunNptApi thread Start
        NNT_NPT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(&g_RunApiThread,
                                     ThreadRunNptApi,
                                     nullptr,
                                     g_Stack,
                                     sizeof(g_Stack),
                                     nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_RunApiThread);

        NN_LOG("### Wainting for Thraed Start \n");
        nn::os::WaitEvent(&g_BootUpEvent);
        NN_LOG("### Thread Start Signal Detected !! \n");
        return;
    }

    void TestMainTagDropInProgress()
    {
        NN_LOG("# TestMainTagDropInProgress \n");

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE DROP TAG Ready... 5 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" 4 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" 3 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" 2 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" 1 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" NOW!! \n");
        NN_LOG("==================================================\n");

        nn::os::SignalEvent(&g_RunApiEvent);

        // DeactivateEventが発生する
        CheckDeactive(true);

        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::npt::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState() );

        // Flushスレッドの終了まで待つ
        nn::os::WaitThread(&g_RunApiThread);
        nn::os::DestroyThread(&g_RunApiThread);

        NNT_NPT_EXPECT_RESULT_EQUAL(nn::nfc::ResultTagNotFound(), g_RunApiThreadResult);

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE ATTACH a TAG!!\n");
        NN_LOG("==================================================\n");
        nnt::npt::DoActivate();
        NN_LOG(" TAG Attached !!\n");
        EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::npt::wrapper::GetDeviceState());

        return;
    }

    void TestMainBluetoothCutInProgress()
    {
        NN_LOG("# TestMainBluetoothCutInProgress \n");

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE CUT BT SIGNAL Ready... 5 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" 4 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" 3 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" 2 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" 1 ");
        nnt::npt::Sleep(WaitTime_CountDown);
        NN_LOG(" NOW!! \n");
        NN_LOG("==================================================\n");

        nn::os::SignalEvent(&g_RunApiEvent);

        // DeactivateEventが発生する
        CheckDeactive(true);

        EXPECT_EQ(nn::nfc::DeviceState_Unexpected, nnt::npt::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState() );

        // Flushスレッドの終了まで待つ
        nn::os::WaitThread(&g_RunApiThread);
        nn::os::DestroyThread(&g_RunApiThread);

        NNT_NPT_EXPECT_RESULT_EQUAL(nn::nfc::ResultNfcDeviceNotFound(), g_RunApiThreadResult);

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE RESTORE BT MODULE (Within %dms) ...       \n",WaitTime_BluetoothSignalRestore);
        NN_LOG("==================================================\n");
        nnt::npt::Sleep(WaitTime_BluetoothSignalRestore);

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE ATTACH a TAG!!\n");
        NN_LOG("==================================================\n");
        nnt::npt::DoActivate();
        NN_LOG(" TAG Attached !!\n");
        EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::npt::wrapper::GetDeviceState());

        return;
    }

    void TestFinalizeRunNptApiThread()
    {
        if(g_RunApiThread._state == nn::os::ThreadType::State_Started)
        {
            nn::os::DestroyThread(&g_RunApiThread);
        }

        NNT_NPT_EXPECT_RESULT_SUCCESS( nnt::npt::FinalizeSystem() );
        nn::os::FinalizeEvent(&g_BootUpEvent);
        nn::os::FinalizeEvent(&g_RunApiEvent);
    }

    void TestInitTransPreRead()
    {
        NN_LOG("TestInitTransPreRead\n");
        // TestAPI設定
        g_RunNptApi = RunNptApi_Read;
        if(g_isStateKeep)
        {
            // Keep状態に遷移
            nnt::npt::DoKeep();
        }
        else
        {
            // Active状態に遷移
            nnt::npt::DoActivate();
        }
        // TestAPI準備
        TestInitRunApiThreadWait();
    }

    void TestInitTransPreWrite()
    {
        NN_LOG("TestInitTransPreWrite\n");
        // TestAPI設定
        g_RunNptApi = RunNptApi_Write;
        if(g_isStateKeep)
        {
            // Keep状態に遷移
            nnt::npt::DoKeep();
        }
        else
        {
            // Active状態に遷移
            nnt::npt::DoActivate();
        }
        // TestAPI準備
        TestInitRunApiThreadWait();
    }

    bool WaitForActivate(int ms, nn::os::SystemEventType activateEvent) NN_NOEXCEPT
    {
        if((activateEvent._state != nn::os::SystemEventType::State_NotInitialized) &&
                (activateEvent._state != nn::os::SystemEventType::State_NotInitialized))
        {
            return nn::os::TimedWaitSystemEvent(&activateEvent, nn::TimeSpan::FromMilliSeconds(ms));
        }
        else
        {
            NN_LOG("WaitForActivate() ActiveEvent DeactivateEvent is NotInitialized \n");
            return false;
        }
    }

    bool WaitForDeactivate(int ms, nn::os::SystemEventType deactivateEvent) NN_NOEXCEPT
    {
        return nn::os::TimedWaitSystemEvent(&deactivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
    }

    void DestroyDefaultActivateEvent(nn::os::SystemEventType activateEvent) NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&activateEvent);
        return;
    }

    void DestroyDefaultDeactivateEvent(nn::os::SystemEventType deactivateEvent) NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&deactivateEvent);
        return;
    }

    void DeviceTestThread(void* pArg) NN_NOEXCEPT
    {
        threadinfo* threaditem = static_cast<threadinfo*>(pArg);
        int deviceNum = threaditem->deviceNum;
        for(int i = 0; i < TestLoop; i++)
        {
            nn::os::SystemEventType activateEvent[DeviceCountMax] = {};
            nn::os::SystemEventType deactivateEvent[DeviceCountMax] = {};
            NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::AttachActivateEvent(
                    &activateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));
            NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::AttachDeactivateEvent(
                    &deactivateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));

            NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));
            if (!WaitForActivate(WaitTimeout, activateEvent[deviceNum]))
            {
                NN_LOG("StartDetection err deviceNum=%d Loop=%d\n", deviceNum, i);
                EXPECT_TRUE(false);
                break;
            }

            EXPECT_EQ(nn::nfc::DeviceState_Active,nnt::npt::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));

            // 送信データ(READ コマンド)
            nn::Bit8 sendReadCommandData[2];
            // READ コマンドに対するレスポンスデータ
            size_t sendReadCommandResponseDataSize;
            nn::Bit8 sendReadCommandResponseData[16];
            // 送信データ(WRITE コマンド)
            nn::Bit8 sendWriteCommandData[6];
            // WRITE コマンドに対するレスポンスデータ
            size_t sendWriteCommandResponseDataSize;
            nn::Bit8 sendWriteCommandResponseData[1];
            nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);

            nnt::npt::PrepareSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
            nnt::npt::PrintSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));

            NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::SendCommandByPassThrough(sendWriteCommandResponseData, &sendWriteCommandResponseDataSize,
                                                                     sendWriteCommandData, sizeof(sendWriteCommandData),
                                                                     sizeof(sendWriteCommandResponseData), timeout));
            nnt::npt::PrepareSendReadCommandData(sendReadCommandData, sizeof(sendReadCommandData));
            NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::SendCommandByPassThrough(sendReadCommandResponseData, &sendReadCommandResponseDataSize,
                                                                     sendReadCommandData, sizeof(sendReadCommandData),
                                                                     sizeof(sendReadCommandResponseData), timeout));
            nnt::npt::PrintSendReadCommandResponseData(sendReadCommandResponseData, sendReadCommandResponseDataSize);
            // 書き込んだデータを読み込めたか確認
            nnt::npt::CheckReadData(sendReadCommandResponseData, sendReadCommandResponseDataSize,
                                    sendWriteCommandData, sizeof(sendWriteCommandData));

            NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StopDetection(g_CurrentDeviceHandleMulti[deviceNum]));
            if (!WaitForDeactivate(WaitTimeout, deactivateEvent[deviceNum]))
            {
                NN_LOG("StopDetection err deviceNum=%d Loop=%d\n", deviceNum, i);
                EXPECT_TRUE(false);
                break;
            }
            DestroyDefaultActivateEvent(activateEvent[deviceNum]);
            DestroyDefaultDeactivateEvent(deactivateEvent[deviceNum]);
        }
    }

} // end of anonymous namespace

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NptManual : public nnt::npt::TestFramework
{
protected:

    NptManual() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::npt::InitializeHidController();
        // API呼び出しログ出力フラグ
        //nnt::npt::wrapper::SetApiCallLoggingMode(nnt::npt::wrapper::LogMode_Aging);
    }

    ~NptManual() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::npt::wrapper::GetState() == nn::nfc::State_Init)
        {
            nnt::npt::FinalizeSystem();
        }
    }
};

//================================================================================
// テストケースの実装です。
//================================================================================
TEST_F(NptManual, TestCaseGoodOperationDeviceDeactive)
{
    // デバイス状態がDeactive時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationDeviceDeactive start\n");

    // 状態をDeviceDeactiveに遷移させる
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH NPT TAG (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::npt::Sleep(WaitTime_TagSwitch);
    nnt::npt::DoActivate();
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE REMOVE a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::npt::WaitForDeactivate();

    // ----------------------------------------
    // テスト対象 : nn::nfc::StartDetection()
    //              nn::nfc::GetDeviceState()
    //              nn::nfc::GetNpadId()
    //              nn::nfc::ListDevices()
    //              nn::nfc::AttachActivateEvent()
    //              nn::nfc::AttachDeactivateEvent()
    //              nn::nfc::StopDetection()
    // ----------------------------------------
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::npt::wrapper::GetDeviceState());
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::GetNpadId(&npadId));
    nnt::npt::CheckNPadId(npadId);
    int outCount;
    nn::nfc::DeviceHandle currentDeviceHandle;
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::npt::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::AttachDeactivateEvent(&deactivateEvent));
    nn::os::DestroySystemEvent(&activateEvent);
    nn::os::DestroySystemEvent(&deactivateEvent);
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StartDetection());

    // 状態をDeviceDeactiveに遷移させる
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE ATTACH a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::npt::WaitForActivate();
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::npt::wrapper::GetDeviceState());
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE REMOVE a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::npt::WaitForDeactivate();

    // ----------------------------------------
    // テスト対象 : nn::nfc::StopDetection()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StopDetection());
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::npt::wrapper::GetDeviceState());
}
TEST_F(NptManual, TestCaseInvalidOperationDeviceDeactive)
{
    // デバイス状態がDeactive時に呼び出すことができないAPIのテストをします。
    NN_LOG("TestCaseInvalidOperationDeviceDeactive start\n");

    // 状態をDeviceDeactiveに遷移させる
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH NPT TAG (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::npt::Sleep(WaitTime_TagSwitch);
    nnt::npt::DoActivate();
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE REMOVE a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::npt::WaitForDeactivate();
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::npt::wrapper::GetDeviceState());

    // ----------------------------------------
    // テスト対象 : nn::nfc::SendCommandByPassThrough()
    //              nn::nfc::KeepPassThroughSession()
    //              nn::nfc::ReleasePassThroughSession()
    //              nn::nfc::GetTagInfo()
    // ----------------------------------------
    // 送信データ(WRITE コマンド)
    nn::Bit8 sendWriteCommandData[6];
    // WRITE コマンドに対するレスポンスデータ
    size_t sendWriteCommandResponseDataSize;
    nn::Bit8 sendWriteCommandResponseData[1];
    nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);
    nnt::npt::PrepareSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    nnt::npt::PrintSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultTagNotFound,
                              nnt::npt::wrapper::SendCommandByPassThrough(sendWriteCommandResponseData, &sendWriteCommandResponseDataSize,
                                                             sendWriteCommandData, sizeof(sendWriteCommandData),
                                                             sizeof(sendWriteCommandResponseData), timeout));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultTagNotFound,
                              nnt::npt::wrapper::KeepPassThroughSession());
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultTagNotFound,
                              nnt::npt::wrapper::ReleasePassThroughSession());
    nn::nfc::TagInfo tagInfo;
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultTagNotFound,
            nnt::npt::wrapper::GetTagInfo(&tagInfo));
}

TEST_F(NptManual, TestCaseStartDetectionNfcProtocolFilter)
{
    NN_LOG("TestCaseStartDetectionNfcProtocolFilter start\n");

    struct ProtocolData
    {
        nn::nfc::NfcProtocol protocol;
        const char* name;
    } tbl[] = {
        {
            nn::nfc::NfcProtocol_TypeA,
            "TypeA"
        },
        {
            nn::nfc::NfcProtocol_TypeB,
            "TypeB"
        },
        {
            nn::nfc::NfcProtocol_TypeF,
            "TypeF"
        },
        {
            nn::nfc::NfcProtocol_Type15693,
            "Type15693"
        },
    };

    auto printFilter = [&tbl](nn::Bit32 filter)
        {
            NN_LOG("NfcProtocolFilter = ");
            if(filter == 0)
            {
                NN_LOG("0");
            }
            else
            {
                bool flg = false;
                for(auto& protocolData : tbl)
                {
                    if(protocolData.protocol & filter)
                    {
                        if(flg)
                        {
                            NN_LOG(" | ");
                        }
                        NN_LOG("%s", protocolData.name);
                        flg = true;
                    }
                }
            }
            NN_LOG("\n");
        };

    nn::Bit32 maxFilter = 0;
    for(auto& protocolData : tbl)
    {
        maxFilter |= protocolData.protocol;
    }

    for(auto& protocolData : tbl)
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH *** %s *** TAG (Within %dms) ...    \n",
               protocolData.name, WaitTime_AttachTag);
        NN_LOG("========================================================================\n");
        nnt::npt::Sleep(WaitTime_AttachTag);
        nnt::npt::DoSearch();
        nnt::npt::WaitForActivate();
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::npt::wrapper::GetDeviceState());
        nn::nfc::TagInfo tagInfo;
        NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::GetTagInfo(&tagInfo));
        EXPECT_EQ(protocolData.protocol, tagInfo.protocol);
        NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StopDetection());
        EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::npt::wrapper::GetDeviceState());

        for(nn::Bit32 filter = 0; filter <= maxFilter; filter++)
        {
            printFilter(filter);
            NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StartDetection(static_cast<nn::nfc::NfcProtocol>(filter)));
            nnt::npt::WaitForActivate();
            if(filter == 0 || filter & protocolData.protocol)
            {
                EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::npt::wrapper::GetDeviceState());
            }
            else
            {
                EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::npt::wrapper::GetDeviceState());
            }
            NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StopDetection());
        }
    }
}

TEST_F(NptManual, TestCaseStateTransWhenReadWrite)
{
    // 書き込み／読み込み中にNptタグを離すテストをします。
    NN_LOG("TestCaseStateTransWhenReadWrite start\n");

    // スレッド準備
    struct TestCaseProc
    {
        const char* pStateName;
        const char* pEventName;
        void  (*pFuncTestInitial)();
        void  (*pFuncTestMain)();
        void  (*pFuncTestFinal)();
    };

    enum StateKind
    {
        StateKind_Active  = 0,
        StateKind_Keep,
        StateKind_Max
    };

    enum EventKind
    {
        EventKind_TagDropInProgressRead  = 0,
        EventKind_TagDropInProgressWrite,
        EventKind_BluetoothCutInProgressRead,
        EventKind_BluetoothCutInProgressWrite,
        EventKind_Max
    };

    static const TestCaseProc testCaseTbl[StateKind_Max][EventKind_Max] =
    {
        /************************************************** ACTIVE ***********************************************************/
        {
            // StateName EventName   Before Test            Main                            After
            {  "ACTIVE", "Tag Drop", TestInitTransPreRead,  TestMainTagDropInProgress,      TestFinalizeRunNptApiThread },
            {  "ACTIVE", "Tag Drop", TestInitTransPreWrite, TestMainTagDropInProgress,      TestFinalizeRunNptApiThread },
            {  "ACTIVE", "BT Off",   TestInitTransPreRead,  TestMainBluetoothCutInProgress, TestFinalizeRunNptApiThread },
            {  "ACTIVE", "BT Off",   TestInitTransPreWrite, TestMainBluetoothCutInProgress, TestFinalizeRunNptApiThread },
        },
        /************************************************** KEEP ***********************************************************/
        {
            // StateName EventName   Before Test            Main                            After
            {  "KEEP",   "Tag Drop", TestInitTransPreRead,  TestMainTagDropInProgress,      TestFinalizeRunNptApiThread },
            {  "KEEP",   "Tag Drop", TestInitTransPreWrite, TestMainTagDropInProgress,      TestFinalizeRunNptApiThread },
            {  "KEEP",   "BT Off",   TestInitTransPreRead,  TestMainBluetoothCutInProgress, TestFinalizeRunNptApiThread },
            {  "KEEP",   "BT Off",   TestInitTransPreWrite, TestMainBluetoothCutInProgress, TestFinalizeRunNptApiThread },
        },
    };

    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE ATTACH a TAG (Within %dms) !! \n",WaitTime_AttachTag);
    NN_LOG("==================================================\n");

    nnt::npt::Sleep(WaitTime_AttachTag);

    for(uint8_t stateCount = 0; stateCount < StateKind_Max; stateCount++)
    {
        g_isStateKeep = (stateCount == StateKind_Keep);
        for(uint8_t eventCount = 0; eventCount < EventKind_Max; eventCount++)
        {
            const TestCaseProc* pTestCaseTbl =
                    &testCaseTbl[stateCount][eventCount];

            if( pTestCaseTbl->pFuncTestInitial == nullptr )
            {
                continue;
            }

            NN_LOG("**************************************************\n");
            NN_LOG("*   TestState : %s \n", pTestCaseTbl->pStateName );
            NN_LOG("*   TestEvent : %s \n", pTestCaseTbl->pEventName );
            NN_LOG("**************************************************\n");

            // 事前処理
            pTestCaseTbl->pFuncTestInitial();
            nnt::npt::Sleep(WaitTime_NextStep);

            // テストケース
            pTestCaseTbl->pFuncTestMain();
            nnt::npt::Sleep(WaitTime_NextStep);

            // あとしまつ
            pTestCaseTbl->pFuncTestFinal();
            nnt::npt::Sleep(WaitTime_NextStep);
        }
    }
}

TEST_F(NptManual, TestCaseMultiDeviceAccess)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please Pairing Multi Devices and Setting Npt Tags on All Device \n");
    NN_LOG("========================================================================\n");
    nnt::npt::Sleep(WaitTimeout);

    // Npt初期化
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::InitializeSystem());

    // 4つのデバイスを取得
    int outCount;
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::ListDevices(
            g_CurrentDeviceHandleMulti, &outCount, DeviceCountMax));
    EXPECT_TRUE( outCount == DeviceCountMax);
    if(outCount != DeviceCountMax)
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" Please Check to Connect Devices\n");
        NN_LOG("========================================================================\n");
        // 4台のデバイスが取得できなかったのでアサートさせる
        ASSERT_TRUE(false);
    }

    for(int i = 0; i < outCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                &g_DeviceThread[i], DeviceTestThread, &g_threadParam[i], g_DeviceThreadStack[i],
                sizeof(g_DeviceThreadStack[i]), nn::os::DefaultThreadPriority));
        // コアを指定する
        // コアは0～2までの3つが指定できるので4つ目のスレッドには0を指定する
        int idealCore = i;
        if (idealCore >= MaxCore)
        {
            idealCore = 0;
        }
        nn::os::SetThreadCoreMask(&g_DeviceThread[i], idealCore, 0x1LL << idealCore);
    }

    for(int i = 0; i < outCount; i++)
    {
        nn::os::StartThread(&g_DeviceThread[i]);
    }

    for(int i = 0; i < outCount; i++)
    {
        nn::os::WaitThread(&g_DeviceThread[i]);
        nn::os::DestroyThread(&g_DeviceThread[i]);
    }
}
