﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNpt_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    // エイジングの回数を指定します。タグの書き込み回数には最大 110,000 回の制限があります。
    const int AgingCountMax = 10000;


    // 送信データ(READ コマンド)
    nn::Bit8 g_sendReadCommandData[2];

    // READ コマンドに対するレスポンスデータ
    size_t g_sendReadCommandResponseDataSize;
    nn::Bit8 g_sendReadCommandResponseData[16];

    // 送信データ(WRITE コマンド)
    nn::Bit8 g_sendWriteCommandData[6];

    // WRITE コマンドに対するレスポンスデータ
    size_t g_sendWriteCommandResponseDataSize;
    nn::Bit8 g_sendWriteCommandResponseData[1];
} // end of anonymous namespace

//================================================================================
// タグの共有領域やアプリケーション専用領域の読み書きを繰り返すエイジングです。
// 必ず新しいタグを設置してからテストを開始してください。
// 使い込んだタグでは耐性を超えてしまい、書き込みの失敗が頻発する可能性があります。
//================================================================================

//================================================================================
// テスト用のユーティリティです。
//================================================================================

//================================================================================
// エイジングテスト用のフレームワークです。
//================================================================================

class NptAgingStableTag : public nnt::npt::TestFramework
{
protected:

    NptAgingStableTag() NN_NOEXCEPT
    {
        //amiibo設定実行用にoeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::npt::InitializeHidController();
        nnt::npt::wrapper::SetApiCallLoggingMode(nnt::npt::wrapper::LogMode_Aging);
    }

    void TestReadWrite() NN_NOEXCEPT;
};

void NptAgingStableTag::TestReadWrite()
{
    NN_LOG("Npt AGING: - TestReadWrite Start\n");
    nnt::npt::PrepareSendWriteCommandData(g_sendWriteCommandData, sizeof(g_sendWriteCommandData));
    nnt::npt::PrintSendWriteCommandData(g_sendWriteCommandData, sizeof(g_sendWriteCommandData));

    nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::SendCommandByPassThrough(g_sendWriteCommandResponseData, &g_sendWriteCommandResponseDataSize,
                                                             g_sendWriteCommandData, sizeof(g_sendWriteCommandData),
                                                             sizeof(g_sendWriteCommandResponseData), timeout));
    nnt::npt::PrepareSendReadCommandData(g_sendReadCommandData, sizeof(g_sendReadCommandData));
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::SendCommandByPassThrough(g_sendReadCommandResponseData, &g_sendReadCommandResponseDataSize,
                                                             g_sendReadCommandData, sizeof(g_sendReadCommandData),
                                                             sizeof(g_sendReadCommandResponseData), timeout));
    nnt::npt::PrintSendReadCommandResponseData(g_sendReadCommandResponseData, g_sendReadCommandResponseDataSize);
    // 書き込んだデータを読み込めたか確認
    nnt::npt::CheckReadData(g_sendReadCommandResponseData, g_sendReadCommandResponseDataSize,
                            g_sendWriteCommandData, sizeof(g_sendWriteCommandData));
    return;
}

//================================================================================
// エイジングの実装です。
//================================================================================

TEST_F(NptAgingStableTag,TestCaseAgingStableTag)
{
    NNT_NPT_ASSERT_RESULT_SUCCESS_FATAL(nnt::npt::InitializeWithRetry());
    nnt::npt::DoSearchDeviceHandle();

    // タグの耐用回数までエイジングを行います。
    for (int i = 0; i < AgingCountMax; ++i)
    {
        NN_LOG("NPT AGING: @COUNT = %d/%d\n", i + 1, AgingCountMax);
        //StopDetection()が実行可能な状態の時のみStopDetection()を実行する
        nn::nfc::DeviceState deviceState = nnt::npt::wrapper::GetDeviceState();
        if((deviceState != nn::nfc::DeviceState_Init) &&
           (deviceState != nn::nfc::DeviceState_Unexpected))
        {
            nn::Result stopResult = nnt::npt::wrapper::StopDetection();
            if (stopResult.IsFailure())
            {
                NNT_NPT_AGING_ASSERT_RESULT_SUCCESS(stopResult);
                continue;
            }
        }

        // タグの検出を開始します。
        nn::Result startResult = nnt::npt::wrapper::StartDetection();
        if (startResult.IsFailure())
        {
            NNT_NPT_AGING_ASSERT_RESULT_SUCCESS(startResult);
            continue;
        }

        // タグが見つかるまで待機します。
        bool isActivated = nnt::npt::WaitForActivate();
        if (!isActivated)
        {
            NNT_NPT_AGING_ASSERT_EQUAL_N(true, isActivated);
            continue;
        }

        // データ領域の読み書きに関するテストです。
        TestReadWrite();

        NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::KeepPassThroughSession());
        TestReadWrite();
        NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::ReleasePassThroughSession());
    }

    NNT_NPT_ASSERT_RESULT_SUCCESS_FATAL(nnt::npt::FinalizeSystem());
}
