﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NfpPermissionSys : public nnt::nfp::TestFramework
{

protected:

    NfpPermissionSys() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
    }

    ~NfpPermissionSys() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::nfp::wrapper::GetState() == nn::nfp::State_Init)
        {
            // sys権限の試験なので、FinalizeSystem()でライブラリを終了します。
            nnt::nfp::wrapper::FinalizeSystem();
        }
    }
};

//================================================================================
// NfpライブラリのAPIの権限テストで共通の処理です。
//================================================================================
namespace {
    bool WaitForActivate(int ms, nn::os::SystemEventType* pActivateEvent) NN_NOEXCEPT
    {
        return nn::os::TimedWaitSystemEvent(pActivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
    }

    bool WaitForDeactivate(int ms, nn::os::SystemEventType* pDeactivateEvent) NN_NOEXCEPT
    {
        return nn::os::TimedWaitSystemEvent(pDeactivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
    }
}

//================================================================================
// テストケースの実装です。
//================================================================================
TEST_F(NfpPermissionSys, TestCasePermissionSysAllow)
{
    // sys権限が必要なAPIが正常に実行できること
    NN_LOG("TestCasePermissionSysAllow start \n");

    const int WaitTimeTagSwitch = 3000;
    // Amiiboタグをかざすメッセージを表示して数秒待つ
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH AMIIBO TAG(Created by NfpManagerEx) (Within %dms) ...    \n",
           WaitTimeTagSwitch);
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTimeTagSwitch);

    // sys権限の試験なので、InitializeSystemを実行する
    // ----------------------------------------
    // テスト対象 : nn::nfp::InitializeSystem()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystem());
    // ----------------------------------------
    // テスト対象 : nn::nfp::ListDevices()
    // ----------------------------------------
    nn::nfp::DeviceHandle currentDeviceHandle;
    int outCount;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(
            &currentDeviceHandle, &outCount, 1));
    // ----------------------------------------
    // テスト対象 : nn::nfp::AttachActivateEvent()
    // ----------------------------------------
    static nn::os::SystemEventType activateEvent = {};
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(
            &activateEvent, currentDeviceHandle));
    // ----------------------------------------
    // テスト対象 : nn::nfp::AttachDeactivateEvent()
    // ----------------------------------------
    static nn::os::SystemEventType deactivateEvent = {};
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(
            &deactivateEvent, currentDeviceHandle));
    // ----------------------------------------
    // テスト対象 : nn::nfp::StartDetection()
    // ----------------------------------------
    const int WaitTime = 5000;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(currentDeviceHandle));
    EXPECT_TRUE(WaitForActivate(WaitTime, &activateEvent));
    // ----------------------------------------
    // テスト対象 : nn::nfp::Mount()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(
            currentDeviceHandle, nn::nfp::ModelType_Amiibo, nn::nfp::MountTarget_All));
    // ----------------------------------------
    // テスト対象 : nn::nfp::GetState()
    // ----------------------------------------
    EXPECT_TRUE(nnt::nfp::wrapper::GetState() == nn::nfp::State_Init);
    // ----------------------------------------
    // テスト対象 : nn::nfp::GetDeviceState()
    // ----------------------------------------
    EXPECT_TRUE(nnt::nfp::wrapper::GetDeviceState(currentDeviceHandle) == nn::nfp::DeviceState_Mount);
    // ----------------------------------------
    // テスト対象 : nn::nfp::Unmount()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(currentDeviceHandle));
    // ----------------------------------------
    // テスト対象 : nn::nfp::StopDetection()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection(currentDeviceHandle));
    EXPECT_TRUE(WaitForDeactivate(WaitTime, &deactivateEvent));

    // Mount状態にする
    nnt::nfp::DoMount();
    // ----------------------------------------
    // テスト対象 : nn::nfp::GetAdminInfo()
    // ----------------------------------------
    nn::nfp::AdminInfo adminInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    // ----------------------------------------
    // テスト対象 : nn::nfp::GetTagInfo()
    // ----------------------------------------
    nn::nfp::TagInfo tagInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    // ----------------------------------------
    // テスト対象 : nn::nfp::GetRegisterInfo()
    //              nn::nfp::GetRegisterInfo()非公開版
    // ----------------------------------------
    nn::nfp::RegisterInfo regInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    nn::nfp::RegisterInfoPrivate regInfoPrivate;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfoPrivate));
    // ----------------------------------------
    // テスト対象 : nn::nfp::GetCommonInfo()
    // ----------------------------------------
    nn::nfp::CommonInfo commonInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    // ----------------------------------------
    // テスト対象 : nn::nfp::GetModelInfo()
    // ----------------------------------------
    nn::nfp::ModelInfo modelInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    // ----------------------------------------
    // テスト対象 : nn::nfp::DeleteRegisterInfo()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteRegisterInfo());
    // ----------------------------------------
    // テスト対象 : nn::nfp::SetRegisterInfo()
    // ----------------------------------------
    nn::nfp::RegisterInfoPrivate regInfoSet;
#if defined(NNT_NFP_LIB_MII_ENABLE)
    nnt::nfp::BuildMiiData(regInfoSet.miiData);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    std::memcpy(&regInfoSet.nickname,
            nnt::nfp::NormalTagNickName, sizeof(nnt::nfp::NormalTagNickName));
    const nn::Bit8 NormalTagFontRegion = nn::nfp::FontRegion_Taiwan;
    regInfoSet.fontRegion = NormalTagFontRegion;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    // ----------------------------------------
    // テスト対象 : nn::nfp::Flush()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Flush());
    // ----------------------------------------
    // テスト対象 : nn::nfp::DeleteApplicationArea()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteApplicationArea());
    // ----------------------------------------
    // テスト対象 : nn::nfp::ExistsApplicationArea()
    // ----------------------------------------
    bool outValue;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ExistsApplicationArea(&outValue));
    EXPECT_TRUE(outValue == false);
    // ----------------------------------------
    // テスト対象 : nn::nfp::FinalizeSystem()
    // ----------------------------------------
    nnt::nfp::wrapper::FinalizeSystem();

    // 一度FinalizeSystem()を実行したので残りの試験を行うために
    // 再度InitializeSystem()を行います。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystem());
    // UtilのDoSearch()を使用してActive状態までの試験を実施
    // InitializeSystem()を行っているため、UtilではInitializedebug()は実行されない
    nnt::nfp::DoSearch();
    // ----------------------------------------
    // テスト対象 : nn::nfp::GetNpadId()
    // ----------------------------------------
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId));

    nnt::nfp::DoActivate();
    // ----------------------------------------
    // テスト対象 : nn::nfp::Restore()
    // ----------------------------------------
    nn::Result restoreResult;
    restoreResult = nnt::nfp::wrapper::Restore();
    // タグが破壊されているわけではないので、ResultNotBroken()を期待する
    NNT_NFP_RESULT_EQUAL(nn::nfp::ResultNotBroken(), restoreResult);
    // ----------------------------------------
    // テスト対象 : nn::nfp::Format()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Format());
    // ----------------------------------------
    // テスト対象 : nn::nfp::Mount (...)
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::nfp::wrapper::Mount(nnt::nfp::GetCurrentDeviceHandle(),
            nn::nfp::ModelType_Amiibo));
    // 最後にFinalizeSystem()を実行しておく
    nnt::nfp::wrapper::FinalizeSystem();
    NN_LOG("TestCasePermissionSysAllow end \n");
} // NOLINT(impl/function_size)

TEST_F(NfpPermissionSys, TestCasePermissionSysDeniedDeathInitialize)
{
// -------------------------------------------------------------------------
// 権限なしでAPIがAbortすることを確認する試験はInitialize系のみを実施します。
// 本試験の期待動作は権限がないAPIの呼び出しなので Abort が発生することです。
// -------------------------------------------------------------------------
    NN_LOG("TestCasePermissionSysDeniedDeathInitialize start \n");
    NN_LOG("========================================================================\n");
    NN_LOG(" THIS TEST IS EXPECTED TO ABORT \n");
    NN_LOG("========================================================================\n");
    // ----------------------------------------
    // テスト対象 : nn::nfp::Initialize()
    // ----------------------------------------
    nn::Result initializeResult;
    initializeResult = nnt::nfp::wrapper::Initialize();
    EXPECT_TRUE(false);
    NN_LOG("Not Expect ! %s\n", nnt::nfp::GetNfpResultTypeString(initializeResult));
    NN_LOG("TestCasePermissionSysDeniedDeathInitialize end \n");
}

TEST_F(NfpPermissionSys, TestCasePermissionSysDeniedDeathInitializeDebug)
{
// -------------------------------------------------------------------------
// 権限なしでAPIがAbortすることを確認する試験はInitialize系のみを実施します。
// 本試験の期待動作は権限がないAPIの呼び出しなので Abort が発生することです。
// -------------------------------------------------------------------------
    NN_LOG("TestCasePermissionSysDeniedDeathInitializeDebug start \n");
    NN_LOG("========================================================================\n");
    NN_LOG(" THIS TEST IS EXPECTED TO ABORT \n");
    NN_LOG("========================================================================\n");
    // ----------------------------------------
    // テスト対象 : nn::nfp::InitializeDebug()
    // ----------------------------------------
    nn::Result initializeResult;
    initializeResult = nnt::nfp::wrapper::InitializeDebug();
    EXPECT_TRUE(false);
    NN_LOG("Not Expect ! %s\n", nnt::nfp::GetNfpResultTypeString(initializeResult));
    NN_LOG("TestCasePermissionSysDeniedDeathInitializeDebug end \n");
}

TEST_F(NfpPermissionSys, TestCasePermissionSysDeniedDeath)
{
#if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
// -------------------------------------------------------------------------
// 権限なしでAPIがAbortすることを確認する試験はNXで実施できるようになるまで保留とします。
// Win版として試験実装は残しますがWin版では権限関係なく実行できてしまうため
// 下記の実装では試験が行えません。
// -------------------------------------------------------------------------
    // sys権限では実行できないAPIが失敗すること
    NN_LOG("TestCasePermissionSysDeniedDeath start \n");

    const int WaitTime_TagSwitch = 3000;
    //Amiiboタグをかざすメッセージを表示して数秒待つ
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH AMIIBO TAG(Created by NfpManagerEx) (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTime_TagSwitch);
    // ----------------------------------------
    // テスト対象 : Initialize()
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::Initialize());
    // ----------------------------------------
    // テスト対象 : Finalize()
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::Finalize());
    // ----------------------------------------
    // テスト対象 : InitializeDebug()
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::InitializeDebug());
    // ----------------------------------------
    // テスト対象 : FinalizeDebug()
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::FinalizeDebug());

    // sys権限の試験なので、InitializeSystemを実行する
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystem());
    // StartNicknameAndOwnerSettingsのためにListDevicesを実行する
    nn::nfp::DeviceHandle deviceHandle;
    int outCount;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
    // ----------------------------------------
    // テスト対象 : StartNicknameAndOwnerSettings (..., const TagInfo &tagInfo, const RegisterInfo &registerInfo)
    // ----------------------------------------
    nn::nfp::TagInfo tagInfo;
    nn::nfp::AmiiboSettingsStartParam startParam;
    nn::nfp::RegisterInfo regInfo;
    bool isRegistered;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            nullptr, &isRegistered, &regInfo, startParam, tagInfo, regInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, nullptr, &regInfo, startParam, tagInfo, regInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, &isRegistered, nullptr, startParam, tagInfo, regInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            nullptr, &isRegistered, &regInfo, startParam, tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, nullptr, &regInfo, startParam, tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, &isRegistered, nullptr, startParam, tagInfo));
    // ----------------------------------------
    // テスト対象 : StartNicknameAndOwnerSettings (..., const TagInfo &tagInfo)
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            nullptr, &isRegistered, &regInfo, startParam, tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, nullptr, &regInfo, startParam, tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, &isRegistered, nullptr, startParam, tagInfo));
    // ----------------------------------------
    // テスト対象 : StartGameDataEraser (..., const TagInfo &tagInfo)
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartGameDataEraser(
            nullptr, startParam, tagInfo));
    // ----------------------------------------
    // テスト対象 : StartRestorer (..., const TagInfo &tagInfo)
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartRestorer(nullptr, startParam, tagInfo));

    // Mountする
    nnt::nfp::DoMount();
    // ----------------------------------------
    // テスト対象 : GetAll()
    // ----------------------------------------
    nn::nfp::NfpData nfpDataGet;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetAll(&nfpDataGet));
    // ----------------------------------------
    // テスト対象 : SetAll()
    // ----------------------------------------
    nn::nfp::NfpData nfpDataSet;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::SetAll(nfpDataSet));
    // ----------------------------------------
    // テスト対象 : FlushDebug()
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::FlushDebug());
    // ----------------------------------------
    // テスト対象 : BreakTag()
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::BreakTag(nn::nfp::BreakType_Activation));
    // ----------------------------------------
    // テスト対象 : OpenApplicationArea()
    // ----------------------------------------
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    // ----------------------------------------
    // テスト対象 : GetApplicationArea()
    // ----------------------------------------
    nn::Bit8 readBuffer[nn::nfp::ApplicationAreaSizeV2];
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetApplicationArea(readBuffer,
            nn::nfp::ApplicationAreaSizeV2));
    // ----------------------------------------
    // テスト対象 : SetApplicationArea()
    // ----------------------------------------
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_ASSERT_DEATH(
            nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::NormalTagData,
            sizeof(nnt::nfp::NormalTagData),
            tagInfo.tagId));
    // ----------------------------------------
    // テスト対象 : CreateApplicationArea()
    // ----------------------------------------
    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    std::memcpy(tagData,nnt::nfp::ZeroTagData,nn::nfp::ApplicationAreaSizeV2);
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo = { nnt::nfp::ZeroTagId,
                                                       nn::nfp::ApplicationAreaSizeV2,
                                                       tagData };
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));

    // 最後にFinalizeSystem()を実行しておく
    nnt::nfp::wrapper::FinalizeSystem();
    NN_LOG("TestCasePermissionSysDeniedDeath end \n");
#endif  // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
} // NOLINT(impl/function_size)
