﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>
//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    // エージングテストの実行回数です
    const uint32_t  AgingRepeat     = 2000;
    const size_t    ThreadStackSize = 8192;

    // スレッド用の変数群です
    nn::os::ThreadType  g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ ThreadStackSize ];   // 1つ目のイベント操作スレッドのスタック

    // 各エラーの発生回数です
    uint32_t g_ResultSuccessTimes = 0; // ResultSuccessのエラー回数
    uint32_t g_ResultWifiOffTimes = 0; // ResultWifiOffのエラー回数

    // Restoreの発生回数です
    uint32_t g_RestoreTimes = 0;
    uint32_t g_SuccessTimes = 0;

    // Flush実行を通知するSignalです
    nn::os::EventType g_BootUpEvent;
    nn::os::EventType g_FlushEvent;

    //================================================================================
    // NFPタグへランダム値を書込む関数です
    //================================================================================
    void FlushNfpThread(void* pArg) NN_NOEXCEPT
    {
        NN_UNUSED(pArg);
        nn::Result result;
        NN_LOG("### %s Activated !!\n", NN_CURRENT_FUNCTION_NAME);

        if(nn::os::TryWaitEvent(&g_BootUpEvent))
        {
            NN_LOG("### Illeagal Event State !!\n");
            NNT_NFP_ASSERT(false);
            return;
        }

        NN_LOG("### Indicate Thread Boot!!\n");
        nn::os::SignalEvent(&g_BootUpEvent);

        NN_LOG("### Waiting for Flush Signal...\n");
        nn::os::WaitEvent(&g_FlushEvent);

        result = nnt::nfp::FlushWithRetry();
        NN_LOG("### Indicate End Flush Signal !!\n");
        nn::os::ClearEvent(&g_FlushEvent);

        do
        {
            if(result.IsSuccess())
            {
                g_ResultSuccessTimes++;
                NN_LOG("### Flush() returned ResultSuccess \n");
                break;
            }

            if(result <= nn::nfp::ResultNfcDisabled())
            {
                g_ResultWifiOffTimes++;
                NN_LOG("### Flush() returned ResultWifiOFF \n");
                break;
            }
            /* RESULT_ERROR_INVALID_DEVICE_STATE or ResultSuccess以外はエラー出力 */
            NNT_NFP_ASSERT(false);

        } while(NN_STATIC_CONDITION(false));

        nn::os::ClearEvent(&g_BootUpEvent);
        return;
    }

    //================================================================================
    // 他スレッド待機用関数です。
    //================================================================================
    void StartFlushNfpThread() NN_NOEXCEPT
    {
        // タグ書き込み用スレッドを生成する
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::CreateThread(&g_Thread, FlushNfpThread, nullptr,
                g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority)
        );
        nn::os::StartThread(&g_Thread);
    }

    void WaitForThreadStart() NN_NOEXCEPT
    {
        NN_LOG("### Wainting for Thraed Start \n");

        nn::os::WaitEvent(&g_BootUpEvent);

        NN_LOG("### Thread Start Signal Detected !! \n");
    }


    void WaitFortThreadFinish() NN_NOEXCEPT
    {
        nn::os::WaitThread(&g_Thread);
        nn::os::DestroyThread(&g_Thread);
    }

} // end of anonymous namespace

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================
class NfpAgingWifiOffWhenFlush : public nnt::nfp::TestFramework
{
protected:

    NfpAgingWifiOffWhenFlush() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
        nnt::nfp::wrapper::SetApiCallLoggingMode(nnt::nfp::wrapper::LogMode_Aging);
    }
};

//================================================================================
// テストケースの実装です。
//================================================================================
TEST_F(NfpAgingWifiOffWhenFlush, TestCaseWifiOffWhenFlush)
{
    nn::Result result;
    nn::Bit8 writeData[nn::nfp::ApplicationAreaSizeV2]   = {0};    // Write DataBuf
    nn::Bit8 readData[nn::nfp::ApplicationAreaSizeV2]    = {0};    // Read DataBuf
    nn::Bit8 tmpData[nn::nfp::ApplicationAreaSizeV2]     = {0};

#if !defined(NNT_NFP_PLATFORM_NX)
    /* Wifi Off時間の調整はWifi制御仕様公開後に対応 */
    const uint32_t WaitUSecBase1    = 0;    // WifiOff後のSleep時間の基本値(usec)
    const uint32_t WaitUSecBase2    = 500;  // WifiOn後のSleep時間の基本値(usec)
    const uint32_t WaitSleepRange1  = 1000; // WifiOff後のSleep時間の幅(1000usec)
    const uint32_t WaitSleepRange2  = 10;   // WifiOff後のSleep時間の幅(10usec)
    uint32_t SleepUSec = WaitUSecBase1;     // WifiOff後のSleep時間(usec)

    // テスト前処理
    nnt::nfp::WifiOn();
    nnt::nfp::DoCreateNormalTagAndEnd();


    //Sleep時間検索処理1 10usec刻みで検索
    for(uint32_t i = 0; i < WaitSleepRange1; i += WaitSleepRange2)
    {
        // イベントを初期化する
        nn::os::InitializeEvent(&g_BootUpEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&g_FlushEvent, false, nn::os::EventClearMode_ManualClear);

        /* 1. NFP StartDetection */
        nnt::nfp::DoActivate();

        result = nnt::nfp::MountWithRetry();
        if(result <= nn::nfp::ResultNeedRestore())
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::RestoreWithRetry());
        }
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        nnt::nfp::RegisterNfp();

        // NFPタグ読込
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(readData, nn::nfp::ApplicationAreaSizeV2));

        // Create random data
        nnt::nfp::CreateRandomNumberSequence(writeData, nn::nfp::ApplicationAreaSizeV2);

        // 最大容量 nn::nfp::ApplicationAreaSizeV2 Byte の書き込み
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(writeData, nn::nfp::ApplicationAreaSizeV2));

        // タグ書き込み用スレッドを起動する
        StartFlushNfpThread();

        // タグ書き込みスレッド起動を待つ
        WaitForThreadStart();

        // 無線状態変更後にタグ書き込み用イベントを発行する。
        NN_LOG("### WIFI OFF Sleep = %d usec \n", SleepUSec + i);
        nnt::nfp::WifiOff();
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(SleepUSec++));
        nn::os::SignalEvent(&g_FlushEvent);

        // タグ書き込みスレッドの終了を待つ
        WaitFortThreadFinish();

        NN_LOG("### WIFI ON \n");
        nnt::nfp::WifiOn();
        nnt::nfp::Sleep(WaitUSecBase2);

        // タグ検知中にWifiOFFされるとDeviceState_Init状態に遷移すること
        NNT_NFP_ASSERT_EQUAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        NNT_NFP_ASSERT_EQUAL(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

        if(g_ResultWifiOffTimes > 0)
        {
            if(i != 0)
            {
                SleepUSec = (SleepUSec + i) - (WaitSleepRange2 - 1);
            }
            break;
        }
    }

    //Sleep時間検索処理2 Sleep時間検索処理1で見つかった時間から9usecマイナスし、1usec刻みで検索
    if(0 >= g_ResultWifiOffTimes)
    {
        NN_LOG("###  Of Sleep times search failure ### \n");
        NNT_NFP_ASSERT(false);
        return;
    }

    g_ResultSuccessTimes = 0;
    g_ResultWifiOffTimes = 0;

    for(uint32_t i = 0; i < WaitSleepRange2; i++)
    {
        nn::os::ClearEvent(&g_BootUpEvent);
        nn::os::ClearEvent(&g_FlushEvent);

        // NFPタグをマウントする
        nnt::nfp::DoActivate();

        result = nnt::nfp::MountWithRetry();
        if(result <= nn::nfp::ResultNeedRestore())
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::RestoreWithRetry());
        }
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        nnt::nfp::RegisterNfp();

        // NFPタグ読込
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(readData, nn::nfp::ApplicationAreaSizeV2));

        // Create random data
        nnt::nfp::CreateRandomNumberSequence(writeData, nn::nfp::ApplicationAreaSizeV2);

        // 最大容量 nn::nfp::ApplicationAreaSizeV2 Byte の書き込み
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(writeData, nn::nfp::ApplicationAreaSizeV2));

        // タグ書き込み用スレッドを起動する
        StartFlushNfpThread();

        // タグ書き込みスレッド起動を待つ
        WaitForThreadStart();

        // 無線状態変更後にタグ書き込み用イベントを発行する。
        NN_LOG("### WIFI OFF Sleep = %d usec \n", SleepUSec);
        nnt::nfp::WifiOff();
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(SleepUSec++));
        nn::os::SignalEvent(&g_FlushEvent);

        // タグ書き込みスレッドの終了を待つ
        WaitFortThreadFinish();

        NN_LOG("### WIFI ON \n");
        nnt::nfp::WifiOn();
        nnt::nfp::Sleep(WaitUSecBase2);

        // タグ検知中にWifiOFFされるとDeviceState_Init状態に遷移すること
        NNT_NFP_ASSERT_EQUAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

        if(g_ResultWifiOffTimes > 0)
        {
            break;
        }
    }

    SleepUSec--;
    g_ResultSuccessTimes = 0;
    g_ResultWifiOffTimes = 0;

    //実試験開始
    NN_LOG("### WifiOffWhenFlush Start ### \n");

    for(uint32_t i = 0; i < AgingRepeat; i++)
    {

        nn::os::ClearEvent(&g_BootUpEvent);
        nn::os::ClearEvent(&g_FlushEvent);

        // NFPタグをマウントする
        nnt::nfp::DoActivate();

        result = nnt::nfp::MountWithRetry();
        if(result <= nn::nfp::ResultNeedRestore())
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Restore());
        }
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        nnt::nfp::RegisterNfp();

        // NFPタグ読込
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(readData, nn::nfp::ApplicationAreaSizeV2));

        // Create random data
        nnt::nfp::CreateRandomNumberSequence(writeData, nn::nfp::ApplicationAreaSizeV2);

        // 最大容量 nn::nfp::ApplicationAreaSizeV2 Byte の書き込み
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(writeData, nn::nfp::ApplicationAreaSizeV2));

        // タグ書き込み用スレッドを起動する
        StartFlushNfpThread();

        // タグ書き込みスレッド起動を待つ
        WaitForThreadStart();

        // 無線状態変更後にタグ書き込み用イベントを発行する。
        NN_LOG("### WIFI OFF Sleep = %d usec \n", SleepUSec);
        nnt::nfp::WifiOff();
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(SleepUSec++));
        nn::os::SignalEvent(&g_FlushEvent);

        // タグ書き込みスレッドの終了を待つ
        WaitFortThreadFinish();

        NN_LOG("### WIFI ON \n");
        nnt::nfp::WifiOn();
        nnt::nfp::Sleep(WaitUSecBase2);

        // タグ検知中にWifiOFFされるとDeviceState_Init状態に遷移すること
        NNT_NFP_ASSERT_EQUAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        NNT_NFP_ASSERT_EQUAL(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());

        // ライブラリを終了するとState_None状態に遷移すること
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        NNT_NFP_ASSERT_EQUAL(nn::nfp::State_None, nnt::nfp::wrapper::GetState());

        // Mount後、DeviceStateがDeviceState_Mount状態に遷移すること
        nnt::nfp::DoMount();
        NNT_NFP_ASSERT_EQUAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        NNT_NFP_ASSERT_EQUAL(nn::nfp::DeviceState_Mount, nnt::nfp::wrapper::GetDeviceState());

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(tmpData, nn::nfp::ApplicationAreaSizeV2));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

    }


    NN_LOG("### Total ResultSuccess %d Times \n", g_ResultSuccessTimes);
    NN_LOG("### Total ResultWifiOFF %d Times \n", g_ResultWifiOffTimes);

    NNT_NFP_ASSERT(0 < g_ResultSuccessTimes);
    NNT_NFP_ASSERT(0 < g_ResultWifiOffTimes);
#else // !defined(NNT_NFP_PLATFORM_NX)

    // NXは下記の応答を期待します。
    // flush後 600ms 付近まで Nfc OFF -> ResultNfcDisabledが返却され Mountは Success
    // flush後 700ms 付近から Nfc OFF -> ResultNfcDisabledが返却され Mountは ResultNeedRestore
    // flush後 1500ms 以降は Nfc OFF にしてもすでに成功済みとなる
    // 試験は flush と Nfc OFF の間隔を 0ms ～ 2000ms で行います

    /* Wifi Off時間の調整はWifi制御仕様公開後に対応 */
    const uint32_t WaitUSecBase1    = 0;    // WifiOff後のSleep時間の基本値(usec)
    const uint32_t WaitUSecBase2    = 500;  // WifiOn後のSleep時間の基本値(usec)
    const uint32_t WaitSleepRange1  = 1000; // WifiOff後のSleep時間の幅(1000usec)
    uint32_t SleepUSec = WaitUSecBase1;     // WifiOff後のSleep時間(usec)

    // テスト前処理
    nnt::nfp::InitializeNfcSystem();
    nnt::nfp::WifiOn();
    nnt::nfp::DoCreateNormalTagAndEnd();

    g_ResultSuccessTimes = 0;
    g_ResultWifiOffTimes = 0;
    g_RestoreTimes       = 0;
    g_SuccessTimes       = 0;

    // イベントを初期化する
    nn::os::InitializeEvent(&g_BootUpEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&g_FlushEvent, false, nn::os::EventClearMode_ManualClear);

    //実試験開始
    NN_LOG("### WifiOffWhenFlush Start ### \n");

    for(uint32_t i = 0; i < AgingRepeat; i++)
    {

        nn::os::ClearEvent(&g_BootUpEvent);
        nn::os::ClearEvent(&g_FlushEvent);

        // NFPタグをマウントする
        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        nnt::nfp::RegisterNfp();

        // NFPタグ読込
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(readData, nn::nfp::ApplicationAreaSizeV2));

        // Create random data
        nnt::nfp::CreateRandomNumberSequence(writeData, nn::nfp::ApplicationAreaSizeV2);

        // 最大容量 nn::nfp::ApplicationAreaSizeV2 Byte の書き込み
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(writeData, nn::nfp::ApplicationAreaSizeV2));

        // タグ書き込み用スレッドを起動する
        StartFlushNfpThread();

        // タグ書き込みスレッド起動を待つ
        WaitForThreadStart();

        // 無線状態変更後にタグ書き込み用イベントを発行する。
        SleepUSec = SleepUSec + WaitSleepRange1;
        NN_LOG("### WIFI OFF Sleep = %d usec \n", SleepUSec);
        nn::os::SignalEvent(&g_FlushEvent);
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(SleepUSec));
        nnt::nfp::WifiOff();

        // タグ書き込みスレッドの終了を待つ
        WaitFortThreadFinish();

        NN_LOG("### WIFI ON \n");
        nnt::nfp::WifiOn();
        nnt::nfp::Sleep(WaitUSecBase2);

        // タグ検知中にWifiOFFされるとDeviceState_Init状態に遷移すること
        NNT_NFP_ASSERT_EQUAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        NNT_NFP_ASSERT_EQUAL(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());

        // ライブラリを終了するとState_None状態に遷移すること
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        NNT_NFP_ASSERT_EQUAL(nn::nfp::State_None, nnt::nfp::wrapper::GetState());

        // Mount後、DeviceStateがDeviceState_Mount状態に遷移すること
        // タグの修復が必要な場合にはリストアする
        nnt::nfp::DoActivate();
        result = nnt::nfp::MountWithRetry();
        if(result <= nn::nfp::ResultNeedRestore())
        {
            g_RestoreTimes++;
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Restore());
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
        }
        else if(result.IsSuccess())
        {
            g_SuccessTimes++;
        }
        NNT_NFP_ASSERT_EQUAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        NNT_NFP_ASSERT_EQUAL(nn::nfp::DeviceState_Mount, nnt::nfp::wrapper::GetDeviceState());

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(tmpData, nn::nfp::ApplicationAreaSizeV2));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

    }
    nnt::nfp::FinalizeNfcSystem();

    NN_LOG("### Total ResultSuccess %d Times \n", g_ResultSuccessTimes);
    NN_LOG("### Total ResultWifiOFF %d Times \n", g_ResultWifiOffTimes);
    NN_LOG("### Total Success %d Times \n", g_SuccessTimes);
    NN_LOG("### Total Restore %d Times \n", g_RestoreTimes);

    NNT_NFP_ASSERT(0 < g_ResultSuccessTimes);
    NNT_NFP_ASSERT(0 < g_ResultWifiOffTimes);
    NNT_NFP_ASSERT(0 < g_RestoreTimes);
    NNT_NFP_ASSERT(0 < g_SuccessTimes);
#endif // !defined(NNT_NFP_PLATFORM_NX)

}   // NOLINT(impl/function_size)

