﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nfc/nfc_PrivateApi.h>
#include <nnt/nfp/testNpt_Common.h>

namespace {

    uint32_t g_LoggingMode = 0;

    const char* RemoveNameSpaceString(const char* pApiName) NN_NOEXCEPT
    {
        const char* NameSpaceString = "nnt::npt::wrapper::";
        if((std::strlen(pApiName) > std::strlen(NameSpaceString)) &&
           (std::memcmp(pApiName,NameSpaceString,std::strlen(NameSpaceString)) == 0))
        {
            // namespace名分の文字列をスキップ
            return pApiName + std::strlen(NameSpaceString);
        }
        return pApiName;
    }


    void PrintApiName(const char* pApiName) NN_NOEXCEPT
    {
        if(g_LoggingMode & nnt::npt::wrapper::LogMode_Aging)
        {
            NN_LOG("[NPT_API_CALL]nn::nfc::pt::%s() ,",RemoveNameSpaceString(pApiName));
        }
    }

    void PrintApiCallLog(const char* pApiName, nn::Result result, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::npt::wrapper::LogMode_Aging)
        {
            NN_LOG("Description = %s ,call count = %lld\n",
                   nnt::npt::GetNptResultTypeString(result),++count);
        }
    }

    void PrintApiCallLog(const char* pApiName,
            nn::nfc::State state, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::npt::wrapper::LogMode_Aging)
        {
            const char* pString;
            switch(state)
            {
            case nn::nfc::State_Init:
                {
                    pString = "State_Init";
                }
                break;
            case nn::nfc::State_None:
                {
                    pString = "State_None";
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            NN_LOG("State = %s ,call count = %lld\n",
                   pString,++count);
        }
    }

    void PrintApiCallLog(const char* pApiName,
            nn::nfc::DeviceState deviceState, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::npt::wrapper::LogMode_Aging)
        {
            const char* pString;
            switch(deviceState)
            {
            case nn::nfc::DeviceState_Init:
                {
                    pString = "DeviceState_Init";
                }
                break;
            case nn::nfc::DeviceState_Search:
                {
                    pString = "DeviceState_Search";
                }
                break;
            case nn::nfc::DeviceState_Active:
                {
                    pString = "DeviceState_Active";
                }
                break;
            case nn::nfc::DeviceState_Deactive:
                {
                    pString = "DeviceState_Deactive";
                }
                break;
            case nn::nfc::DeviceState_Keep:
                {
                    pString = "DeviceState_Keep";
                }
                break;
            case nn::nfc::DeviceState_Unexpected:
                {
                    pString = "DeviceState_Unexpected";
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            NN_LOG("DeviceState = %s ,call count = %lld\n",
                   pString,++count);
        }
    }
} // namespace

namespace nnt { namespace npt { namespace wrapper {
    //==================================================
    // API実行時のログ設定
    //==================================================

    void SetApiCallLoggingMode(const uint32_t mode) NN_NOEXCEPT
    {
        g_LoggingMode = mode;
    }

    //NptライブラリAPI実行用ラッパー関数
    nn::Result Initialize() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::Initialize();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        /* テストプログラムの互換性の為、nn::ResultSuccess()を返す */

        return nn::ResultSuccess();
    }

    void Finalize() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::Finalize();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
    }

    nn::Result ListDevices(nn::nfc::DeviceHandle* pOutBuffer,
            int* pOutCount, int bufferCount) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::ListDevices(pOutBuffer, pOutCount, bufferCount);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result StartDetection(const nn::nfc::DeviceHandle& deviceHandle, nn::nfc::NfcProtocol protocolFilter) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::StartDetection(deviceHandle, protocolFilter);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result StartDetection(const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        return nnt::npt::wrapper::StartDetection(deviceHandle, nn::nfc::NfcProtocol_All);
    }

    nn::Result StartDetection(nn::nfc::NfcProtocol protocolFilter) NN_NOEXCEPT
    {
        return nnt::npt::wrapper::StartDetection(GetCurrentDeviceHandle(), protocolFilter);
    }

    nn::Result StartDetection() NN_NOEXCEPT
    {
        return nnt::npt::wrapper::StartDetection(GetCurrentDeviceHandle(), nn::nfc::NfcProtocol_All);
    }

    nn::Result StopDetection(const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::StopDetection(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result StopDetection() NN_NOEXCEPT
    {
        return nnt::npt::wrapper::StopDetection(GetCurrentDeviceHandle());
    }

    nn::Result GetTagInfo(nn::nfc::TagInfo* pOutTagInfo,
            const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::GetTagInfo(pOutTagInfo, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetTagInfo(nn::nfc::TagInfo* pOutTagInfo) NN_NOEXCEPT
    {
        return nnt::npt::wrapper::GetTagInfo(pOutTagInfo, GetCurrentDeviceHandle());
    }

    nn::Result AttachActivateEvent(nn::os::SystemEventType* pEvent,
            const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::AttachActivateEvent(pEvent, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result AttachActivateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        return nnt::npt::wrapper::AttachActivateEvent(pEvent, GetCurrentDeviceHandle());
    }

    nn::Result AttachDeactivateEvent(nn::os::SystemEventType* pEvent,
            const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::AttachDeactivateEvent(pEvent, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result AttachDeactivateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        return nnt::npt::wrapper::AttachDeactivateEvent(pEvent, GetCurrentDeviceHandle());
    }

    nn::Result AttachAvailabilityChangeEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::AttachAvailabilityChangeEvent(pEvent);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return nn::ResultSuccess();
    }

    nn::nfc::State GetState() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::State state = nn::nfc::GetState();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,state,++count);
        return state;
    }

    nn::nfc::DeviceState GetDeviceState(
            const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::DeviceState deviceState = nn::nfc::GetDeviceState(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,deviceState,++count);
        return deviceState;
    }

    nn::nfc::DeviceState GetDeviceState() NN_NOEXCEPT
    {
        return nnt::npt::wrapper::GetDeviceState(GetCurrentDeviceHandle());
    }

    nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId,
            const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::GetNpadId(pOutNpadId, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId) NN_NOEXCEPT
    {
        return nnt::npt::wrapper::GetNpadId(pOutNpadId, GetCurrentDeviceHandle());
    }

    nn::Result SendCommandByPassThrough(void* pOutBuffer, size_t* pOutSize,
                                        const nn::nfc::DeviceHandle& deviceHandle,
                                        const void* pData, size_t dataSize,
                                        size_t bufferSize, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::SendCommandByPassThrough(pOutBuffer, pOutSize,
                                                              deviceHandle,
                                                              pData, dataSize,
                                                              bufferSize, timeout);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result SendCommandByPassThrough(void* pOutBuffer, size_t* pOutSize,
                                        const void* pData, size_t dataSize,
                                        size_t bufferSize, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        return nnt::npt::wrapper::SendCommandByPassThrough(pOutBuffer, pOutSize,
                                                           GetCurrentDeviceHandle(),
                                                           pData, dataSize,
                                                           bufferSize, timeout);
    }

    nn::Result KeepPassThroughSession(const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::KeepPassThroughSession(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result KeepPassThroughSession() NN_NOEXCEPT
    {
        return nnt::npt::wrapper::KeepPassThroughSession(GetCurrentDeviceHandle());
    }

    nn::Result ReleasePassThroughSession(const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfc::ReleasePassThroughSession(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result ReleasePassThroughSession() NN_NOEXCEPT
    {
        return nnt::npt::wrapper::ReleasePassThroughSession(GetCurrentDeviceHandle());
    }

    nn::Result InitializeSystem() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::InitializeSystem();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return nn::ResultSuccess();
    }

    void FinalizeSystem() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        static uint64_t count = 0;
        nn::nfc::FinalizeSystem();
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return;
    }

    nn::Result InitializeDebug() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfc::InitializeSystem();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        /* テストプログラムの互換性の為、nn::ResultSuccess()を返す */
        return nn::ResultSuccess();
    }

    void FinalizeDebug() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        static uint64_t count = 0;
        nn::nfc::FinalizeSystem();
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return;
    }
}}} //// end of namespace nnt::npt::wrapper
