﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkText.h>
#include <nn/hid.h>
#include <nn/mii.h>
#include <nn/nfc.h>
#include <nn/nfp/nfp_PrivateResult.h>
#include <nn/nfc/nfc_PrivateApi.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/mii/mii_StoreDataAccessor.h>
#include <nn/mii/mii_StoreDataContext.h>

#include <nnt/result/testResult_Assert.h>
#include <nnt/nfp/testNfp_Common.h>

#if defined(NNT_NFP_PLATFORM_NX)
#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/time/time_ApiForSystem.h>
#endif

namespace nnt { namespace nfp {

    //======================================================
    // テストで使用する定数です。
    //======================================================

    const uint32_t ZeroTagId            = 0x12345678;
    const uint32_t NormalTagId          = 0x87654321;
    const uint32_t NfpManagerUsedTagId  = 0x0FF41E00;

    const uint32_t FlushTrialCount      = 10;
    const uint32_t FlushTrialDuringTime = 100;

    const nn::Bit8 NormalTagFontRegion = nn::nfp::FontRegion_Taiwan;

    //======================================================
    // ユーティリティ関数で使用する使用する変数です。
    //======================================================

    nn::nfp::ApplicationAreaCreateInfo g_AppAreaInfo;
    nn::nfp::CommonInfo                g_CommonInfo;
    nn::nfp::AdminInfo                 g_AdminInfo;
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnect機能は実装されない
    nn::nfp::ConnectionStatus          g_ConnectionStatus;
#endif // !defined(NNT_NFP_PLATFORM_NX)

    nn::nfp::Date                      g_NfpDate;
    nn::nfp::RegisterInfo              g_RegInfo;
    nn::nfp::RegisterInfoPrivate       g_RegInfoSet;
    nn::nfp::TagInfo                   g_TagInfo;
    nn::nfp::ModelInfo                 g_ModelRomInfo;
    nn::nfp::DeviceHandle              g_CurrentDeviceHandle;

#if defined(NNT_NFP_PLATFORM_CAFE)
    OSEvent                            g_ActivateEvent;
    OSEvent                            g_DeactivateEvent;
#elif defined(NNT_NFP_PLATFORM_CTR)
    nn::os::Event                      g_ActivateEvent;
    nn::os::Event                      g_DeactivateEvent;
#elif defined(NNT_NFP_PLATFORM_NX)
    nn::os::SystemEventType            g_ActivateEvent = {};
    nn::os::SystemEventType            g_DeactivateEvent = {};
    nn::os::SystemEventType            g_AvailabilityChangeEvent = {};
#endif // defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
    nn::fnd::DateTime s_Now;
#endif // !defined(NNT_NFP_PLATFORM_NX)
    nn::TimeSpan g_Span;

    //======================================================
    // テストで使用するデータです。
    //======================================================

    const nn::Bit8 ZeroTagData[nn::nfp::ApplicationAreaSizeV2] = {};

    const nn::Bit8 NormalTagData[nn::nfp::ApplicationAreaSizeV2] =
    {
        1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,
        27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,
        50,51,52,53,54,55,56,57,58,59,60,61,62,63,64,65,66,67,68,69,70,71,72,
        73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,91,92,93,94,95,
        96,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,
        114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,
        131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,
        148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,
        165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,
        182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,
        199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216
    };
    const nn::Bit8 NormalTagData1[nn::nfp::ApplicationAreaSizeV2] =
    {
        216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,
        198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,
        181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,
        164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,
        147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,
        130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,
        113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,
        95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,
        72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,
        49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,
        26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4,3,2,1
    };
    const nn::Bit8 NormalTagData2[nn::nfp::ApplicationAreaSizeV2] =
    {
        96,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,
        114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,
        131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,
        148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,
        165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,
        182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,
        199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216,
        1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,
        27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,
        50,51,52,53,54,55,56,57,58,59,60,61,62,63,64,65,66,67,68,69,70,71,72,
        73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,91,92,93,94,95
    };
    const nn::Bit8 NormalTagData3[nn::nfp::ApplicationAreaSizeV2] =
    {
        95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,
        72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,
        49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,
        26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4,3,2,1,
        216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,
        198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,
        181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,
        164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,
        147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,
        130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,
        113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96
    };
    const nn::Bit8 NormalTagMiiData[nnt::nfp::TestMiiDataSize] =
    {
        0x21, 0x7f, 0x00, 0x01, 0x08, 0x01, 0x13, 0x00, 0x08, 0x02, 0x17, 0x8c, 0x06, 0x01, 0x69, 0x6d,
        0x8a, 0x6a, 0x82, 0x14, 0x02, 0x00, 0x20, 0x64, 0x72, 0x44, 0x04, 0x04, 0xaa, 0x30, 0xfc, 0x30,
        0xca, 0x30, 0xfc, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x9a, 0xa2, 0xfc, 0xe5, 0x31, 0x33, 0x4f, 0xa0, 0x86, 0xb6, 0xb4, 0xd6, 0xb4, 0x13, 0x33, 0x6c,
        0x4a, 0xb5, 0x2f, 0x2f,
    };

    // 下記の記述方法だとwindowsのビルドを行うとShift_JIS文字扱いとなるため
    // Utf-8文字列としてTagNickNameはバイト配列で定義する。
    // const char NormalTagNickName[nn::nfp::NicknameLengthMax * 4 + 1] = "任天堂NFPTEST";
    const unsigned char NormalTagNickName[nn::nfp::NicknameLengthMax * 4 + 1] =
    {
        0xe4, 0xbb, 0xbb, 0xe5, 0xa4, 0xa9, 0xe5, 0xa0, 0x82, 0x4e,
        0x46, 0x50, 0x54, 0x45, 0x53, 0x54
    };

    // 下記の記述方法だとwindowsのビルドを行うとShift_JIS文字扱いとなるため
    // Utf-8文字列としてTagNickNameはバイト配列で定義する。
    // const char NormalTagNickName2[nn::nfp::NicknameLengthMax * 4 + 1] = "NFPTEST任天堂";
    const unsigned char NormalTagNickName2[nn::nfp::NicknameLengthMax * 4 + 1] =
    {
        0x4e, 0x46, 0x50, 0x54, 0x45, 0x53, 0x54, 0xe4, 0xbb, 0xbb,
        0xe5, 0xa4, 0xa9, 0xe5, 0xa0, 0x82
    };

    // 下記の記述方法だとwindowsのビルドを行うとShift_JIS文字扱いとなるため
    // Utf-8文字列としてTagNickNameはバイト配列で定義する。
    // const char NormalTagNickName2[nn::nfp::NicknameLengthMax * 4 + 1] = "NFPTEST";
    const unsigned char NormalTagNickName3[nn::nfp::NicknameLengthMax * 4 + 1] =
    {
        0x4e, 0x46, 0x50, 0x54, 0x45, 0x53, 0x54
    };

    // 下記の記述方法だとwindowsのビルドを行うとShift_JIS文字扱いとなるため
    // Utf-8文字列としてTagNickNameはバイト配列で定義する。
    // const char NormalTagNickName2[nn::nfp::NicknameLengthMax * 4 + 1] = "任天堂";
    const unsigned char NormalTagNickName4[nn::nfp::NicknameLengthMax * 4 + 1] =
    {
        0xe4, 0xbb, 0xbb, 0xe5, 0xa4, 0xa9, 0xe5, 0xa0, 0x82
    };

    // Miiデータ
    // ニックネーム : オーナー
    // お気に入りの色 : 黄色
    // 性別 : 男性
    // 身長 : 最も高い
    // 体格 : 最も細い
    const nn::Bit8 MiiData2[nnt::nfp::TestMiiDataSize] =
    {
        0x44, 0x7f, 0x00, 0x32, 0x08, 0x00, 0x13, 0x00, 0x08, 0x02, 0x17, 0x8c, 0x06, 0x01, 0x69, 0x6d,
        0x8a, 0x6a, 0x82, 0x14, 0x00, 0x02, 0x04, 0x20, 0x64, 0x72, 0x44, 0x44, 0xaa, 0x30, 0xfc, 0x30,
        0xca, 0x30, 0xfc, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x51, 0x6e, 0x08, 0xd0, 0x89, 0x55, 0x44, 0x39, 0x93, 0xee, 0x87, 0xa5, 0x0b, 0xb4, 0x4a, 0x5c,
        0xe8, 0x10, 0x2f, 0x2f,
    };

    // ニックネームに使用できる文字一覧
    // 数字の0～9
    // アルファベット大文字小文字のA～Z
    const unsigned char AllowAsciiCode[62] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
        0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a,
        0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54,
        0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x61, 0x62, 0x63, 0x64,
        0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e,
        0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
        0x79, 0x7A
    };

    //======================================================
    // コントローラ初期化に関連する変数です。
    //======================================================
    const nn::hid::NpadIdType NpadIds[] = {nn::hid::NpadId::No1,
                                       nn::hid::NpadId::No2,
                                       nn::hid::NpadId::No3,
                                       nn::hid::NpadId::No4,
                                       nn::hid::NpadId::Handheld,
                                       };
    const int NpadIdCountMax = sizeof(NpadIds) / sizeof(nn::hid::NpadIdType);

    //======================================================
    // プラットフォーム毎に異なる処理をラップする関数です。
    //======================================================


    nn::Result InitializeSystem() NN_NOEXCEPT
    {
#if defined(NNT_NFP_PLATFORM_CTR)
        return nnt::nfp::wrapper::InitializeSystem();
#elif defined(NNT_NFP_PLATFORM_NX)  // NXではInitializeSystemを多重呼出しするとアボートする
        if( nnt::nfp::wrapper::GetState() == nn::nfp::State_None )
        {
            //NXではInitializeSystemで初期化した場合に実行できないAPIがあるのでInitializeDebugを使用する
            return nnt::nfp::wrapper::InitializeDebug();
        }
        else
        {
            NN_LOG("* Duplicate function call : InitializeDebug() \n");
            return nn::ResultSuccess();
        }
#else
            return nnt::nfp::wrapper::Initialize();
#endif // defined(NNT_NFP_PLATFORM_CTR)) || (defined(NNT_NFP_PLATFORM_NX)
    }

    nn::Result FinalizeSystem() NN_NOEXCEPT
    {
#if defined(NNT_NFP_PLATFORM_CTR)
        return nnt::nfp::wrapper::FinalizeSystem();
#elif defined(NNT_NFP_PLATFORM_NX)  // NXではFinalizeSystemを多重呼出しするとアボートする
        if( nnt::nfp::wrapper::GetState() != nn::nfp::State_None )
        {
            nnt::nfp::wrapper::FinalizeDebug();
        }
        else
        {
            NN_LOG("* Duplicate function call : FinalizeDebug()");
        }
        return nn::ResultSuccess();
#else
        return nnt::nfp::wrapper::Finalize();
#endif
    }

    nn::Result SetDefaultActivateEvent() NN_NOEXCEPT
    {
        if(g_ActivateEvent._state != nn::os::SystemEventType::State_NotInitialized)
        {
            DestroyDefaultActivateEvent();
        }
        nn::Result result;
        result = nnt::nfp::wrapper::AttachActivateEvent(&g_ActivateEvent);
        if(!result.IsSuccess())
        {
            std::memset(&g_ActivateEvent, 0x00, sizeof(g_ActivateEvent));
        }
        return result;
    }

    nn::Result SetDefaultDeactivateEvent() NN_NOEXCEPT
    {
        if(g_DeactivateEvent._state != nn::os::SystemEventType::State_NotInitialized)
        {
            DestroyDefaultDeactivateEvent();
        }
        nn::Result result;
        result = nnt::nfp::wrapper::AttachDeactivateEvent(&g_DeactivateEvent);
        if(!result.IsSuccess())
        {
            std::memset(&g_DeactivateEvent, 0x00, sizeof(g_DeactivateEvent));
        }
        return result;
    }

    nn::Result SetDefaultAvailabilityChangeEvent() NN_NOEXCEPT
    {
        if(g_AvailabilityChangeEvent._state != nn::os::SystemEventType::State_NotInitialized)
        {
            DestroyDefaultAvailabilityChangeEvent();
        }
        nn::Result result;
        result = nnt::nfp::wrapper::AttachAvailabilityChangeEvent(&g_AvailabilityChangeEvent);
        if(!result.IsSuccess())
        {
            std::memset(&g_AvailabilityChangeEvent, 0x00, sizeof(g_AvailabilityChangeEvent));
        }
        return result;
    }

    void DestroyDefaultActivateEvent() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&g_ActivateEvent);
        return;
    }

    void DestroyDefaultDeactivateEvent() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&g_DeactivateEvent);
        return;
    }

    void DestroyDefaultAvailabilityChangeEvent() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&g_AvailabilityChangeEvent);
        return;
    }

    bool WaitForActivate(int ms) NN_NOEXCEPT
    {
        #if defined(NNT_NFP_PLATFORM_CTR)
            return g_ActivateEvent.Wait(nn::fnd::TimeSpan::FromMilliSeconds(ms));
        #elif defined(NNT_NFP_PLATFORM_CAFE)
            OSWaitEvent(&g_ActivateEvent);
        #elif defined(NNT_NFP_PLATFORM_NX)
            if((g_ActivateEvent._state != nn::os::SystemEventType::State_NotInitialized) &&
                    (g_DeactivateEvent._state != nn::os::SystemEventType::State_NotInitialized))
            {
                return nn::os::TimedWaitSystemEvent(&g_ActivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
            }
            else
            {
                NN_LOG("WaitForActivate() ActiveEvent DeactivateEvent is NotInitialized \n");
                return false;
            }
        #endif // defined(NNT_NFP_PLATFORM_NX)
    }

    bool WaitForActivate() NN_NOEXCEPT
    {
        return WaitForActivate(ActivateTimeout);
    }

    bool WaitForDeactivate(int ms) NN_NOEXCEPT
    {
        #if defined(NNT_NFP_PLATFORM_CTR)
            return g_DeactivateEvent.Wait(nn::fnd::TimeSpan::FromMilliSeconds(ms));
        #elif defined(NNT_NFP_PLATFORM_CAFE)
            return OSWaitEventWithTimeout(&g_DeactivateEvent, ms * 1000000LL) == TRUE;
        #elif defined(NNT_NFP_PLATFORM_NX)
            return nn::os::TimedWaitSystemEvent(&g_DeactivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
        #endif // defined(NNT_NFP_PLATFORM_NX)
    }

    bool WaitForDeactivate() NN_NOEXCEPT
    {
        return WaitForDeactivate(DeactivateTimeout);
    }

    bool WaitForAvailabilityChange(int ms) NN_NOEXCEPT
    {
        #if defined(NNT_NFP_PLATFORM_NX)
            return nn::os::TimedWaitSystemEvent(&g_AvailabilityChangeEvent, nn::TimeSpan::FromMilliSeconds(ms));
        #endif // defined(NNT_NFP_PLATFORM_NX)
    }

    bool WaitForAvailabilityChange() NN_NOEXCEPT
    {
        return WaitForAvailabilityChange(AvailabilityChangeTimeout);
    }

    bool WaitForConnect(int ms) NN_NOEXCEPT
    {
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されない
        const int  period = 10;
        int        sum    = 0;
        nn::Result result;
        nn::nfp::TargetConnectionStatus status;

        while (sum + period < ms)
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&status));
            if (status == nn::nfp::TARGET_CONNECTED)
            {
                return true;
            }
            Sleep(period);
            sum += period;
        }

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&status));
        return status == nn::nfp::TARGET_CONNECTED;
#else
        NN_UNUSED(ms);
        return true;
#endif // !defined(NNT_NFP_PLATFORM_NX)
    }


    bool WaitForConnect() NN_NOEXCEPT
    {
        return WaitForConnect(ConnectTimeout);
    }

    bool WaitForDisconnect(int ms) NN_NOEXCEPT
    {
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されない
        const int  period = 100;
        int        sum    = 0;
        nn::Result result;
        nn::nfp::TargetConnectionStatus status;

        while (sum + period < ms)
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&status));
            if (status == nn::nfp::TARGET_DISCONNECTED)
            {
                return true;
            }
            Sleep(period);
            sum += period;
        }

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&status));
        return status == nn::nfp::TARGET_DISCONNECTED;
#else
        NN_UNUSED(ms);
        return true;
#endif // !defined(NNT_NFP_PLATFORM_NX)
    }

    bool WaitForDisconnect() NN_NOEXCEPT
    {
        return WaitForDisconnect(DisconnectTimeout);
    }

    //==================================================
    // NFP ライブラリの API をラップしてリトライに対応した関数です。
    //==================================================

    nn::Result InitializeWithRetry() NN_NOEXCEPT
    {
        nn::Result result;

#if defined(NNT_NFP_PLATFORM_NX)    // NXはInitializeの二重呼出不可
        if( nnt::nfp::wrapper::GetState() == nn::nfp::State_Init )
        {
            return nn::ResultSuccess();
        }
#endif  // defined(NNT_NFP_PLATFORM_NX)

        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::Initialize();
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result InitializeSystemWithRetry() NN_NOEXCEPT
    {
        nn::Result result;

#if defined(NNT_NFP_PLATFORM_NX)    // NXはInitializeの二重呼出不可
        if( nnt::nfp::wrapper::GetState() == nn::nfp::State_Init )
        {
            return nn::ResultSuccess();
        }
#endif  // defined(NNT_NFP_PLATFORM_NX)
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = InitializeSystem();
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result MountWithRetry() NN_NOEXCEPT
    {
        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::Mount();
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result MountRomWithRetry() NN_NOEXCEPT
    {
        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::MountRom();
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result FlushWithRetry() NN_NOEXCEPT
    {
        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::Flush();
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result FlushDebugWithRetry() NN_NOEXCEPT
    {
        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::FlushDebug();
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result RestoreWithRetry() NN_NOEXCEPT
    {
        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::Restore();
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result FormatWithRetry(const nn::Bit8* pDefaultData, int32_t defaultDataSize) NN_NOEXCEPT
    {

        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
#if !defined(NNT_NFP_PLATFORM_NX) //NXではFormatは引数を持たない
            result = nnt::nfp::wrapper::Format(pDefaultData,defaultDataSize);
#else
            NN_UNUSED(pDefaultData);
            NN_UNUSED(defaultDataSize);
            result = nnt::nfp::wrapper::Format();
#endif // !defined(NNT_NFP_PLATFORM_NX)
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

#if defined(NNT_NFP_PLATFORM_NX) //NX用の引数なしFormat
    nn::Result FormatWithRetry() NN_NOEXCEPT
    {
        return FormatWithRetry(nullptr,0);
    }
#endif // defined(NNT_NFP_PLATFORM_NX)

    nn::Result CreateApplicationAreaWithRetry(const nn::nfp::ApplicationAreaCreateInfo& info) NN_NOEXCEPT
    {
        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::CreateApplicationArea(info);
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result DeleteApplicationAreaWithRetry() NN_NOEXCEPT
    {
        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::DeleteApplicationArea();
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result DeleteNfpRegisterInfoWithRetry() NN_NOEXCEPT
    {
        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::DeleteRegisterInfo();
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    nn::Result BreakTagWithRetry(nn::nfp::BreakType breakType) NN_NOEXCEPT
    {
        nn::Result result;
        for (int i = 0; i < RetryCountMax; ++i)
        {
            result = nnt::nfp::wrapper::BreakTag(breakType);
            if (!(result <= nn::nfp::ResultTimeOutError()))
            {
                break;
            }
            Sleep(RetryInterval);
        }
        return result;
    }

    //==================================================
    // 頻繁に利用されるシーケンスをまとめた関数です。
    //==================================================

    nn::nfp::DeviceHandle GetCurrentDeviceHandle() NN_NOEXCEPT
    {
        return g_CurrentDeviceHandle;
    }

    void SetCurrentDeviceHandle(nn::nfp::DeviceHandle deviceHandle) NN_NOEXCEPT
    {
        g_CurrentDeviceHandle = deviceHandle;
    }

    nn::Result SearchDeviceHandle() NN_NOEXCEPT
    {
        int outCount;
        nn::nfp::State state = nnt::nfp::wrapper::GetState();

        //デバイスリストが取得できない状態でこの関数が実行された場合は、テスト手順が不正
        EXPECT_NE(nn::nfp::State_None,state);

        return nnt::nfp::wrapper::ListDevices(&g_CurrentDeviceHandle, &outCount, 1);
    }

    void DoSearchDeviceHandle() NN_NOEXCEPT
    {
        nn::nfp::State state;
        state = nnt::nfp::wrapper::GetState();
        if (state == nn::nfp::State_None)
        {
            NNT_EXPECT_RESULT_SUCCESS(InitializeSystemWithRetry());
        }

        //NXではConnectの代わりにデバイスハンドルの取得を行う
        NNT_EXPECT_RESULT_SUCCESS(SearchDeviceHandle());

        //イベント取得準備
        NNT_EXPECT_RESULT_SUCCESS(SetDefaultActivateEvent());
        NNT_EXPECT_RESULT_SUCCESS(SetDefaultDeactivateEvent());

        return;
    }

#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されない
    void DoConnect()
    {
        nn::nfp::DeviceState state;
        state = nnt::nfp::wrapper::GetDeviceState();
        if (state == nn::nfp::NONE)
        {
            NNT_EXPECT_RESULT_SUCCESS(InitializeSystemWithRetry());
        }

        state = nnt::nfp::wrapper::GetDeviceState();
        if (state == nn::nfp::INIT)
        {
            NNT_EXPECT_RESULT_SUCCESS(SetDefaultActivateEvent());
            NNT_EXPECT_RESULT_SUCCESS(SetDefaultDeactivateEvent());
            nn::nfp::TargetConnectionStatus targetConnectionStatus;
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetConnectionStatus));
            if( targetConnectionStatus == nn::nfp::TARGET_DISCONNECTED )
            {
                nn::Result connectionResult;
                NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());
                EXPECT_TRUE(WaitForConnect());
                NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectionResult));
                NNT_EXPECT_RESULT_SUCCESS(connectionResult);
            }
        }
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではFWアップデートは実装されない
    void DoConnectForFwUpdate()
    {
        nn::nfp::DeviceState state;
        state = nnt::nfp::wrapper::GetDeviceState();
        if (state == nn::nfp::NONE)
        {
            NNT_EXPECT_RESULT_SUCCESS(InitializeSystemWithRetry());
        }

        state = nnt::nfp::wrapper::GetDeviceState();
        if (state == nn::nfp::NONE)
        {
            NNT_EXPECT_RESULT_SUCCESS(SetDefaultActivateEvent());
            NNT_EXPECT_RESULT_SUCCESS(SetDefaultDeactivateEvent());
            nn::nfp::TargetConnectionStatus targetConnectionStatus;
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetConnectionStatus));
            if( targetConnectionStatus == nn::nfp::TARGET_DISCONNECTED )
            {
                while(NN_STATIC_CONDITION(true))
                {
                    nn::Result connectionResult;

                    // ConnectForFwUpdate()での接続はFangate Firmwareが壊れている必要がある
                    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());
                    if(WaitForConnect())
                    {
                        // CONNEC_FW状態の確認方法はないため
                        // ConnecctForFwUpdate()コール後にnn::nfp::TARGET_CONNECTEDが返されたら遷移OKとする
                        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectionResult));
                        NNT_EXPECT_RESULT_SUCCESS(connectionResult);
                        break;
                    }
                    else
                    {
                        // ConnectForFwUpdateコール後に接続できなかった場合は、ファームウェア正常として
                        // TargetFirmwareを破壊する
                        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::BreakTargetFirmware());
                        continue;
                    }
                }
            }
        }
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

    void DoSearch() NN_NOEXCEPT
    {
#if !defined(NNT_NFP_PLATFORM_NX)
        DoConnect();
#else
        DoSearchDeviceHandle();
#endif // !defined(NNT_NFP_PLATFORM_NX)
        nn::nfp::DeviceState state = nnt::nfp::wrapper::GetDeviceState();

#if !defined(NNT_NFP_PLATFORM_NX)
        if (state == nn::nfp::INIT || state == nn::nfp::RW_DEACTIVE)
#else
        nn::nfp::State libState = nnt::nfp::wrapper::GetState();
        if ((libState == nn::nfp::State_Init)
            && (state == nn::nfp::DeviceState_Init || state == nn::nfp::DeviceState_Deactive))
#endif // !defined(NNT_NFP_PLATFORM_NX)
        {
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されない
            nn::nfp::TargetConnectionStatus targetConnectionStatus;
            nnt::nfp::wrapper::GetTargetConnectionStatus(&targetConnectionStatus);
#endif // !defined(NNT_NFP_PLATFORM_NX)
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection());
        }
    }

    void DoActivate() NN_NOEXCEPT
    {
        DoSearch();
        nn::nfp::DeviceState state = nnt::nfp::wrapper::GetDeviceState();
#if !defined(NNT_NFP_PLATFORM_NX)
        if (state == nn::nfp::RW_SEARCH)
#else
        nn::nfp::State libState = nnt::nfp::wrapper::GetState();
        if ((libState == nn::nfp::State_Init)
            && (state == nn::nfp::DeviceState_Search))
#endif // !defined(NNT_NFP_PLATFORM_NX)
        {
            EXPECT_TRUE(WaitForActivate());
            EXPECT_EQ(nn::nfp::DeviceState_Active,nnt::nfp::wrapper::GetDeviceState());
        }
    }

    void DoMount() NN_NOEXCEPT
    {
        //既にマウントされている場合、Ram/Rom指定が不明なので一旦アンマウントする
        if(nn::nfp::DeviceState_Mount == nnt::nfp::wrapper::GetDeviceState())
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
        }

        DoActivate();
        NNT_EXPECT_RESULT_SUCCESS(MountWithRetry());
        EXPECT_EQ(nn::nfp::DeviceState_Mount,nnt::nfp::wrapper::GetDeviceState());
    }

    void DoMountRom() NN_NOEXCEPT
    {
        //既にマウントされている場合、Ram/Rom指定が不明なので一旦アンマウントする
        if(nn::nfp::DeviceState_Mount == nnt::nfp::wrapper::GetDeviceState())
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
        }

        DoActivate();
        NNT_EXPECT_RESULT_SUCCESS(MountRomWithRetry());
        EXPECT_EQ(nn::nfp::DeviceState_Mount,nnt::nfp::wrapper::GetDeviceState());
    }

    void DoOpenApplicationArea(nn::Bit32 accessId) NN_NOEXCEPT
    {
        DoMount();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(accessId));
    }

    void DoFormat(const nn::Bit8* pDefaultData, int32_t defaultDataSize) NN_NOEXCEPT
    {
        DoActivate();
        NNT_EXPECT_RESULT_SUCCESS(FormatWithRetry(pDefaultData, defaultDataSize));
    }

    void DoCreateNormalTag() NN_NOEXCEPT
    {
        DoFormat(ZeroTagData, sizeof(ZeroTagData));
        NNT_EXPECT_RESULT_SUCCESS(MountWithRetry());

        nn::nfp::RegisterInfoPrivate regInfoSet;
#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されている
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_LIB_VERSION_BETA)
    #if defined(NNT_NFP_LIB_MII_ENABLE)
        nnt::nfp::BuildMiiData(regInfoSet.miiData);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // !defined(NNT_NFP_LIB_VERSION_BETA)
        std::memcpy(&regInfoSet.nickname, NormalTagNickName, sizeof(NormalTagNickName));
        regInfoSet.fontRegion = NormalTagFontRegion;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));

        nn::nfp::ApplicationAreaCreateInfo info;
#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されている
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&info));
#endif // !defined(NNT_NFP_PLATFORM_NX)
        nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2];
        std::memcpy(tagData,NormalTagData,nn::nfp::ApplicationAreaSizeV2);
        info.accessId        = NormalTagId;
        info.pInitialData    = tagData;
        info.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
        NNT_EXPECT_RESULT_SUCCESS(CreateApplicationAreaWithRetry(info));
    }

    void DoCreateZeroTag() NN_NOEXCEPT
    {
        DoFormat(ZeroTagData, sizeof(ZeroTagData));
        NNT_EXPECT_RESULT_SUCCESS(MountWithRetry());

        nn::nfp::ApplicationAreaCreateInfo info;
#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されている
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&info));
#endif //  !defined(NNT_NFP_PLATFORM_NX)
        nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2];
        std::memcpy(tagData,ZeroTagData,nn::nfp::ApplicationAreaSizeV2);
        info.accessId        = ZeroTagId;
        info.pInitialData    = tagData;
        info.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
        NNT_EXPECT_RESULT_SUCCESS(CreateApplicationAreaWithRetry(info));
    }

    void DoMountAndEnd() NN_NOEXCEPT
    {
        DoMount();
        NNT_EXPECT_RESULT_SUCCESS(FinalizeSystem());
    }

    void DoMountRomAndEnd() NN_NOEXCEPT
    {
        DoMountRom();
        NNT_EXPECT_RESULT_SUCCESS(FinalizeSystem());
    }

    void DoFormatAndEnd(const nn::Bit8* pDefaultData, int32_t defaultDataSize) NN_NOEXCEPT
    {
        DoFormat(pDefaultData, defaultDataSize);
        NNT_EXPECT_RESULT_SUCCESS(FinalizeSystem());
    }

    void DoCreateNormalTagAndEnd() NN_NOEXCEPT
    {
        DoCreateNormalTag();
        NNT_EXPECT_RESULT_SUCCESS(FinalizeSystem());
    }

    void DoCreateZeroTagAndEnd() NN_NOEXCEPT
    {
        DoCreateZeroTag();
        NNT_EXPECT_RESULT_SUCCESS(FinalizeSystem());
    }

    bool DoExistsApplicationArea() NN_NOEXCEPT
    {
        bool outValue = false;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ExistsApplicationArea(&outValue));
        return outValue;
    }

    //==================================================
    // その他のユーティリティです。
    //==================================================

    void Sleep(int ms) NN_NOEXCEPT
    {
        #if defined(NNT_NFP_PLATFORM_CTR)
            nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(ms));
        #elif defined(NNT_NFP_PLATFORM_CAFE)
            OSSleepMilliseconds(ms);
        #elif defined(NNT_NFP_PLATFORM_NX)
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ms));
        #endif // defined(NNT_NFP_PLATFORM_NX)
    }

    void* AlignBuffer(void* pBuffer, int alignment) NN_NOEXCEPT
    {
        uintptr_t address = reinterpret_cast<uintptr_t>(pBuffer);
        address = (address + alignment - 1) & ~(alignment - 1);
        return reinterpret_cast<void*>(address);
    }

    nn::nfp::Date GetCurrentDate()
    {
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::nfp::Date date;
        nn::fnd::DateTime dateTime = nn::fnd::DateTime::GetNow();
        date.year  = dateTime.GetYear();
        date.month = dateTime.GetMonth();
        date.day   = dateTime.GetDay();
        return date;
#else // !defined(NNT_NFP_PLATFORM_NX)
        nn::nfp::Date date;

        // 時刻ライブラリの初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::InitializeForSystem() );

        // 協定世界時 (UTC) の 西暦1970年1月1日午前0時0分0秒 からの経過秒数で現在時刻を取得します。
        nn::time::PosixTime posixTime;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );
        // PosixTime を CalendarTime へ変換します。
        // 計算に利用されるタイムゾーンはデバイスに設定されたものを利用します。
        nn::time::CalendarTime calendarTime;
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
        // nfpのDate構造体にコピーする
        std::memcpy(&date.year, &calendarTime.year, sizeof(date.year));
        std::memcpy(&date.month, &calendarTime.month, sizeof(date.month));
        std::memcpy(&date.day, &calendarTime.day, sizeof(date.day));

        // 時刻ライブラリの終了
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );
        return date;
#endif // !defined(NNT_NFP_PLATFORM_NX)
    }

#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    nn::cfg::CfgCountryCode GetCountry()
    {
        nn::cfg::CTR::init::Initialize();
        nn::cfg::CfgCountryCode myCountry = nn::cfg::CTR::GetCountry();
        nn::cfg::CTR::init::Finalize();
        return myCountry;
    }
#endif //!defined(NNT_NFP_PLATFORM_NX)

    void SetNextDate()
    {
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::cfg::CTR::init::Initialize();
        NNT_EXPECT_RESULT_SUCCESS(nn::ptm::InitializeForSystemMenu());
        nn::ptm::CTR::SetUserTime(nn::fnd::DateTime::GetNow() + nn::TimeSpan::FromDays(1));
        nn::ptm::FinalizeForSystemMenu();
        nn::cfg::CTR::init::Finalize();
#else
        nn::time::PosixTime posixTime;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );
        posixTime += nn::TimeSpan::FromDays(1);
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockCurrentTime(posixTime) );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
#endif // !defined(NNT_NFP_PLATFORM_NX)
    }

    void SetPreviousDate()
    {
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::cfg::CTR::init::Initialize();
        NNT_EXPECT_RESULT_SUCCESS(nn::ptm::InitializeForSystemMenu());
        nn::ptm::CTR::SetUserTime(nn::fnd::DateTime::GetNow() - nn::TimeSpan::FromDays(1));
        nn::ptm::FinalizeForSystemMenu();
        nn::cfg::CTR::init::Finalize();
#else
        nn::time::PosixTime posixTime;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );
        posixTime -= nn::TimeSpan::FromDays(1);
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockCurrentTime(posixTime) );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
#endif // !defined(NNT_NFP_PLATFORM_NX)
    }

    bool IsAllZero( const void* pBuffer, int size ) NN_NOEXCEPT
    {
        for ( int i = 0; i < size; ++i )
        {
            if (static_cast<const nn::Bit8*>(pBuffer)[i] != 0)
            {
                return false;
            }
        }
        return true;
    }

    void WifiOn() NN_NOEXCEPT
    {
#if defined(NNT_NFP_PLATFORM_CTR)
        nn::nwm::InitializeExtControl();
        nn::nwm::Ext::SetWifiOn();
        nn::nwm::FinalizeExtControl();
        nn::os::Thread::Sleep( nn::fnd::TimeSpan::FromMilliSeconds(100) );
#elif defined(NNT_NFP_PLATFORM_NX)
        if(nn::nfc::GetState() == nn::nfc::State_None)
        {
            nn::nfc::InitializeSystem();
            nn::nfc::SetNfcEnabled(true);
            nn::nfc::FinalizeSystem();
        }
        else
        {
            nn::nfc::SetNfcEnabled(true);
        }
#endif  // defined(NNT_NFP_PLATFORM_NX)
    }

    void WifiOff() NN_NOEXCEPT
    {
#if defined(NNT_NFP_PLATFORM_CTR)
        nn::nwm::InitializeExtControl();
        nn::nwm::Ext::SetWifiOff();
        nn::nwm::FinalizeExtControl();
        nn::os::Thread::Sleep( nn::fnd::TimeSpan::FromMilliSeconds(100) );
#elif defined(NNT_NFP_PLATFORM_NX)
        if(nn::nfc::GetState() == nn::nfc::State_None)
        {
            nn::nfc::InitializeSystem();
            nn::nfc::SetNfcEnabled(false);
            nn::nfc::FinalizeSystem();
        }
        else
        {
            nn::nfc::SetNfcEnabled(false);
        }
#endif  // defined(NNT_NFP_PLATFORM_NX)
    }

    void CreateRandomNumberSequence(nn::Bit8* pOut, int size) NN_NOEXCEPT
    {
        std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
        for(int i = 0; i < size; i++)
        {
            pOut[i] = std::rand() & 0xFF;
        }
    }

    void CreateRandomNumberSequence(nn::Bit16* pOut, int size) NN_NOEXCEPT
    {
        std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
        for(int i = 0; i < size; i++)
        {
            pOut[i] = std::rand() & 0xFFFF;
        }
    }

    void CreateRandomAllowAsciiCodeSequence(nn::Bit8* pOut, int size) NN_NOEXCEPT
    {
        std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
        for(int i = 0; i < size; i++)
        {
            int index = std::rand() % sizeof(AllowAsciiCode);
            pOut[i] = AllowAsciiCode[index];
        }
    }

    void* MallocAndGetAlignedPtr(void* pOutAllocMem, uint32_t mallocSize, uint32_t alignment) NN_NOEXCEPT
    {
        pOutAllocMem = std::malloc(mallocSize + alignment - 1);
        if (pOutAllocMem == nullptr)
        {
            return nullptr;
        }
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
        uint64_t addrAsuint64 = reinterpret_cast<uint64_t>(pOutAllocMem);
        uint64_t alignedAddr = (addrAsuint64 + alignment - 1) & (~(alignment - 1));
#else
        uint32_t addrAsuint32 = reinterpret_cast<uint32_t>(pOutAllocMem);
        uint32_t alignedAddr = (addrAsuint32 + alignment - 1) & (~(alignment - 1));
#endif // defined(NN_BUILD_CONFIG_ADDRESS_64)
        void *pAligned  =  reinterpret_cast<void *>( alignedAddr );
        return pAligned;
    }

    void UpdateTime() NN_NOEXCEPT
    {
#if defined(NNT_NFP_PLATFORM_CAFE)

        OSCalendarTime now;
        __OSSetAbsoluteSystemTime(OSGetTime() + 0xfffffffffff);
        OSTicksToCalendarTime(OSGetTime(), &now);
        NN_SDK_LOG("Updated Time: %d/%d/%d\n",now.year,now.mon,now.mday);

#elif defined(NNT_NFP_PLATFORM_CTR)

        nn::cfg::CTR::init::Initialize();
        NNT_EXPECT_RESULT_SUCCESS(nn::ptm::InitializeForSystemMenu());
        s_Now += g_Span.FromDays(1);
        nn::ptm::CTR::SetUserTime(s_Now);
        NN_SDK_LOG("UpdatedTime: %04d/%02d/%02d\n",s_Now.GetYear(),s_Now.GetMonth(),s_Now.GetDay());
        nn::ptm::FinalizeForSystemMenu();
        nn::cfg::CTR::init::Finalize();
#endif // defined(NNT_NFP_PLATFORM_CTR)
    }

    void FlushNfp() NN_NOEXCEPT
    {
        nn::Result result;
        for(int i = 0; i < FlushTrialCount; i++)
        {
            result = nnt::nfp::wrapper::Flush();
            if(result.IsSuccess())
            {
                return;
            }
            else if(result <= nn::nfp::ResultNeedRetry())
            {
                // ResultNeedRetry を返した場合はリトライ
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(FlushTrialDuringTime) );
            }
            else
            {
                EXPECT_TRUE(false);
            }
        }
        EXPECT_TRUE(false);
    }

    void MountNfp() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( InitializeSystemWithRetry() );
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&g_ActivateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&g_DeactivateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection());
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::os::WaitEvent(&g_ActivateEvent);
#else
        nn::os::WaitSystemEvent(&g_ActivateEvent);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount());
    }

    void MountRomNfp() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( InitializeSystemWithRetry() );
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&g_ActivateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&g_DeactivateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection());
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::os::WaitEvent(&g_ActivateEvent);
#else
        nn::os::WaitSystemEvent(&g_ActivateEvent);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::MountRom());
    }

    void RegisterNfp() NN_NOEXCEPT
    {
#if defined(NNT_NFP_PLATFORM_CAFE)
        nn::Bit8 miiData[nnt::nfp::TestMiiDataSize] = {};
        CreateRandomNumberSequence(miiData, nnt::nfp::TestMiiDataSize);
        nn::Bit8 nickNameRandNumSeq[nn::nfp::NicknameLengthMax + 1] = {};
        CreateRandomNumberSequence(miiData, nn::nfp::NicknameLengthMax + 1);
        g_RegInfoSet.fontRegion  = 0x01;
        std::memcpy(g_RegInfoSet.miiData.data, miiData, nnt::nfp::TestMiiDataSize);
        std::memcpy(g_RegInfoSet.nickName, nickNameRandNumSeq, nn::nfp::NicknameLengthMax + 1);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(g_RegInfoSet));
        FlushNfp();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&g_RegInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&g_TagInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&g_CommonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&g_AdminInfo));
#elif defined(NNT_NFP_PLATFORM_CTR)
        // HOME メニューの  Mii スタジオで作成した自分の  MiiData を取得します。
        // テスト実行前に Mii データを作成してください。
        NNT_EXPECT_RESULT_SUCCESS(nn::friends::Initialize());
        NNT_EXPECT_RESULT_SUCCESS(nn::friends::GetMyMii(&g_RegInfoSet.miiData));

        g_RegInfoSet.fontRegion  = 0x01;
        std::memcpy(g_RegInfoSet.nickName, L"Nintendo", sizeof(L"Nintendo"));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(g_RegInfoSet));
        FlushNfp();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&g_RegInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&g_TagInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&g_CommonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&g_AdminInfo));
#elif defined(NNT_NFP_PLATFORM_NX)
        #if defined(NNT_NFP_LIB_MII_ENABLE)
        nnt::nfp::BuildMiiData(g_RegInfoSet.miiData);
        #endif // defined(NNT_NFP_LIB_MII_ENABLE)
        g_RegInfoSet.fontRegion  = nn::nfp::FontRegion_China;
        std::memcpy(g_RegInfoSet.nickname, "Nintendo", sizeof("Nintendo"));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(g_RegInfoSet));
        FlushNfp();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&g_RegInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&g_TagInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&g_CommonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&g_AdminInfo));
#endif // defined(NNT_NFP_PLATFORM_NX)
    }

    void RegisterNfpType1() NN_NOEXCEPT
    {
        g_RegInfoSet.fontRegion  = nn::nfp::FontRegion_JpUsEu;
#if defined(NNT_NFP_LIB_VERSION_BETA)
    #if defined(NNT_NFP_LIB_MII_ENABLE)
        nnt::nfp::BuildMiiData(g_RegInfoSet.miiData);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
        std::memcpy(g_RegInfoSet.nickname, NN_TEXT("任天堂NFPテスト"), sizeof(NN_TEXT("任天堂NFPテスト")));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(g_RegInfoSet));
    }

    void RegisterNfpType2() NN_NOEXCEPT
    {
        g_RegInfoSet.fontRegion  = nn::nfp::FontRegion_China;
#if defined(NNT_NFP_LIB_VERSION_BETA)
    #if defined(NNT_NFP_LIB_MII_ENABLE)
        std::memcpy(g_RegInfoSet.miiData._data, MiiData2, sizeof(g_RegInfoSet.miiData._data));
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
        std::memcpy(g_RegInfoSet.nickname, "A", sizeof("A"));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(g_RegInfoSet));
    }

    nn::nfp::TagInfo GetTagInfoNfp() NN_NOEXCEPT
    {
        return g_TagInfo;
    }

    void StartNfp(uint32_t accessId, nn::Bit8* pAppData, uint32_t appSize) NN_NOEXCEPT
    {
        MountNfp();
        if(nnt::nfp::wrapper::OpenApplicationArea(accessId).IsFailure())
        {
            g_AppAreaInfo.accessId        = accessId;
            g_AppAreaInfo.pInitialData    = pAppData;
            g_AppAreaInfo.initialDataSize = appSize;
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::CreateApplicationArea(g_AppAreaInfo));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(accessId));
        }

        RegisterNfp();
    }

    void EndNfp() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
        NNT_EXPECT_RESULT_SUCCESS( FinalizeSystem() );
    }

#if !defined(NNT_NFP_PLATFORM_NX)
    void FormatNfp() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( InitializeSystemWithRetry() );
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&g_ActivateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&g_DeactivateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection());
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::os::WaitEvent(&g_ActivateEvent);
#else
        nn::os::WaitSystemEvent(&g_ActivateEvent);
#endif // !defined(NNT_NFP_PLATFORM_NX)

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Format(ZeroTagData, nn::nfp::ApplicationAreaSizeV2));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
        NNT_EXPECT_RESULT_SUCCESS( FinalizeSystem() );
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_CAFE)
    nn::Result CheckBackupNfp()
    {
        nn::nfp::BackupEntryInfo entryInfo;
        nn::nfp::BackupHeaderInfo headerInfo;

        uint32_t   allocatedAddress;
        void* alignedAddress;
        const uint32_t BackupBufferSize = 2048 * 1024;  //2M Byte
        alignedAddress = MallocAndGetAlignedPtr(&allocatedAddress, BackupBufferSize, FS_IO_BUFFER_ALIGN);

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(alignedAddress, BackupBufferSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&headerInfo, alignedAddress, BackupBufferSize));
        if(headerInfo.entryNum != 0)
        {
            return nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 0, alignedAddress, BackupBufferSize);
        }
        else
        {
            result = nnt::nfp::wrapper::Flush();
            if(result == nn::ResultSuccess())
            {
                return;
            }
            else if(result <= nn::nfp::ResultOperationFailed())
            {
                nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(100));
            }
            else
            {
                NNT_NFP_ASSERT_FATAL(false);
            }
        }
        NNT_NFP_ASSERT_FATAL(false);
    }
#elif defined(NNT_NFP_PLATFORM_CTR)
    bool CheckBackupNfp()
    {
        // バックアップデータを格納するために必要なバッファサイズを取得します。
        // このバッファサイズは BACKUP_BUFFER_ALIGNMENT に調整されています。
        uint32_t requiredSize = nnt::nfp::wrapper::GetBackupSaveDataSize();

        // バックアップバッファを確保します。 BACKUP_BUFFER_ALIGNMENT の境界に合わせます。
        uint32_t alignment = nn::nfp::BACKUP_BUFFER_ALIGNMENT;
        uintptr_t nfpHeap = reinterpret_cast<uintptr_t>(new nn::Bit8[requiredSize + alignment - 1]);
        void* alignedNfpHeap = reinterpret_cast<void*>((nfpHeap + alignment - 1) & ~(alignment - 1));

        // バックアップデータを一括して読み込みます。完了まで 1 秒程度かかります。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(alignedNfpHeap, requiredSize));

        // バックアップデータのヘッダ情報を取得します。
        nn::nfp::BackupHeaderInfo header;
        nn::nfp::BackupEntryInfo entry;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&header, alignedNfpHeap, requiredSize));
        EXPECT_TRUE(header.entryNum != 0);
        EXPECT_TRUE(header.formatVersion == 0);   //formatVersion は 0 固定
        EXPECT_TRUE(header.nextEntryIndex == header.entryNum ||
                (header.nextEntryIndex < header.entryNum == 1000));   //entryNum は 1000 が上限、nextEntryIndex は循環する
        for(int i = 0; i < header.entryNum; i++)
        {
            nnt::nfp::wrapper::GetBackupEntryFromMemory(&entry, 0, alignedNfpHeap, requiredSize);
            if(std::memcmp(&entry.registerInfo, &g_RegInfo, sizeof(nn::nfp::RegisterInfo)) == 0
             && std::memcmp(&entry.commonInfo, &g_CommonInfo, sizeof(nn::nfp::CommonInfo)) == 0
             && std::memcmp(&entry.adminInfo, &g_AdminInfo, sizeof(nn::nfp::AdminInfo)) == 0
             && entry.isEnableRegisterInfo == true
             && entry.isMoveReserved == false
             && std::memcmp(&entry.tagId.uid, &g_TagInfo.tagId.uid, entry.tagId.length) == 0)
            {
                return header.isUpdated;
            }
        }
        EXPECT_TRUE(false);
        return header.isUpdated;
    }
#endif // defined(NNT_NFP_PLATFORM_CTR)

#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    void SetCountry(nn::cfg::CTR::CfgCountryCode country)
    {
        using namespace nn::cfg::CTR;
        using namespace nn::cfg::CTR::detail;

        SimpleAddressIdCfgData simpleAddressId;
        TwlCountryCodeCfgData countryData;

        nn::cfg::CTR::init::Initialize();

        NNT_EXPECT_RESULT_SUCCESS(nn::cfg::CTR::init::GetConfig(&simpleAddressId, sizeof(SimpleAddressIdCfgData), GET_CFG_KEY(NN_CFG_SIMPLE_ADDRESS, NN_CFG_SIMPLE_ADDRESS_ID)));
        NNT_EXPECT_RESULT_SUCCESS(nn::cfg::CTR::init::GetConfig(&countryData, sizeof(TwlCountryCodeCfgData), GET_CFG_KEY(NN_CFG_TWL, NN_CFG_TWL_COUNTRY_CODE)));
        nn::cfg::CTR::init::Finalize();

        simpleAddressId.id = (country << CFG_SIMPLE_ADDRESS_ID_COUNTRY_SHIFT) | (1 << CFG_SIMPLE_ADDRESS_ID_REGION_SHIFT);
        countryData.country=country;

        nn::cfg::CTR::init::Initialize();
        NNT_EXPECT_RESULT_SUCCESS(nn::cfg::CTR::init::SetConfig(GET_CFG_KEY(NN_CFG_SIMPLE_ADDRESS, NN_CFG_SIMPLE_ADDRESS_ID), &simpleAddressId, sizeof(SimpleAddressIdCfgData)));
        NNT_EXPECT_RESULT_SUCCESS(nn::cfg::CTR::init::SetConfig(GET_CFG_KEY(NN_CFG_TWL, NN_CFG_TWL_COUNTRY_CODE), &countryData, sizeof(TwlCountryCodeCfgData)));
        nn::cfg::CTR::init::FlushConfig();
        nn::cfg::CTR::init::Finalize();
    }
#endif //!defined(NNT_NFP_PLATFORM_NX)

    uint16_t GetSystemWriteCounter() NN_NOEXCEPT
    {
        nn::nfp::NfpData nfpData;
        nnt::nfp::wrapper::GetAll(&nfpData);
        return nfpData.systemInfo.systemWriteCounter;
    }

#if !defined(NNT_NFP_PLATFORM_NX) //NX の NFP ライブラリでは BirthPlatform を制御することはない(Mii ライブラリに完全依存）
    nn::mii::BirthPlatform GetBirthPlatform()
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&g_RegInfo));
        nn::mii::StoreDataAccessor storeData(&g_RegInfo.miiData);
        return storeData.GetBirthPlatform();
    }
#endif

    nn::Bit32 Crc32(const void* pData, size_t dataSize) NN_NOEXCEPT
    {
        const nn::Bit8* c = reinterpret_cast<const nn::Bit8*>(pData);
        const nn::Bit32 CrcPoly2 = 0xEDB88320;
        const size_t CharBit = 8;
        nn::Bit32 r;
        size_t i, j;

        r = 0xFFFFFFFF;
        for(i = 0; i < dataSize; i++)
        {
            r ^= c[i];

            for (j = 0; j < CharBit; j++)
            {
                if (r & 1)
                {
                    r = (r >> 1) ^ CrcPoly2;
                }
                else
                {
                    r >>= 1;
                }
            }
        }
        return r ^ 0xFFFFFFFF;
    }

    void CreateDummyBackupData(int count, void* pBackupBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        // システムセーブデータを一旦削除します。
#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#else
        //NX の NFPライブラリにはシステムセーブデータを削除する機能がない別の方法で削除
        nnt::nfp::DeleteSystemSaveData();
#endif //!defined(NNT_NFP_PLATFORM_NX)

        // バッファサイズを確認します。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(count <= nn::nfp::BACKUP_ENTRY_MAX);
        EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
        EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < bufferSize);
#else
        EXPECT_TRUE(count <= nn::nfp::BackupEntryMax);
        EXPECT_TRUE(0 < sizeof(nn::nfp::BackupData));
        EXPECT_TRUE(sizeof(nn::nfp::BackupData) < bufferSize);
#endif //!defined(NNT_NFP_PLATFORM_NX)

        // バックアップデータを読み込みます。
#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(pBackupBuffer, bufferSize));
        nn::nfp::BackupHeaderInfo& header = *reinterpret_cast<nn::nfp::BackupHeaderInfo*>(pBackupBuffer);
#else
        size_t readSize;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(pBackupBuffer, &readSize, bufferSize));
        EXPECT_EQ(sizeof(nn::nfp::BackupData), readSize);
        nn::nfp::BackupDataHeader& header = *reinterpret_cast<nn::nfp::BackupDataHeader*>(pBackupBuffer);
#endif //!defined(NNT_NFP_PLATFORM_NX)

        // 指定された数のバックアップデータが存在するかのように偽造します。
#if !defined(NNT_NFP_PLATFORM_NX)
        header.entryNum       = count < nn::nfp::BACKUP_ENTRY_MAX ? (uint16_t)count : nn::nfp::BACKUP_ENTRY_MAX;
        header.nextEntryIndex = count % nn::nfp::BACKUP_ENTRY_MAX;
        header.hash           = 0U;   //nn::util::Crc32::Calculate(&header, sizeof(nn::nfp::BackupDataHeader) - 4);   //TODO Siglo には CRC は実装されていない
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetAllBackupSaveData(pBackupBuffer, bufferSize));
#else
        header.entryNum       = count < nn::nfp::BackupEntryMax ? (uint16_t)count : nn::nfp::BackupEntryMax;
        header.nextEntryIndex = count % nn::nfp::BackupEntryMax;
        header.hash           = Crc32(&header, sizeof(nn::nfp::BackupDataHeader) - 4);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(pBackupBuffer, sizeof(nn::nfp::BackupData)));
#endif //!defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }

#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    void UpdateCountry()
    {
        uint32_t country;

        #if defined(NNT_NFP_PLATFORM_CAFE)
            SCIGetCafeCountry(&country);
        #elif defined(NNT_NFP_PLATFORM_CTR)
            nn::cfg::CTR::init::Initialize();
            nn::cfg::CfgCountryCode countryCode = nn::cfg::GetCountry();
            country = static_cast<uint32_t>(countryCode);
        #endif    /* TARGET_PLATFORM */
        country++;
        if(country > 1 && country < 8)
        {
            country = 8;
        }
        else if(country > 52 && country < 64)
        {
            country = 64;
        }
        else if(country > 127)
        {
            country = 1;
        }
        NN_SDK_LOG("Update Country=%d\n",country);

        #if defined(NNT_NFP_PLATFORM_CAFE)
            SCISetCafeCountry(country);
        #elif defined(NNT_NFP_PLATFORM_CTR)
            countryCode = static_cast<nn::cfg::CfgCountryCode>(country);
            SetCountry( countryCode );
            nn::cfg::CTR::init::Finalize();
        #endif    /* TARGET_PLATFORM */
    }
#endif //!defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
    void InitSpecificTest() NN_NOEXCEPT
    {
        #if defined(NNT_NFP_PLATFORM_CTR)
            WifiOn();
        #endif //  defined(NNT_NFP_PLATFORM_CTR)
        FormatNfp();
    }
#endif //!defined(NNT_NFP_PLATFORM_NX)

    const char* GetNfpResultTypeString(nn::Result result) NN_NOEXCEPT
    {
        const char* pString = nullptr;
        #define NFP_GET_RESULT_STRING(resultType,result,pOut)                               \
            else if ((nn::nfp::resultType().GetModule() == (result).GetModule()) &&         \
                     (nn::nfp::resultType().GetDescription() == (result).GetDescription())) \
            { \
                pOut = #resultType;\
            }

        if (result.IsSuccess())
        {
            pString = "ResultSuccess";
        }
        //private
        NFP_GET_RESULT_STRING(ResultBadRequest,result,pString)
        NFP_GET_RESULT_STRING(ResultInvalidNickname,result,pString)
        NFP_GET_RESULT_STRING(ResultInvalidMii,result,pString)
        NFP_GET_RESULT_STRING(ResultBackupError,result,pString)
        NFP_GET_RESULT_STRING(ResultBackupReadFailed,result,pString)
        NFP_GET_RESULT_STRING(ResultBackupWriteFailed,result,pString)
        NFP_GET_RESULT_STRING(ResultBackupCrcBroken,result,pString)
        NFP_GET_RESULT_STRING(ResultNfcDeviceError,result,pString)
        NFP_GET_RESULT_STRING(ResultInvalidFormat,result,pString)
        NFP_GET_RESULT_STRING(ResultInvalidTag,result,pString)
        NFP_GET_RESULT_STRING(ResultAuthenticationError,result,pString)
        NFP_GET_RESULT_STRING(ResultInvalidRomArea,result,pString)
        NFP_GET_RESULT_STRING(ResultUidMisMatch,result,pString)
        NFP_GET_RESULT_STRING(ResultInvalidDeviceState,result,pString)
        NFP_GET_RESULT_STRING(ResultTimeOutError,result,pString)
        NFP_GET_RESULT_STRING(ResultOperationFailed,result,pString)
        NFP_GET_RESULT_STRING(ResultTagNotFound,result,pString)

        NFP_GET_RESULT_STRING(ResultSleep,result,pString)
        NFP_GET_RESULT_STRING(ResultNotForeground,result,pString)
        //public
        NFP_GET_RESULT_STRING(ResultNeedRetry,result,pString)
        NFP_GET_RESULT_STRING(ResultNotSupported,result,pString)
        NFP_GET_RESULT_STRING(ResultNeedRestore,result,pString)
        NFP_GET_RESULT_STRING(ResultNeedFormat,result,pString)
        NFP_GET_RESULT_STRING(ResultNeedCreate,result,pString)
        NFP_GET_RESULT_STRING(ResultNeedRegister,result,pString)
        NFP_GET_RESULT_STRING(ResultAlreadyCreated,result,pString)
        NFP_GET_RESULT_STRING(ResultAccessIdMisMatch,result,pString)
        NFP_GET_RESULT_STRING(ResultNotBroken,result,pString)
        NFP_GET_RESULT_STRING(ResultInvalidFormatVersion,result,pString)
        NFP_GET_RESULT_STRING(ResultNfcDisabled,result,pString)
        NFP_GET_RESULT_STRING(ResultNotUpdated,result,pString)
        NFP_GET_RESULT_STRING(ResultNfcDeviceNotFound,result,pString)
        NFP_GET_RESULT_STRING(ResultMaxNfcDeviceActivated,result,pString)
        NFP_GET_RESULT_STRING(ResultConflictFunction,result,pString)
        NFP_GET_RESULT_STRING(ResultNeedRestart,result,pString)
        else
        {
            //不明な場合はエラーコードを表示する
            static char unknownErrorStringBuffer[64];
            std::memset(unknownErrorStringBuffer,0x00,sizeof(unknownErrorStringBuffer));
            std::sprintf(unknownErrorStringBuffer,
                         "Unknown(Module=%d,Description=%d)",result.GetModule(),
                                                             result.GetDescription());
            pString = unknownErrorStringBuffer;
        }
        #undef NFP_GET_RESULT_STRING

        return pString;
    }

    bool CheckNpadStyle(nn::hid::NpadIdType npadId) NN_NOEXCEPT
    {
        bool ret = false;

        // 取得したNpad IDを指定してGetNpadStyleSetを実行する
        nn::hid::NpadStyleSet style;
        style = nn::hid::GetNpadStyleSet(npadId);
        if ((style.Test<nn::hid::NpadStyleFullKey>() == true) |
                (style.Test<nn::hid::NpadStyleHandheld>() == true) |
                (style.Test<nn::hid::NpadStyleJoyDual>() == true) |
                (style.Test<nn::hid::NpadStyleJoyLeft>() == true) |
                (style.Test<nn::hid::NpadStyleJoyRight>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystem>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystemExt>() == true))
        {
            ret = true;
        }
        return ret;
    }

    bool IsNfcEnable() NN_NOEXCEPT
    {
        bool ret;

        if(nn::nfc::GetState() == nn::nfc::State_None)
        {
            nn::nfc::Initialize();
            ret = nn::nfc::IsNfcEnabled();
            nn::nfc::Finalize();
        }
        else
        {
            ret = nn::nfc::IsNfcEnabled();
        }

        return ret;
    }

    nn::Result GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo)
    {
        // 結果格納用
        nn::Result result;
        nn::Result resultPrivate;
        // 非公開RegisterInfo構造体
        nn::nfp::RegisterInfoPrivate regInfoPrivate = {};
#if defined(NNT_NFP_LIB_MII_ENABLE)
        // miiData確認用
        nn::mii::CharInfo charInfoPrivate;
        const int MiiCheckflags =
                nn::mii::CheckFlag_Nickname |
                nn::mii::CheckFlag_Gender |
                nn::mii::CheckFlag_Height |
                nn::mii::CheckFlag_Build |
                nn::mii::CheckFlag_FavoriteColor;
#endif // defined(NNT_NFP_LIB_MII_ENABLE)

        // 通常GetRegisterInfoを実行して結果を格納
        result = nnt::nfp::wrapper::GetRegisterInfo(pOutRegisterInfo);
        // 結果を判定
        if(result.IsSuccess())
        {
            // 成功の場合は非公開版で取得した値が同じになるか確認する
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfoPrivate));
#if defined(NNT_NFP_LIB_MII_ENABLE)
            nn::mii::StoreDataAccessor storeData(&regInfoPrivate.miiData);
            storeData.GetCharInfo(&charInfoPrivate);
            EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                    pOutRegisterInfo->miiData, charInfoPrivate) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
            EXPECT_TRUE(std::memcmp(&pOutRegisterInfo->registerDate, &regInfoPrivate.registerDate, sizeof(regInfoPrivate.registerDate)) == 0);
            EXPECT_TRUE(std::memcmp(pOutRegisterInfo->nickname, regInfoPrivate.nickname, sizeof(regInfoPrivate.nickname)) == 0);
            EXPECT_TRUE(std::memcmp(&pOutRegisterInfo->fontRegion, &regInfoPrivate.fontRegion, sizeof(regInfoPrivate.fontRegion)) == 0);
        }
        else
        {
            // 失敗の場合、失敗戻り値が非公開版と同じかどうか確認する
            resultPrivate = nnt::nfp::wrapper::GetRegisterInfo(&regInfoPrivate);
            EXPECT_TRUE(NNT_NFP_RESULT_EQUAL(resultPrivate, result));
        }
        return result;
    }

    void InitializeHidController() NN_NOEXCEPT
    {
        nn::hid::InitializeNpad();

        //使用する操作形態を設定
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask |
                nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);

        // 使用する Npad を設定
        nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
    }

    // サンプル用の Mii データを作成します。
    void BuildMiiData(nn::mii::StoreData& miiData) NN_NOEXCEPT
    {
#if defined(NNT_NFP_LIB_MII_ENABLE)
        nn::mii::StoreDataContext context;
        nn::mii::StoreDataAccessor accessor(&miiData);
        accessor.BuildDefault(&context, 0);

        // ニックネーム : オーナー
        const char16_t* nickname_utf16 = u"オーナー";
        nn::mii::Nickname nickname;
        nickname.Set(reinterpret_cast<uint16_t*>(const_cast<char16_t*>(nickname_utf16)));
        accessor.SetNickname(nickname, nn::mii::FontRegion_JpUsEu);

        // お気に入りの色 : 黄色
        // 性別 : 男性
        // 身長 : 最も高い
        // 体格 : 最も細い
        accessor.SetFavoriteColor(nn::mii::FavoriteColor_Yellow);
        accessor.SetGender(nn::mii::Gender_Male);
        accessor.SetHeight(nn::mii::HeightMax);
        accessor.SetBuild(nn::mii::BuildMin);

#else // defined(NNT_NFP_LIB_MII_ENABLE)
        NN_UNUSED(miiData);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    }

    // サンプル用の Mii データを作成します。
    void BuildAnotherMiiData(nn::mii::StoreData& miiData) NN_NOEXCEPT
    {
#if defined(NNT_NFP_LIB_MII_ENABLE)
        nn::mii::StoreDataContext context;
        nn::mii::StoreDataAccessor accessor(&miiData);
        accessor.BuildDefault(&context, 0);

        // ニックネーム : オーナー
        const char16_t* nickname_utf16 = u"オーナー";
        nn::mii::Nickname nickname;
        nickname.Set(reinterpret_cast<uint16_t*>(const_cast<char16_t*>(nickname_utf16)));
        accessor.SetNickname(nickname, nn::mii::FontRegion_JpUsEu);

        // お気に入りの色 : 青色
        // 性別 : 女性
        // 身長 : 最も低い
        // 体格 : 最も細い
        accessor.SetFavoriteColor(nn::mii::FavoriteColor_Blue);
        accessor.SetGender(nn::mii::Gender_Female);
        accessor.SetHeight(nn::mii::HeightMin);
        accessor.SetBuild(nn::mii::BuildMin);

#else // defined(NNT_NFP_LIB_MII_ENABLE)
        NN_UNUSED(miiData);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    }

    // サンプル用の Mii データを作成します。
    void BuildMiiData(nn::mii::StoreData& miiData, MiiDataPattern pattern) NN_NOEXCEPT
    {
#if defined(NNT_NFP_LIB_MII_ENABLE)
        nn::mii::StoreDataContext context;
        nn::mii::StoreDataAccessor accessor(&miiData);
        accessor.BuildDefault(&context, 0);

        // ニックネーム : オーナー
        const char16_t* nickname_utf16 = u"オーナー";
        nn::mii::Nickname nickname;
        nickname.Set(reinterpret_cast<uint16_t*>(const_cast<char16_t*>(nickname_utf16)));
        accessor.SetNickname(nickname, nn::mii::FontRegion_JpUsEu);

        switch(pattern)
        {
        case MiiDataPattern_0:
            {
                // お気に入りの色 : 黄色
                // 性別 : 男性
                // 身長 : 最も高い
                // 体格 : 最も細い
                accessor.SetFavoriteColor(nn::mii::FavoriteColor_Yellow);
                accessor.SetGender(nn::mii::Gender_Male);
                accessor.SetHeight(nn::mii::HeightMax);
                accessor.SetBuild(nn::mii::BuildMin);
            }
            break;
        case MiiDataPattern_1:
            {
                // お気に入りの色 : 青色
                // 性別 : 女性
                // 身長 : 最も低い
                // 体格 : 最も細い
                accessor.SetFavoriteColor(nn::mii::FavoriteColor_Blue);
                accessor.SetGender(nn::mii::Gender_Female);
                accessor.SetHeight(nn::mii::HeightMin);
                accessor.SetBuild(nn::mii::BuildMin);
            }
            break;
        case MiiDataPattern_2:
            {
                // お気に入りの色 : 黄色
                // 性別 : 女性
                // 身長 : 最も高い
                // 体格 : 最も細い
                accessor.SetFavoriteColor(nn::mii::FavoriteColor_Yellow);
                accessor.SetGender(nn::mii::Gender_Female);
                accessor.SetHeight(nn::mii::HeightMax);
                accessor.SetBuild(nn::mii::BuildMin);
            }
            break;
        case MiiDataPattern_3:
            {
                // お気に入りの色 : 青色
                // 性別 : 男性
                // 身長 : 最も低い
                // 体格 : 最も細い
                accessor.SetFavoriteColor(nn::mii::FavoriteColor_Blue);
                accessor.SetGender(nn::mii::Gender_Male);
                accessor.SetHeight(nn::mii::HeightMin);
                accessor.SetBuild(nn::mii::BuildMin);
            }
            break;
        default:
            {
                // お気に入りの色 : 黄色
                // 性別 : 男性
                // 身長 : 最も高い
                // 体格 : 最も細い
                accessor.SetFavoriteColor(nn::mii::FavoriteColor_Yellow);
                accessor.SetGender(nn::mii::Gender_Male);
                accessor.SetHeight(nn::mii::HeightMax);
                accessor.SetBuild(nn::mii::BuildMin);
            }
            break;
        }

#else // defined(NNT_NFP_LIB_MII_ENABLE)
        NN_UNUSED(miiData);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    }

    nn::Result InitializeNfcSystem() NN_NOEXCEPT
    {
        EXPECT_EQ(nn::nfc::State_None, nn::nfc::GetState());
        nn::nfc::InitializeSystem();
        EXPECT_EQ(nn::nfc::State_Init, nn::nfc::GetState());
        return nn::ResultSuccess();
    }

    nn::Result FinalizeNfcSystem() NN_NOEXCEPT
    {
        EXPECT_EQ(nn::nfc::State_Init, nn::nfc::GetState());
        nn::nfc::FinalizeSystem();
        EXPECT_EQ(nn::nfc::State_None, nn::nfc::GetState());
        return nn::ResultSuccess();
    }

#if defined(NNT_NFP_PLATFORM_NX)
    void DeleteSystemSaveData() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(nnt::nfp::wrapper::GetState() == nn::nfp::State_None);

        const nn::fs::SystemSaveDataId SystemSaveDataId = 0x8000000000000020;

        // 必ずシステムセーブデータが作成された状態にするため Initialize
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteSystemSaveData(nn::fs::SaveDataSpaceId::System, SystemSaveDataId, nn::fs::InvalidUserId));
    }
#endif
}} // end of namespace nnt::nfp
