﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nfp.h>
#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt/nfp/testNfp_Common.h>


namespace {

#if defined(NNT_NFP_LIB_VERSION_ALPHA) && !defined(NNT_NFP_LIB_VERSION_BETA)
    const uint32_t NfpLibraryInitializeWaitTime = 1500; //ライブラリ初期化後にデバイスが準備できるまで待つ時間(ms)
#endif //  defined(NNT_NFP_LIB_VERSION_ALPHA) && !defined(NNT_NFP_LIB_VERSION_BETA)
    uint32_t g_LoggingMode = 0;

    const char* RemoveNameSpaceString(const char* pApiName) NN_NOEXCEPT
    {
        const char* NameSpaceString = "nnt::nfp::wrapper::";
        if((std::strlen(pApiName) > std::strlen(NameSpaceString)) &&
           (std::memcmp(pApiName,NameSpaceString,std::strlen(NameSpaceString)) == 0))
        {
            // namespace名分の文字列をスキップ
            return pApiName + std::strlen(NameSpaceString);
        }
        return pApiName;
    }


    void PrintApiName(const char* pApiName) NN_NOEXCEPT
    {
        if(g_LoggingMode & nnt::nfp::wrapper::LogMode_Aging)
        {
            NN_LOG("[NFP_API_CALL]nn::nfp::%s() ,",RemoveNameSpaceString(pApiName));
        }
    }

    void PrintApiCallLog(const char* pApiName, nn::Result result, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::nfp::wrapper::LogMode_Aging)
        {
            NN_LOG("Description = %s ,call count = %lld\n",
                   nnt::nfp::GetNfpResultTypeString(result),++count);
        }
    }

    void PrintApiCallLog(const char* pApiName, nn::nfp::State state, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::nfp::wrapper::LogMode_Aging)
        {
            const char* pString;
            switch(state)
            {
            case nn::nfp::State_Init:
                {
                    pString = "State_Init";
                }
                break;
            case nn::nfp::State_None:
                {
                    pString = "State_None";
                }
                break;
            case nn::nfp::State_Unexpected:
                {
                    pString = "State_Unexpected";
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            NN_LOG("State = %s ,call count = %lld\n",
                   pString,++count);
        }
    }

    void PrintApiCallLog(const char* pApiName, nn::nfp::DeviceState deviceState, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::nfp::wrapper::LogMode_Aging)
        {
            const char* pString;
            switch(deviceState)
            {
            case nn::nfp::DeviceState_Init:
                {
                    pString = "DeviceState_Init";
                }
                break;
            case nn::nfp::DeviceState_Search:
                {
                    pString = "DeviceState_Search";
                }
                break;
            case nn::nfp::DeviceState_Active:
                {
                    pString = "DeviceState_Active";
                }
                break;
            case nn::nfp::DeviceState_Deactive:
                {
                    pString = "DeviceState_Deactive";
                }
                break;
            case nn::nfp::DeviceState_Mount:
                {
                    pString = "DeviceState_Mount";
                }
                break;
            case nn::nfp::DeviceState_Unexpected:
                {
                    pString = "DeviceState_Unexpected";
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            NN_LOG("DeviceState = %s ,call count = %lld\n",
                   pString,++count);
        }
    }

    void PrintApiCallLog(const char* pApiName, bool result, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::nfp::wrapper::LogMode_Aging)
        {
            NN_LOG("bool = %s ,call count = %lld\n",
                   result ? "true" : "false",++count);
        }
    }

    void PrintApiCallLog(const char* pApiName, uint64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(pApiName);
        if(g_LoggingMode & nnt::nfp::wrapper::LogMode_Aging)
        {
            NN_LOG("call count = %lld\n",++count);
        }
    }
} // namespace

namespace nnt { namespace nfp { namespace wrapper {
    //==================================================
    // API実行時のログ設定
    //==================================================

    void SetApiCallLoggingMode(const uint32_t mode) NN_NOEXCEPT
    {
        g_LoggingMode = mode;
    }

    //NFPライブラリAPI実行用ラッパー関数
    nn::Result Initialize() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfp::Initialize();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        /* テストプログラムの互換性の為、nn::ResultSuccess()を返す */

#if  defined(NNT_NFP_LIB_VERSION_ALPHA) && !defined(NNT_NFP_LIB_VERSION_BETA)
        nnt::nfp::Sleep(NfpLibraryInitializeWaitTime);
#endif //  defined(NNT_NFP_LIB_VERSION_ALPHA) && !defined(NNT_NFP_LIB_VERSION_BETA)
        return nn::ResultSuccess();
    }

    void Finalize() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfp::Finalize();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
    }

    nn::Result ListDevices(nn::nfp::DeviceHandle* pOutBuffer, int* pOutCount, int bufferCount) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::ListDevices(pOutBuffer, pOutCount, bufferCount);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result StartDetection(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::StartDetection(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result StartDetection() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::StartDetection(GetCurrentDeviceHandle());
    }

    nn::Result StopDetection(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::StopDetection(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result StopDetection() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::StopDetection(GetCurrentDeviceHandle());
    }

    nn::Result Mount(const nn::nfp::DeviceHandle& deviceHandle, nn::nfp::ModelType modelType) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::Mount(deviceHandle, modelType);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result Mount(const nn::nfp::DeviceHandle& deviceHandle, nn::nfp::ModelType modelType, nn::nfp::MountTarget mountTarget) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::Mount(deviceHandle, modelType, mountTarget);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result Mount() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::Mount(GetCurrentDeviceHandle(),nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_All);
    }

    nn::Result MountRom() NN_NOEXCEPT
    {
        //NXではMountRomの代わりにnn::nfp::MountTarget_Rom指定でマウントを実行する
        return nnt::nfp::wrapper::Mount(GetCurrentDeviceHandle(),nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_Rom);
    }

    nn::Result Unmount(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::Unmount(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result Unmount() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::Unmount(GetCurrentDeviceHandle());
    }

    nn::Result OpenApplicationArea(const nn::nfp::DeviceHandle& deviceHandle, nn::Bit32 accessId) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::OpenApplicationArea(deviceHandle, accessId);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result OpenApplicationArea(nn::Bit32 accessId) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::OpenApplicationArea(GetCurrentDeviceHandle(), accessId);
    }

    nn::Result CreateApplicationArea(const nn::nfp::DeviceHandle& deviceHandle, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::CreateApplicationArea(deviceHandle, createInfo);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result CreateApplicationArea(const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::CreateApplicationArea(GetCurrentDeviceHandle(), createInfo);
    }

    nn::Result GetApplicationArea(void* pOutBuffer, const nn::nfp::DeviceHandle& deviceHandle, size_t bufferSize) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        static uint64_t count = 0;
#if !defined(NNT_NFP_PLATFORM_NX) // NXでは読み出しサイズ出力用変数が追加された
        nn::Result result = nn::nfp::GetApplicationArea(pOutBuffer, deviceHandle, bufferSize);
#else
        size_t readSize = 0;
        nn::Result result = nn::nfp::GetApplicationArea(pOutBuffer, &readSize, deviceHandle, bufferSize);
#endif  // !defined(NNT_NFP_PLATFORM_NX)
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetApplicationArea(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
    {
#if !defined(NNT_NFP_PLATFORM_NX) // NXでは読み出しサイズ出力用変数が追加された
        return nnt::nfp::wrapper::GetApplicationArea(pOutBuffer, GetCurrentDeviceHandle(), bufferSize);
#else
        size_t readSize = 0;
        return nnt::nfp::wrapper::GetApplicationArea(pOutBuffer, &readSize, GetCurrentDeviceHandle(), bufferSize);
#endif  // !defined(NNT_NFP_PLATFORM_NX)
    }

    nn::Result GetApplicationArea(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetApplicationArea(pOutBuffer, pOutSize, GetCurrentDeviceHandle(), bufferSize);
    }

    nn::Result GetApplicationArea(void* pOutBuffer, size_t* pOutSize, const nn::nfp::DeviceHandle& deviceHandle, size_t bufferSize) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        static uint64_t count = 0;
        nn::Result result = nn::nfp::GetApplicationArea(pOutBuffer, pOutSize, deviceHandle, bufferSize);
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result SetApplicationArea(const nn::nfp::DeviceHandle& deviceHandle, const void* pData, size_t dataSize) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::SetApplicationArea(deviceHandle, pData, dataSize);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result SetApplicationArea(const void* pData, size_t dataSize) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::SetApplicationArea(GetCurrentDeviceHandle(), pData, dataSize);
    }

    nn::Result SetApplicationArea(const void* pData, size_t dataSize, nn::nfp::TagId tagId) NN_NOEXCEPT
    {
#if defined(NNT_NFP_PLATFORM_NX) //NXでは引数でtagIdを受け取らない
        NN_UNUSED(tagId);
#endif // defined(NNT_NFP_PLATFORM_NX)
        return nnt::nfp::wrapper::SetApplicationArea(GetCurrentDeviceHandle(), pData, dataSize);
    }

    nn::Result RecreateApplicationArea(const nn::nfp::DeviceHandle& deviceHandle, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::RecreateApplicationArea(deviceHandle, createInfo);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result RecreateApplicationArea(const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::RecreateApplicationArea(GetCurrentDeviceHandle(), createInfo);
    }

    nn::Result Flush(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::Flush(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result Flush() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::Flush(GetCurrentDeviceHandle());
    }

    nn::Result Restore(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::Restore(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result Restore() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::Restore(GetCurrentDeviceHandle());
    }

    nn::Result GetTagInfo(nn::nfp::TagInfo* pOutTagInfo, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::GetTagInfo(pOutTagInfo, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetTagInfo(nn::nfp::TagInfo* pOutTagInfo) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetTagInfo(pOutTagInfo, GetCurrentDeviceHandle());
    }

    nn::Result GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::GetRegisterInfo(pOutRegisterInfo, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetRegisterInfo(pOutRegisterInfo, GetCurrentDeviceHandle());
    }

    nn::Result GetRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::GetRegisterInfo(pOutRegisterInfo, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetRegisterInfo(pOutRegisterInfo, GetCurrentDeviceHandle());
    }

    nn::Result GetCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::GetCommonInfo(pOutCommonInfo, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetCommonInfo(pOutCommonInfo, GetCurrentDeviceHandle());
    }

    nn::Result GetModelInfo(nn::nfp::ModelInfo* pOutModelInfo, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::GetModelInfo(pOutModelInfo, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetModelInfo(nn::nfp::ModelInfo* pOutModelInfo) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetModelInfo(pOutModelInfo, GetCurrentDeviceHandle());
    }

    nn::Result AttachActivateEvent(nn::os::SystemEventType* pEvent, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::AttachActivateEvent(pEvent, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result AttachActivateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::AttachActivateEvent(pEvent, GetCurrentDeviceHandle());
    }

    nn::Result AttachDeactivateEvent(nn::os::SystemEventType* pEvent, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::AttachDeactivateEvent(pEvent, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result AttachDeactivateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::AttachDeactivateEvent(pEvent, GetCurrentDeviceHandle());
    }

    nn::Result AttachAvailabilityChangeEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfp::AttachAvailabilityChangeEvent(pEvent);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return nn::ResultSuccess();
    }

    nn::nfp::State GetState() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfp::State state = nn::nfp::GetState();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,state,++count);
        return state;
    }

    nn::nfp::DeviceState GetDeviceState(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfp::DeviceState deviceState = nn::nfp::GetDeviceState(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,deviceState,++count);
        return deviceState;
    }

    nn::nfp::DeviceState GetDeviceState() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetDeviceState(GetCurrentDeviceHandle());
    }

    nn::Result GetAll(nn::nfp::NfpData* pOutNfpData, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::GetAll(pOutNfpData, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetAll(nn::nfp::NfpData* pOutNfpData) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetAll(pOutNfpData, GetCurrentDeviceHandle());
    }

    nn::Result SetAll(const nn::nfp::DeviceHandle& deviceHandle, const nn::nfp::NfpData& nfpData) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::SetAll(deviceHandle, nfpData);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result SetAll(const nn::nfp::NfpData& nfpData) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::SetAll(GetCurrentDeviceHandle(), nfpData);
    }

    nn::Result FlushDebug(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::FlushDebug(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result FlushDebug() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::FlushDebug(GetCurrentDeviceHandle());
    }

    nn::Result BreakTag(const nn::nfp::DeviceHandle& deviceHandle, nn::nfp::BreakType breakType) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::BreakTag(deviceHandle, breakType);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result BreakTag(nn::nfp::BreakType breakType) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::BreakTag(GetCurrentDeviceHandle(), breakType);
    }

    nn::Result InitializeSystem() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfp::InitializeSystem();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        /* テストプログラムの互換性の為、nn::ResultSuccess()を返す */
#if defined(NNT_NFP_LIB_VERSION_ALPHA) && !defined(NNT_NFP_LIB_VERSION_BETA)
        nnt::nfp::Sleep(NfpLibraryInitializeWaitTime);
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA) && !defined(NNT_NFP_LIB_VERSION_BETA)
        return nn::ResultSuccess();
    }

    void FinalizeSystem() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        static uint64_t count = 0;
        nn::nfp::FinalizeSystem();
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return;
    }

    nn::Result InitializeDebug() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfp::InitializeDebug();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        /* テストプログラムの互換性の為、nn::ResultSuccess()を返す */
        return nn::ResultSuccess();
    }

    void FinalizeDebug() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        static uint64_t count = 0;
        nn::nfp::FinalizeDebug();
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,nn::ResultSuccess(),++count);
        return;
    }

    nn::Result Format(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::Format(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result Format() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::Format(GetCurrentDeviceHandle());
    }

    nn::Result Format(const nn::Bit8* pDefaultData, int32_t defaultDataSize) NN_NOEXCEPT
    {
#if defined(NNT_NFP_PLATFORM_NX)
        //NXの場合はFormat時に引数を受け取らない
        NN_UNUSED(pDefaultData);
        NN_UNUSED(defaultDataSize);
#endif // defined(NNT_NFP_PLATFORM_NX)
        return nnt::nfp::wrapper::Format(GetCurrentDeviceHandle());
    }

    nn::Result GetAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::GetAdminInfo(pOutAdminInfo, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetAdminInfo(pOutAdminInfo,GetCurrentDeviceHandle());
    }

    nn::Result SetRegisterInfo(const nn::nfp::DeviceHandle& deviceHandle, const nn::nfp::RegisterInfoPrivate& regInfo) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::SetRegisterInfo(deviceHandle, regInfo);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result SetRegisterInfo(const nn::nfp::RegisterInfoPrivate& regInfo) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::SetRegisterInfo(GetCurrentDeviceHandle(), regInfo);
    }

    nn::Result DeleteRegisterInfo(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::DeleteRegisterInfo(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result DeleteRegisterInfo() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::DeleteRegisterInfo(GetCurrentDeviceHandle());
    }

    nn::Result DeleteApplicationArea(const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::DeleteApplicationArea(deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result DeleteApplicationArea() NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::DeleteApplicationArea(GetCurrentDeviceHandle());
    }

    nn::Result ExistsApplicationArea(bool *pOutValue, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::ExistsApplicationArea(pOutValue, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result ExistsApplicationArea(bool *pOutValue) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::ExistsApplicationArea(pOutValue, GetCurrentDeviceHandle());
    }

#if defined(NNT_NFP_LIB_VERSION_BETA)
    nn::Result StartNicknameAndOwnerSettings(nn::nfp::DeviceHandle* pOutDeviceHandle, bool* pOutIsRegistered, nn::nfp::RegisterInfo* pOutRegisterInfo, const nn::nfp::AmiiboSettingsStartParam& startParam, const nn::nfp::TagInfo& tagInfo, const nn::nfp::RegisterInfo& registerInfo) NN_NOEXCEPT
    {
        return nn::nfp::StartNicknameAndOwnerSettings(pOutDeviceHandle, pOutIsRegistered, pOutRegisterInfo, startParam, tagInfo, registerInfo);
    }

    nn::Result StartNicknameAndOwnerSettings(nn::nfp::DeviceHandle* pOutDeviceHandle, bool* pOutIsRegistered, nn::nfp::RegisterInfo* pOutRegisterInfo, const nn::nfp::AmiiboSettingsStartParam& startParam, const nn::nfp::TagInfo& tagInfo) NN_NOEXCEPT
    {
        return nn::nfp::StartNicknameAndOwnerSettings(pOutDeviceHandle, pOutIsRegistered, pOutRegisterInfo, startParam, tagInfo);
    }

    nn::Result StartGameDataEraser(nn::nfp::DeviceHandle* pOutDeviceHandle, const nn::nfp::AmiiboSettingsStartParam& startParam, const nn::nfp::TagInfo& tagInfo) NN_NOEXCEPT
    {
        return nn::nfp::StartGameDataEraser(pOutDeviceHandle, startParam, tagInfo);
    }

    nn::Result StartRestorer(nn::nfp::DeviceHandle* pOutDeviceHandle, const nn::nfp::AmiiboSettingsStartParam& startParam, const nn::nfp::TagInfo& tagInfo) NN_NOEXCEPT
    {
        return nn::nfp::StartRestorer(pOutDeviceHandle, startParam, tagInfo);
    }

    nn::Result StartNicknameAndOwnerSettings(nn::nfp::TagInfo* pOutTagInfo, nn::nfp::DeviceHandle* pOutDeviceHandle, bool* pOutIsRegistered, nn::nfp::RegisterInfo* pOutRegisterInfo, const nn::nfp::AmiiboSettingsStartParam& startParam) NN_NOEXCEPT
    {
        return nn::nfp::StartNicknameAndOwnerSettings(pOutTagInfo, pOutDeviceHandle, pOutIsRegistered, pOutRegisterInfo, startParam);
    }

    nn::Result StartGameDataEraser(nn::nfp::TagInfo* pOutTagInfo, nn::nfp::DeviceHandle* pOutDeviceHandle, const nn::nfp::AmiiboSettingsStartParam& startParam) NN_NOEXCEPT
    {
        return nn::nfp::StartGameDataEraser(pOutTagInfo, pOutDeviceHandle, startParam);
    }

    nn::Result StartRestorer(nn::nfp::TagInfo* pOutTagInfo, nn::nfp::DeviceHandle* pOutDeviceHandle, const nn::nfp::AmiiboSettingsStartParam& startParam) NN_NOEXCEPT
    {
        return nn::nfp::StartRestorer(pOutTagInfo, pOutDeviceHandle, startParam);
    }

    nn::Result StartFormatter(nn::nfp::TagInfo* pOutTagInfo, nn::nfp::DeviceHandle* pOutDeviceHandle, const nn::nfp::AmiiboSettingsStartParam& startParam) NN_NOEXCEPT
    {
        return nn::nfp::StartFormatter(pOutTagInfo, pOutDeviceHandle, startParam);
    }
#endif

#if !defined(NNT_NFP_PLATFORM_NX)
    nn::Result DeleteSystemSaveData() NN_NOEXCEPT
    {
        return nn::ResultSuccess();
    }

    nn::Result SetAllBackupSaveData(const void* pDataBuf, unsigned int writeSize) NN_NOEXCEPT
    {
        NN_UNUSED(pDataBuf);
        NN_UNUSED(writeSize);
        return nn::ResultSuccess();
    }

    nn::Result InitializeRegisterInfoSet( nn::nfp::RegisterInfoPrivate* pRegInfo ) NN_NOEXCEPT
    {
        NN_UNUSED(pRegInfo);
        return nn::ResultSuccess();
    }

    unsigned int GetBackupSaveDataSize() NN_NOEXCEPT
    {
        return 0;
    }

    nn::Result GetAllBackupSaveData(void* pDataBuf, unsigned int bufSize) NN_NOEXCEPT
    {
        NN_UNUSED(pDataBuf);
        NN_UNUSED(bufSize);
        return nn::ResultSuccess();
    }

    nn::Result GetBackupHeaderFromMemory(nn::nfp::BackupHeaderInfo* pHeaderInfo,
                                             const void* pBackupData,
                                             unsigned int backupDataSize ) NN_NOEXCEPT
    {
        NN_UNUSED(pHeaderInfo);
        NN_UNUSED(pBackupData);
        NN_UNUSED(backupDataSize);
        return nn::ResultSuccess();
    }

    nn::Result GetBackupEntryFromMemory( nn::nfp::BackupEntryInfo* pEntryInfo,
                                         unsigned short entryIndex,
                                         const void* pBackupData,
                                         unsigned int backupDataSize ) NN_NOEXCEPT
    {
        NN_UNUSED(pEntryInfo);
        NN_UNUSED(entryIndex);
        NN_UNUSED(pBackupData);
        NN_UNUSED(backupDataSize);
        return nn::ResultSuccess();
    }
#else
    nn::Result ReadBackupData(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
    {
        return nn::nfp::ReadBackupData(pOutBuffer, pOutSize, bufferSize);
    }
    nn::Result WriteBackupData(const void* pData, size_t dataSize) NN_NOEXCEPT
    {
        return nn::nfp::WriteBackupData(pData, dataSize);
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

    nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId, const nn::nfp::DeviceHandle& deviceHandle) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::Result result = nn::nfp::GetNpadId(pOutNpadId, deviceHandle);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }

    nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId) NN_NOEXCEPT
    {
        return nnt::nfp::wrapper::GetNpadId(pOutNpadId, GetCurrentDeviceHandle());
    }

    void SetErrorVisible(bool visible) NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        nn::nfp::SetErrorVisible(visible);
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,++count);
    }

    bool IsErrorVisible() NN_NOEXCEPT
    {
        PrintApiName(NN_CURRENT_FUNCTION_NAME);
        bool result = nn::nfp::IsErrorVisible();
        static uint64_t count = 0;
        PrintApiCallLog(NN_CURRENT_FUNCTION_NAME,result,++count);
        return result;
    }
}}} //// end of namespace nnt::nfp::wrapper
