﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"

// パスフレーズを購読処理後に行って問題なくダウンロードされるかどうかのテスト
#define PASSPHRASE_DELAY_SET_TEST 0 // NOLINT(preprocessor/const)

// アプリがネットワーク帯域の確保要求を出した時にニュースのダウンロードが中断されるかどうかのテスト
#define NETWORK_CONNECTION_TEST 0 // NOLINT(preprocessor/const)

namespace
{
    nn::ApplicationId g_ApplicationId = {0x0100000000001000};
    const char* g_Passphrase = "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx";

    const char* g_TopicId1 = "nx_news_sp_siglo01";
    const char* g_TopicId2 = "nx_news_sp_siglo02";
    const char* g_TopicIdDummy = "nx_news_sp_siglo_dummy";
}

TEST(Downloader, Initialize)
{
    nn::nifm::Initialize();
}

#if PASSPHRASE_DELAY_SET_TEST == 0

TEST(Downloader, SetPassphrase)
{
    ASSERT_RESULT_SUCCESS(nn::news::SetPassphrase(g_ApplicationId, g_Passphrase));
}

#endif

TEST(Downloader, Subscribe1)
{
    ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicId1, nn::news::SubscriptionStatus_Subscribed));

#if NETWORK_CONNECTION_TEST == 1
    nn::nifm::NetworkConnection connection;
#endif

    NN_LOG("Wait 10 seconds...\n");

    for (int i = 10; i > 0; i--)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

#if NETWORK_CONNECTION_TEST == 1

        // 5 秒後に帯域の確保を宣言する。
        // この時、ニュースのダウンロード処理は中断される。
        // 本関数を抜けると、ダウンロード処理は再開されるはず。
        if (i == 25)
        {
            NN_LOG("NetworkConnection::SubmitRequest...\n");
            nn::nifm::SetRequestRequirementPreset(connection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort);
            connection.SubmitRequest();
            NN_LOG("NetworkConnection::SubmitRequest done!\n");
        }

#endif

        NN_LOG("%d\n", i);
    }

    NN_LOG("Done!\n");

    nn::news::Database db;
    nn::news::NewsRecord record;
    int count;

    ASSERT_RESULT_SUCCESS(db.Open());
    ASSERT_RESULT_SUCCESS(db.GetList(&count, &record, "", "", 0, 1));

    if (count == 1)
    {
        nn::news::Data data;

        ASSERT_RESULT_SUCCESS(data.Open(record));

        int64_t size;
        ASSERT_RESULT_SUCCESS(data.GetSize(&size));

        NN_LOG("DataSize = %lld\n", size);
    }

#if NETWORK_CONNECTION_TEST == 1
    connection.CancelRequest();
#endif
}

TEST(Downloader, Subscribe2)
{
    ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicId2, nn::news::SubscriptionStatus_Subscribed));

    // データを開いている間は、ダウンロード後のコミットが行われない。
    {
        nn::news::Database db;
        nn::news::NewsRecord record;
        int count;

        ASSERT_RESULT_SUCCESS(db.Open());
        ASSERT_RESULT_SUCCESS(db.GetList(&count, &record, "", "", 0, 1));

        if (count == 1)
        {
            NN_LOG("Lock storage 10 seconds...\n");

            nn::news::Data data;
            ASSERT_RESULT_SUCCESS(data.Open(record));

            for (int i = 10; i > 0; i--)
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                NN_LOG("%d\n", i);
            }

            data.Close();

            NN_LOG("Done!\n");
        }
    }

    NN_LOG("Wait 10 seconds...\n");

    for (int i = 10; i > 0; i--)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG("%d\n", i);
    }

    NN_LOG("Done!\n");
}

TEST(Downloader, Subscribe3)
{
    ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicIdDummy, nn::news::SubscriptionStatus_Subscribed));

    NN_LOG("Wait 10 seconds...\n");

    for (int i = 10; i > 0; i--)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG("%d\n", i);
    }

    NN_LOG("Done!\n");
}

TEST(Downloader, RequestImmediateReception1)
{
    ASSERT_RESULT_SUCCESS(nn::news::RequestImmediateReception(g_TopicId1));

    NN_LOG("Wait 10 seconds...\n");

    for (int i = 10; i > 0; i--)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG("%d\n", i);
    }

    NN_LOG("Done!\n");
}

TEST(Downloader, RequestImmediateReception3)
{
    ASSERT_RESULT_SUCCESS(nn::news::RequestImmediateReception(g_TopicIdDummy));

    NN_LOG("Wait 10 seconds...\n");

    for (int i = 10; i > 0; i--)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG("%d\n", i);
    }

    NN_LOG("Done!\n");
}

#if PASSPHRASE_DELAY_SET_TEST == 1

TEST(Downloader, SetPassphrase)
{
    // 後からパスフレーズを設定することで、パスフレーズ未設定エラーを回復。
    ASSERT_RESULT_SUCCESS(nn::news::SetPassphrase(g_ApplicationId, g_Passphrase));

    NN_LOG("Wait 60 seconds...\n");

    for (int i = 60; i > 0; i--)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG("%d\n", i);
    }

    NN_LOG("Done!\n");
}

#endif

TEST(Downloader, GetList)
{
    nn::news::Database db;
    nn::news::NewsRecord records[10];
    int count;

    ASSERT_RESULT_SUCCESS(db.Open());
    ASSERT_RESULT_SUCCESS(db.GetList(&count, records, "", "received_at DESC", 0, 10));

    NN_LOG("--------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------\n");
        NN_LOG("NewsRecord[%d]\n", i);
        NN_LOG("    - NewsId: %s\n", records[i].newsId.value);
        NN_LOG("    - UserId: %s\n", records[i].userId.value);
        NN_LOG("    - ReceivedTime: %lld\n", records[i].receivedTime.value);
        NN_LOG("    - Read: %d\n", records[i].read);
        NN_LOG("    - Newly: %d\n", records[i].newly);
        NN_LOG("    - Displayed: %d\n", records[i].displayed);
    }

    NN_LOG("--------------------------------------------------\n");
}

TEST(Downloader, UnsubscribeWhileDownloading)
{
    ASSERT_RESULT_SUCCESS(nn::news::ClearStorage());
    ASSERT_RESULT_SUCCESS(nn::news::ClearSubscriptionStatusAll());

    ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicId1, nn::news::SubscriptionStatus_Subscribed));

    // データを開いている間は、ダウンロード後のコミットが行われない。
    {
        nn::news::Database db;

        ASSERT_RESULT_SUCCESS(db.Open());

        NN_LOG("Lock storage 10 seconds...\n");

        for (int i = 10; i > 0; i--)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            NN_LOG("%d\n", i);
        }

        NN_LOG("Done!\n");

        // ロックが解除される前に購読を解除する。
        ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicId1, nn::news::SubscriptionStatus_Unsubscribed));
    }

    // コミットされるまで少し待つ。
    NN_LOG("Wait 3 seconds...\n");

    for (int i = 3; i > 0; i--)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG("%d\n", i);
    }

    NN_LOG("Done!\n");

    nn::news::Database db;
    int count;

    ASSERT_RESULT_SUCCESS(db.Open());
    ASSERT_RESULT_SUCCESS(db.Count(&count, ""));

    EXPECT_TRUE(count == 0);
}

TEST(Downloader, Clear)
{
    ASSERT_RESULT_SUCCESS(nn::news::ClearStorage());
    ASSERT_RESULT_SUCCESS(nn::news::ClearSubscriptionStatusAll());
}

TEST(Downloader, Finalize)
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    nn::news::service::StopServer();
#endif
}
