﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"

#include <nn/bcat/service/bcat_TypesService.h>

namespace
{
    const nn::ApplicationId ApplicationId = {0x0100000000001000};
    const char* Passphrase = "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx";
}

namespace
{
    nn::os::SystemEvent g_NewlyArrivedEvent;

    nn::Bit8 g_Encoded[1 * 1024 * 1024] = {};
    size_t g_EncodedSize = 0;

    nn::Bit8 g_Decoded[1 * 1024 * 1024] = {};
    size_t g_DecodedSize = 0;

    nn::Bit8 g_Answer[1 * 1024 * 1024] = {};
    size_t g_AnswerSize = 0;
}

namespace
{
    void PostLocalNews(const char* path) NN_NOEXCEPT
    {
        g_NewlyArrivedEvent.Clear();

        NNT_ASSERT_RESULT_SUCCESS(nnt::news::PostLocalNews(path));

        EXPECT_TRUE(g_NewlyArrivedEvent.TryWait());
    }

    void LoadDecodeTestFiles() NN_NOEXCEPT
    {
        static nn::Bit8 s_MountRomCacheBuffer[4096];

        size_t mountRomCacheUseSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));

        NN_ABORT_UNLESS_LESS_EQUAL(mountRomCacheUseSize, sizeof (s_MountRomCacheBuffer));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("rom", s_MountRomCacheBuffer, sizeof (s_MountRomCacheBuffer)));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("rom");
        };

        {
            nn::fs::FileHandle handle = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, "rom:/list.baf", nn::fs::OpenMode_Read));

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&g_EncodedSize, handle, 0, g_Encoded, sizeof (g_Encoded)));
        }

        {
            nn::fs::FileHandle handle = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, "rom:/list.baf.msgpack", nn::fs::OpenMode_Read));

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&g_AnswerSize, handle, 0, g_Answer, sizeof (g_Answer)));
        }
    }
}

class NewsBasic : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
#if defined (NN_BUILD_CONFIG_OS_WIN)
        nn::time::Initialize();
        nn::nifm::Initialize();
#endif

        NNT_ASSERT_RESULT_SUCCESS(nn::news::RegisterNewlyArrivedEvent(&g_NewlyArrivedEvent));
        NNT_ASSERT_RESULT_SUCCESS(nn::news::ClearStorage());
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
#if defined (NN_BUILD_CONFIG_OS_WIN)
        nn::news::service::StopServer();
        nn::time::Finalize();
        nn::time::Finalize(); // TORIAEZU: sqlite で time::Initialize している分。
#endif
    }
};

TEST_F(NewsBasic, Post)
{
    ASSERT_NO_FATAL_FAILURE(PostLocalNews("rom:/news1.msgpack"));
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1200));

    ASSERT_NO_FATAL_FAILURE(PostLocalNews("rom:/news1_overwrite.msgpack"));
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1200));

    ASSERT_NO_FATAL_FAILURE(PostLocalNews("rom:/news2.msgpack"));
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1200));

    ASSERT_NO_FATAL_FAILURE(PostLocalNews("rom:/news3_no_id.msgpack"));
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1200));

    ASSERT_NO_FATAL_FAILURE(PostLocalNews("rom:/news3_no_id.msgpack"));

    NNT_ASSERT_RESULT_FAILURE(nn::news::ResultInvalidFormat, nn::news::PostLocalNews("dummy", 5));
}

TEST_F(NewsBasic, Update)
{
    nn::news::Database db;
    nn::news::NewsRecord record;
    int count;

    NNT_ASSERT_RESULT_SUCCESS(db.Open());
    NNT_ASSERT_RESULT_SUCCESS(db.GetList(&count, &record, "", "", 0, 1));

    EXPECT_EQ(count, 1);

    char wherePhrase[64] = {};
    nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase), "news_id = '%s'", record.newsId.value);

    NNT_ASSERT_RESULT_SUCCESS(db.UpdateWithAddition("read", 1, wherePhrase));
    NNT_ASSERT_RESULT_SUCCESS(db.UpdateWithAddition("read", -1, wherePhrase));

    NNT_ASSERT_RESULT_SUCCESS(db.Update("newly", 0, wherePhrase));

    NNT_ASSERT_RESULT_SUCCESS(db.Update("extra_1", 100, wherePhrase));
    NNT_ASSERT_RESULT_SUCCESS(db.Update("extra_2", 200, wherePhrase));
}

TEST_F(NewsBasic, GetList)
{
    nn::news::Database db;
    nn::news::NewsRecord records[10];
    int count;

    NNT_ASSERT_RESULT_SUCCESS(db.Open());
    NNT_ASSERT_RESULT_SUCCESS(db.GetList(&count, records, "", "", 0, NN_ARRAY_SIZE(records)));

    ASSERT_EQ(count, 4);

    // ひとつ前のテストで先頭のレコードを書き換えている。
    EXPECT_EQ(records[0].read, 0);
    EXPECT_EQ(records[0].newly, 0);
    EXPECT_EQ(records[0].extra1, 100);
    EXPECT_EQ(records[0].extra2, 200);

    NN_LOG("--------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------\n");
        NN_LOG("NewsRecord[%d]\n", i);
        NN_LOG("    - NewsId: %s\n", records[i].newsId.value);
        NN_LOG("    - UserId: %s\n", records[i].userId.value);
        NN_LOG("    - TopicId: %s\n", records[i].topicId.value);
        NN_LOG("    - ReceivedTime: %lld\n", records[i].receivedTime.value);
        NN_LOG("    - Read: %d\n", records[i].read);
        NN_LOG("    - Newly: %d\n", records[i].newly);
        NN_LOG("    - Displayed: %d\n", records[i].displayed);
        NN_LOG("    - Extra1: %d\n", records[i].extra1);
        NN_LOG("    - Extra2: %d\n", records[i].extra2);

        // テストデータのトピックはすべて nx_news になっている。
        EXPECT_STREQ(records[i].topicId.value, "nx_news");
    }

    NN_LOG("--------------------------------------------------\n");
}

TEST_F(NewsBasic, Count)
{
    nn::news::Database db;
    int count;

    NNT_ASSERT_RESULT_SUCCESS(db.Open());

    NNT_ASSERT_RESULT_SUCCESS(db.Count(&count, "N_PMATCH(news_id, 'LS')"));
    EXPECT_EQ(count, 2);

    NNT_ASSERT_RESULT_SUCCESS(db.Count(&count, "N_PMATCH(news_id, 'NS')"));
    EXPECT_EQ(count, 0);

    NNT_ASSERT_RESULT_SUCCESS(db.Count(&count, "topic_id='nx_news'"));
    EXPECT_EQ(count, 4);

    NNT_ASSERT_RESULT_SUCCESS(db.Count(&count, true, "read", ""));
    EXPECT_EQ(count, 1);
}

TEST_F(NewsBasic, ReadData)
{
    nn::news::Database db;
    nn::news::NewsRecord record;
    int count = 0;

    NNT_ASSERT_RESULT_SUCCESS(db.Open());
    NNT_ASSERT_RESULT_SUCCESS(db.GetList(&count, &record, "", "", 0, 1));

    EXPECT_EQ(count, 1);

    nn::news::Data data;
    NNT_ASSERT_RESULT_SUCCESS(data.Open(record));

    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(data.GetSize(&size));

    NN_LOG("DataSize = %lld\n", size);

    nn::Bit8 end[16];
    size_t readSize;

    NNT_ASSERT_RESULT_SUCCESS(data.Read(&readSize, size - sizeof (end), end, sizeof (end)));

    EXPECT_EQ(readSize, 16u);
}

TEST_F(NewsBasic, TopicList)
{
    static nn::news::TopicId s_TopicIds[nn::news::TopicCountMax] = {};
    int count = 0;

    NNT_ASSERT_RESULT_SUCCESS(nn::news::ClearSubscriptionStatusAll());

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_None));
    EXPECT_EQ(count, 0);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Subscribed));
    EXPECT_EQ(count, 0);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Unsubscribed));
    EXPECT_EQ(count, 0);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus("nx_test_01", nn::news::SubscriptionStatus_Subscribed));
    NNT_ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus("nx_test_02", nn::news::SubscriptionStatus_AutoSubscribed));

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, 1,
        nn::news::SubscriptionStatusFilter_None));
    EXPECT_EQ(count, 1);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_None));
    EXPECT_EQ(count, 2);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Subscribed));
    EXPECT_EQ(count, 2);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Unsubscribed));
    EXPECT_EQ(count, 0);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus("nx_test_02", nn::news::SubscriptionStatus_Subscribed));

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_None));
    EXPECT_EQ(count, 2);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Subscribed));
    EXPECT_EQ(count, 2);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Unsubscribed));
    EXPECT_EQ(count, 0);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus("nx_test_03", nn::news::SubscriptionStatus_Unsubscribed));

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_None));
    EXPECT_EQ(count, 3);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Subscribed));
    EXPECT_EQ(count, 2);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Unsubscribed));
    EXPECT_EQ(count, 1);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus("nx_test_03", nn::news::SubscriptionStatus_Subscribed));

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_None));
    EXPECT_EQ(count, 3);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Subscribed));
    EXPECT_EQ(count, 3);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
        nn::news::SubscriptionStatusFilter_Unsubscribed));
    EXPECT_EQ(count, 0);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::ClearSubscriptionStatusAll());
}

TEST_F(NewsBasic, ClearStorage)
{
    ASSERT_RESULT_SUCCESS(nn::news::ClearStorage());
}

TEST_F(NewsBasic, ArchiveFile1)
{
    LoadDecodeTestFiles();

    NNT_ASSERT_RESULT_SUCCESS(nn::news::SetPassphrase(ApplicationId, Passphrase));

    NNT_ASSERT_RESULT_SUCCESS(nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), g_Encoded, g_EncodedSize));

    EXPECT_EQ(g_DecodedSize, g_AnswerSize);
    EXPECT_TRUE(std::memcmp(g_Decoded, g_Answer, g_AnswerSize) == 0);

    NNT_ASSERT_RESULT_SUCCESS(nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, g_AnswerSize, g_Encoded, g_EncodedSize));

    EXPECT_EQ(g_DecodedSize, g_AnswerSize);
    EXPECT_TRUE(std::memcmp(g_Decoded, g_Answer, g_AnswerSize) == 0);

    NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
        nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), g_Encoded, 1));

    NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
        nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), g_Encoded, sizeof (nn::bcat::service::ArchiveHeader) - 1));

    NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
        nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), g_Encoded, sizeof (nn::bcat::service::ArchiveHeader)));

    NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
        nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), g_Encoded, g_EncodedSize - 1));

    NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
        nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), g_Encoded, g_EncodedSize + 1));

    NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
        nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, g_AnswerSize - 1, g_Encoded, g_EncodedSize));
}

TEST_F(NewsBasic, ArchiveFile2)
{
    static nn::Bit8 s_BrokenFile[1 * 1024 * 1024];

    nn::bcat::service::ArchiveHeader header;

    // ヘッダ書き換え（マジック）
    {
        std::memcpy(&header, g_Encoded, sizeof (header));
        header.magic[0] = 'a';

        std::memcpy(s_BrokenFile, g_Encoded, g_EncodedSize);
        std::memcpy(s_BrokenFile, &header, sizeof (header));

        NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
            nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), s_BrokenFile, g_EncodedSize));
    }

    // ヘッダ書き換え（暗号種別）
    {
        std::memcpy(&header, g_Encoded, sizeof (header));

        switch (header.encryptionType)
        {
        case nn::bcat::service::EncryptionType_NoEncrypted:
            header.encryptionType = nn::bcat::service::EncryptionType_AesCtr128;
            break;
        case nn::bcat::service::EncryptionType_AesCtr128:
            header.encryptionType = nn::bcat::service::EncryptionType_AesCtr192;
            break;
        case nn::bcat::service::EncryptionType_AesCtr192:
            header.encryptionType = nn::bcat::service::EncryptionType_AesCtr256;
            break;
        case nn::bcat::service::EncryptionType_AesCtr256:
            header.encryptionType = nn::bcat::service::EncryptionType_NoEncrypted;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        std::memcpy(s_BrokenFile, g_Encoded, g_EncodedSize);
        std::memcpy(s_BrokenFile, &header, sizeof (header));

        NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
            nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), s_BrokenFile, g_EncodedSize));
    }

    // ヘッダ書き換え（署名種別）
    {
        std::memcpy(&header, g_Encoded, sizeof (header));

        switch (header.signatureType)
        {
        case nn::bcat::service::SignatureType_RsaPkcs1Sha1:
            header.signatureType = nn::bcat::service::SignatureType_RsaPkcs1Sha256;
            break;
        case nn::bcat::service::SignatureType_RsaPkcs1Sha256:
            header.signatureType = nn::bcat::service::SignatureType_RsaPssSha1;
            break;
        case nn::bcat::service::SignatureType_RsaPssSha1:
            header.signatureType = nn::bcat::service::SignatureType_RsaPssSha256;
            break;
        case nn::bcat::service::SignatureType_RsaPssSha256:
            header.signatureType = nn::bcat::service::SignatureType_RsaPkcs1Sha1;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        std::memcpy(s_BrokenFile, g_Encoded, g_EncodedSize);
        std::memcpy(s_BrokenFile, &header, sizeof (header));

        NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
            nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), s_BrokenFile, g_EncodedSize));
    }

    // ヘッダ書き換え（署名）
    {
        std::memcpy(&header, g_Encoded, sizeof (header));

        header.signature[0]++;

        std::memcpy(s_BrokenFile, g_Encoded, g_EncodedSize);
        std::memcpy(s_BrokenFile, &header, sizeof (header));

        NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
            nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), s_BrokenFile, g_EncodedSize));
    }

    // 実データ部書き換え
    {
        std::memcpy(s_BrokenFile, g_Encoded, g_EncodedSize);
        s_BrokenFile[g_EncodedSize - 1]++;

        NNT_ASSERT_RESULT_FAILURE(nn::news::ResultDataVerificationFailed,
            nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), s_BrokenFile, g_EncodedSize));
    }
}

TEST_F(NewsBasic, ArchiveFile3)
{
    static nn::Bit8 s_UnsupportedFile[1 * 1024 * 1024];

    nn::bcat::service::ArchiveHeader header;

    // ヘッダ書き換え（バージョン）
    {
        std::memcpy(&header, g_Encoded, sizeof (header));

        header.version = 255;

        std::memcpy(s_UnsupportedFile, g_Encoded, g_EncodedSize);
        std::memcpy(s_UnsupportedFile, &header, sizeof (header));

        NNT_ASSERT_RESULT_FAILURE(nn::news::ResultUnsupportedFormatDetected,
            nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), s_UnsupportedFile, g_EncodedSize));
    }

    // ヘッダ書き換え（暗号種別）
    {
        std::memcpy(&header, g_Encoded, sizeof (header));

        header.encryptionType = static_cast<nn::bcat::service::EncryptionType>
            (static_cast<int>(nn::bcat::service::EncryptionType::EncryptionType_AesCtr256) + 1);

        std::memcpy(s_UnsupportedFile, g_Encoded, g_EncodedSize);
        std::memcpy(s_UnsupportedFile, &header, sizeof (header));

        NNT_ASSERT_RESULT_FAILURE(nn::news::ResultUnsupportedFormatDetected,
            nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), s_UnsupportedFile, g_EncodedSize));
    }

    // ヘッダ書き換え（署名種別）
    {
        std::memcpy(&header, g_Encoded, sizeof (header));

        header.signatureType = static_cast<nn::bcat::service::SignatureType>
            (static_cast<int>(nn::bcat::service::SignatureType::SignatureType_RsaPssSha256) + 1);

        std::memcpy(s_UnsupportedFile, g_Encoded, g_EncodedSize);
        std::memcpy(s_UnsupportedFile, &header, sizeof (header));

        NNT_ASSERT_RESULT_FAILURE(nn::news::ResultUnsupportedFormatDetected,
            nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), s_UnsupportedFile, g_EncodedSize));
    }

    // ヘッダ書き換え（ソルトインデックス）
    {
        std::memcpy(&header, g_Encoded, sizeof (header));

        header.saltIndex = 255;

        std::memcpy(s_UnsupportedFile, g_Encoded, g_EncodedSize);
        std::memcpy(s_UnsupportedFile, &header, sizeof (header));

        NNT_ASSERT_RESULT_FAILURE(nn::news::ResultUnsupportedFormatDetected,
            nn::news::DecodeArchiveFile(&g_DecodedSize, g_Decoded, sizeof (g_Decoded), s_UnsupportedFile, g_EncodedSize));
    }
}
