﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NetTest/Horizon/NetTest_Types.h"
#include "NetTest/Horizon/NetTest_OS.h"
#include "Utils/CommandLineParser.h"

#include <nnt/nntest.h>

#include <cstring>
#include <cstdio>
#include <cstdlib> // For malloc

#include <windows.h>

namespace NATF {
namespace Utils {
int ReadLine(FILE* pFile, char* pBuffer, int bufferLen, bool dropWide) NN_NOEXCEPT;
}}

namespace
{
    const uint32_t MaxPath = 2 * 1024;
    const uint32_t MaxCmdBufLen = 64 + MaxPath;
    const uint32_t MaxCmdTypeButLen = 32;
    const char* const AllPidsFile = "AllPids.txt";
    const char* const TmpFile = "c_tmp.txt";
    char g_pProcessPath[MaxPath];
}

TEST(natf, SystemCmd)
{
    bool defaultIngnoreFailure = false;
    bool ignoreFailure = defaultIngnoreFailure;
    char pCmdBuffer[MaxCmdBufLen];
    char pPidsFile[MaxPath];
    int argC = 0;
    const char * const * pArgV = nullptr;
    NATF::Utils::ParserGroup parser;
    char pCommandType[MaxCmdTypeButLen];

    memset(pCmdBuffer, 0, MaxCmdBufLen);
    memset(pPidsFile, 0, MaxPath);

    NETTEST_GET_ARGS(argC, pArgV);

    parser.AddParser(NATF::Utils::StringParser("--Command", nullptr, pCmdBuffer, sizeof(pCmdBuffer)));
    parser.AddParser(NATF::Utils::StringParser("--CommandType", nullptr, pCommandType, sizeof(pCommandType)));
    parser.AddParser(NATF::Utils::StringParser("--PidsFile", "", pPidsFile, sizeof(pPidsFile)));
    parser.AddParser(NATF::Utils::BoolParser  ("--IgnoreFailure", &defaultIngnoreFailure, ignoreFailure));

    if (!parser.Parse(argC, pArgV))
    {
        NN_NETTEST_LOG(" * Failed to parse command line arguements!\n\n");
        FAIL();
        return;
    }

    NN_LOG("Command buffer: %s\n", pCmdBuffer);

    NetTest::StrUpr(pCommandType);
    if( strcmp(pCommandType, "CONSOLE") == 0 )
    {
        NN_LOG("Running console command...\n");
        int rval = system(pCmdBuffer);
        NN_LOG("Return code: %d\n", rval);

        if(!ignoreFailure)
        {
            EXPECT_EQ(rval, 0);
        }
    }
    else if( strcmp(pCommandType, "PROCESS") == 0 )
    {
        STARTUPINFOA si;
        PROCESS_INFORMATION pi;

        ZeroMemory( &pi, sizeof(pi) );
        ZeroMemory( &si, sizeof(si) );
        si.cb = sizeof(si);
        char windowTitle[] = "NATF";
        si.lpTitle = windowTitle;

        do
        {

            NN_LOG("Creating new process...\n");
            // Start the child process.
            if( !CreateProcessA( nullptr,    // No module name (use command line)
                                pCmdBuffer, // Command line
                                nullptr,    // Process handle not inheritable
                                nullptr,    // Thread handle not inheritable
                                FALSE,      // Set handle inheritance to FALSE
                                DETACHED_PROCESS,          // No creation flags
                                nullptr,    // Use parent's environment block
                                nullptr,    // Use parent's starting directory
                                &si,        // Pointer to STARTUPINFO structure
                                &pi ) )     // Pointer to PROCESS_INFORMATION structure
            {
                NN_LOG( "CreateProcess failed (%d).\n\n", GetLastError() );
                FAIL();
                break;
            }

            // If pid file name is passed, write pid to file.
            if(pPidsFile[0] != '\0')
            {
                char pBuffer[80];
                snprintf(pBuffer, sizeof(pBuffer), "wmic process where processId=%u get ExecutablePath > %s", pi.dwProcessId, TmpFile);

                int rval = system(pBuffer);
                if(rval != 0)
                {
                    NN_LOG(" COMMAND FAILED: %s. Err: %d\n\n", pBuffer, rval);
                    FAIL();
                    break;
                }

                FILE* pProcessPathFile = fopen(TmpFile, "rt");
                if(nullptr == pProcessPathFile)
                {
                    NN_LOG(" Failed to open file for read: %s. Err: %d\n\n", TmpFile, ferror(pProcessPathFile));
                    FAIL();
                    break;
                }

                // Read and ignore first line.
                int bytesRead = NATF::Utils::ReadLine(pProcessPathFile, g_pProcessPath, sizeof(g_pProcessPath), true);
                if(bytesRead <= 0)
                {
                    fclose(pProcessPathFile);
                    NN_LOG(" Failed to read 1st line: %s. ret: %d\n\n", TmpFile, bytesRead);
                    FAIL();
                    break;
                }

                // Read and ignore second line.
                bytesRead = NATF::Utils::ReadLine(pProcessPathFile, g_pProcessPath, sizeof(g_pProcessPath), true);
                if(bytesRead <= 0)
                {
                    fclose(pProcessPathFile);
                    NN_LOG(" Failed to read 2nd line: %s ret: %d.\n\n", TmpFile, bytesRead);
                    FAIL();
                    break;
                }

                // Read third line and keep it.
                bytesRead = NATF::Utils::ReadLine(pProcessPathFile, g_pProcessPath, sizeof(g_pProcessPath), true);
                if(bytesRead <= 0)
                {
                    fclose(pProcessPathFile);
                    NN_LOG(" Failed to read 2nd line: %s ret: %d.\n\n", TmpFile, bytesRead);
                    FAIL();
                    break;
                }

                NN_LOG(" Process Path: %*s\n", bytesRead, g_pProcessPath);

                fclose(pProcessPathFile);

                FILE* pFile = fopen(pPidsFile, "wt");
                if( !pFile )
                {
                    NN_LOG( "Failed to open file: %s. Err: %d\n\n", pPidsFile, ferror(pFile));
                    FAIL();
                    break;
                }

                fprintf(pFile, "%d:%s\n", pi.dwProcessId, g_pProcessPath);
                fclose(pFile);
            }

            FILE* pFile = fopen(AllPidsFile, "at");
            if( !pFile )
            {
                NN_LOG( "Failed to open file: %s. Err: %d\n\n", AllPidsFile, ferror(pFile));
                FAIL();
                break;
            }

            fprintf(pFile, "%d:%s\n", pi.dwProcessId, g_pProcessPath);
            fclose(pFile);
        } while(NN_STATIC_CONDITION(false));

        sprintf(pCmdBuffer, "del %s", TmpFile);
        system(pCmdBuffer);
    }
    else
    {
        NN_LOG("\n * Error: Invalid --CommandType! Valid values: Console, Process\n\n");
        FAIL();
    }

    NN_LOG("Exiting...\n");
} // NOLINT(impl/function_size)
