﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "CommandParsers.h"
#include "GenTestlist_Win.h"
#include "nn/nn_Log.h"

using namespace ParserHelpers;

namespace ParserHelpers
{
    size_t LineLen(const char* pStr) NN_NOEXCEPT
    {
        const char* pPos = pStr;

        if(nullptr == pStr)
        {
            return 0;
        }

        while(*pPos != '\n' && *pPos != '\0')
        {
            ++pPos;
        }

        // Include the end of line char
        if(*pPos == '\n')
        {
            ++pPos;
        }

        return pPos - pStr;
    }

    bool ContainsNonWhiteSpace(const char* pLine, size_t lineLen) NN_NOEXCEPT
    {
        const char* pCurChar = pLine;

        while(static_cast<size_t>(pCurChar - pLine) < lineLen && *pCurChar != '\n' && *pCurChar != '\0')
        {
            if(*pCurChar != ' ' && *pCurChar != '\t')
            {
                return true;
            }

            ++pCurChar;
        }

        return false;
    }
}

bool ParseInfo::CheckForCommand(const char* pCommand, size_t commandLen) NN_NOEXCEPT
{
    if(lineLen >= commandLen && memcmp(pLine, pCommand, commandLen) == 0)
    {
        PRINT_LINE_INFO(*this, "Found: %s\n", pCommand);
        return true;
    }

    return false;
}

char* BufferLineReader::ReadLine(char* pOutBuffer, uint32_t bufLen) NN_NOEXCEPT
{
    uint32_t maxReadLen = (bufLen - 1) < (m_bufLen - m_readPos) ? (bufLen - 1) : (m_bufLen - m_readPos); // -1 for null terminator
    uint32_t bytesRead = 0;

    while(m_pReadBuffer[m_readPos] != '\n' &&
            m_pReadBuffer[m_readPos] != '\0' &&
            bytesRead < maxReadLen)
    {
        pOutBuffer[bytesRead] = m_pReadBuffer[m_readPos];
        ++m_readPos;
        ++bytesRead;
    }

    // Read the end of line char
    if(m_pReadBuffer[m_readPos] == '\n')
    {
        pOutBuffer[bytesRead] = m_pReadBuffer[m_readPos];
        ++m_readPos;
        ++bytesRead;
    }

    // Null terminate the line
    pOutBuffer[bytesRead] = '\0';

    if(bytesRead > 0)
    {
        return pOutBuffer;
    }
    else
    {
        return nullptr;
    }
}

int32_t TestlistGenerator::IgnorePlatformDefinition(ParseInfo& parseInfo) NN_NOEXCEPT
{
    uint32_t startLine = parseInfo.lineNumber;

    // Don't write platform definitions to out file.
    if(parseInfo.CheckForCommand(CommandBeginDefinePlatform, CommandBeginDefinePlatformLen))
    {
        std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
        if(nullptr == pReadBuf)
        {
            NN_LOG("Allocation error!\n\n");
            return -1;
        }

        // Skip to the end of the definition
        while(false == parseInfo.lineReader.IsEndOfStream())
        {
            char* pRet = parseInfo.lineReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
            if( pRet )
            {
                ++parseInfo.lineNumber;
                size_t lineLen = LineLen(pRet);

                // Check for end command
                if(lineLen >= CommandEndDefinePlatformLen && memcmp(pRet, CommandEndDefinePlatform, CommandEndDefinePlatformLen) == 0)
                {
                    return static_cast<int32_t>(CommandEndDefinePlatformLen);
                }
            }
        }

        NN_LOG("Error: Found end of file before '%s'. Looking to close %s at line %d\n", CommandEndDefinePlatform, CommandBeginDefinePlatform, startLine);
        return -1;
    }

    return 0;
}

int32_t TestlistGenerator::CheckForDisablePlatform(ParseInfo& parseInfo) NN_NOEXCEPT
{
    if(parseInfo.CheckForCommand(CommandDisablePlatform, CommandDisablePlatformLen))
    {
        // Check if line is long enough to have a platform name
        if(parseInfo.lineLen < CommandDisablePlatformLen + 2) // +2 is for a space and the first letter of the platform name
        {
            PRINT_LINE_INFO(parseInfo, "ERROR: A platform name must be specified after %s\n\n", CommandDisablePlatform);
            return -1;
        }

        const char* pPlatformName = parseInfo.pLine + CommandDisablePlatformLen + 1; // +1 for space
        size_t platformNameLen = LineLen(pPlatformName);
        if(pPlatformName[platformNameLen - 1] == '\n')
        {
            --platformNameLen;
        }

        // Find platform name and remove it from array.
        for(uint32_t iPlatform = 0; iPlatform < m_platformCount; ++iPlatform)
        {
            size_t curPlatformNameLen = strlen(m_pPlatforms[iPlatform].pName);
            NN_LOG("platform: %s. Len1: %d, Len2: %d\n", m_pPlatforms[iPlatform].pName, platformNameLen, curPlatformNameLen);
            if(platformNameLen == curPlatformNameLen && memcmp(m_pPlatforms[iPlatform].pName, pPlatformName, platformNameLen) == 0)
            {
                m_pPlatforms[iPlatform].isEnabled = false;
                break;
            }

            if(iPlatform + 1 == m_platformCount)
            {
                PRINT_LINE_INFO(parseInfo, "ERROR: Could not find platform name: %.*s\n\n", platformNameLen, pPlatformName);
                return -1;
            }
        }

        return static_cast<int32_t>(parseInfo.lineLen);
    }

    return 0;
}

int32_t TestlistGenerator::CheckForIgnoreEmptyLine(ParseInfo& parseInfo) NN_NOEXCEPT
{
    if(parseInfo.CheckForCommand(CommandBeginIgnoreEmptyLines, CommandBeginIgnoreEmptyLinesLen))
    {
        uint32_t startLine = parseInfo.lineNumber;

        bool saveIsIgnoreEmptyLinesEnabled = parseInfo.isIgnoreEmptyLinesEnabled;
        parseInfo.isIgnoreEmptyLinesEnabled = true;

        std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
        if(nullptr == pReadBuf)
        {
            NN_LOG("Allocation error!\n\n");
            return -1;
        }

        do
        {
            parseInfo.pLine = parseInfo.lineReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
            if( parseInfo.pLine )
            {
                parseInfo.lineLen = LineLen(parseInfo.pLine);
                ++parseInfo.lineNumber;

                if(parseInfo.lineLen >= CommandEndIgnoreEmptyLinesLen && memcmp(parseInfo.pLine, CommandEndIgnoreEmptyLines, CommandEndIgnoreEmptyLinesLen) == 0)
                {
                    PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandEndIgnoreEmptyLines);

                    parseInfo.isIgnoreEmptyLinesEnabled = saveIsIgnoreEmptyLinesEnabled;
                    return static_cast<int32_t>(parseInfo.lineLen);
                }

                parseInfo.isIgnoreBeginningWhiteSpaceEnabled = true;
                if(false == ParseAndWriteLine(parseInfo))
                {
                    return -1;
                }
            }
        } while(false == parseInfo.lineReader.IsEndOfStream());

        PRINT_LINE_INFO(parseInfo, "ERROR: Reached end of file before finding %s. Trying to close %s from line %d\n", CommandEndIgnoreEmptyLines, CommandBeginIgnoreEmptyLines, startLine);

        return -1;
    }

    return 0;
}

int32_t TestlistGenerator::CheckForInclude(ParseInfo& parseInfo) NN_NOEXCEPT
{
    if(parseInfo.CheckForCommand(CommandInclude, CommandIncludeLen))
    {
        if(parseInfo.lineLen <= 2)
        {
            PRINT_LINE_INFO(parseInfo, "ERROR: A file name must be specified after %s\n\n", CommandInclude);
            return -1;
        }

        std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
        if(nullptr == pReadBuf)
        {
            NN_LOG("Allocation error!\n\n");
            return -1;
        }

        char pIncludeFileName[MaxFileNameBufLen];
        memset(pIncludeFileName, 0 , MaxFileNameBufLen);

        const char* pFileName = parseInfo.pLine + CommandIncludeLen + 1; // +1 for space
        strncpy(pIncludeFileName, pFileName, MaxFileNameBufLen);

        // Remove newline char from the end.
        size_t fileNameLen = strlen(pIncludeFileName);
        if(pIncludeFileName[fileNameLen - 1] == '\n')
        {
            pIncludeFileName[fileNameLen - 1] = '\0';
        }

        FILE* pIncludeFile = fopen(pIncludeFileName, "rt");
        if(nullptr == pIncludeFile)
        {
            PRINT_LINE_INFO(parseInfo, "ERROR: Failed to open %s\n", pIncludeFileName);
            return -1;
        }

        FileLineReader includeFileReader(pIncludeFile);
        uint32_t lineNumber = 0;
        int32_t rval = static_cast<int32_t>(parseInfo.lineLen);

        do
        {
            char* pRet = includeFileReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
            if( pRet )
            {
                ++lineNumber;

                ParseInfo newParseInfo(includeFileReader, pRet, LineLen(pRet), parseInfo.pPrevIter, parseInfo.pCurIter, lineNumber, parseInfo.isIgnoreEmptyLinesEnabled, true, pIncludeFileName);
                if(false == ParseAndWriteLine(newParseInfo))
                {
                    rval = -1;
                    break;
                }
            }
        } while(false == includeFileReader.IsEndOfStream());

        fclose(pIncludeFile);
        return rval;
    }

    return 0;
}

int32_t TestlistGenerator::CheckForComment(ParseInfo& parseInfo) NN_NOEXCEPT
{
    if(parseInfo.CheckForCommand(CommandBeginComment, CommandBeginCommentLen))
    {
        uint32_t startLine = parseInfo.lineNumber;
        uint32_t nestedCommentCount = 1;

        // Check to see if comment ends on same line.
        const char* pFindCommand = strchr(&parseInfo.pLine[CommandBeginCommentLen], CommandChar);
        if(nullptr != pFindCommand)
        {
            size_t commandLen = strlen(pFindCommand);
            if(commandLen >= CommandEndCommentLen && memcmp(pFindCommand, CommandEndComment, CommandEndCommentLen) == 0)
            {
                PRINT_LINE_INFO(parseInfo, "Found: %s : %s\n", CommandEndComment, parseInfo.pLine);
                return static_cast<int32_t>(parseInfo.lineLen);
            }
        }

        std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
        if(nullptr == pReadBuf)
        {
            NN_LOG("Allocation error!\n\n");
            return -1;
        }

        // Keep reading until end command is found.
        //  Will only write if doWriteBlock is true
        while(false == parseInfo.lineReader.IsEndOfStream())
        {
            const char* pRet = parseInfo.lineReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
            if( pRet )
            {
                ++parseInfo.lineNumber;

                do
                {
                    pFindCommand = strchr(pRet, CommandChar);
                    if(nullptr != pFindCommand)
                    {
                        size_t restOfLineLen = LineLen(pFindCommand);

                        // Check for nested comment
                        if(restOfLineLen >= CommandBeginCommentLen && memcmp(pFindCommand, CommandBeginComment, CommandBeginCommentLen) == 0)
                        {
                            PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandBeginComment);
                            ++nestedCommentCount;
                        }
                        // Check for end command
                        else if(restOfLineLen >= CommandEndCommentLen && memcmp(pFindCommand, CommandEndComment, CommandEndCommentLen) == 0)
                        {
                            PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandEndComment);

                            --nestedCommentCount;
                            if(nestedCommentCount == 0)
                            {
                                return static_cast<int32_t>(parseInfo.lineLen);
                            }
                        }
                    }

                    pRet = pFindCommand + 1;
                } while(pFindCommand != nullptr); // Continue to look for commands on this line
            }
        }

        PRINT_LINE_INFO(parseInfo, "ERROR: Found end of file before finding %s. Trying to end %s at line %d\n", CommandEndComment, CommandBeginComment, startLine);
        return -1;
    }

    return 0;
}

int32_t TestlistGenerator::CheckForBuild(ParseInfo& parseInfo) NN_NOEXCEPT
{
    if(parseInfo.CheckForCommand(CommandBuild, CommandBuildLen))
    {
        size_t buildTypeLen = strlen(m_pBuildTypes[m_curBuildType]);

        size_t bytesWritten = fwrite(m_pBuildTypes[m_curBuildType], 1, buildTypeLen, m_pWriteFile);
        if( bytesWritten != buildTypeLen )
        {
            NN_LOG("File error while writing! err: %d\n\n", ferror(m_pWriteFile));
            return -1;
        }

        // If there is more to parse, then parse it.
        if(parseInfo.lineLen > CommandBuildLen)
        {
            ParseInfo newParseInfo(&parseInfo.pLine[CommandBuildLen], parseInfo.lineLen - CommandBuildLen, parseInfo);
            newParseInfo.isIgnoreBeginningWhiteSpaceEnabled = false;
            if(false == ParseAndWriteLine(newParseInfo))
            {
                return -1;
            }
        }
        else
        {
            ++parseInfo.lineNumber;
        }

        return static_cast<int32_t>(parseInfo.lineLen);
    }

    return 0;
}

int32_t TestlistGenerator::CheckForPlatform(ParseInfo& parseInfo) NN_NOEXCEPT
{
    if(parseInfo.CheckForCommand(CommandPlatform, CommandPlatformLen))
    {
        size_t platformLen = strlen(m_pPlatforms[m_curPlatform].pName);

        size_t bytesWritten = fwrite(m_pPlatforms[m_curPlatform].pName, 1, platformLen, m_pWriteFile);
        if( bytesWritten != platformLen )
        {
            NN_LOG("File error while writing! err: %d\n\n", ferror(m_pWriteFile));
            return -1;
        }

        // If there is more to parse, then parse it.
        if(parseInfo.lineLen > CommandPlatformLen)
        {
            if(parseInfo.pLine[CommandPlatformLen] == '\n')
            {
                bytesWritten = fwrite("\n", 1, 1, m_pWriteFile);
                if( bytesWritten != 1 )
                {
                    NN_LOG("File error while writing! err: %d\n\n", ferror(m_pWriteFile));
                    return false;
                }
            }
            else
            {
                ParseInfo newParseInfo(&parseInfo.pLine[CommandPlatformLen], parseInfo.lineLen - CommandPlatformLen, parseInfo);
                newParseInfo.isIgnoreBeginningWhiteSpaceEnabled = false;
                if(false == ParseAndWriteLine(newParseInfo))
                {
                    return -1;
                }
            }
        }

        return static_cast<int32_t>(parseInfo.lineLen);
    }

    return 0;
}

int32_t TestlistGenerator::CheckForConstants(ParseInfo& parseInfo) NN_NOEXCEPT
{
    // Check for platform constants command
    if(parseInfo.CheckForCommand(CommandPlatformConstants, CommandPlatformConstantsLen))
    {
        uint32_t platformLine = m_pPlatforms[m_curPlatform].startLine;
        size_t constantsLen = strlen(m_pPlatforms[m_curPlatform].pConstants);

        std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
        if(nullptr == pReadBuf)
        {
            NN_LOG("Allocation error!\n\n");
            return -1;
        }

        BufferLineReader constantsReader(m_pPlatforms[m_curPlatform].pConstants, static_cast<uint32_t>(constantsLen));
        do
        {
            const char* pLine = constantsReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
            if(pLine)
            {
                ++platformLine;
                ParseInfo newParseInfo(constantsReader, pLine, LineLen(pLine), nullptr, nullptr, platformLine, parseInfo.isIgnoreEmptyLinesEnabled, true, parseInfo.pFileName);
                if(false == ParseAndWriteLine(newParseInfo))
                {
                    return -1;
                }
            }
        } while(false == constantsReader.IsEndOfStream());

        // If there is more to parse, then parse it.
        if(parseInfo.lineLen > CommandPlatformConstantsLen)
        {
            if(parseInfo.pLine[CommandPlatformConstantsLen] == '\n')
            {
                size_t bytesWritten = fwrite("\n", 1, 1, m_pWriteFile);
                if( bytesWritten != 1 )
                {
                    NN_LOG("File error while writing! err: %d\n\n", ferror(m_pWriteFile));
                    return -1;
                }
            }
            else
            {
                ParseInfo newParseInfo(&parseInfo.pLine[CommandPlatformConstantsLen], parseInfo.lineLen - CommandPlatformConstantsLen, parseInfo);
                newParseInfo.isIgnoreBeginningWhiteSpaceEnabled = false;
                if(false == ParseAndWriteLine(newParseInfo))
                {
                    return -1;
                }
            }
        }

        return static_cast<int32_t>(parseInfo.lineLen);
    }

    return 0;
}

bool TestlistGenerator::CheckLoopComponentCommands(ParseInfo& parseInfo, LoopState& loopState) NN_NOEXCEPT
{
    loopState.doWriteLine = true;

    if(parseInfo.CheckForCommand(CommandParamLoop, CommandParamLoopLen))
    {
        PRINT_LINE_INFO(parseInfo, "Nested Loop!\n");
        ++loopState.nestedLoopCounter;
    }
    // Check for end loop command
    else if(parseInfo.CheckForCommand(CommandEndLoop, CommandEndLoopLen))
    {
        RETURN_FALSE_ON_FALSE(FoundEndLoopCommand(parseInfo, loopState));
    }
    else if(loopState.nestedLoopCounter == 0)
    {
        loopState.doWriteLine = false;

        //  Check for end parameter command
        if(parseInfo.CheckForCommand(CommandEndParam, CommandEndParamLen))
        {
            RETURN_FALSE_ON_FALSE(FoundEndParamCommand(parseInfo, loopState));
        }
        //  Check for begin parameter command
        else if(parseInfo.CheckForCommand(CommandNextParam, CommandNextParamLen))
        {
            RETURN_FALSE_ON_FALSE(FoundBeginParamCommand(parseInfo, loopState));
        }
        // Check for end iteration command. CommandCloseIteration
        else if(parseInfo.CheckForCommand(CommandCloseIteration, CommandCloseIterationLen))
        {
            RETURN_FALSE_ON_FALSE(FoundEndIterationCommand(parseInfo, loopState));
        }
        //  Check for begin iteration command
        else if(parseInfo.CheckForCommand(CommandNextIteration, CommandNextIterationLen))
        {
            RETURN_FALSE_ON_FALSE(FoundBeginIterationCommand(parseInfo, loopState));
        }
        // Check for loop body command
        else if(parseInfo.CheckForCommand(CommandLoopBody, CommandLoopBodyLen))
        {
            RETURN_FALSE_ON_FALSE(FoundLoopBodyCommand(parseInfo, loopState));
        }
        else
        {
            loopState.doWriteLine = true;
        }
    }

    return true;
}

bool TestlistGenerator::FoundEndLoopCommand(ParseInfo& parseInfo, LoopState& loopState) NN_NOEXCEPT
{
    if(loopState.nestedLoopCounter > 0)
    {
        PRINT_LINE_INFO(parseInfo, "Unnesting Loop!\n");
        --loopState.nestedLoopCounter;
    }
    else
    {
        PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandEndLoop);

        if(nullptr == loopState.pLoopBody)
        {
            PRINT_LINE_INFO(parseInfo, "ERROR: Found %s before %s\n", CommandEndLoop, CommandLoopBody);
            return false;
        }

        loopState.doWriteLine   = false;
        loopState.isAtEndOfLoop = true;
    }

    return true;
}

bool TestlistGenerator::FoundEndParamCommand(ParseInfo& parseInfo, LoopState& loopState) NN_NOEXCEPT
{
    PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandEndParam);

    if(loopState.eLoopState != LoopReadState::ReadingParam)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s\n\n", CommandEndParamLen);
        return false;
    }

    loopState.eLoopState = LoopReadState::ReadingNothing;

    return true;
}

bool TestlistGenerator::FoundBeginParamCommand(ParseInfo& parseInfo, LoopState& loopState) NN_NOEXCEPT
{
    PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandNextParam);

    if(false == loopState.isInIteration)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s. Must be inside of a %s block.\n\n", CommandNextParam, CommandNextIteration);
        return false;
    }

    if(loopState.eLoopState == LoopReadState::ReadingParam)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s. The current param must first be closed with %s\n\n", CommandNextParam, CommandEndParam);
        return false;
    }
    else if(loopState.eLoopState != LoopReadState::ReadingNothing)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s\n\n", CommandNextParam);
        return false;
    }

    LoopParam* pNewParam = new LoopParam;
    if(nullptr == pNewParam)
    {
        NN_LOG("ERROR: Failed to allocate LoopParam data.\n");
        return false;
    }

    if(nullptr == loopState.pRootParam)
    {
        loopState.pRootParam = loopState.pCurIter = pNewParam;
    }
    else if(loopState.isNewIteration)
    {
        loopState.pCurIter->pNextIter = pNewParam;
        loopState.pCurIter = pNewParam;
    }
    else
    {
        loopState.pCurParam->pNextParam = pNewParam;
    }

    loopState.eLoopState = LoopReadState::ReadingParam;
    loopState.isNewIteration = false;

    loopState.pCurParam = pNewParam;
    loopState.pCurParam->startLine = parseInfo.lineNumber;

    return true;
}

bool TestlistGenerator::FoundEndIterationCommand(ParseInfo& parseInfo, LoopState& loopState) NN_NOEXCEPT
{
    PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandCloseIteration);

    if(false == loopState.isInIteration)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s. Iteration block must first be opened with %s\n\n", CommandCloseIteration, CommandNextIteration);
        return false;
    }

    if(loopState.eLoopState != LoopReadState::ReadingNothing)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s.\n\n", CommandNextIteration);
        return false;
    }

    // If this is still a new iteration(meaning no params were specified),
    //   create an empty param just for the iteration.
    if(loopState.isNewIteration)
    {
        LoopParam* pNewParam = new LoopParam;
        if(nullptr == pNewParam)
        {
            NN_LOG("ERROR: Failed to allocate LoopParam data.\n");
            return false;
        }

        if(nullptr == loopState.pRootParam)
        {
            loopState.pRootParam = loopState.pCurIter = loopState.pCurParam = pNewParam;
        }
        else
        {
            loopState.pCurIter->pNextIter = pNewParam;
            loopState.pCurIter = loopState.pCurParam = pNewParam;
        }

        loopState.isNewIteration = false;
    }

    loopState.isInIteration = false;

    return true;
}

bool TestlistGenerator::FoundBeginIterationCommand(ParseInfo& parseInfo, LoopState& loopState) NN_NOEXCEPT
{
    PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandNextIteration);

    if(true == loopState.isInIteration)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s. Iteration block must first be closed with %s\n\n", CommandNextIteration, CommandCloseIteration);
        return false;
    }

    if(loopState.eLoopState == LoopReadState::ReadingParam)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s. The current param must first be closed with %s\n\n", CommandNextIteration, CommandEndParam);
        return false;
    }
    else if(loopState.eLoopState != LoopReadState::ReadingNothing)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s.\n\n", CommandNextIteration);
        return false;
    }

    loopState.isNewIteration = true;
    loopState.isInIteration = true;
    ++loopState.iterationCount;

    return true;
}

bool TestlistGenerator::FoundLoopBodyCommand(ParseInfo& parseInfo, LoopState& loopState) NN_NOEXCEPT
{
    PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandLoopBody);

    if(nullptr != loopState.pLoopBody)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Found two %s before finding %s\n", CommandLoopBody, CommandEndLoop);
        return false;
    }

    if(loopState.iterationCount <= 1)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Found %s before finding %s\n", CommandLoopBody, CommandNextIteration);
        return false;
    }

    if(true == loopState.isInIteration)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s. The iteration block must first be closed with %s\n", CommandLoopBody, CommandCloseIteration);
        return false;
    }

    if(loopState.eLoopState == LoopReadState::ReadingParam)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s. %s needs to be closed first with %s\n\n", CommandLoopBody, CommandNextParam, CommandEndParam);
        return false;
    }
    else if(loopState.eLoopState != LoopReadState::ReadingNothing)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Unexpected command %s.\n\n", CommandLoopBody);
        return false;
    }

    loopState.pLoopBody.reset(new char[MaxLoopBodyBufLen]);
    if(nullptr == loopState.pLoopBody)
    {
        NN_LOG("ERROR: Failed to allocate loop body data.\n");
        return false;
    }

    loopState.eLoopState = LoopReadState::ReadingBody;
    loopState.bodyLine = parseInfo.lineNumber;

    return true;
}

bool TestlistGenerator::ReadLoopComponentsIntoBuffers(ParseInfo& parseInfo, LoopState& loopState) NN_NOEXCEPT
{
    uint32_t startLine = parseInfo.lineNumber;

    std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
    if(nullptr == pReadBuf)
    {
        NN_LOG("Allocation error!\n\n");
        goto cleanup;
    }

    // Read loop parameters and loop body into buffers
    while(false == parseInfo.lineReader.IsEndOfStream())
    {
        char* pRet = parseInfo.lineReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
        loopState.doWriteLine = true;
        if( pRet )
        {
            ++parseInfo.lineNumber;
            size_t lineLen = strlen(pRet);

            char* pFindCommand = strchr(pRet, CommandChar);
            if(nullptr != pFindCommand)
            {
                ParseInfo newParseInfo(pFindCommand, LineLen(pFindCommand), parseInfo);
                if( false == CheckLoopComponentCommands(newParseInfo, loopState) )
                {
                    goto cleanup;
                }

                if(true == loopState.isAtEndOfLoop)
                {
                    break;
                }
            }

            if(loopState.doWriteLine)
            {
                // Are are writing param data?
                if(loopState.eLoopState == LoopReadState::ReadingParam && nullptr != loopState.pCurParam)
                {
                    if((lineLen + loopState.pCurParam->dataLen) > (MaxLoopParamBufLen - 1))
                    {
                        NN_LOG(" File %s - ERROR: Loop param size is too large. Max param size: %d. Param started at line %d\n", parseInfo.pFileName, (MaxLoopParamBufLen - 1), loopState.pCurParam->startLine);
                        goto cleanup;
                    }

                    memcpy(&loopState.pCurParam->pData[loopState.pCurParam->dataLen], pRet, lineLen);
                    loopState.pCurParam->dataLen += static_cast<uint32_t>(lineLen);
                    loopState.pCurParam->pData[loopState.pCurParam->dataLen] = '\0'; // Null terminate
                }
                // Are we writing Loop body data?
                else if(loopState.eLoopState == LoopReadState::ReadingBody && nullptr != loopState.pLoopBody)
                {
                    if(loopState.loopBodyLen + lineLen > (MaxLoopBodyBufLen - 1))
                    {
                        NN_LOG(" File: %s - ERROR: Loop body size is too large. Max body size: %d. Body started at line %d\n", parseInfo.pFileName, (MaxLoopBodyBufLen - 1), loopState.bodyLine);
                        goto cleanup;
                    }

                    memcpy(&loopState.pLoopBody[loopState.loopBodyLen], pRet, lineLen);
                    loopState.loopBodyLen += static_cast<uint32_t>(lineLen);
                    loopState.pLoopBody[loopState.loopBodyLen] = '\0'; // Null terminate
                }
                else if(loopState.eLoopState == LoopReadState::ReadingNothing && ContainsNonWhiteSpace(pRet, lineLen))
                {
                    PRINT_LINE_INFO(parseInfo, "ERROR: Expecting loop command.\n");
                    goto cleanup;
                }
            }
        }
    }

    if(false == loopState.isAtEndOfLoop)
    {
        PRINT_LINE_INFO(parseInfo, "ERROR: Found end of file before finding %s. Loop started at line %d\n", CommandEndLoop, startLine);
        goto cleanup;
    }

    if(loopState.iterationCount <= 1)
    {
        PRINT_LINE_INFO(parseInfo, "Error after %s. Could not find %s. Loop started at line %d\n", CommandEndLoop, CommandNextIteration, startLine);
        goto cleanup;
    }

    return true;

cleanup:

    // Clean up loop iterations
    if(nullptr != loopState.pRootParam)
    {
        delete loopState.pRootParam; // Recursively deletes the whole multi-list scructure.
        loopState.pRootParam = nullptr;
    }

    return false;
}

int32_t TestlistGenerator::CheckForParamLoop(ParseInfo& parseInfo) NN_NOEXCEPT
{
    int32_t bytesRead = 0;
    LoopState loopState;

    // Check for loop command
    if(parseInfo.CheckForCommand(CommandParamLoop, CommandParamLoopLen))
    {
        bool isSuccess = ReadLoopComponentsIntoBuffers(parseInfo, loopState);
        if(false == isSuccess)
        {
            bytesRead = -1;
            goto out;
        }

        bytesRead = static_cast<int32_t>(parseInfo.lineLen);

        // Loop through each loop iteration
        for(LoopParam* pIter = loopState.pRootParam; pIter != nullptr; pIter = pIter->pNextIter)
        {
            // Loop through each param
            for(LoopParam* pParam = pIter; pParam != nullptr; pParam = pParam->pNextParam)
            {
                // Strip new line char off the end of each param.
                if(pParam->dataLen > 0 && pParam->pData[pParam->dataLen - 1] == '\n')
                {
                    --pParam->dataLen;
                    pParam->pData[pParam->dataLen] = '\0';
                }
            }
        }

        std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
        if(nullptr == pReadBuf)
        {
            NN_LOG("Allocation error!\n\n");
            bytesRead = -1;
            goto out;
        }

        // Write loop iterations to file and insert loop parameters when necessary
        LoopParam* pCurIter = loopState.pRootParam;
        while(nullptr != pCurIter)
        {
            BufferLineReader bufferReader(loopState.pLoopBody.get(), loopState.loopBodyLen);
            uint32_t loopIterationLine = loopState.bodyLine;

            do
            {
                char* pLine = bufferReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
                if(nullptr == pLine)
                {
                    continue;
                }

                ++loopIterationLine;

                ParseInfo newParseInfo(bufferReader, pLine, LineLen(pLine), parseInfo.pCurIter, pCurIter, loopIterationLine, parseInfo.isIgnoreEmptyLinesEnabled, true, parseInfo.pFileName);
                if( false == ParseAndWriteLine(newParseInfo) )
                {
                    bytesRead = -1;
                    goto out;
                }
            } while(false == bufferReader.IsEndOfStream());

            pCurIter = pCurIter->pNextIter;
        }
    }

out:

    // Clean up loop iterations
    if(nullptr != loopState.pRootParam)
    {
        delete loopState.pRootParam;
        loopState.pRootParam = nullptr;
    }

    return bytesRead;
}

int32_t TestlistGenerator::CheckForLoopParam(ParseInfo& parseInfo) NN_NOEXCEPT
{
    // Check for loop parameter
    if(parseInfo.CheckForCommand(CommandLoopParam, CommandLoopParamLen))
    {
        const char* pLine = parseInfo.pLine + CommandLoopParamLen + 1; // +1 for space
        const char* pParamNum = pLine;
        ++pLine; // Move past digit

        if(*pParamNum <= '0' || *pParamNum > '9')
        {
            PRINT_LINE_INFO(parseInfo, "Error after %s. A single digit number between 1 and 9 must be specified that corresponds to the parameter number.\n\n", CommandLoopParam);
            return -1;
        }

        // Convert char number to int number
        uint8_t paramNum = *pParamNum - '0';
        PRINT_LINE_INFO(parseInfo, "Found: %d\n", paramNum);

        uint8_t curParamNum = 0;
        for(LoopParam* pCurParam = parseInfo.pCurIter; nullptr != pCurParam; pCurParam = pCurParam->pNextParam)
        {
            ++curParamNum;
            uint32_t paramLine = parseInfo.pCurIter->startLine;

            if(curParamNum == paramNum)
            {
                std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
                if(nullptr == pReadBuf)
                {
                    NN_LOG("Allocation error!\n\n");
                    return -1;
                }

                BufferLineReader paramReader(pCurParam->pData, pCurParam->dataLen);
                do
                {
                    char* pParamLine = paramReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
                    if(nullptr == pParamLine)
                    {
                        continue;
                    }

                    ++paramLine;

                    ParseInfo newParseInfo(paramReader, pParamLine, LineLen(pParamLine), nullptr, parseInfo.pPrevIter, paramLine, parseInfo.isIgnoreEmptyLinesEnabled, true, parseInfo.pFileName);
                    if(false == ParseAndWriteLine(newParseInfo))
                    {
                        return -1;
                    }
                } while(false == paramReader.IsEndOfStream());

                break;
            }
        }

        if(parseInfo.lineLen > CommandLoopParamLen + 2) // +2 for space and digit
        {
            // pLine was already moved forward past the parsed chars.
            ParseInfo newParseInfo(pLine, LineLen(pLine), parseInfo);
            newParseInfo.isIgnoreBeginningWhiteSpaceEnabled = false;
            if(false == ParseAndWriteLine(newParseInfo))
            {
                return -1;
            }
        }

        return static_cast<int32_t>(parseInfo.lineLen);
    }

    return 0;
}

int32_t TestlistGenerator::CheckOnlyPlatforms(ParseInfo& parseInfo) NN_NOEXCEPT
{
    int32_t bytesRead  = 0;
    uint32_t startLine = parseInfo.lineNumber;

    // Check for only platforms command
    if(parseInfo.CheckForCommand(CommandBeginOnlyPlatforms, CommandBeginOnlyPlatformsLen))
    {
        bool isLastChar = false;
        bool doWriteBlock = false;
        const char* pPlatform = (parseInfo.pLine + CommandBeginOnlyPlatformsLen + 1);

        bytesRead = static_cast<int32_t>(parseInfo.lineLen);

        do // Parse platforms
        {
            const char* pEndToken = pPlatform;

            do
            {
                // Found end of token and line
                if(*pEndToken == '\n' || *pEndToken == '\r' || *pEndToken == '\0')
                {
                    isLastChar = true;
                    break;
                }
                else if(*pEndToken == ' ') // found end of token
                {
                    break;
                }

                ++pEndToken;
            } while(NN_STATIC_CONDITION(true));

            size_t checkLen    = pEndToken - pPlatform;
            size_t platformLen = strlen(m_pPlatforms[m_curPlatform].pName);
            PRINT_LINE_INFO(parseInfo, "Found: %.*s\n", checkLen, pPlatform);

            // Check if we are generating the given platform
            if(checkLen >= platformLen && memcmp(pPlatform, m_pPlatforms[m_curPlatform].pName, platformLen) == 0)
            {
                doWriteBlock = true;
                break;
            }

            // Skip a space between platform names
            pPlatform = pEndToken + 1;
        } while(isLastChar == false);

        std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
        if(nullptr == pReadBuf)
        {
            NN_LOG("Allocation error!\n\n");
            return -1;
        }

        bool isWriteNewLine = false;
        int nestedCounter = 1;

        // Keep reading until end command is found.
        //  Will only write if doWriteBlock is true
        while(false == parseInfo.lineReader.IsEndOfStream())
        {
            char* pRet = parseInfo.lineReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
            if( pRet )
            {
                ++parseInfo.lineNumber;
                size_t lineLen = strlen(pRet);

                char* pFindCommand = strchr(pRet, CommandChar);
                if(nullptr != pFindCommand)
                {
                    size_t restOfLineLen = LineLen(pFindCommand);

                    // Check for nested command
                    if(false == doWriteBlock && restOfLineLen >= CommandBeginOnlyPlatformsLen && memcmp(pFindCommand, CommandBeginOnlyPlatforms, CommandBeginOnlyPlatformsLen) == 0)
                    {
                        ++nestedCounter;
                    }
                    // Check for end command
                    else if(restOfLineLen >= CommandEndOnlyPlatformsLen && memcmp(pFindCommand, CommandEndOnlyPlatforms, CommandEndOnlyPlatformsLen) == 0)
                    {
                        --nestedCounter;

                        if(nestedCounter == 0)
                        {
                            PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandEndOnlyPlatforms);

                            return bytesRead;
                        }
                    }
                }

                // Don't pre-write a new line char the first time.
                if(isWriteNewLine && doWriteBlock)
                {
                    size_t bytesWritten = fwrite("\n", 1, 1, m_pWriteFile);
                    if( bytesWritten != 1 )
                    {
                        NN_LOG("File error while writing! err: %d\n\n", ferror(m_pWriteFile));
                        return -1;
                    }
                }

                isWriteNewLine = true;

                // Don't write the last new line char.
                //  We are pre-writing them so we can leave off the last new line char.
                size_t bytesToWrite = lineLen;
                if(lineLen > 0 && pRet[lineLen - 1] == '\n')
                {
                    --bytesToWrite;
                    pRet[bytesToWrite] = '\0';
                }

                if(bytesToWrite > 0 && doWriteBlock)
                {
                    ParseInfo newParseInfo(pRet, bytesToWrite, parseInfo);
                    newParseInfo.isIgnoreBeginningWhiteSpaceEnabled = true;
                    if(false == ParseAndWriteLine(newParseInfo))
                    {
                        return -1;
                    }
                }
            }
        }

        PRINT_LINE_INFO(parseInfo, "ERROR: Found end of file before finding %s. Trying to end %s at line %d\n", CommandEndOnlyPlatforms, CommandBeginOnlyPlatforms, startLine);
        return -1;
    }

    return bytesRead;
}

bool TestlistGenerator::FindOrCreateDefinition(ParseInfo& parseInfo, const char* pDefineName, size_t newDefineNameLen, DefineParam*& pFoundDefine) NN_NOEXCEPT
{
    pFoundDefine = nullptr;

    // Check for empty list
    if(nullptr == m_pRootDefine)
    {
        m_pRootDefine = pFoundDefine = new DefineParam;
        if(nullptr == pFoundDefine)
        {
            NN_LOG("ERROR: Allocation failed for %s\n\n", CommandBeginDefine);
            return false;
        }
    }
    else
    {
        DefineParam* pLastParam = nullptr;
        DefineParam* pIterParam = m_pRootDefine;
        while(nullptr != pIterParam)
        {
            size_t nameLen = strlen(&pIterParam->pName[1]) - 1; // [1] to ignore '<' and -1 to ignore '>'

            // Compare platform names
            if(nameLen == newDefineNameLen && memcmp(&pIterParam->pName[1], pDefineName, nameLen) == 0)
            {
                PRINT_LINE_INFO(parseInfo, "Overwriting %s from line %d\n", pIterParam->pName, pIterParam->startLine);

                pFoundDefine = pIterParam;
                pFoundDefine->dataLen = 0;
                pFoundDefine->startLine = parseInfo.lineNumber;
                strncpy(pFoundDefine->pFileName, parseInfo.pFileName, MaxFileNameBufLen);
                break;
            }

            pLastParam = pIterParam;
            pIterParam = pIterParam->pNextParam;
        }

        if(nullptr == pFoundDefine)
        {
            pLastParam->pNextParam = pFoundDefine = new DefineParam;
            if(nullptr == pFoundDefine)
            {
                NN_LOG("ERROR: Allocation failed for %s\n\n", CommandBeginDefine);
                return false;
            }
        }
    }

    return true;
}

bool TestlistGenerator::ReadDefinitionIntoBuffer(ParseInfo& parseInfo, DefineParam* pDefine, uint32_t startLine) NN_NOEXCEPT
{
    std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
    if(nullptr == pReadBuf)
    {
        NN_LOG("Allocation error!\n\n");
        return false;
    }

    int nestedDefineCounter = 1;

    // Read block definition into buffer
    while(false == parseInfo.lineReader.IsEndOfStream())
    {
        char* pRet = parseInfo.lineReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
        if(pRet)
        {
            ++parseInfo.lineNumber;
            size_t lineLen = LineLen(pRet);
            const char* pCommand = strchr(pRet, CommandChar);

            if(nullptr != pCommand)
            {
                size_t restOfLineLen = LineLen(pCommand);

                // Check for nested defines
                if(restOfLineLen >= CommandBeginDefineLen && memcmp(pCommand, CommandBeginDefine, CommandBeginDefineLen) == 0)
                {
                    PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandBeginDefine);

                    // Only increment nested counter if its a multi line define.
                    const char* pIsSingleLineDefine = strchr(pCommand, '=');
                    if(nullptr == pIsSingleLineDefine)
                    {
                        ++nestedDefineCounter;
                    }
                }
                // Check for end command
                else if(restOfLineLen >= CommandEndDefineLen && memcmp(pCommand, CommandEndDefine, CommandEndDefineLen) == 0)
                {
                    PRINT_LINE_INFO(parseInfo, "Found: %s\n", CommandEndDefine);

                    --nestedDefineCounter;
                    if(nestedDefineCounter == 0)
                    {
                        // Strip last new line char
                        if(pDefine->pData[pDefine->dataLen - 1] == '\n')
                        {
                            --pDefine->dataLen;
                            pDefine->pData[pDefine->dataLen] = '\0';
                        }

                        return true;
                    }
                }
            }

            if((pDefine->dataLen + lineLen) > MaxDefineBufLen - 1) // -1 for null terminator
            {
                NN_LOG("Error: Data for %s is too large. %s started at line %d\n", CommandBeginDefine, CommandBeginDefine, startLine);
                return false;
            }

            memcpy(&pDefine->pData[pDefine->dataLen], pRet, lineLen);
            pDefine->dataLen += static_cast<uint32_t>(lineLen);
            pDefine->pData[pDefine->dataLen] = '\0';
        }
    }

    NN_LOG("Error: Reached end of file before finding %s. Trying to close block from line %d\n", CommandEndDefine, startLine);
    return false;
}

int32_t TestlistGenerator::CheckForDefinitions(ParseInfo& parseInfo) NN_NOEXCEPT
{
    if(parseInfo.CheckForCommand(CommandBeginDefine, CommandBeginDefineLen))
    {
        uint32_t startLine = parseInfo.lineNumber;
        const char* pDefineName = parseInfo.pLine + CommandBeginDefineLen + 1; // +1 for space
        size_t restOfLine = LineLen(pDefineName);
        size_t newDefineNameLen = 0;
        DefineParam* pCurParam = nullptr;

        if(restOfLine < 1 || pDefineName[0] == '\n')
        {
            PRINT_LINE_INFO(parseInfo, "Error: Cannot find name after %s\n", CommandBeginDefine);
            return -1;
        }

        size_t newLineToIgnore = 0;
        if(pDefineName[restOfLine - 1] == '\n')
        {
            newLineToIgnore = 1;
        }

        // Check for single line definition => <DEFINE> NEW_VAR=value
        const char* pFound = strchr(pDefineName, '=');
        bool isSingleLine = (nullptr != pFound);

        // -newLineToIgnore from restOfLine to ignore possible end line char.
        newDefineNameLen = isSingleLine ? (pFound - pDefineName) : (restOfLine - newLineToIgnore);

        // -3 from MaxDefineNameBufLen to leave room for null terminator, '<', and '>'.
        if(newDefineNameLen > MaxDefineNameBufLen - 3)
        {
            PRINT_LINE_INFO(parseInfo, "Error: Name after %s is too long! Max name length is %d\n", CommandBeginDefine, MaxDefineNameBufLen - 3);
            return -1;
        }

        PRINT_LINE_INFO(parseInfo, "Found: %.*s\n", newDefineNameLen, pDefineName);

        if( false == FindOrCreateDefinition(parseInfo, pDefineName, newDefineNameLen, pCurParam) || nullptr == pCurParam)
        {
            return -1;
        }

        // Wrap the define name in <>.
        pCurParam->pName[0] = CommandChar;
        memcpy(&pCurParam->pName[1], pDefineName, newDefineNameLen);
        pCurParam->pName[newDefineNameLen + 1] = CommandCharEnd;
        pCurParam->pName[newDefineNameLen + 2] = '\0';
        pCurParam->startLine = startLine;
        strncpy(pCurParam->pFileName, parseInfo.pFileName, MaxFileNameBufLen);

        if(isSingleLine)
        {
            const char* pDefineValue = pFound + 1;
            size_t valueLen = LineLen(pDefineValue);
            if(valueLen == 0)
            {
                PRINT_LINE_INFO(parseInfo, "A value must be specified for %.*s\n", newDefineNameLen, pDefineName);
                return -1;
            }

            if(pDefineValue[valueLen - 1] == '\n')
            {
                --valueLen;
            }

            memcpy(pCurParam->pData, pDefineValue, valueLen);
            pCurParam->dataLen = static_cast<uint32_t>(valueLen);
            pCurParam->pData[pCurParam->dataLen] = '\0';

            return static_cast<int32_t>(parseInfo.lineLen);
        }

        if(ReadDefinitionIntoBuffer(parseInfo, pCurParam, startLine))
        {
            return static_cast<int32_t>(parseInfo.lineLen);
        }

        return -1;
    }

    return 0;
}

int32_t TestlistGenerator::CheckForDefineReference(ParseInfo& parseInfo) NN_NOEXCEPT
{
    for(DefineParam* pCurDefine = m_pRootDefine; nullptr != pCurDefine; pCurDefine = pCurDefine->pNextParam)
    {
        size_t commandLen = strlen(pCurDefine->pName);
        if(parseInfo.CheckForCommand(pCurDefine->pName, commandLen))
        {
            uint32_t defineLine = pCurDefine->startLine;

            std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
            if(nullptr == pReadBuf)
            {
                NN_LOG("Allocation error!\n\n");
                return -1;
            }

            BufferLineReader defineReader(pCurDefine->pData, pCurDefine->dataLen);
            do
            {
                const char* pLine = defineReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
                if(pLine)
                {
                    ++defineLine;
                    ParseInfo newParseInfo(defineReader, pLine, LineLen(pLine), parseInfo.pPrevIter, parseInfo.pCurIter, defineLine, parseInfo.isIgnoreEmptyLinesEnabled, true, pCurDefine->pFileName);
                    if(false == ParseAndWriteLine(newParseInfo))
                    {
                        return -1;
                    }
                }
            } while(false == defineReader.IsEndOfStream());

            // If there is more to parse, then parse it.
            if(parseInfo.lineLen > commandLen)
            {
                if(parseInfo.pLine[commandLen] == '\n')
                {
                    size_t bytesWritten = fwrite("\n", 1, 1, m_pWriteFile);
                    if( bytesWritten != 1 )
                    {
                        NN_LOG("File error while writing! err: %d\n\n", ferror(m_pWriteFile));
                        return false;
                    }
                }
                else
                {
                    ParseInfo newParseInfo(&parseInfo.pLine[commandLen], parseInfo.lineLen - commandLen, parseInfo);
                    newParseInfo.isIgnoreBeginningWhiteSpaceEnabled = false;
                    if(false == ParseAndWriteLine(newParseInfo))
                    {
                        return -1;
                    }
                }
            }

            return static_cast<int32_t>(parseInfo.lineLen);
        }
    }

    return 0;
}

int32_t TestlistGenerator::CheckForUnexpectedCommand(ParseInfo& parseInfo) NN_NOEXCEPT
{
    if(parseInfo.CheckForCommand(CommandEndDefinePlatform, CommandEndDefinePlatformLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandEndDefinePlatform);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandEndDefine, CommandEndDefineLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandEndDefine);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandEndComment, CommandEndCommentLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandEndComment);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandEndIgnoreEmptyLines, CommandEndIgnoreEmptyLinesLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandEndIgnoreEmptyLines);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandEndOnlyPlatforms, CommandEndOnlyPlatformsLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandEndOnlyPlatforms);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandLoopParam, CommandLoopParamLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandLoopParam);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandNextParam, CommandNextParamLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandNextParam);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandEndParam, CommandEndParamLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandEndParam);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandNextIteration, CommandNextIterationLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandNextIteration);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandCloseIteration, CommandCloseIterationLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandCloseIteration);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandLoopBody, CommandLoopBodyLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandLoopBody);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandEndLoop, CommandEndLoopLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandEndLoop);
        return -1;
    }
    else if(parseInfo.CheckForCommand(CommandLoopParam, CommandLoopParamLen))
    {
        PRINT_LINE_INFO(parseInfo, "Error unexpected command: %s\n\n", CommandLoopParam);
        return -1;
    }

    return 0;
}

