﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include "NetTest_Port.h"
#include "Api/testNet_ApiCommon.h" // ERROR_IF

#include "Tests/ResolverGetNameInfoTest.h"
#include "Utils/CommandLineParser.h"

#include <nnt/nntest.h>

#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <cstring> // strcpy

namespace {

    // ParseOptions
    bool ParseOptions(char* testName, nn::util::Uuid* pNetProfile) NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::Result rval;

        NATF::Utils::ParserGroup parser;
        nn::util::Uuid netProfileLocal;
        char testNameLocal[NATF::BaseTest::NameBufferLen];

        int argC;
        const char * const * argV;

        ERROR_IF(nullptr == testName, "TestSetup: testName nullptr");
        ERROR_IF(nullptr == pNetProfile, "TestSetup: pNetProfile nullptr");

        NETTEST_GET_ARGS(argC, argV);

        parser.AddParser(NATF::Utils::StringParser  ("--Name", nullptr, testNameLocal, sizeof(testNameLocal)));
        parser.AddParser(NATF::Utils::UuidParser    ("--NetProfile", &nn::util::InvalidUuid, netProfileLocal));

        ERROR_IF( !parser.Parse(argC, argV), "Failed to parse command line arguments");

        strcpy(testName, testNameLocal);
        *pNetProfile = netProfileLocal;

out:
        return isSuccess;
    }

    // ResolverGetNameInfo_CheckNoError
    TEST(natf, ResolverGetNameInfo_CheckNoError)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckNoError: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckNoError();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiAgain
    TEST(natf, ResolverGetNameInfo_CheckEaiAgain)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetAddrInfo_CheckEaiSystem: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiAgain();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiBadflags
    TEST(natf, ResolverGetNameInfo_CheckEaiBadflags)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckEaiBadflags: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiBadflags();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiFail
    TEST(natf, ResolverGetNameInfo_CheckEaiFail)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckEaiFail: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiFail();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiFamilyWithUnrecognizedAddressFamily
    TEST(natf, ResolverGetNameInfo_CheckEaiFamilyWithUnrecognizedAddressFamily)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckEaiFamilyWithUnrecognizedAddressFamily: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiFamilyWithUnrecognizedAddressFamily();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiFamilyWithInvalidAddressLength
    TEST(natf, ResolverGetNameInfo_CheckEaiFamilyWithInvalidAddressLength)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckEaiFamilyWithInvalidAddressLength: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiFamilyWithInvalidAddressLength();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiMemory
    TEST(natf, ResolverGetNameInfo_CheckEaiMemory)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckEaiMemory: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiMemory();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiNonameWithNoArgs
    TEST(natf, ResolverGetNameInfo_CheckEaiNonameWithNoArgs)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckEaiNonameWithNoArgs: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiNonameWithNoArgs();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiNonameWithNiNamereqd
    TEST(natf, ResolverGetNameInfo_CheckEaiNonameWithNiNamereqd)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckEaiNonameWithNiNamereqd: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiNonameWithNiNamereqd();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiOverflow
    TEST(natf, ResolverGetNameInfo_CheckEaiOverflow)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckEaiOverflow: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiOverflow();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

    // ResolverGetNameInfo_CheckEaiSystem
    TEST(natf, ResolverGetNameInfo_CheckEaiSystem)
    {
        bool isSuccess;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];
        NATF::Tests::ResolverGetNameInfoTest *pResolverGetNameInfoTest = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "ResolverGetNameInfo_CheckEaiSystem: ParseOptions failed");

        NN_NETTEST_LOG("\nEnter NATF Process\n\n");

        pResolverGetNameInfoTest = new NATF::Tests::ResolverGetNameInfoTest(testName, netProfile);
        ERROR_IF(nullptr == pResolverGetNameInfoTest, "NATF::Tests::ResolverGetNameInfoTest constructor failed");

        pResolverGetNameInfoTest->CheckEaiSystem();
        isSuccess = pResolverGetNameInfoTest->Run();

out:
        if( nullptr != pResolverGetNameInfoTest )
        {
            delete pResolverGetNameInfoTest;
            pResolverGetNameInfoTest = nullptr;
        }
        EXPECT_EQ(isSuccess, true);

        NN_NETTEST_LOG("\nExit NATF Process\n\n");
    }

} // unnamed
